// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

// this: #include <Common/Base/Thread/Concurrency/hkConcurrency.h>
#include <Common/Base/Thread/TaskQueue/hkTask.h>
#include <Common/Base/Thread/TaskQueue/hkTaskQueue.h>
#include <Common/Base/Thread/Atomic/hkIntegerDistributor.h>

/// Havok concurrency class.
class HK_EXPORT_COMMON hkConcurrency : public hkReferencedObject
{
public:
    HK_DECLARE_CLASS( hkConcurrency, New, Singleton );

    /// AsyncHandle.
    typedef hkTaskQueue::Handle AsyncHandle;

    /// Iterator for parallelForEx.
    typedef hkAtomicIntIterator Iterator;

    /// Constructor the m_taskQueue to null, so if setTaskQueue is not called, all processing will be single-threaded.
    hkConcurrency();

    /// Set the task queue to use.
    void setTaskQueue(_In_opt_ hkTaskQueue* taskQueue );

    /// Get the currently set task queue.
    HK_INLINE _Ret_maybenull_ hkTaskQueue* getTaskQueue() const;

    /// Get the currently set task queue or HK_NULL if single-threaded.
    HK_INLINE hkTaskQueue* getTaskQueueIfMultithreaded() const;

    /// Set single-threaded (global).
    void setSingleThreaded( bool singleThreaded );

    /// Get single-threaded (global).
    bool isSingleThreaded() const;

    // -------------------------------------------------------------------------------------------------------------
    // Parallel loops
    // -------------------------------------------------------------------------------------------------------------

    /// Call function for each indexed item.
    /// FUNCTION must implement operator()(int index).
    template <typename FUNCTION>
    HK_ALWAYS_INLINE static void parallelFor( int numItems, const FUNCTION& function ) { parallelFor( numItems, false, function ); }

    /// Call function for each indexed item.
    /// FUNCTION must implement operator()(int index).
    /// Execute on the calling thread if singleThreaded is true.
    template <typename FUNCTION>
    HK_INLINE static void parallelFor( int numItems, bool singleThreaded, const FUNCTION& function );

    /// Call function that consumes \p numItems items.
    /// FUNCTION must implement operator()( hkConcurrency::Iterator& items ).
    template <typename FUNCTION>
    HK_ALWAYS_INLINE static void parallelForEx( int numItems, const FUNCTION& function ) { parallelForEx( numItems, 1, false, function ); }

    /// Call function that consumes \p numItems items.
    /// FUNCTION must implement operator()( hkConcurrency::Iterator& items ).
    /// Execute on the calling thread if singleThreaded is true.
    template <typename FUNCTION>
    HK_ALWAYS_INLINE static void parallelForEx( int numItems, bool singleThreaded, const FUNCTION& function ) { parallelForEx( numItems, 1, singleThreaded, function ); }

    /// Call function that consumes \p numItems items.
    /// FUNCTION must implement operator()( hkConcurrency::Iterator& items ).
    /// hintBatchSize is the ideal batch size
    template <typename FUNCTION>
    HK_INLINE static void parallelForEx( int numItems, int hintBatchSize, bool singleThreaded, const FUNCTION& function );

    // -------------------------------------------------------------------------------------------------------------
    // Async
    // -------------------------------------------------------------------------------------------------------------


    /// Call a function without return value asynchronously.
    template <typename FUNCTION>
    HK_INLINE static AsyncHandle async( const FUNCTION& function );

    /// Call a function without return value asynchronously.
    /// WARNING: function argument is passed by value.
    template <typename FUNCTION, typename A0>
    HK_INLINE static AsyncHandle async( const FUNCTION& function, const A0& a0 );

    /// Call a function without return value asynchronously.
    /// WARNING: function arguments are passed by value.
    template <typename FUNCTION, typename A0, typename A1>
    HK_INLINE static AsyncHandle async( const FUNCTION& function, const A0& a0, const A1& a1 );

    /// Call a function without return value asynchronously.
    /// WARNING: function arguments are passed by value.
    template <typename FUNCTION, typename A0, typename A1, typename A2>
    HK_INLINE static AsyncHandle async( const FUNCTION& function, const A0& a0, const A1& a1, const A2& a2 );

    /// Call a function without return value asynchronously.
    /// WARNING: function arguments are passed by value.
    template <typename FUNCTION, typename A0, typename A1, typename A2, typename A3>
    HK_INLINE static AsyncHandle async( const FUNCTION& function, const A0& a0, const A1& a1, const A2& a2, const A3& a3 );

    /// Wait for the completion of an asynchronous task and free the handle.
    /// Returns false if blocking is set to false and the task was not finished, true otherwise.
    static bool await( AsyncHandle& asyncHandle, bool blocking = true );

    /// Wait for the completion of an asynchronous tasks and free the handles.
    template <typename CONTAINER>
    HK_INLINE static void awaitAll( CONTAINER& asyncHandles );

    // -------------------------------------------------------------------------------------------------------------
    // Fields
    // -------------------------------------------------------------------------------------------------------------

protected:

    /// Task queue to use, if null (default), all processing will be single-threaded.
    hkTaskQueue* m_taskQueue;

    /// Force single-threaded operations if true.
    bool m_singleThreaded;
};

#include <Common/Base/Thread/Concurrency/hkConcurrency.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
