// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Types/hkBaseDefs.h>
#include <Common/Base/Thread/Atomic/hkAtomicPrimitives.h>

/// A helper class to deal with multiple threads trying to increment a shared counter.
struct hkIntegerDistributor
{
public:

    /// Constructor. Creates \a maxNumThreads queues and tries to distribute the items evenly amongst the threads.
    HK_EXPORT_COMMON void initThreadInfo( int maxNumThreads, int initialMaxCounter );

    /// Get a single item from a queue; currentThreadId is a hint to speed up seeking of a queue to take the item from.
    /// Returns HK_FAILURE if there are no more items left.
    HK_INLINE hkResult getItem( int currentThreadId, _Out_ hkUint32* indexOut );

    /// Add a list of more items to queue 0, this increases max value.
    /// This function is multi threading safe and also can be called while other threads are calling getItem().
    HK_EXPORT_COMMON void addMoreItems( int numItems );

    /// Get the max value
    HK_INLINE hkUint32 getMaxValue() { return maxValueOf( m_queues[ 0 ].m_counterAndMax ); }

    /// Get multiple items, to be used with the hkAtomicIntIterator
    HK_EXPORT_COMMON hkResult getItems(int currentThreadId, _Out_ hkUint32* numRequestedInOut, _Out_ hkUint32* startIndexOut, _Out_ hkUint32* maxIndexOut);

protected:

    enum
    {
        MIN_QUEUE_LEN_FOR_WORK_STEALING = 16
    };

    struct Queue
    {
        void set( hkUint32 counter, hkUint32 maxValue )
        {
            hkUint64 newCounterAndMax = ( hkUint64( maxValue ) << 32 ) | hkUint64( counter );
            #ifdef HK_ARCH_X64
            m_counterAndMax = newCounterAndMax;
            #else
            while ( true )
            {
                hkUint64 oldCounterAndMax = m_counterAndMax;
                if ( hkAtomic::compareAndSwap64( &m_counterAndMax, oldCounterAndMax, newCounterAndMax ) )
                {
                    break;
                }
            }
            #endif
        }

        hkUint64 m_counterAndMax;
        hkUchar  m_padding[ hkAtomic::CACHELINE_SIZE - sizeof( hkUint64 ) ];    // avoid false sharing
    };

    // per thread data, padded to avoid false sharing
    struct ThreadData
    {
        hkUint32 m_lastSuccesfulQueue;
        hkUchar  m_padding[ hkAtomic::CACHELINE_SIZE - sizeof( hkUint32 ) ];    // avoid false sharing
    };

    HK_EXPORT_COMMON hkResult refillLocalCounterAndGetItem( int currentThreadId, _Out_ hkUint32* indexOut );

    static HK_INLINE hkUint32 counterOf( hkUint64 i ) { return hkUint32( i ); }
    static HK_INLINE hkUint32 maxValueOf( hkUint64 i ) { return hkUint32( i >> 32 ); }

    // -------------------------------------------------------------------------------------------------------------
    // Fields
    // -------------------------------------------------------------------------------------------------------------

    hkUint32 m_enableWorkStealing;
    hkUchar  m_padding[ hkAtomic::CACHELINE_SIZE ]; // avoid false sharing

    // Queue 0 is special; it is not used by any thread; instead, it is the queue that each thread attempts to steal from first.
    hkInplaceArray<ThreadData, 5> m_threadData;
    hkInplaceArray<Queue, 5> m_queues;
};

/// Helper class to even speed up the hkIntegerDistributor by getting batches of integers
struct hkAtomicIntIterator
{
    /// Iterator value.
    struct Value
    {
        HK_ALWAYS_INLINE Value() : m_provider( HK_NULL ), m_value( -1 ) {}
        HK_ALWAYS_INLINE Value( hkAtomicIntIterator* provider ) : m_provider( provider ), m_value( -1 ) { operator++(); }
        HK_ALWAYS_INLINE bool operator!=( const Value& other ) const { return m_provider != other.m_provider; }
        HK_ALWAYS_INLINE int operator++() { const int value = m_value; m_provider = m_provider->getItem( (hkUint32*)&m_value ).isSuccess() ? m_provider : HK_NULL; return value; }
        HK_ALWAYS_INLINE int operator*() const { return m_value; }

        hkAtomicIntIterator* m_provider;
        int m_value;
    };

    /// Ctor
    HK_INLINE hkAtomicIntIterator( hkIntegerDistributor* distributor, int threadId );

    /// Get one element
    HK_INLINE hkResult getItem(_Out_ hkUint32* indexOut );

    /// Iterator support.
    HK_ALWAYS_INLINE Value begin() { return Value( this ); }

    /// Iterator support.
    HK_ALWAYS_INLINE Value end() const { return Value(); }


    hkIntegerDistributor* m_integers;
    hkUint32 m_current;
    hkUint32 m_max;
    hkUint32 m_batchSize;
    hkUint32 m_minBatchSize;
    int m_threadId;
};

#include <Common/Base/Thread/Atomic/hkIntegerDistributor.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
