// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

// this: #include <Common/Base/Thread/Atomic/hkAtomicPrimitives.h>

#include <Common/Base/Types/hkBaseDefs.h>

/// A list of atomic functions, implemented for 32 bit.
/// Also implemented for 64 bit if HK_ATOMIC_SUPPORT_64BIT is #defined.
namespace hkAtomic
{
    enum
    {
        CACHELINE_SIZE = 64     ///< The minimum size of a cache line
    };

    /// VC++ 2012's __declspec(align()) expects a literal alignment, something
    /// like __declspec(align(hkAtomic::CACHELINE_SIZE) won't work. For this
    /// reason, we add this HK_CACHELINE_SIZE macro.
#   define HK_CACHELINE_SIZE 64

    namespace Detail
    {
        template <int> struct Storage { };

        template <> struct Storage<1> { typedef hkUint8 Type; static const int Alignment = 1; };
        template <> struct Storage<2> { typedef hkUint16 Type; static const int Alignment = 2; };
        template <> struct Storage<4> { typedef hkUint32 Type; static const int Alignment = 4; };
        template <> struct Storage<8> { typedef hkUint64 Type; static const int Alignment = 8; };

        template <int size>
        HK_INLINE bool HK_CALL compareAndSwap(
            _Inout_ typename Storage<size>::Type* address,
            _Inout_ typename Storage<size>::Type* oldValue,
            typename Storage<size>::Type newValue);

        template <int size>
        HK_INLINE bool HK_CALL compareAndSwapBool(
            _Inout_ typename Storage<size>::Type* address,
            typename Storage<size>::Type oldValue,
            typename Storage<size>::Type newValue);

        template <int size>
        HK_INLINE typename Storage<size>::Type HK_CALL compareAndSwapVal(
            _Inout_ typename Storage<size>::Type* address,
            typename Storage<size>::Type oldValue,
            typename Storage<size>::Type newValue);

        template <int size>
        HK_INLINE typename Storage<size>::Type exchangeAdd(
            _Inout_ typename Storage<size>::Type* address,
            typename Storage<size>::Type b);

        template <int size>
        HK_INLINE typename Storage<size>::Type swap(
            _Inout_ typename Storage<size>::Type* address,
            typename Storage<size>::Type newvalue);

        template <int size> HK_ALWAYS_INLINE typename Storage<size>::Type loadRelaxed(_In_ const typename Storage<size>::Type* address);
        template <int size> HK_ALWAYS_INLINE typename Storage<size>::Type loadAcquire(_In_ const typename Storage<size>::Type* address);
        template <int size> HK_ALWAYS_INLINE typename Storage<size>::Type loadSeqCst(_In_ const typename Storage<size>::Type* address);

        template <int size> HK_ALWAYS_INLINE void storeRelaxed(_Out_ typename Storage<size>::Type* address, typename Storage<size>::Type value);
        template <int size> HK_ALWAYS_INLINE void storeRelease(_Out_ typename Storage<size>::Type* address, typename Storage<size>::Type value);
        template <int size> HK_ALWAYS_INLINE void storeSeqCst(_Out_ typename Storage<size>::Type* address, typename Storage<size>::Type value);

        /// Fallback implementation which uses compareAndSwap. The compiler will
        /// prefer specializations, which are in the platform/compiler specific headers.
        template <int size>
        HK_INLINE typename Storage<size>::Type exchangeAdd(
            _Inout_ typename Storage<size>::Type* address,
            typename Storage<size>::Type b)
        {
            typedef typename Storage<size>::Type StorageType;
            StorageType oldVal = *address;
            while(true)
            {
                StorageType newVal = oldVal + (StorageType)b;

                // Loop until we can set the change being applied
                if(compareAndSwap<size>(address, &oldVal, newVal))
                {
                    return oldVal;
                }
            }
        }
    }
}


// -------------------------------------------------------------------------------------------------------------
// Implementations
// -------------------------------------------------------------------------------------------------------------

#ifndef __HAVOK_PARSER__
#   if      defined(HK_PLATFORM_WIN32)|| \
            (defined(HK_PLATFORM_NX) && defined(_WIN32))
#       include <Common/Base/Thread/Atomic/hkAtomicPrimitives_Win32.inl>
#   elif    defined(HK_PLATFORM_PSVITA) || \
            defined(HK_PLATFORM_ANDROID)|| \
            defined(HK_PLATFORM_TIZEN)  || \
            defined(HK_PLATFORM_PS4)    || \
            defined(HK_PLATFORM_LINUX)  || \
            defined(HK_PLATFORM_IOS)    || \
            defined(HK_PLATFORM_MAC)    || \
            defined(HK_PLATFORM_NX)
#       include <Common/Base/Thread/Atomic/hkAtomicPrimitives_Gcc.inl>
#   elif    defined(HK_PLATFORM_WIIU)
#       include <Common/Base/Thread/Atomic/hkAtomicPrimitives_WiiU.inl>
#   else
#       error Missing implementation of atomic primitives!!
#   endif
#endif

namespace hkAtomic
{
    /// Compare and swap. Returns true if the compare and swap succeeds, false
    /// otherwise. If it failed, then curValue will be updated to hold the
    /// current value of address. This corresponds to the following code,
    /// performed atomically:
    ///
    ///     if(*address == *oldValue)
    ///     {
    ///         *address = newValue;
    ///         return true;
    ///     }
    ///     else
    ///     {
    ///         *oldValue = *address;
    ///         return false;
    ///     }
    template <typename T>
    HK_ALWAYS_INLINE bool HK_CALL compareAndSwapFull(_Inout_ T* address, _Inout_ T* curValue, T newValue)
    {
        static_assert(hkTrait::IsPodType<T>::result, "hkAtomic::compareAndSwap should only be used with POD types.");
        typedef typename Detail::Storage<sizeof(T)>::Type StorageType;
        return Detail::compareAndSwap<sizeof(T)>(
            reinterpret_cast<StorageType*>(address),
            reinterpret_cast<StorageType*>(curValue),
            reinterpret_cast<const StorageType&>(newValue));
    }

    /// Compare and swap, returning true if the compare and swap succeeded,
    /// false otherwise. This corresponds to the following code, performed atomically:
    ///
    ///     if(*address == oldValue)
    ///     {
    ///         *address = newValue;
    ///         return true;
    ///     }
    ///     return false;
    template <typename T>
    HK_ALWAYS_INLINE bool HK_CALL compareAndSwap(_Inout_ T* address, T oldValue, T newValue)
    {
        static_assert(hkTrait::IsPodType<T>::result, "hkAtomic::compareAndSwap should only be used with POD types.");
        typedef typename Detail::Storage<sizeof(T)>::Type StorageType;
        return Detail::compareAndSwapBool<sizeof(T)>(
            reinterpret_cast<StorageType*>(address),
            reinterpret_cast<const StorageType&>(oldValue),
            reinterpret_cast<const StorageType&>(newValue));
    }

    /// Compare and swap, returning the old value. This corresponds to the
    /// following code, performed atomically:
    ///
    ///     if(*address == oldValue)
    ///     {
    ///         T ret = *address;
    ///         *address = newValue;
    ///         return ret;
    ///     }
    ///     return *address;
    template <typename T>
    HK_ALWAYS_INLINE T HK_CALL compareAndSwapVal(_Inout_ T* address, T oldValue, T newValue)
    {
        static_assert(hkTrait::IsPodType<T>::result, "hkAtomic::compareAndSwapVal should only be used with POD types.");
        typedef typename Detail::Storage<sizeof(T)>::Type StorageType;
        StorageType ret = Detail::compareAndSwapVal<sizeof(T)>(
            reinterpret_cast<StorageType*>(address),
            reinterpret_cast<const StorageType&>(oldValue),
            reinterpret_cast<const StorageType&>(newValue));
        return reinterpret_cast<const T&>(ret);
    }

    /// Adds \p b to the value at \p address and returns the old value.
    HK_ALWAYS_INLINE hkUint32 exchangeAdd(_Inout_ hkUint32* address, hkUint32 b) { return Detail::exchangeAdd<4>(address, b); }
    HK_ALWAYS_INLINE hkInt32 exchangeAdd(_Inout_ hkInt32* address, hkInt32 b) { return (hkInt32)Detail::exchangeAdd<4>((hkUint32*)address, b); }
    HK_ALWAYS_INLINE hkUint64 exchangeAdd(_Inout_ hkUint64* address, hkUint64 b) { return Detail::exchangeAdd<8>(address, b); }
    HK_ALWAYS_INLINE hkInt64 exchangeAdd(_Inout_ hkInt64* address, hkInt64 b) { return (hkUint64)Detail::exchangeAdd<8>((hkUint64*)address, (hkUint64)b); }

    /// Adds \p to the value at \p address.
    HK_ALWAYS_INLINE void add(_Inout_ hkUint32* address, hkUint32 b) { Detail::exchangeAdd<4>(address, b); }
    HK_ALWAYS_INLINE void add(_Inout_ hkInt32* address, hkInt32 b) { Detail::exchangeAdd<4>((hkUint32*)address, b); }
    HK_ALWAYS_INLINE void add(_Inout_ hkUint64* address, hkUint64 b) { Detail::exchangeAdd<8>(address, b); }
    HK_ALWAYS_INLINE void add(_Inout_ hkInt64* address, hkInt64 b) { Detail::exchangeAdd<8>((hkUint64*)address, (hkUint64)b); }

    /// Swap the value at \p address with newValue, returning the old value.
    /// This corresponds to the following code, performed atomically:
    ///
    ///     T ret = *address;
    ///     *address = newValue;
    ///     return ret;
    template <typename T>
    HK_ALWAYS_INLINE T swap(_Inout_ T* address, T newValue)
    {
        static_assert(hkTrait::IsPodType<T>::result, "hkAtomic::swap should only be used with POD types.");
        typedef typename Detail::Storage<sizeof(T)>::Type StorageType;
        StorageType ret = Detail::swap<sizeof(T)>(
            reinterpret_cast<StorageType*>(address),
            reinterpret_cast<const StorageType&>(newValue));
        return reinterpret_cast<const T&>(ret);
    }

    template <typename T>
    HK_ALWAYS_INLINE T loadRelaxed(_In_ const T* address)
    {
        static_assert(hkTrait::IsPodType<T>::result, "hkAtomic::loadRelaxed should only be used with POD types.");
        typedef typename Detail::Storage<sizeof(T)>::Type StorageType;
        StorageType ret = Detail::loadRelaxed<sizeof(T)>(reinterpret_cast<const StorageType*>(address));
        return reinterpret_cast<const T&>(ret);
    }

    template <typename T>
    HK_ALWAYS_INLINE T loadAcquire(_In_ const T* address)
    {
        static_assert(hkTrait::IsPodType<T>::result, "hkAtomic::loadAcquire should only be used with POD types.");
        typedef typename Detail::Storage<sizeof(T)>::Type StorageType;
        StorageType ret = Detail::loadAcquire<sizeof(T)>(reinterpret_cast<const StorageType*>(address));
        return reinterpret_cast<const T&>(ret);
    }

    template <typename T>
    HK_ALWAYS_INLINE T loadSeqCst(_In_ const T* address)
    {
        static_assert(hkTrait::IsPodType<T>::result, "hkAtomic::loadSeqCst should only be used with POD types.");
        typedef typename Detail::Storage<sizeof(T)>::Type StorageType;
        StorageType ret = Detail::loadSeqCst<sizeof(T)>(reinterpret_cast<const StorageType*>(address));
        return reinterpret_cast<const T&>(ret);
    }

    template <typename T>
    HK_ALWAYS_INLINE void storeRelaxed(_In_ T* address, T value)
    {
        static_assert(hkTrait::IsPodType<T>::result, "hkAtomic::storeRelaxed should only be used with POD types.");
        typedef typename Detail::Storage<sizeof(T)>::Type StorageType;
        Detail::storeRelaxed<sizeof(T)>(reinterpret_cast<StorageType*>(address),
            reinterpret_cast<const StorageType&>(value));
    }

    template <typename T>
    HK_ALWAYS_INLINE void storeRelease(_In_ T* address, T value)
    {
        static_assert(hkTrait::IsPodType<T>::result, "hkAtomic::storeRelease should only be used with POD types.");
        typedef typename Detail::Storage<sizeof(T)>::Type StorageType;
        Detail::storeRelease<sizeof(T)>(reinterpret_cast<StorageType*>(address),
            reinterpret_cast<const StorageType&>(value));
    }

    template <typename T>
    HK_ALWAYS_INLINE void storeSeqCst(_In_ T* address, T value)
    {
        static_assert(hkTrait::IsPodType<T>::result, "hkAtomic::storeSeqCst should only be used with POD types.");
        typedef typename Detail::Storage<sizeof(T)>::Type StorageType;
        Detail::storeSeqCst<sizeof(T)>(reinterpret_cast<StorageType*>(address),
            reinterpret_cast<const StorageType&>(value));
    }

    /// Put the processor in a pause state for a few cycles
    HK_INLINE void pauseTicks( int numTicks = 1 );

    /// Forces a memory barrier providing sequential consistency.
    HK_INLINE void HK_CALL readWriteBarrier();

    /// Pauses the current thread for increasingly longer time intervals, eventually yielding it
    template <hkUint32 LoopsBeforeYield = 8>
    struct Backoff
    {
        HK_INLINE Backoff() : m_loopCount(0)    {}

        HK_INLINE void pause()
        {
            pauseTicks(1 << m_loopCount);
            m_loopCount = ( m_loopCount < LoopsBeforeYield )? (m_loopCount + 1) : m_loopCount;
        }

        hkUint32 m_loopCount;
    };

    /// Waits until the value is equal.
    /// Please avoid using this type of busy wait: On some platforms the OS scheduler might
    /// let this thread do busy wait while the predecessor thread is put asleep.
    template <typename T>
    HK_INLINE void waitUntilEqual(_In_ T* address, const T& valueToCheck)
    {
        while(loadSeqCst(address) != valueToCheck)
        {
            pauseTicks();
        }
    }

    /// Waits for the given condition to become true.
    /// Please avoid using this type of busy wait: On some platforms the OS scheduler might
    /// let this thread do busy wait while the predecessor thread is put asleep.
    template <typename T, typename Condition>
    HK_INLINE void waitUntilTrue(_In_ T* address, Condition c)
    {
        while(!c(loadSeqCst(address)))
        {
            pauseTicks();
        }
    }

    /// An atomic variable. Note that it may be stored on more than sizeof(T) bytes.
    /// It only supports native atomics (see HK_NATIVE_ATOMIC_ADD).
    /// Specific specializations for numeric types provide extra features (op++...).
    template <class T>
    class Variable
    {
    public:
        HK_DECLARE_CLASS(Variable, New, Reflect);

        HK_ALWAYS_INLINE Variable() { }
        HK_ALWAYS_INLINE Variable(const T& srcVal)
            : m_val(srcVal)
        {
        }

        /// Retrieves the value
        HK_ALWAYS_INLINE operator T() const
        {
            return hkAtomic::loadSeqCst(&m_val);
        }

        /// Sets the value
        HK_ALWAYS_INLINE void operator=(const T& other)
        {
            hkAtomic::storeSeqCst(&m_val, other);
        }

        HK_ALWAYS_INLINE T loadRelaxed() const { return hkAtomic::loadRelaxed(&m_val); }
        HK_ALWAYS_INLINE T loadAcquire() const { return hkAtomic::loadAcquire(&m_val); }
        HK_ALWAYS_INLINE T loadSeqCst() const { return hkAtomic::loadSeqCst(&m_val); }

        HK_ALWAYS_INLINE void storeRelaxed(T newVal) { hkAtomic::storeRelaxed(&m_val, newVal); }
        HK_ALWAYS_INLINE void storeRelease(T newVal) { hkAtomic::storeRelease(&m_val, newVal); }
        HK_ALWAYS_INLINE void storeSeqCst(T newVal) { hkAtomic::storeSeqCst(&m_val, newVal); }

        HK_INLINE T operator++()
        {
            return hkAtomic::exchangeAdd(&m_val, (T)1) + 1;
        }

        HK_INLINE T operator++(int)
        {
            return hkAtomic::exchangeAdd(&m_val, (T)1);
        }

        HK_INLINE T operator--()
        {
            return hkAtomic::exchangeAdd(&m_val, (T)-1) - 1;
        }

        HK_INLINE T operator--(int)
        {
            return hkAtomic::exchangeAdd(&m_val, (T)-1);
        }

        T exchangeAdd(T b)
        {
            return hkAtomic::exchangeAdd(&m_val, b);
        }

        HK_INLINE bool compareAndSwap(T& curValue, T newValue)
        {
            return hkAtomic::compareAndSwapFull(&m_val, &curValue, newValue);
        }

        HK_INLINE T swap(T newValue)
        {
            return hkAtomic::swap(&m_val, newValue);
        }

    private:
        T m_val;
    };
}

// Deprecated
namespace hkAtomic
{
    HK_ALWAYS_INLINE bool compareAndSwap32(_In_ hkUint32* address, hkUint32 oldValue, hkUint32 newValue) { return compareAndSwap(address, oldValue, newValue); }
    HK_ALWAYS_INLINE bool compareAndSwap64(_In_ hkUint64* address, hkUint64 oldValue, hkUint64 newValue) { return compareAndSwap(address, oldValue, newValue); }

    HK_ALWAYS_INLINE hkUint32 exchangeAdd32(_In_ hkUint32* address, hkUint32 b) { return exchangeAdd(address, b); }
    HK_ALWAYS_INLINE hkUint64 exchangeAdd64(_In_ hkUint64* address, hkUint64 b) { return exchangeAdd(address, b); }
}

#include <Common/Base/_Auto/TemplateTypes/hkAtomicPrimitives_Types.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
