// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>

class hkAsyncThreadPool;
class hkSemaphore;

/// An 'simple' hkAsyncThreadPool aware semaphore. This is similar to a
/// \ref hkSemaphore, but differs from it in 2 ways. First of all, it's
/// hkAsyncThreadPool aware, which means that an hkAsyncThreadPool thread
/// waiting for this semaphore won't block the hkAsyncThreadPool from changing
/// between foreground and background mode. Second, there's the restriction
/// which makes it simple, which is that the acquire function must always be
/// called from the same hkAsyncThreadPool thread. Its release function can
/// still be called from any of the hkAsyncThreadPool's thread, or from the main thread.
class HK_EXPORT_COMMON hkAsyncThreadPoolSimpleSemaphore
{
public:
    HK_DECLARE_CLASS(hkAsyncThreadPoolSimpleSemaphore, New);

    HK_INLINE hkAsyncThreadPoolSimpleSemaphore()
        : m_count(0)
    {
    }

    /// Decreases the semaphore's count, and if it becomes negative, the thread
    /// will until the count becomes non-negative again.
    ///
    /// The \p threadPool and \p threadIdx arguments must be those of the
    /// current thread.
    void acquire(hkAsyncThreadPool* threadPool, int threadIdx);

    /// Increases the semaphore's count. If the semaphore got increased from
    /// negative to non-negative, then the waiting thread will be woken up.
    ///
    /// The \p threadPool and \p threadIdx arguments must be those of the thread
    /// the semaphore belongs to (that is, the thread which makes the acquire calls).
    void release(hkAsyncThreadPool* threadPool, int threadIdx);

private:
    hkAtomic::Variable<hkInt32> m_count;
};

/// An hkAsyncThreadPool aware mutex. This is similar to \ref hkCriticalSection,
/// but has the additional feature that it won't block an hkAsyncThreadPool's
/// thread from switching between foreground and background mode when the mutex
/// is taken.
class HK_EXPORT_COMMON hkAsyncThreadPoolMutex
{
public:
    HK_DECLARE_CLASS(hkAsyncThreadPoolMutex, New);

    hkAsyncThreadPoolMutex();
    ~hkAsyncThreadPoolMutex();

    /// Lock this mutex.
    void enter();

    /// Unlock this mutex.
    void leave();

private:
    // The number of threads which are currently contending for this mutex.
    hkAtomic::Variable<hkInt32> m_numContenders;

    struct WaitingQueueItem
    {
        WaitingQueueItem* m_next;

        /// The worker thread index of the thread this WaitingQueueItem belongs,
        /// or -1 if the thread doesn't belong to an hkAsyncThreadPool. If
        /// m_threadPoolThreadIdx >= 0, we know that the m_threadPool pointer is
        /// valid, otherwise we know that the m_semaphore pointer is valid.
        int m_threadPoolThreadIdx;
        union
        {
            hkAsyncThreadPool* m_threadPool;
            hkSemaphore* m_semaphore;
        };
    };

    // The following is a linked list of the threads waiting for this mutex. The
    // item at m_queueHead is the next one to be woken up, new items should be
    // appended after m_queueTail. This list may only be modified when the
    // m_queueLock is taken.
    //
    // We also maintain the following invariant:
    // If m_numContenders is greater than 1, then one of the following
    // two must be the case:
    //  - m_queueLock is taken.
    //  - m_queueHead and m_queueTail point to valid items.
    hkCriticalSection m_queueLock;
    WaitingQueueItem* m_queueHead;
    WaitingQueueItem* m_queueTail;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
