// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Types/hkBaseDefs.h>
#include <Common/Base/Thread/Semaphore/hkSemaphore.h>

namespace hkAsyncThreadPoolDetail
{
    /// A fixed size message queue which is used by the \ref hkAsyncThreadPool
    /// to let the worker threads tell the main thread when it's time for them
    /// to go to background mode.
    class ToBackgroundQueue
    {
    public:
        HK_DECLARE_CLASS(ToBackgroundQueue, New);
        
        /// Constructs a new ToBackgroundQueue for a hkAsyncThreadPool with the
        /// given number of threads.
        ToBackgroundQueue(int numThreads);

        /// Resets the queue, that is, it restores it the same state it would
        /// have just after it was constructed.
        void reset();

        /// Pushes a message onto the queue which indicates that the worker
        /// thread with the given index wants to go to background mode.
        ///
        /// This function may only be called once per thread, unless the queue
        /// is reset in the meanwhile.
        void pushToBackground(int threadIdx);

        /// Pushes a message onto the queue which indicates that all foreground
        /// threads which are waiting for a semaphore should be brought to
        /// background mode.
        void pushAllSemaphoresToBackground();

        /// A class to hold the result of the pop function. This can be either a
        /// 'all semaphores to background' message, or the index of a thread
        /// which should be brought to background mode.
        class PopResult
        {
        public:
            /// Constructs a new PopResult. Pass -1 to make it an 'all
            /// semaphores to background' message.
            PopResult(int threadIndex)
                : m_threadIndex(threadIndex)
            {
            }

            /// Returns whether this is an 'all semaphores to background' message.
            bool isAllSemaphoresToBackground() const { return m_threadIndex == -1; }

            /// Returns the index of the thread which should be brought to
            /// background mode. You should only call this when
            /// isAllSemaphoresToBackground returns false.
            int getThreadIndex() const
            {
                HK_ASSERT_NO_MSG(0xd5ffe4f, !isAllSemaphoresToBackground());
                return m_threadIndex;
            }

        private:
            int m_threadIndex;
        };
        
        /// Pops one message from this queue, or waits when none is available.
        PopResult pop();
        
    private:
        /// The semaphore which should be released when a new message is
        /// available. When it's released, it's guaranteed that either
        /// m_numAllSemaphoresToBackground != 0, or that there's an element in
        /// m_toBackground within the range [m_head, m_tail), which is > 0.
        hkSemaphore m_semaphore;

        /// The head of the queue. This is a lower bound on the indices of the
        /// upcoming messages in m_toBackground.
        int m_head;

        /// The tail of the queue, that is, the location in m_toBackground where
        /// the next message should be written to.
        hkAtomic::Variable<int> m_tail;

        /// The queue which holds the thread indices of the 'to background' messages.
        ///
        /// Besides thread indices, there are 2 special values:
        /// - The value -1 indicates that the slot hasn't been used yet, so a
        ///   new element can still be written to it.
        /// - The value -2 is written to a slot after its value is consumed.
        hkArray<int> m_toBackground;

        /// The number of times we have the the 'all waiting for semaphore to
        /// background' message.
        hkAtomic::Variable<int> m_numAllSemaphoresToBackground;
    };
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
