// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Types/hkBaseDefs.h>
#include <Common/Base/Object/hkReferencedObject.h>
#include <Common/Base/Thread/TaskQueue/hkTaskQueue.h>
#include <Common/Base/Thread/Atomic/hkAtomicPrimitives.h>
#include <Common/Base/Thread/Thread/hkThread.h>
#include <Common/Base/Config/hkConfigMonitors.h>
#include <Common/Base/Types/hkScopedPtr.h>

/// Set this to non-zero to enable tendToThreadPool instrumentation. This
/// measures the average times between adjacent calls of tendToThreadPool.
/// Instrumentation comes at a performance cost and can take up quite a bit of
/// memory, so it should certainly be disabled by default. Don't ever release a
/// game with this enabled.
#define HK_INSTRUMENT_TEND_TO_THREADPOOL 0

class hkAsyncThreadPoolInstrumenter;
class hkDefaultTaskQueue;

namespace hkAsyncThreadPoolDetail
{
    class ToBackgroundQueue;
}

/// A hkAsyncThreadPool is a thread pool which supports fine grained thread
/// priority management.
///
/// The threads in an \ref hkAsyncThreadPool can be in either foreground mode or
/// background mode. When threads are in foreground mode, they have the normal
/// thread priority, when they are in background mode, they have the priority
/// specified through \ref hkAsyncThreadPoolCinfo::m_threadBackgroundPriorities
/// (which includes a suspended mode). By default, threads are in background
/// mode, but they can all be brought to foreground mode for a specific amount
/// of time using a call to processForeground. After the given time has elapsed,
/// the threads will automatically go back to background mode again.
///
/// To be able to have fine grained control over the moment when a thread
/// changes priority or goes to suspend mode, the hkAsyncThreadPool needs some
/// help from the code running in its threads. Specifically, the threads need to
/// make sure that the tendToCurrentThreadPool function is called at a regular
/// interval (we aim for once each 1/10 ms). If thread pool threads also make
/// sure that they don't call tendToThreadPool from inside locks (unless it's a
/// \ref hkAsyncThreadPoolLock), then a thread can't go to background mode while
/// it's waiting for a lock either, which means we don't have to worry about
/// things like priority inversion.
///
/// This thread pool is ideal for work which needs to be performed over the span
/// of multiple frames, where during each frame, each thread is given a specific
/// amount of processing time it can use, regardless of whether the work was
/// finished or not. On top of that, by using a background priority other than
/// suspend, these worker threads can still jump and use the processing time of
/// cores which would be idle otherwise.
class HK_EXPORT_COMMON hkAsyncThreadPool : public hkReferencedObject
{
public:
    HK_DECLARE_CLASS(hkAsyncThreadPool, NewOpaque);

    /// A ThreadPriority is used to indicate the priority a thread should be
    /// given when the hkAsyncThreadPool is in background mode. The priorities
    /// NORMAL and LOW mean that the thread will still be running, but possibly
    /// at a lower priority, the SUSPEND priority means the thread will be fully
    /// suspended when the hkAsyncThreadPool is running in background mode.
    ///
    /// The actual hkThread::setPriority priority corresponding to NORMAL and
    /// LOW can be set through the m_normalPriority and m_lowPriority of the
    /// thread pools Cinfo.
    enum class ThreadPriority : hkUint8
    {
        /// This is the priority the non background Havok threads are expected
        /// to have.
        ///
        /// The actual hkThread priority a background thread will be given can
        /// be specified through \ref Cinfo::m_normalPriority.
        NORMAL = 0,

        /// Lower than NORMAL, so threads with NORMAL priority will be
        /// preferred over a thread with this priority, yet it will still fill
        /// in the gaps when cores would otherwise be idle.
        ///
        /// The actual hkThread priority a background thread will be given can
        /// be specified through \ref Cinfo::m_lowPriority.
        LOW = 1,

        /// The thread is suspended and doesn't run at all, even when there are
        /// free cores available.
        SUSPENDED = 2,
    };

    /// How 'Cinfo::m_hardwareThreadMasksOrIds' is interpreted to execute this
    /// pool's threads on specific hardware threads.
    enum class HardwareThreadBinding : hkUint8
    {
        /// Threads may run on any hardware thread; expects
        /// m_hardwareThreadMasksOrIds to be empty.
        UNSPECIFIED,

        /// Threads request execution on specific hardware thread ID, solely
        /// via scheduler hint. Windows platforms only.
        SCHEDULER_HINT,

        /// Threads hard-affinitized to execute only on masked hardware
        /// thread IDs.
        HARD_AFFINITY
    };

    /// Construction info for \ref hkAsyncThreadPool.
    struct HK_EXPORT_COMMON Cinfo
    {
        HK_DECLARE_CLASS(Cinfo, New, Reflect);

        Cinfo();

        HardwareThreadBinding m_hardwareThreadBinding;

        /// The number of threads to create.
        /// Defaults to 1.
        int m_numThreads;

        /// The hardware thread IDs or masks to use for each thread (see
        /// HardwareThreadBinding::SCHEDULER_HINT/HARD_AFFINITY, respectively).
        /// If empty array, assumes UNSPECIFIED hardware binding and assigns
        /// hardware thread IDs spanning all cores before assigning to more
        /// hardware threads in same core (see
        /// hkCpuThreadPool::calcHardwareThreadMaskOrId()) These defaults assume
        /// that you are using hardware thread 0 as your main calling thread.
        hkArray<hkUint32> m_hardwareThreadMasksOrIds;

        /// The program stack size to use for each thread, if applicable.
        ///  - On Windows and Xbox360, this defaults to 0, meaning the thread
        ///    inherits the executable's stack size.
        ///  - On PlayStation(R)3, this defaults to 256K.
        ///  - On PlayStation(R)4, this defaults to 512K in Debug/Dev
        ///    configurations and 128K in Release.
        ///  - This has no effect on other platforms (e.g., posix).
        int m_stackSize;

        /// The thread name to be passed to each hkThread::startThread().
        hkStringPtr m_threadName;

        /// The hkThread priority corresponding to ThreadPriority::NORMAL. This
        /// is the priority we pass to \ref hkThread::setPriority.
        ///
        /// This priority is used both for thread in background mode, if their
        /// background priority is ThreadPriority::NORMAL, and for any thread
        /// when it's running in foreground mode. The default value is 0.
        int m_normalPriority;

        /// The hkThread priority corresponding to Threadpriority::LOW. This is
        /// the priority we pass to \ref hkThread::setPriority. The default
        /// value is -1.
        int m_lowPriority;


        /// The background priority for each worker thread. This array should have
        /// size m_numThreads.
        hkArray<ThreadPriority> m_threadBackgroundPriorities;

        /// The buffer size to allocate to each thread for collecting timer
        /// information. To view timers in the Visual Debugger, you must set
        /// this buffer size to a non zero value. 2MB is a good recommended
        /// size. Smaller buffers are ok, but you may lose some timer info.
        /// Defaults to 0, which means that timers will be disabled.
        int m_timerBufferPerThreadAllocation;
    };

    /// Sets up the thread pool. To actually start the threads, call \ref startThreads.
    hkAsyncThreadPool(const Cinfo& cinfo);

    /// Destructs this hkAsyncThreadPool. Note that you should explicitly call
    /// \ref stopThreads to terminate the threads before you can destruct the
    /// hkAsyncThreadPool. The main reason why we require this is because we
    /// want to be able to guarantee that the full instance is valid for as long
    /// as the user's implementation of \ref workerThreadMain is running. If
    /// we'd terminate threads in this class' destructor instead, the
    /// destructors of derived classes would already have run when we terminate
    /// threads, which means the workerThreadMain function might end up trying
    /// to access data which was already destructed.
    virtual ~hkAsyncThreadPool();

    /// Starts the worker threads in background mode.
    /// \param cinfo The construction info of this hkAsyncThreadPool. This must
    ///    be the same cinfo as the one which was used to construct this hkAsyncThreadPool.
    void startThreads(const Cinfo& cinfo);

    /// Stops the worker threads in this thread pool, and will return after they
    /// all have terminated.
    ///
    /// You can restart the threads again with a call to \ref startThreads.
    ///
    /// Note that this function doesn't abort the \ref workerThreadMain
    /// function, so it's still up to user code to make sure that the worker
    /// threads return from their workerThreadMain function. The stopThreads
    /// function does bring all worker threads to foreground mode until all
    /// worker threads have finished, so you don't have to worry about worker
    /// threads having low priority or being suspended.
    ///
    /// It's ok to call this function multiple times. If the threads have
    /// already been stopped, it will just return immediately. It's also allowed
    /// to start the threads again using \ref startThreads.
    virtual void stopThreads();

    /// Returns the number of worker threads in this hkAsyncThreadPool.
    HK_INLINE int getNumThreads() const { return m_workerThreads.getSize(); }

    /// Sets the background priorities of the threads in this thread pool.
    /// \param backgroundPriorities The background priorities. This must be a
    ///    pointer to an array whose size is (at least) the number of threads
    ///    in this hkAsyncThreadPool.
    void setBackgroundPriorities(const ThreadPriority* backgroundPriorities);

    /// Runs the threads in foreground mode for the given amount of time. This
    /// function will block until all the threads have been returned to
    /// background mode. Threads will go back to background mode either when the
    /// given amount of time has passed, or if \ref cancelForeground is called
    /// (which obviously must be done from a different thread).
    /// \param processingTimeSeconds The amount of time for which to run
    ///    foreground processing.
    void processForeground(hkReal processingTimeSeconds);

    /// Similar to \ref processForeground, but keeps the threads in foreground
    /// mode indefinitely or, as will usually be the case, until
    /// \ref cancelForeground is called.
    void processForegroundIndefinitely();

    /// Cancels foreground processing. This function can be called from any
    /// thread while the main thread is inside \ref processForground, and will
    /// have the effect that processForeground will end and return immediately.
    /// If cancelForeground is called when we're not doing foreground
    /// processing, nothing happens.
    void cancelForeground();

    /// Returns a pointer to the \ref hkAsyncThreadPool of the current worker
    /// thread if this thread is a worker thread of an \ref hkAsyncThreadPool,
    /// returns HK_NULL otherwise.
    static hkAsyncThreadPool* getCurrentThreadPool();

    /// Returns the index of the current worker thread, within the
    /// \ref hkAsyncThreadPool the thread belongs to.
    static int getCurrentThreadIndex();

    /// Calls \ref tendToThreadPool() on the current thread's thread pool (as
    /// returned by \ref getCurrentThreadPool()), or nothing if it's called from
    /// a thread which doesn't belong to a \ref hkAsyncThreadPool.
    static void tendToCurrentThreadPool(hkUint32 id);

    /// The function which each thread should call periodically. This is the
    /// function which checks if the thread should go to background mode when
    /// it's running in foreground mode.
    /// \param id A unique identifier for this tendToThreadPool call. This is
    ///    used by the instrumenter to match up timings with tendToThreadPool calls.
    void tendToThreadPool(hkUint32 id);

    /// Returns the current priority of the worker thread with the given index.
    /// This is the thread's background priority if the thread is currently
    /// running in background mode and ThreadPriority::NORMAL if it's running
    /// in foreground mode.
    ThreadPriority getThreadPriority(int threadIdx) const;

    /// Returns the thread status of the worker thread at the given index. This
    /// is the same value as the one returned by the thread's \ref
    /// hkThread::getStatus() function.
    hkThread::Status getThreadStatus(int threadIdx) const;

    /// \name Thread pool aware semaphores
    ///
    /// Each thread has a thread pool aware semaphore associated with it. This
    /// semaphore works similarly to a normal OS semaphore, but with the main
    /// difference that the hkAsyncThreadPool knows when a worker thread is
    /// waiting for its semaphore and for that reason is able to change the
    /// thread from foreground mode to background mode and vice versa when a
    /// worker thread is waiting for this semaphore. This is different with
    /// normal semaphores, because with normal semaphores, the thread pool will
    /// just keep waiting until the next call to tendToCurrentThreadPool, which
    /// won't be until after the semaphore gets released.
    ///
    /// Functionally, this semaphore is equivalent to a spinning implementation,
    /// where the spinning loop continuously calls tendToCurrentThreadPool (but
    /// of course this is not how we actually implement it).
    ///
    /// One other difference with a normal semaphore is that each semaphore is
    /// strictly associated with a certain thread, which means that only that
    /// thread is allowed to acquire it. The semaphore can still be released
    /// from any of the worker threads of this thread pool, or from the main
    /// thread (the thread making the processForeground calls that is).
    ///
    /// Note that worker threads will rarely use this semaphore directly,
    /// instead you should use the thread pool aware synchronization primitives
    /// like \ref hkAsyncThreadPoolSimpleSemaphore, \ref
    /// hkAsyncThreadPoolSemaphore and \ref hkAsyncThreadPoolLock which are
    /// implemented on top of the functions below.
    ///
    /// \{

    /// Acquires the semaphore of the current worker thread. This function can
    /// only be called from one of the worker threads.
    void acquireThreadSemaphore();

    /// Releases the semaphore of the thread with the given index. This function
    /// can be called from any thread.
    void releaseThreadSemaphore(int threadIdx);

    /// Releases all the semaphores of all the worker threads in this thread pool.
    void releaseAllThreadSemaphores();

    /// Calls releases on the semaphores of all the worker threads which are
    /// currently waiting for theirs.
    void releaseAllWaitingForSemaphore();

    /// \}

    /// Gets the instrumentation report. See \ref hkAsyncThreadPoolInstrumenter
    /// for more details. This function gives a warning and doesn't do anything
    /// else if HK_INSTRUMENT_TEND_TO_THREADPOOL is 0.
    /// \ref report[out] The hkStringBuf which will contain the report on
    ///    return. It should be empty when the function is called.
    void getInstrumentationReport(hkStringBuf& reportOut) const;

private:
    HK_CLASSALIGN(struct, HK_CACHELINE_SIZE) HK_EXPORT_COMMON WorkerThread;

    /// Copying not supported (we need to specify this to stop the compiler from
    /// auto-generating one, which would break the DLL build).
    hkAsyncThreadPool(const hkAsyncThreadPool& src);
    void operator = (const hkAsyncThreadPool& src);

    virtual void workerThreadMain(int threadIdx) = 0;

    void processForegroundImpl(hkUint64 toBackgroundTicks);

    /// Drives the process of brining threads to foreground mode. If
    /// m_toForegroundIdx != m_numThreads, then this function will keep
    /// incrementing it and updating the priorities of the corresponding
    /// threads, until m_toForegroundIdx == m_numThreads.
    ///
    /// This is a collaborative function, which means many threads can run it
    /// simultaneously, in which case they will automatically distribute the
    /// work between the different threads.
    void driveToForeground();

    /// Drives the process of bringing threads to background mode.
    void driveToBackground();

    /// Simple forwarder to the member function.
    static void* HK_CALL threadMainForwarder(void* workerThreadData);

    void threadMain(int threadIndex);

    /// Implementation of releaseThreadSemaphore. Assumes the relevant locks are
    /// taken (that is, the lock of the semaphore thread and the thread which is
    /// doing the releasing, if it's a worker thread).
    void releaseThreadSemaphoreImpl(WorkerThread& semaphoreThread);

    /// The hkThread priority corresponding to ThreadPriority::NORMAL. This
    /// is the priority we pass to \ref hkThread::setPriority.
    int m_normalPriority;

    /// The hkThread priority corresponding to Threadpriority::LOW. This is
    /// the priority we pass to \ref hkThread::setPriority.
    int m_lowPriority;

    /// Indicates whether the threads are running. This is set to true in
    /// startThreads and to false in stopThreads.
    bool m_threadsRunning;

    /// The worker threads in this hkAsyncThreadPool.
    hkArray<WorkerThread>::Aligned m_workerThreads;

    /// While we're in the process of bringing worker threads to foreground
    /// mode, this holds the index of the next thread to bring to foreground
    /// mode. This should be equal to the number of threads otherwise.
    hkAtomic::Variable<hkUint32> m_toForegroundIdx;

    /// The cached result of hkSystemClock::getTicksPerSecond().
    hkUint64 m_ticksPerSecond;

    /// The tick count (as returned by hkSystemClock::getTickCounter()) of the
    /// time when foreground processing expires and worker threads need to be
    /// brought back to background mode. This is checked when threads are
    /// running in foreground mode (state 2) and is ignored otherwise.
    hkUint64 m_toBackgroundTicks;

    /// The queue which the worker threads use to let the main thread know that
    /// it's time for them to go to background mode.
    hkScopedPtr<hkAsyncThreadPoolDetail::ToBackgroundQueue> m_toBackgroundQueue;

    /// The number of threads which are currently running in foreground mode
    /// (that is, have STATE_FOREGROUND, this does not include threads with
    /// STATE_FOREGROUND_ACQUIRE_SEMAPHORE). The purpose of this variable is to
    /// catch the situation where all threads have started waiting for a semaphore.
    hkAtomic::Variable<hkInt32> m_numInForeground;

    /// The size allocated for each thread's monitor stream.
    int m_timerBufferPerThreadAllocation;

#if HK_CONFIG_MONITORS == HK_CONFIG_MONITORS_ENABLED
    hkUint64 m_toForegroundStartTicks;
    hkUint64 m_toForegroundEndTicks;
#endif
};


/// A hkAsyncThreadPool which runs a task queue. Tasks which are submitted to
/// the task queue returned by \ref getTaskQueue will be executed by this thread pool.
class HK_EXPORT_COMMON hkAsyncTaskQueueThreadPool : public hkAsyncThreadPool
{
public:
    hkAsyncTaskQueueThreadPool(const Cinfo& cinfo);
    virtual ~hkAsyncTaskQueueThreadPool();

    virtual void stopThreads() override;

    /// Returns the task queue this thread pool is executing. Submit tasks to it
    /// to run them in this thread pools worker threads.
    hkTaskQueue* getTaskQueue();

    /// Keeps running the thread pool in foreground mode until the given handle
    /// is finished. The function will return after that.
    /// \param handle The handle of a task on this thread pool's task queue.
    void processUntilFinished(hkTaskQueue::Handle handle);

    /// Get the timer data collected so far (up to the most recent safe point).
    void appendTimerData(hkArray<hkTimerData>& timerDataOut) const;

    /// Clear the timer data as soon as possible.
    void clearTimerData();

protected:
    virtual void workerThreadMain(int threadIdx) override;

private:
    hkRefPtr<hkDefaultTaskQueue> m_taskQueue;
};

HK_REFLECT_ENUM(HK_EXPORT_COMMON, hkAsyncThreadPool::HardwareThreadBinding);
HK_REFLECT_ENUM(HK_EXPORT_COMMON, hkAsyncThreadPool::ThreadPriority);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
