// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Thread/Async/hkAsyncThreadPool.h>
#include <Common/Base/Thread/Thread/hkThread.h>
#include <Common/Base/Thread/Thread/hkWorkerThreadContext.h>
#include <Common/Base/System/Stopwatch/hkSystemClock.h>
#include <Common/Base/System/Hardware/hkHardwareInfo.h>
#include <Common/Base/Thread/TaskQueue/Default/hkDefaultTaskQueue.h>
#include <Common/Base/Thread/Async/hkAsyncThreadPoolInstrumenter.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>
#include <Common/Base/Thread/Semaphore/hkSemaphore.h>
#include <Common/Base/Thread/Async/hkAsyncThreadPoolDetail.h>

#if HK_CONFIG_MONITORS == HK_CONFIG_MONITORS_ENABLED
#define TIMERS_BEGIN(name) HK_MULTI_TIMER_BEGIN(monitorStream, name);
#define TIMERS_END(time) \
    { \
        hkMonitorStream::MultiTimerCommand cmd; \
        cmd.m_timerCommand.m_time0 = (hkMonitorStream::TimerValue)time; \
        cmd.m_callCount = 1; \
        HK_MULTI_TIMER_END(monitorStream, "", cmd); \
    }

#define TIMERS_TIMER(name, time) \
    TIMERS_BEGIN(name); \
    TIMERS_END(time);

#endif

#if defined (_MSC_VER) && (_MSC_VER == 1700 || _MSC_VER == 1800)
static __declspec(thread) int s_threadPoolThreadIndex = -1;
static __declspec(thread) hkAsyncThreadPool* s_threadPool = nullptr;
#elif defined(HK_PLATFORM_IOS) || defined(HK_PLATFORM_NX)
HK_THREAD_LOCAL(int) s_threadPoolThreadIndex;
HK_THREAD_LOCAL(hkAsyncThreadPool*) s_threadPool;
#else
static thread_local int s_threadPoolThreadIndex = -1;
static thread_local hkAsyncThreadPool* s_threadPool = nullptr;
#endif

HK_CLASSALIGN(struct, HK_CACHELINE_SIZE) hkAsyncThreadPool::WorkerThread
{
    HK_DECLARE_CLASS(WorkerThread, NewOpaque);

    /// The thread pool we're in.
    hkAsyncThreadPool* m_threadPool;

    /// The index of this thread in its thread pool.
    int m_threadIndex;

    /// The thread.
    hkThread m_thread;

    /// Hardware thread mask (or ID) onto which this software thread will be
    /// affinitized (or scheduler-hinted)
    int m_hardwareThreadMasksOrId;
    HardwareThreadBinding m_hardwareThreadBinding;

    /// The priority this thread should have when the pool is running in
    /// background mode.
    ThreadPriority m_backgroundPriority;

    enum State
    {
        /// The thread is currently running in background mode. It can
        /// transition to STATE_FOREGROUND in driveToForeground and to
        /// STATE_BACKGROUND_ACQUIRE_SEMAPHORE when the thread tries to acquire
        /// a semaphore.
        STATE_BACKGROUND,

        /// The thread is currently running in foreground mode. It can
        /// transition to STATE_TO_BACKGROUND when tendToThreadPool detects that
        /// the foreground processing time has elapsed, or it can transition to
        /// STATE_FOREGROUND_ACQUIRE_SEMAPHORE when the thread tries to acquire
        /// the semaphore.
        STATE_FOREGROUND,

        /// The thread is still running in foreground mode, but
        /// driveToBackground needs to bring it to background mode instead.
        /// driveToBackground can transition it to STATE_BACKGROUND. Since the
        /// thread itself will wait for the m_resumeSemaphore while it's in
        /// STATE_TO_BACKGROUND, it's impossible to acquire a semaphore while
        /// we're in this state.
        STATE_TO_BACKGROUND,

        /// The thread would be running in background mode, but is waiting for
        /// its semaphore. When the semaphore is released, the thread will
        /// transition to STATE_BACKGROUND, when we enter a processForeground,
        /// the thread will transition to STATE_FOREGROUND_ACQUIRE_SEMAPHORE.
        STATE_BACKGROUND_ACQUIRE_SEMAPHORE,

        /// The thread would be running in foreground mode, but is waiting for its semaphore.
        STATE_FOREGROUND_ACQUIRE_SEMAPHORE,

        /// The thread has terminated, that is, the workerThreadMain function
        /// has returned.
        STATE_TERMINATED,
    };
    
    /// The state the thread is currently in.
    State m_state;

    /// The semaphore count of this thread's hkAsyncThreadPool aware
    /// semaphore. This count is decreased on acquire and increased on
    /// release. If it reaches 0, the worker thread will suspend until it's
    /// increased to above 0 again by one of the release functions.
    /// Initially it has the value 1.
    hkUint32 m_semaphoreCount;

    /// The lock which needs to be taken when we're accessing this WorkerThread.
    hkCriticalSection m_lock;

    /// True if the thread is currently waiting for the m_resumeSemaphore
    /// (or at least, almost, it's set to true before the call to
    /// m_resumeSemaphore.acquire() and set to false after it). The only
    /// function which uses it (that is, read from it, as opposed to write
    /// to it) is getThreadPriority(), which is not used by the async thread
    /// pool class itself, so this value is not important to the functioning
    /// of the thread pool.
    hkBool m_suspended;

    /// For threads with background priority SUSPENDED, this semaphore is
    /// held while it's suspended. For other threads, it's used while going
    /// from foreground to background mode (it's held by the worker thread,
    /// when it has notified the main thread it wants to go to background
    /// mode and released by the main thread when it has done so).
    hkSemaphore m_resumeSemaphore;

    hkMonitorStream* m_monitorStream;

#if HK_INSTRUMENT_TEND_TO_THREADPOOL
    hkRefPtr<hkAsyncThreadPoolInstrumenter> m_instrumenter;
#endif
};

hkAsyncThreadPool::Cinfo::Cinfo()
    : m_hardwareThreadBinding(HardwareThreadBinding::UNSPECIFIED)
    , m_numThreads(1)
    , m_stackSize(hkThread::HK_THREAD_DEFAULT_STACKSIZE)
    , m_threadName("HavokWorkerThread")
    , m_normalPriority(0)
    , m_lowPriority(-1)
    , m_timerBufferPerThreadAllocation(0)
{
}

HK_INLINE static int calcHardwareThreadId( int numCores, int numThreadsPerCore, int threadId )
{
#if defined(HK_PLATFORM_WIN32)
    //X360: { 2,4,1,3,5, 0, 2,4,.. }
    int procGroup = ( threadId % numCores ) * numThreadsPerCore;
    return procGroup + ( numThreadsPerCore > 1 ? ( ( threadId / numCores ) % numThreadsPerCore ) : 0 );
#else
    // Default affinity is trying to assign a physical core to each logical thread
    return threadId;
#endif
}

hkAsyncThreadPool::hkAsyncThreadPool(const Cinfo& cinfo)
    : m_normalPriority(cinfo.m_normalPriority)
    , m_lowPriority(cinfo.m_lowPriority)
    , m_threadsRunning(false)
    , m_workerThreads(cinfo.m_numThreads)
    , m_timerBufferPerThreadAllocation(cinfo.m_timerBufferPerThreadAllocation)
{
    HK_ASSERT(0x17543071, cinfo.m_numThreads >= 1, "cinfo.m_numThreads must at least be 1.");
    HK_ASSERT(0x5668c5, cinfo.m_threadBackgroundPriorities.getSize() == cinfo.m_numThreads,
        "A background priority must be specified for each thread");

    m_ticksPerSecond = hkSystemClock::getTicksPerSecond();

    // Get cores info for thread affinity
    int numCores = hkHardwareInfo::getNumHardwareThreads();

    for(int i = 0; i < cinfo.m_numThreads; i++)
    {
        WorkerThread& workerThread = m_workerThreads[i];

        workerThread.m_threadPool = this;
        workerThread.m_threadIndex = i;

        if(cinfo.m_hardwareThreadMasksOrIds.getSize() > 0)
        {
            HK_ASSERT(0x343d3fc1, cinfo.m_hardwareThreadMasksOrIds.getSize() >= cinfo.m_numThreads,
                "If you initialize hardware thread ids, you must give an ID to all threads.");
            HK_ASSERT(0xa890fe2, cinfo.m_hardwareThreadBinding != HardwareThreadBinding::UNSPECIFIED,
                "You've supplied preferred HW thread IDs or affinity masks, but haven't specified how they should be interpretted.");
            workerThread.m_hardwareThreadBinding = cinfo.m_hardwareThreadBinding;
            workerThread.m_hardwareThreadMasksOrId = cinfo.m_hardwareThreadMasksOrIds[i];
        }
        else
        {
            HK_ASSERT(0xbf9937a, cinfo.m_hardwareThreadBinding == HardwareThreadBinding::UNSPECIFIED,
                "You've requested HW thread affinity/preference but haven't supplied masks/IDs.");
            // Unspecified per-thread masks force SCHEDULER_HINT usage
            workerThread.m_hardwareThreadBinding = HardwareThreadBinding::SCHEDULER_HINT;
            workerThread.m_hardwareThreadMasksOrId = calcHardwareThreadId(numCores, 1, i);
        }

        workerThread.m_backgroundPriority = cinfo.m_threadBackgroundPriorities[i];

#if HK_INSTRUMENT_TEND_TO_THREADPOOL
        workerThread.m_instrumenter.setAndDontIncrementRefCount(new hkAsyncThreadPoolInstrumenter());
#endif
    }
}

hkAsyncThreadPool::~hkAsyncThreadPool()
{
#ifdef HK_DEBUG
    HK_ASSERT(0x507d6e0f, !m_threadsRunning,
        "You must explicitly terminate the threads of a hkAsyncThreadPool by "
        "calling stopThreads() before the hkAsyncThreadPool's destructor can be called.");

    for(WorkerThread& workerThread : m_workerThreads)
    {
        HK_ASSERT_NO_MSG(0xffc9537, workerThread.m_thread.getStatus() == hkThread::THREAD_NOT_STARTED ||
            workerThread.m_thread.getStatus() == hkThread::THREAD_TERMINATED);
    }
#endif
}

void hkAsyncThreadPool::startThreads(const Cinfo& cinfo)
{
    HK_ASSERT(0x4b62ac0, !m_threadsRunning, 
        "hkAsyncThreadPool::startThreads shouldn't be called more than once, "
        "unless there's a call to stopThreads between them.");

    m_toForegroundIdx.storeRelaxed(cinfo.m_numThreads);
    m_toBackgroundTicks = 0;
    m_numInForeground.storeRelaxed(m_workerThreads.getSize());
    m_toBackgroundQueue.reset(new hkAsyncThreadPoolDetail::ToBackgroundQueue(cinfo.m_numThreads));

    for(WorkerThread& workerThread : m_workerThreads)
    {
        HK_ASSERT_NO_MSG(0x2f0d45df, workerThread.m_thread.getStatus() == hkThread::THREAD_NOT_STARTED);

        // Threads start out in foreground state (since they still have normal
        // priority), but will be brought to background mode right after they
        // are created.
        workerThread.m_state = WorkerThread::STATE_FOREGROUND;
        workerThread.m_semaphoreCount = 1;
        workerThread.m_suspended = false;
        workerThread.m_monitorStream = nullptr;

        workerThread.m_thread.startThread(&threadMainForwarder, &workerThread, cinfo.m_threadName, cinfo.m_stackSize);
    }

    // Threads are created in foreground mode, but we've set m_toBackgroundTicks
    // to 0, which means the threads will time-out instantly and then this
    // driveToBackground call will get them to background mode.
    driveToBackground();

    m_threadsRunning = true;
}

void hkAsyncThreadPool::stopThreads()
{
    if(!m_threadsRunning)
    {
        return;
    }

    // Set it to (hkUint64)-1, which is the maximum value a hkUint64 can have,
    // so that m_toBackgroundTicks will never be passed, which has the effect
    // that the worker threads will stay in foreground mode.
    m_toBackgroundTicks = (hkUint64)-1;

#if HK_CONFIG_MONITORS == HK_CONFIG_MONITORS_ENABLED
    m_toForegroundStartTicks = 0;
#endif

    // Bring all threads to foreground mode.
    m_toForegroundIdx.storeRelease(0);
    m_toBackgroundQueue->reset();
    driveToForeground();

    for(WorkerThread& workerThread : m_workerThreads)
    {
        workerThread.m_thread.joinThread();
    }

    m_toBackgroundQueue.reset(nullptr);
    m_threadsRunning = false;
}

void* HK_CALL hkAsyncThreadPool::threadMainForwarder(void* threadDataAsVoid)
{
    WorkerThread* workerThread = static_cast<WorkerThread*>(threadDataAsVoid);
    workerThread->m_threadPool->threadMain(workerThread->m_threadIndex);
    return HK_NULL;
}

void hkAsyncThreadPool::threadMain(int threadIdx)
{
    hkWorkerThreadContext threadContext(1000 + threadIdx);
    WorkerThread& workerThread = m_workerThreads[threadIdx];

    {
        hkMonitorStream* monitorStream = hkMonitorStream::getInstancePtr();
        if(monitorStream)
        {
            // Allocate monitor stream buffer space for this thread (this enables timers)
            if(m_timerBufferPerThreadAllocation > 0)
            {
                monitorStream->resize(m_timerBufferPerThreadAllocation);
            }
        }

        workerThread.m_monitorStream = monitorStream;
    }

#if defined(HK_PLATFORM_IOS) || defined(HK_PLATFORM_NX)
    HK_THREAD_LOCAL_SET(s_threadPoolThreadIndex, threadIdx);
    HK_THREAD_LOCAL_SET(s_threadPool, this);
#else
    s_threadPoolThreadIndex = threadIdx;
    s_threadPool = this;
#endif

    if(workerThread.m_hardwareThreadBinding == HardwareThreadBinding::HARD_AFFINITY)
    {
        workerThread.m_thread.setThreadAffinityMask(workerThread.m_hardwareThreadMasksOrId);
    }
    else
    {
        workerThread.m_thread.setIdealProcessor(workerThread.m_hardwareThreadMasksOrId);
    }

    // If the background mode is suspended, then the actual thread priority
    // never gets updated (we only wait for a semaphore when suspending), so if
    // that's the case, we need to set the threads priority to m_normalPriority.
    // If the background priority is different, there's no need for setting the
    // priority, because it will be changed to background priority in the next
    // tendToThreadPool call anyway.
    if(workerThread.m_backgroundPriority == ThreadPriority::SUSPENDED)
    {
        workerThread.m_thread.setPriority(m_normalPriority);
    }

    // This call is needed to do our part of driveToBackground call in the
    // hkAsyncThreadPool's constructor.
    tendToThreadPool(0xb90a045c);

    workerThreadMain(threadIdx);

    workerThread.m_lock.enter();
    switch(workerThread.m_state)
    {
        HK_NO_DEFAULT_CASE(0x30db7f8f, "This code should only be reachable when "
            "the thread has a state which indicates it's actually running.");

    case WorkerThread::STATE_BACKGROUND:
        workerThread.m_state = WorkerThread::STATE_TERMINATED;
        workerThread.m_lock.leave();
        break;

    case WorkerThread::STATE_FOREGROUND:
        {
            workerThread.m_state = WorkerThread::STATE_TERMINATED;
            workerThread.m_lock.leave();

            hkInt32 numInForeground = --m_numInForeground;
            HK_ASSERT(0x3ea7f3e, numInForeground >= 0, "numInForeground should never be negative.");
            if(numInForeground == 0)
            {
                m_toBackgroundQueue->pushAllSemaphoresToBackground();
            }

            m_toBackgroundQueue->pushToBackground(threadIdx);
        }
        break;
    }
}

void hkAsyncThreadPool::driveToForeground()
{
    hkAtomic::readWriteBarrier();

    // We collaboratively loop over all the threads and update the priority of
    // each one. m_toForegroundIdx is the index of the next thread to be
    // updated. For any given value of m_toForegroundIdx, there's exactly one
    // thread which increments m_toForegroundIdx from that value to the next and
    // it will be this thread (the one which did the increment) which is
    // responsible for updating the priority of the thread with this index.
    hkUint32 toForegroundIdx = m_toForegroundIdx.loadRelaxed();
    while(toForegroundIdx != (hkUint32)m_workerThreads.getSize())
    {
        if(m_toForegroundIdx.compareAndSwap(toForegroundIdx, toForegroundIdx + 1))
        {
            // toForegroundIdx is the index of the thread we have to bring to
            // foreground mode.

            WorkerThread& toForegroundThread = m_workerThreads[toForegroundIdx];
            toForegroundThread.m_lock.enter();

            switch(toForegroundThread.m_state)
            {
                HK_NO_DEFAULT_CASE(0x440d81ba,
                    "States can only be in STATE_BACKGROUND, STATE_BACKGROUND_ACQUIRE_SEMAPHORE "
                    "or STATE_TERMIANTED when trying to bring them to foreground mode.");

            case WorkerThread::STATE_BACKGROUND:
                toForegroundThread.m_state = WorkerThread::STATE_FOREGROUND;
                m_numInForeground++;

                if(toForegroundThread.m_backgroundPriority == ThreadPriority::LOW)
                {
                    toForegroundThread.m_thread.setPriority(m_normalPriority);
                }
                else if(toForegroundThread.m_backgroundPriority == ThreadPriority::SUSPENDED)
                {
                    // Suspended threads still have priority 0, but are waiting for
                    // the m_resumeSemaphore, so releasing that semaphore will bring
                    // them to foreground mode.
                    toForegroundThread.m_resumeSemaphore.release();
                }
                break;

            case WorkerThread::STATE_BACKGROUND_ACQUIRE_SEMAPHORE:
                toForegroundThread.m_state = WorkerThread::STATE_FOREGROUND_ACQUIRE_SEMAPHORE;

                if(toForegroundThread.m_backgroundPriority == ThreadPriority::LOW)
                {
                    toForegroundThread.m_thread.setPriority(m_normalPriority);
                }
                break;

            case WorkerThread::STATE_TERMINATED:
                // The thread isn't running anymore, so just send the message to
                // driveToBackground immediately, so that it still gets the full
                // number of messages.
                m_toBackgroundQueue->pushToBackground(toForegroundIdx);
                break;
            }

            toForegroundThread.m_lock.leave();

#if HK_INSTRUMENT_TEND_TO_THREADPOOL
            if(toForegroundThread.m_instrumenter)
            {
                toForegroundThread.m_instrumenter->beginInstrument();
            }
#endif

            toForegroundIdx++;

#if HK_CONFIG_MONITORS == HK_CONFIG_MONITORS_ENABLED
            if(m_toForegroundStartTicks != 0)
            {
                hkUint64 newTicks = hkSystemClock::getTickCounter();
                hkUint64 curTicks = m_toForegroundEndTicks;
                while(newTicks > curTicks)
                {
                    if(hkAtomic::compareAndSwapFull(&m_toForegroundEndTicks, &curTicks, newTicks))
                    {
                        break;
                    }
                }
            }
#endif
        }
    }
}

void hkAsyncThreadPool::driveToBackground()
{
    int numInBackground = 0;
    while(numInBackground != m_workerThreads.getSize())
    {
        auto popResult = m_toBackgroundQueue->pop();
        if(popResult.isAllSemaphoresToBackground())
        {
            // Bring all worker threads which are in state
            // WorkerThread::STATE_FOREGROUND_ACQUIRE_SEMAPHORE to WorkerThread::STATE_BACKGROUND_ACQUIRE_SEMAPHORE.
            for(WorkerThread& workerThread : m_workerThreads)
            {
                if(workerThread.m_state == WorkerThread::STATE_FOREGROUND_ACQUIRE_SEMAPHORE)
                {
                    workerThread.m_lock.enter();
                    if(workerThread.m_state == WorkerThread::STATE_FOREGROUND_ACQUIRE_SEMAPHORE)
                    {
                        if(workerThread.m_backgroundPriority == ThreadPriority::LOW)
                        {
                            workerThread.m_thread.setPriority(m_lowPriority);
                        }

                        workerThread.m_state = WorkerThread::STATE_BACKGROUND_ACQUIRE_SEMAPHORE;
                        numInBackground++;
                    }
                    workerThread.m_lock.leave();
                }
            }
        }
        else
        {
            // Bring the thread with threadIndex to background mode.

            int threadIndex = popResult.getThreadIndex();
            WorkerThread& workerThread = m_workerThreads[threadIndex];

            workerThread.m_lock.enter();
            switch(workerThread.m_state)
            {
                HK_NO_DEFAULT_CASE(0x24ef6ee0, 
                    "Only worker threads which are in STATE_TO_BACKGROUND or "
                    "in STATE_TERMINATED may be pushed onto the 'to background' queue.");

            case WorkerThread::STATE_TO_BACKGROUND:
                {
                    workerThread.m_state = WorkerThread::STATE_BACKGROUND;

                    if(workerThread.m_backgroundPriority == ThreadPriority::NORMAL)
                    {
                        workerThread.m_resumeSemaphore.release();
                    }
                    else if(workerThread.m_backgroundPriority == ThreadPriority::LOW)
                    {
                        workerThread.m_thread.setPriority(m_lowPriority);
                        workerThread.m_resumeSemaphore.release();
                    }

                    workerThread.m_lock.leave();

                    hkInt32 numInForeground = --m_numInForeground;
                    numInBackground++;

                    HK_ASSERT(0x60c05e3c, numInForeground >= 0, "m_numInForeground should never get negative.");

                    if(numInForeground == 0 && numInBackground != m_workerThreads.getSize())
                    {
                        m_toBackgroundQueue->pushAllSemaphoresToBackground();
                    }
                }
                break;

            case WorkerThread::STATE_TERMINATED:
                workerThread.m_lock.leave();
                numInBackground++;
                break;
            }
        }
    }
}

void hkAsyncThreadPool::processForeground(hkReal processingTimeSeconds)
{
    hkUint64 toBackgroundTicks = hkSystemClock::getTickCounter() + (hkUint64)(processingTimeSeconds * (hkDouble64)m_ticksPerSecond);
    processForegroundImpl(toBackgroundTicks);
}

void hkAsyncThreadPool::processForegroundIndefinitely()
{
    processForegroundImpl((hkUint64)-1);
}

void hkAsyncThreadPool::processForegroundImpl(hkUint64 toBackgroundTicks)
{
#if HK_CONFIG_MONITORS == HK_CONFIG_MONITORS_ENABLED
    m_toForegroundStartTicks = hkSystemClock::getTickCounter();
    m_toForegroundEndTicks = m_toForegroundStartTicks;
#endif

    // It's important that m_toBackgroundTicks is set before we start bringing
    // threads to foreground mode (which is triggered by setting
    // m_toForegroundIdx to 0).
    m_toBackgroundTicks = toBackgroundTicks;
    m_toForegroundIdx.storeRelease(0);
    m_toBackgroundQueue->reset();

    driveToForeground();
    driveToBackground();

#if HK_CONFIG_MONITORS == HK_CONFIG_MONITORS_ENABLED
    hkUint64 toBackgroundEndTicks = hkSystemClock::getTickCounter();
    hkUint64 toForegroundEndTicks = m_toForegroundEndTicks;

    hkUint64 toForegroundDuration = toForegroundEndTicks - m_toForegroundStartTicks;

    hkUint64 waitingDuration, toBackgroundDuration;
    if(toBackgroundEndTicks < toBackgroundTicks)
    {
        // We went to background mode sooner than we had to, which can happen
        // when cancelForeground is called, or when all worker threads are
        // suspended. In this case, we ascribe all the foreground time to
        // 'waiting' and nothing to the 'to background' phase.
        waitingDuration = toBackgroundEndTicks - toForegroundEndTicks;
        toBackgroundDuration = 0;
    }
    else
    {
        // The normal case. The 'waiting' time is the time we were supposed to
        // be waiting, the amount by which we exceed it is ascribed to the 'to
        // background' phase.
        waitingDuration = toBackgroundTicks - toForegroundEndTicks;
        toBackgroundDuration = toBackgroundEndTicks - toBackgroundTicks;
    }

    hkUint64 totalDuration = toForegroundDuration + waitingDuration + toBackgroundDuration;

    hkMonitorStream* monitorStream = hkMonitorStream::getInstancePtr();
    TIMERS_BEGIN("processForegroundImpl");
        TIMERS_TIMER("driveToForeground", toForegroundDuration);
        TIMERS_TIMER("waiting", waitingDuration);
        TIMERS_TIMER("driveToBackground", toBackgroundDuration);
    TIMERS_END(totalDuration);
#endif
}

void hkAsyncThreadPool::cancelForeground()
{
    // Simply trigger a timeout.
    m_toBackgroundTicks = 0;
}

void hkAsyncThreadPool::tendToThreadPool(hkUint32 id)
{
    int curThreadIdx = getCurrentThreadIndex();
    WorkerThread& workerThread = m_workerThreads[curThreadIdx];

    driveToForeground();

    if(workerThread.m_state == WorkerThread::STATE_FOREGROUND)
    {
        hkUint64 tickCounter = hkSystemClock::getTickCounter();

#if HK_INSTRUMENT_TEND_TO_THREADPOOL
        if(workerThread.m_instrumenter)
        {
            workerThread.m_instrumenter->newInterval(id);
        }
#endif

        if(tickCounter > m_toBackgroundTicks)
        {
            workerThread.m_lock.enter();
            if(workerThread.m_state == WorkerThread::STATE_FOREGROUND)
            {
                workerThread.m_state = WorkerThread::STATE_TO_BACKGROUND;
                workerThread.m_lock.leave();

#if HK_INSTRUMENT_TEND_TO_THREADPOOL
                if(workerThread.m_instrumenter)
                {
                    workerThread.m_instrumenter->endInstrument();
                }
#endif

                // We just set the state to STATE_TO_BACKGROUND, so we need to
                // release the m_toBackgroundSemaphore, to let the
                // driveToBackground function know that it can bring this thread
                // to background mode.
                m_toBackgroundQueue->pushToBackground(curThreadIdx);

                workerThread.m_suspended = true;
                workerThread.m_resumeSemaphore.acquire();
                workerThread.m_suspended = false;

                if(workerThread.m_backgroundPriority == ThreadPriority::SUSPENDED)
                {
                    driveToForeground();
                }
            }
            else
            {
                workerThread.m_lock.leave();
            }
        }
    }
}

hkAsyncThreadPool* hkAsyncThreadPool::getCurrentThreadPool()
{
#if defined(HK_PLATFORM_IOS) || defined(HK_PLATFORM_NX)
    return HK_THREAD_LOCAL_GET(s_threadPool);
#else
    return s_threadPool;
#endif
}

void hkAsyncThreadPool::tendToCurrentThreadPool(hkUint32 id)
{
    hkAsyncThreadPool* threadPool = getCurrentThreadPool();

    if(threadPool)
    {
        threadPool->tendToThreadPool(id);
    }
}

hkAsyncThreadPool::ThreadPriority hkAsyncThreadPool::getThreadPriority(int threadIdx) const
{
    const WorkerThread& workerThread = m_workerThreads[threadIdx];
    if(workerThread.m_suspended)
    {
        return ThreadPriority::SUSPENDED;
    }
    else
    {
        int threadPriority = workerThread.m_thread.getPriority();
        switch(threadPriority)
        {
        case 0:
            return ThreadPriority::NORMAL;

        case -1:
            return ThreadPriority::LOW;

        default:
            HK_ERROR(0x166ae2ea, threadPriority << " is an invalid priority for a thread in a hkAsyncThreadPool");
            return ThreadPriority::NORMAL;
        }
    }
}

hkThread::Status hkAsyncThreadPool::getThreadStatus(int threadIdx) const
{
    return m_workerThreads[threadIdx].m_thread.getStatus();
}

int hkAsyncThreadPool::getCurrentThreadIndex()
{
#if defined(HK_PLATFORM_IOS) || defined(HK_PLATFORM_NX)
    hkAsyncThreadPool* threadPool = HK_THREAD_LOCAL_GET(s_threadPool);
    int threadPoolThreadIndex = HK_THREAD_LOCAL_GET(s_threadPoolThreadIndex);
    return threadPool ? threadPoolThreadIndex : -1;
#else
    return s_threadPoolThreadIndex;
#endif
}

void hkAsyncThreadPool::setBackgroundPriorities(const ThreadPriority* backgroundPriorities)
{
#if HK_CONFIG_MONITORS == HK_CONFIG_MONITORS_ENABLED
    m_toForegroundStartTicks = 0;
#endif

    // Bring all threads to foreground mode, then update the
    // m_backgroundPriority and then go back to background mode again, this time
    // with the new background priority.
    m_toBackgroundTicks = (hkUint64)-1;
    m_toForegroundIdx.storeRelease(0);
    m_toBackgroundQueue->reset();
    driveToForeground();

    for(int i = 0; i < m_workerThreads.getSize(); i++)
    {
        m_workerThreads[i].m_backgroundPriority = backgroundPriorities[i];
    }

    m_toBackgroundTicks = 0;
    driveToBackground();
}

void hkAsyncThreadPool::acquireThreadSemaphore()
{
    int curThreadIdx = getCurrentThreadIndex();

    HK_ASSERT(0x7f32d324, curThreadIdx != -1 && this == getCurrentThreadPool(),
        "hkAsyncThreadPool::acquireThreadSemaphore should only be called from "
        "one of the worker threads of the thread pool.");

    WorkerThread& curThread = m_workerThreads[curThreadIdx];

    curThread.m_lock.enter();

    curThread.m_semaphoreCount--;
    if(curThread.m_semaphoreCount == 0)
    {
        switch(curThread.m_state)
        {
            HK_NO_DEFAULT_CASE(0x6963be86,
                "This code should be unreachable when the thread is in "
                "STATE_TO_BACKGROUND or one of the acquire states.");

        case WorkerThread::STATE_FOREGROUND:
            {
                curThread.m_state = WorkerThread::STATE_FOREGROUND_ACQUIRE_SEMAPHORE;
                curThread.m_lock.leave();

                hkInt32 numInForeground = --m_numInForeground;
                HK_ASSERT(0x2121fcc1, numInForeground >= 0, "numInForeground should never be negative.");
                if(numInForeground == 0)
                {
                    m_toBackgroundQueue->pushAllSemaphoresToBackground();
                }
            }
            break;

        case WorkerThread::STATE_BACKGROUND:
            curThread.m_state = WorkerThread::STATE_BACKGROUND_ACQUIRE_SEMAPHORE;
            curThread.m_lock.leave();
            break;
        }

        curThread.m_suspended = true;
        curThread.m_resumeSemaphore.acquire();
        curThread.m_suspended = false;

        HK_ASSERT(0xabef776, curThread.m_semaphoreCount > 0,
            "The m_semaphoreCount must be > 0 when the m_resumeSemaphore is released.");
    }
    else
    {
        curThread.m_lock.leave();
    }
}

void hkAsyncThreadPool::releaseThreadSemaphore(int threadIdx)
{
    WorkerThread& semaphoreThread = m_workerThreads[threadIdx];

    int callingThreadIdx = getCurrentThreadIndex();

    if(callingThreadIdx == -1)
    {
        semaphoreThread.m_lock.enter();
        releaseThreadSemaphoreImpl(semaphoreThread);
        semaphoreThread.m_lock.leave();
    }
    else
    {
        WorkerThread& callingThread = m_workerThreads[callingThreadIdx];
        hkCriticalSection* locks[2] = { &semaphoreThread.m_lock, &callingThread.m_lock };
        if(locks[0] > locks[1])
        {
            hkMath::swap(locks[0], locks[1]);
        }

        locks[0]->enter();
        locks[1]->enter();

        releaseThreadSemaphoreImpl(semaphoreThread);

        locks[0]->leave();
        locks[1]->leave();
    }
}

void hkAsyncThreadPool::releaseThreadSemaphoreImpl(WorkerThread& semaphoreThread)
{
    if(semaphoreThread.m_semaphoreCount == 0)
    {
        semaphoreThread.m_semaphoreCount = 1;

        switch(semaphoreThread.m_state)
        {
            HK_NO_DEFAULT_CASE(0x59a760c8, "This code should only be reached when the thread's state is one of the two acquire states.");

        case WorkerThread::STATE_FOREGROUND_ACQUIRE_SEMAPHORE:
            semaphoreThread.m_state = WorkerThread::STATE_FOREGROUND;
            m_numInForeground++;
            semaphoreThread.m_resumeSemaphore.release();
            break;

        case WorkerThread::STATE_BACKGROUND_ACQUIRE_SEMAPHORE:
            semaphoreThread.m_state = WorkerThread::STATE_BACKGROUND;

            if (semaphoreThread.m_backgroundPriority != ThreadPriority::SUSPENDED)
            {
                semaphoreThread.m_resumeSemaphore.release();
            }
            break;
        }
    }
    else
    {
        semaphoreThread.m_semaphoreCount++;
    }
}

void hkAsyncThreadPool::releaseAllThreadSemaphores()
{
    for(int i = 0; i < m_workerThreads.getSize(); i++)
    {
        releaseThreadSemaphore(i);
    }
}

void hkAsyncThreadPool::releaseAllWaitingForSemaphore()
{
    int curThreadIdx = hkAsyncThreadPool::getCurrentThreadIndex();
    if(curThreadIdx == -1)
    {
        // We can assume we're the main thread (the one doing the
        // processForeground calls, but we're obviously not in processForeground
        // atm, so we can assume we're currently in background mode.

        for(WorkerThread& semaphoreThread : m_workerThreads)
        {
            semaphoreThread.m_lock.enter();
            if(semaphoreThread.m_state == WorkerThread::STATE_FOREGROUND_ACQUIRE_SEMAPHORE ||
                semaphoreThread.m_state == WorkerThread::STATE_BACKGROUND_ACQUIRE_SEMAPHORE)
            {
                releaseThreadSemaphoreImpl(semaphoreThread);
            }
            semaphoreThread.m_lock.leave();
        }
    }
    else
    {
        WorkerThread& callingThread = m_workerThreads[curThreadIdx];
        for(int i = 0; i < m_workerThreads.getSize(); i++)
        {
            if(i == curThreadIdx)
            {
                continue;
            }

            WorkerThread& semaphoreThread = m_workerThreads[i];
            hkCriticalSection* locks[2] = { &semaphoreThread.m_lock, &callingThread.m_lock };
            if(locks[0] > locks[1])
            {
                hkMath::swap(locks[0], locks[1]);
            }

            locks[0]->enter();
            locks[1]->enter();

            if(semaphoreThread.m_state == WorkerThread::STATE_FOREGROUND_ACQUIRE_SEMAPHORE ||
                semaphoreThread.m_state == WorkerThread::STATE_BACKGROUND_ACQUIRE_SEMAPHORE)
            {
                releaseThreadSemaphoreImpl(semaphoreThread);
            }

            locks[0]->leave();
            locks[1]->leave();
        }
    }
}


void hkAsyncThreadPool::getInstrumentationReport(hkStringBuf& reportOut) const
{
#if HK_INSTRUMENT_TEND_TO_THREADPOOL
    hkLocalArray<const hkAsyncThreadPoolInstrumenter*> instrumenters(m_workerThreads.getSize());
    for(const WorkerThread& thread : m_workerThreads)
    {
        if(thread.m_instrumenter)
        {
            instrumenters.pushBackUnchecked(thread.m_instrumenter);
        }
    }

    hkAsyncThreadPoolInstrumenter combinedInstrumenters(instrumenters);

    combinedInstrumenters.getReport(reportOut);

#else
    HK_WARN_ONCE(0x0b15c927, "Instrumentation disabled. Define HK_INSTRUMENT_TEND_TO_THREADPOOL.");
#endif
}

hkAsyncTaskQueueThreadPool::hkAsyncTaskQueueThreadPool(const Cinfo& cinfo)
    : hkAsyncThreadPool(cinfo)
{
    hkDefaultTaskQueue::Cinfo taskQueueCinfo;
    taskQueueCinfo.m_schedulingMode = hkDefaultTaskQueue::MODE_ASYNC_THREAD_POOL;
    taskQueueCinfo.m_maxNumThreads = cinfo.m_numThreads;
    taskQueueCinfo.m_asyncThreadPool = this;

    m_taskQueue.setAndDontIncrementRefCount(new hkDefaultTaskQueue(taskQueueCinfo));
    m_taskQueue->setNumThreadsHint(cinfo.m_numThreads);

    startThreads(cinfo);
}

hkAsyncTaskQueueThreadPool::~hkAsyncTaskQueueThreadPool()
{
    stopThreads();
}

void hkAsyncTaskQueueThreadPool::stopThreads()
{
    m_taskQueue->close();
    hkAsyncThreadPool::stopThreads();
}

void hkAsyncTaskQueueThreadPool::workerThreadMain(int threadIdx)
{
    m_taskQueue->process();
}

void hkAsyncTaskQueueThreadPool::processUntilFinished(hkTaskQueue::Handle handle)
{
    processForegroundIndefinitely();
    m_taskQueue->processUntilFinished(handle);
    cancelForeground();
}

hkTaskQueue* hkAsyncTaskQueueThreadPool::getTaskQueue()
{
    return m_taskQueue;
}

void hkAsyncTaskQueueThreadPool::appendTimerData(hkArray<hkTimerData>& timerDataOut) const
{
    m_taskQueue->getTimerData(timerDataOut);
}

void hkAsyncTaskQueueThreadPool::clearTimerData()
{
    m_taskQueue->clearTimerData();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
