// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Types/hkBaseDefs.h>
#include <Common/Base/System/Stopwatch/hkSystemClock.h>
#include <Common/Base/Thread/Async/hkRunningApproxMedian.h>
#include <Common/Base/Container/Hash/hkHashMap.h>

namespace hkAsyncHeartbeat
{
    enum class HeartbeatFile : hkUint8;

    /// The class can be used to instrument the calls to several of the
    /// heartbeat functions (tick, forRange...). It will then use the resulting
    /// instrumentation data to automatically generate C++ header files which
    /// contains heartbeat counts, calibrated in such a way that the \ref
    /// heartbeat function will be called at a rate as close as possible to the
    /// specified interval duration (the default duration is 1/10th of a millisecond).
    class HK_EXPORT_COMMON Instrumenter
    {
    public:
        /// Get and set the \ref HeartbeatInstrumenter associated with the
        /// current thread. This can be nullptr.
        ///
        /// If an instrumenter is currently set, then all Heartbeat calls will
        /// be instrumented.
        static Instrumenter* getCurrentInstrumenter();
        static void setCurrentInstrumenter(Instrumenter* instrumenter);

        Instrumenter();

        /// Begin a scope. All instrumentation results will go to the current
        /// scope and all results in a certain scope will be written to the
        /// output file with the given \p fileName and a block, with a comment
        /// with the given \p name above it. Each scope should be closed using a
        /// call to \ref endScope. Note that scopes can be nested. The
        /// instrumenter copies the pointers of the given two strings, so it's
        /// up to the caller to make sure that they stay around for as long as
        /// the instrumenter exists. Usually these are string constants though,
        /// so this is automatically the case.
        void beginScope(hkUint32 id, HeartbeatFile file, const char* name);

        /// Ends the current scope. If the current scope is nested in another
        /// scope, then calling this function will make the parent scope the
        /// active one again.
        void endScope();

        /// Begins instrumenting an interval.
        void beginInterval();

        /// Ends the current interval, The duration of the interval will be used
        /// to compute the heartbeat count for the tick call with the given id.
        void endInterval(hkUint32 id);

        /// Invalidates the current interval. This has the effect that
        void invalidateInterval();

        /// Begins instrumenting a loop.
        void beginLoop();

        /// Instruments a hkAsyncHeartbeat::loopTick call. This call is needed
        /// to record the number of iterations of the current loop.
        void loopTick(int numIterations);

        /// Ends instrumenting a loop. The duration of both the interval before
        /// beginLoop, as well as the duration of running the full loop, will be
        /// used to compute the heartbeat counts for the loop with the given id.
        void endLoop(hkUint32 id);

        /// Writes the result of getHeartbeatCountsCpp to a set of files in the
        /// directory at "../../HeartbeatCounts/".
        void writeHeartbeatCountFiles() const;

        /// Generates the source for C++ headers which contain the heartbeat
        /// counts. Each entry in the output map has the file name (as passed to
        /// \ref Heartbeat::beginScope) as key and the source code as value.
        void getHeartbeatCountsCpp(hkHashMap<hkStringPtr, hkStringPtr>& res) const;

    private:
        /// The start time of the interval we're currently instrumenting.
        hkUint64 m_intervalStart;

        /// The start time and number of iterations of the loop we're currently instrumenting.
        hkUint64 m_loopStartTime;
        int m_loopNumIterations;

        /// If true, then we're currently in an interval which has been
        /// invalidated. This effectively means that the next call to
        /// endInterval should be ignored.
        hkBool m_intervalInvalidated;

        /// The scope ID of the root scope.
        static const hkUint32 ROOT_SCOPE_ID;

        /// The times for a loop (that is, a group of beginLoop, endLoop and
        /// loopTick calls).
        struct LoopTimes
        {
            /// The approx median of the time (in tick counts) of the block
            /// leading up to the beginLoop call.
            hkRunningApproxMedian<hkUint64> m_loopStartTime;

            /// The approx median of the time of a single iteration of the loop.
            hkRunningApproxMedian<hkDouble64> m_loopIterationTime;
        };

        /// The timings for a specific scope. This struct contains the data
        /// which will eventually be written to the Instrumenter's output files.
        struct Scope
        {
            HeartbeatFile m_file;
            const char* m_name;
            hkUint32 m_parentScopeId;
            hkHashMap<hkUint32, hkRunningApproxMedian<hkUint64> > m_intervalTimes;
            hkHashMap<hkUint32, LoopTimes> m_loopTimes;
        };
        hkHashMap<hkUint32, Scope> m_scopes;

        hkUint32 m_curScopeId;
        Scope* m_curScope;
    };
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
