// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Thread/Async/hkAsyncHeartbeatInstrumenter.h>
#include <Common/Base/Thread/Async/hkAsyncHeartbeat.h>
#include <Common/Base/Thread/Atomic/hkAtomicPrimitives.h>
#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>

#if defined (_MSC_VER) && (_MSC_VER == 1700 || _MSC_VER == 1800)
static __declspec(thread) hkAsyncHeartbeat::Instrumenter* s_curInstrumenter = nullptr;
#elif defined(HK_PLATFORM_IOS)
HK_THREAD_LOCAL(hkAsyncHeartbeat::Instrumenter*) s_curInstrumenter;
#else
static thread_local hkAsyncHeartbeat::Instrumenter* s_curInstrumenter = nullptr;
#endif

hkAsyncHeartbeat::Instrumenter* hkAsyncHeartbeat::Instrumenter::getCurrentInstrumenter()
{
#ifdef HK_PLATFORM_IOS
    return HK_THREAD_LOCAL_GET(s_curInstrumenter);
#else
    return s_curInstrumenter;
#endif
}

void hkAsyncHeartbeat::Instrumenter::setCurrentInstrumenter(Instrumenter* instrumenter)
{
#ifdef HK_PLATFORM_IOS
    HK_THREAD_LOCAL_SET(s_curInstrumenter, instrumenter);
#else
    s_curInstrumenter = instrumenter;
#endif

    if(instrumenter)
    {
        instrumenter->invalidateInterval();
    }
}

hkAsyncHeartbeat::Instrumenter::Instrumenter()
{
    Scope globalScope;
    globalScope.m_file = HeartbeatFile::GLOBAL;
    globalScope.m_name = "<global>";

    // Just set the parent of the root scope to be the root scope as well. This
    // scope should never be ended anyway, but if you do, then it just won't
    // have any effect.
    globalScope.m_parentScopeId = ROOT_SCOPE_ID;
    auto it = m_scopes.findOrInsertKey(ROOT_SCOPE_ID, globalScope);

    m_curScopeId = ROOT_SCOPE_ID;
    m_curScope = &m_scopes.getValue(it);
}

void hkAsyncHeartbeat::Instrumenter::beginScope(hkUint32 id, HeartbeatFile file, const char* name)
{
    Scope defScope;
    defScope.m_file = file;
    defScope.m_name = name;
    auto it = m_scopes.findOrInsertKey(id, defScope);

    m_curScope = &m_scopes.getValue(it);
    m_curScope->m_parentScopeId = m_curScopeId;

    m_curScopeId = m_scopes.getKey(it);

    beginInterval();
}

void hkAsyncHeartbeat::Instrumenter::endScope()
{
    m_curScopeId = m_curScope->m_parentScopeId;
    m_curScope = &m_scopes.getValue(m_scopes.find(m_curScopeId));
}

void hkAsyncHeartbeat::Instrumenter::beginInterval()
{
    m_intervalInvalidated = false;
    m_intervalStart = hkSystemClock::getTickCounter();
    hkAtomic::readWriteBarrier();
}

void hkAsyncHeartbeat::Instrumenter::endInterval(hkUint32 id)
{
    if(!m_intervalInvalidated)
    {
        hkAtomic::readWriteBarrier();
        hkUint64 curTime = hkSystemClock::getTickCounter();
        hkUint64 intervalTime = curTime - m_intervalStart;

        auto it = m_curScope->m_intervalTimes.findOrInsertKey(id, hkRunningApproxMedian<hkUint64>());
        auto& medianTime = m_curScope->m_intervalTimes.getValue(it);
        medianTime.addSample(intervalTime);
    }
}

void hkAsyncHeartbeat::Instrumenter::invalidateInterval()
{
    m_intervalInvalidated = true;
}

void hkAsyncHeartbeat::Instrumenter::beginLoop()
{
    hkAtomic::readWriteBarrier();

    hkUint64 curTime = hkSystemClock::getTickCounter();
    m_loopStartTime = curTime - m_intervalStart;
    m_loopNumIterations = 0;
    m_intervalStart = curTime;

    hkAtomic::readWriteBarrier();
}

void hkAsyncHeartbeat::Instrumenter::loopTick(int numIterations)
{
    m_loopNumIterations += numIterations;
}

void hkAsyncHeartbeat::Instrumenter::endLoop(hkUint32 id)
{
    if(m_loopNumIterations)
    {
        hkAtomic::readWriteBarrier();

        hkUint64 curTime = hkSystemClock::getTickCounter();
        hkDouble64 iterationTime = (hkDouble64)(curTime - m_intervalStart) / (hkDouble64)m_loopNumIterations;

        auto it = m_curScope->m_loopTimes.findOrInsertKey(id, LoopTimes());
        auto& loopTimes = m_curScope->m_loopTimes.getValue(it);
        loopTimes.m_loopStartTime.addSample(m_loopStartTime);
        loopTimes.m_loopIterationTime.addSample(iterationTime);
    }
}

void hkAsyncHeartbeat::Instrumenter::writeHeartbeatCountFiles() const
{
    hkHashMap<hkStringPtr, hkStringPtr> heartbeatCountsCpp;
    getHeartbeatCountsCpp(heartbeatCountsCpp);

    hkFileSystem& fileSystem = hkFileSystem::getInstance();
    fileSystem.mkdirRecursive("../../HeartbeatCounts/", hkFileSystem::ALLOW_EXISTING);
    for(const auto file : heartbeatCountsCpp.viewItems())
    {
        hkStringBuf path;
        path.setJoin("../../HeartbeatCounts/", file.m_0);
        hkRefPtr<hkStreamWriter> writer = fileSystem.openWriter(path);
        if(writer)
        {
            writer->write(file.m_1.cString(), file.m_1.getLength());
        }
        else
        {
            HK_ERROR(0x443347a, "Failed to open heartbeat count file " << path << ".");
        }
    }
}

namespace hkAsyncHeartbeat
{
    static const char HEARTBEAT_FILE_PROLOG[] =
        "// TKBMS v1.0 -----------------------------------------------------\n"
        "//\n"
        "// PLATFORM   : ALL\n"
        "// PRODUCT   : {}\n"
        "// VISIBILITY   : PUBLIC\n"
        "//\n"
        "// ------------------------------------------------------TKBMS v1.0\n"
        "\n"
        "#pragma once\n"
        "\n"
        "#include <Common/Base/Thread/Async/hkAsyncHeartbeat.h>\n"
        "\n"
        "// This file contains the heartbeat counts for the module it belongs to. It was\n"
        "// auto-generated using the hkAsyncHeartbeatInstrumenter class. For a\n"
        "// description of the meaning of these values, see the hkAsyncHeartbeat class\n"
        "// and the declarations of the HK_HEARTBEAT_TIME and HK_HEARTBEAT_LOOP_TIMES macros.\n"
        "//\n"
        "namespace hkAsyncHeartbeat\n"
        "{{\n";

    struct HeartbeatFileInfo
    {
        hkAsyncHeartbeat::HeartbeatFile m_heartbeatFile;
        const char* m_fileName;
        const char* m_products;
    };

    static const HeartbeatFileInfo s_heartbeatFileInfos[] =
    {
        { HeartbeatFile::GLOBAL, "hkGlobalHeartbeat.h", "COMMON" },
        { HeartbeatFile::GEOMETRY, "hkGeometryHeartbeat.h", "COMMON" },
        { HeartbeatFile::CD_INTERNAL, "hkcdInternalHeartbeat.h", "COMMON" },
        { HeartbeatFile::BASE, "hkBaseHeartbeat.h", "COMMON" },
        { HeartbeatFile::AI_NAVMESH_GENERATION, "hkaiNavMeshGenerationHeartbeat.h", "AI" },
        { HeartbeatFile::AI_NAVMESH, "hkaiNavMeshHeartbeat.h", "AI" },
        { HeartbeatFile::AI_OLD_NAVMESH_GENERATION, "hkaiOldNavMeshGenerationHeartbeat.h", "AI" },
        { HeartbeatFile::AI_PHYSICS_BRIDGE, "hkaiPhysicsBridgeHeartbeat.h", "AI+PHYSICS" },
        { HeartbeatFile::NP_INTERNAL, "hknpInternalHeartbeat.h", "PHYSICS" },
    };

    static const HeartbeatFileInfo& getHeartbeatFileInfos(hkAsyncHeartbeat::HeartbeatFile heartbeatFile);
}

static const hkAsyncHeartbeat::HeartbeatFileInfo& hkAsyncHeartbeat::getHeartbeatFileInfos(hkAsyncHeartbeat::HeartbeatFile heartbeatFile)
{
#ifdef HK_DEBUG
    for(int i = 0; i < HK_COUNT_OF(s_heartbeatFileInfos); i++)
    {
        HK_ASSERT(0x77f8f96d, s_heartbeatFileInfos[i].m_heartbeatFile == (hkAsyncHeartbeat::HeartbeatFile)i,
            "The m_heartbeatFile of each element in the s_heartbeatFileInfos "
            "array must match its index.");
    }

    HK_ASSERT(0x52ba1fd, (int)heartbeatFile >= 0 && (int)heartbeatFile < HK_COUNT_OF(s_heartbeatFileInfos),
        "No heartbeat file info found for the given heartbeat file.");
#endif

    return s_heartbeatFileInfos[(int)heartbeatFile];
}

void hkAsyncHeartbeat::Instrumenter::getHeartbeatCountsCpp(hkHashMap<hkStringPtr, hkStringPtr>& res) const
{
    if(m_scopes.getSize() == 0)
    {
        return;
    }

    hkUint64 freq = hkSystemClock::getTicksPerSecond();
    hkDouble64 scale = (hkDouble64)(0x100000000U * 20000) / freq;

    hkArray<const Scope*>::Temp scopes;
    scopes.reserve(m_scopes.getSize());
    for(const auto& scopePair : m_scopes.viewItems())
    {
        scopes.pushBackUnchecked(&scopePair.m_1);
    }

    hkSort(scopes.begin(), scopes.getSize(),
        [](const Scope* a, const Scope* b)
        {
            if(a->m_file != b->m_file)
            {
                return a->m_file < b->m_file;
            }
            else
            {
                return hkString::strCmp(a->m_name, b->m_name) < 0;
            }
        });

    const HeartbeatFileInfo* curFileInfo = &getHeartbeatFileInfos(scopes[0]->m_file);

    hkStringBuf fileText;
    fileText.format(HEARTBEAT_FILE_PROLOG, curFileInfo->m_products);
    for(const Scope* scope : scopes)
    {
        if(curFileInfo->m_heartbeatFile != scope->m_file)
        {
            fileText.append("}");

            res.insert(curFileInfo->m_fileName, fileText.cString());
            curFileInfo = &getHeartbeatFileInfos(scope->m_file);

            fileText.format(HEARTBEAT_FILE_PROLOG, curFileInfo->m_products);
        }

        fileText.appendFormat("\t// {}\n", scope->m_name);
        for(auto pair : scope->m_intervalTimes.viewItems())
        {
            hkUint64 mediansSum;
            int numMedians;
            pair.m_1.getMedian(mediansSum, numMedians);
            hkDouble64 median = (hkDouble64)mediansSum / numMedians;

            hkUint32 heartbeatCount = (hkUint32)(median * scale);
            fileText.appendFormat("\tHK_HEARTBEAT_TIME(0x{:x08}, {});\n", pair.m_0, heartbeatCount);
        }

        for(const auto& pair : scope->m_loopTimes.viewItems())
        {
            const LoopTimes& loopTimes = pair.m_1;

            hkUint64 startCount, iterationCount;

            {
                hkUint64 mediansSum;
                int numMedians;

                loopTimes.m_loopStartTime.getMedian(mediansSum, numMedians);
                hkDouble64 median = (hkDouble64)mediansSum / numMedians;
                startCount = (hkUint32)(median * scale);
            }

            {
                hkDouble64 mediansSum;
                int numMedians;

                loopTimes.m_loopIterationTime.getMedian(mediansSum, numMedians);
                hkDouble64 median = mediansSum / numMedians;
                iterationCount = (hkUint32)(median * scale);
            }

            fileText.appendFormat("\tHK_HEARTBEAT_LOOP_TIMES(0x{:x08}, {}, {});\n", pair.m_0, startCount, iterationCount);
        }

        fileText.append("\n");
    }

    fileText.append("}");
    res.insert(curFileInfo->m_fileName, fileText.cString());
}

const hkUint32 hkAsyncHeartbeat::Instrumenter::ROOT_SCOPE_ID = 0;

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
