// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Types/hkBaseDefs.h>

/// Set this to non-zero to enable heartbeat instrumentation. See the \ref
/// hkAsyncHeartbeat::Instrumenter class for more details. Instrumentation will
/// only happen when a heartbeat instrumenter is set (using \ref
/// Instrumenter::setCurrentInstrumenter), but even without an active
/// instrumenter, code will run slightly slower with this value set to a
/// non-zero value, so it should be disabled by default.
#define HK_INSTRUMENT_HEARTBEAT 0

#ifdef HK_INSTRUMENT_HEARTBEAT
#include <Common/Base/Thread/Async/hkAsyncHeartbeatInstrumenter.h>
#endif

namespace hkAsyncHeartbeat
{
    /// The main heartbeat function. Currently, this calls
    /// hkAsyncThreadPool::tendToCurrentThreadPool, but might do other things in
    /// the future as well.
    HK_EXPORT_COMMON void HK_CALL beat(hkUint32 id);

    /// The Heartbeat class can be used to make sure that the heartbeat function
    /// is called at an as regular interval as possible.
    ///
    /// The heartbeat function is a function which heartbeat aware code is
    /// expected to call often enough so that the duration between two
    /// subsequent heartbeat calls never exceeds a certain threshold (We
    /// currently have this threshold at 0.1 ms in Havok). At the same time,
    /// since this function isn't free, calling it too often would lead to too
    /// big a performance impact, so for this reason, we provide the Heartbeat
    /// class, which has many tools which allow you to fine tune the moments
    /// when the heartbeat function gets called, at as low an overhead as possible.
    ///
    /// Internally, the Heartbeat class consists of a single heartbeat counter,
    /// which is a simple integer, to which the various functions in this class
    /// add their values, and whenever this integer wraps around, we know it's
    /// time for another call to the heartbeat function.
    ///
    /// The exact values which by functions need to increment the heartbeat
    /// counter depend on how expensive the surrounding code is, but these
    /// values can be determined automatically using heartbeat instrumentation.
    class HK_EXPORT_COMMON Heartbeat
    {
    public:
        /// Initializes a new Heartbeat instance. This constructor immediately
        /// calls heartbeat() and initializes the heartbeat counter to 0.
        HK_INLINE Heartbeat(hkUint32 id);

        /// Begins a heartbeat scope. This is used by instrumentation to
        /// indicate that the instrumentation result for the upcoming heartbeat
        /// functions should go to the heartbeat count file with the given
        /// filename (use the constants below), and a block (indicated with a
        /// comment) with the given name.
        ///
        /// Scopes should be closed using a call to \ref endScope.
        ///
        /// Scopes can be nested, which means that if beginScope is called from
        /// within another scope, the previous scope becomes the parent scope
        /// and after our call to \ref endScope, this parent scope will become
        /// the active scope again.
        ///
        /// This function is only used when instrumentation is enabled and will
        /// be fully compiled away when HK_INSTRUMENT_HEARTBEAT is 0.
        HK_INLINE void beginScope(hkUint32 id, HeartbeatFile file, const char* name);

        /// Closes the current scope. If the current scope is nested in another
        /// scope, then the parent scope will become the active scope again.
        HK_INLINE void endScope();

        /// This is the main function of the Heartbeat class. It increases the
        /// internal counter by a specific count (depending on the id), and if
        /// the counter wraps around, then a call to the heartbeat function is made.
        ///
        /// The count by which the counter is increased represents the amount of
        /// time elapsed since the last call to tick (or similar functions).
        /// Specifically, it indicates that the amount of time which has elapsed
        /// since the last call to tick (or similar) is approximately
        ///
        ///   count / 2^32 * intervalDuration
        ///
        /// where intervalDuration is the duration we want between consecutive
        /// heartbeat calls.
        ///
        /// Note that these counts are always only going to be an approximation,
        /// but that's ok, it doesn't have to be too precise. It's generally
        /// better to overestimate than to underestimate the time which has elapsed.
        ///
        /// The count used is taken from HeartbeatCount<id>::s_heartbeatCount,
        /// which is 0 by default, but is usually specified specifically for the
        /// given id through a template specialization (see the \ref
        /// HK_HEARTBEAT_TIME macro). A file which contains these
        /// HK_HEARTBEAT_TIME macros is usually generated using the HeartbeatInstrumenter.
        template <hkUint32 id>
        HK_INLINE void tick();

        /// Indicates that the current code path is a "short path". Calling this
        /// function lets the instrumenter know that timings of the current code
        /// path between the last tick call and the upcoming one should not be
        /// included when computing the heartbeat count constant for this
        /// upcoming tick call.
        ///
        /// To get ideal timing, we often have to place many heartbeat.tick()
        /// calls in our code, but in practice, it's often better to use a
        /// little fewer heartbeat.tick() calls at the cost of a slight
        /// overestimation of the time which has elapsed (which means more
        /// heartbeat() calls than strictly necessary).
        ///
        /// One common scenario is a piece of code where there are multiple
        /// constant time code paths between two point in our code. To cut down
        /// on the tick calls, we only place a tick call at the beginning of the
        /// whole block and another one at the end of it. Then, to make sure we
        /// never underestimate the time elapsed, we want second tick to use the
        /// heartbeat count corresponding to the longest path through the
        /// current code block, so to achieve this, we need to let the
        /// instrumenter know that it should ignore the timings of all the code
        /// paths, except for the longest one. To let the instrumenter know that
        /// the current code path is a short path, we need to call
        /// heartbeat.shortPath() at least once on these code paths.
        ///
        /// This function is only used by the instrumenter and won't do anything
        /// if we're not instrumenting. If we're compiling with
        /// HK_INSTRUMENT_HEAERTBEAT disabled, it will be fully optimized away.
        HK_INLINE void shortPath();

        /// \name {Loops}
        ///
        /// A loop can be handled by putting the appropriate tick() calls in the
        /// body of a normal for- or while-loop, but this can often result in
        /// quite a bit of overhead, especially if the body of the loop is
        /// small. In such a case, it's usually a better idea to use the loop
        /// interface instead. This loop interface allows you to group your
        /// loop's iterations into larger blocks of iterations, where each block
        /// requires only a single heartbeat call (\ref loopTick) to handle the
        /// heartbeat count of the whole block. The following code example
        /// illustrates this:
        ///
        /// \code{c++}
        /// int innerLoopNumIts = heartbeat.beginLoop<0x8d969560>();
        /// for(int i = 0; i < num;)
        /// {
        ///     int innerMax = hkMath::min2(i + innerLoopNumIts, num);
        ///     int numIts = innerMax - i;
        ///     for(; i < innerMax; i++)
        ///     {
        ///         body(i);
        ///     }
        ///     heartbeat.loopTick<0x8d969560>(numIts);
        /// }
        /// heartbeat.endLoop<0x8d969560>();
        /// \endcode
        ///
        /// The ideal number of iterations for the inner loop is returned by the
        /// beginLoop function, and the nice thing is that this value is
        /// determined automatically using heartbeat instrumentation.
        ///
        /// Note that it's important that all the functions (beginLoop, endLoop,
        /// loopTick) of a single loop should have the same id template parameter.
        ///
        /// Now the above code isn't very difficult, but it'd certainly add a
        /// lot more clutter compared to a simple for loop, so in practice
        /// you'll often use the convenience \ref forRange function, which would
        /// make the above code look like:

        /// \code{c++}
        /// heartbeat.forRange<0x8d969560>(0, num,
        ///     [](int i)
        ///     {
        ///         body(i);
        ///     });
        /// \endcode
        ///
        /// or even
        ///
        /// \code{c++}
        /// heartbeat.forRange<0x8d969560>(0, num, body);
        /// \endcode
        ///
        /// \{

        /// Begins a loop. This function records the heartbeat count of the code
        /// leading up to this loop (similar to \ref tick) and returns the ideal
        /// number of iterations which should be run before a call to \ref
        /// loopTick is made.
        template <hkUint32 id>
        HK_INLINE int beginLoop();

        /// Ends a loop.
        template <hkUint32 id>
        HK_INLINE void endLoop();

        /// Ends a loop, handling the trailing number of iterations. This is
        /// equivalent to calling loopTick(numIterations), followed by the endLoop().
        template <hkUint32 id>
        HK_INLINE void endLoop(int numIterations);

        /// Increments the heartbeat counter by the count for the given number
        /// of iterations of the current loop. This can only be called from
        /// within a beginLoop/endLoop block.
        template <hkUint32 id>
        HK_INLINE void loopTick(int numIterations);

        /// A convenience function which provides an easy way to write the
        /// heartbeat aware equivalent of a common for loop.
        /// \param begin The first integer in the range to iterate over.
        /// \param end One past the last integer in the range to iterate over.
        /// \param f A functor specifying the for loop's body. This functor
        ///    should take a single integer as parameter, which represents the
        ///    index of the current iteration.
        template <hkUint32 id, typename FUNC>
        HK_INLINE void forRange(int begin, int end, FUNC f);

        /// \}

        /// Forces a heartbeat. This function always calls a heartbeat and
        /// resets the internal counter to 0.
        template <hkUint32 id>
        HK_INLINE void forceHeartbeat();

    private:
        /// The heartbeat counter. Each call to tick (and the similar functions)
        /// increase this counter, and whenever it wraps around, a call to \ref
        /// heartbeat() is made.
        hkUint32 m_heartbeatCounter;

#if HK_INSTRUMENT_HEARTBEAT
        Instrumenter* m_instrumenter;
#endif
    };

    /// An class with a similar interface as Heartbeat, but which doesn't
    /// actually keep track of heartbeat counts. This is useful for implementing
    /// both a Heartbeat aware version of a function as well as one which
    /// doesn't use the heartbeat, for example as in the following code:
    ///
    /// \code{.cpp}
    /// template <typename Heartbeat>
    /// void _func(int arg, Heartbeat& heartbeat) { ... }
    ///
    /// void func(int arg, hkHeartbeat::Heartbeat& heartbeat)
    /// {
    ///    _func(arg, heartbeat);
    /// }
    ///
    /// void func(int arg)
    /// {
    ///    hkHeartbeat::NullHeartbeat heartbeat;
    ///    _func(arg, heartbeat);
    /// }
    /// \endcode
    ///
    /// hkHeartbeat::Heartbeat is designed to be as fast as as possible, but it
    /// still isn't free, in part because the heartbeat code can get in the way
    /// of the compiler performing certain optimizations. For this reason, it
    /// can be desirable to have two implementations of a certain function, one
    /// which handles the heartbeat and another one which doesn't. The above
    /// pattern is a good way of achieving this, without having to duplicate
    /// your code.
    class HK_EXPORT_COMMON NullHeartbeat
    {
    public:
        /// An empty version for \ref Heartbeat::tick.
        template <hkUint32 id>
        HK_ALWAYS_INLINE void tick() { }

        /// An empty implementation of \ref Heartbeat::forceHeartbeat.
        template <hkUint32 id>
        HK_ALWAYS_INLINE void forceHeartbeat() { }

        /// An empty implementation of \ref Heartbeat::shortPath.
        HK_ALWAYS_INLINE void shortPath() { }

        /// An empty implementation of \ref Heartbeat::beginLoop.
        template <hkUint32 id>
        HK_INLINE int beginLoop();

        /// An empty implementation of \ref Heartbeat::endLoop.
        template <hkUint32 id>
        HK_INLINE void endLoop();

        /// An empty implementation of \ref Heartbeat::endLoop.
        template <hkUint32 id>
        HK_INLINE void endLoop(int numIterations);

        /// An implemenation of \ref Heartbeat::forRange which simply compiles
        /// to a normal for loop.
        template <hkUint32 id, typename FUNC>
        HK_ALWAYS_INLINE void forRange(int begin, int end, FUNC f);
    };

    /// A scope guard which calls the given Heartbeat's beginScope and endScope
    /// when entering and leaving its scope. This does nothing (and should be
    /// fully optimized away) if HK_INSTRUMENT_HEARTBEAT is 0.
    ///
    /// It also has a constructor which takes a NullHeartbeat, which makes it
    /// easy to use with functions which take a templated heartbeat (See
    /// \ref NullHeartbeat for a description of this pattern).
    class Scope
    {
    public:
        /// Opens a scope guarded heartbeat scope. \ref See
        /// Heartbeat::beginScope for more details.
        HK_INLINE Scope(Heartbeat& heartbeat, hkUint32 id, HeartbeatFile file, const char* name);

        /// Similar to the above constructor but takes a NullHeartbeat. Note
        /// that even though NullHeartbeat contains beginScope and endScope
        /// functions as well, we know these are empty, so we don't bother
        /// calling them at all.
        HK_INLINE Scope(hkAsyncHeartbeat::NullHeartbeat& heartbeat, hkUint32 id, HeartbeatFile file, const char* name);

#if HK_INSTRUMENT_HEARTBEAT
        /// When the scope closes, call the endScope function.
        HK_INLINE ~Scope();
#endif

    private:
#if HK_INSTRUMENT_HEARTBEAT
        /// The heartbeat of this scope, or a nullptr if it was constructed
        /// using a NullHeartbeat.
        hkAsyncHeartbeat::Heartbeat* m_heartbeat;
#endif
    };

    template <hkUint32 id>
    struct HeartbeatCount
    {
        static const hkUint32 s_heartbeatCount = 0;
    };

    template <hkUint32 id>
    struct HeartbeatLoopCount
    {
        static const hkUint32 s_startCount = 0;
        static const hkUint32 s_iterationCount = 1;
    };

    enum class HeartbeatFile : hkUint8
    {
        GLOBAL,
        GEOMETRY,
        CD_INTERNAL,
        BASE,
        AI_NAVMESH_GENERATION,
        AI_NAVMESH,
        AI_OLD_NAVMESH_GENERATION,
        AI_PHYSICS_BRIDGE,
        NP_INTERNAL,
    };

    /// Use this macro to specify the heartbeat count for the tick call with the
    /// given id. It creates a specialization of the \ref HeartbeatCount
    /// templated struct above.
    ///
    /// This macro is usually used in the auto-generated header files produced
    /// by the hkAsyncHeartbeatInstrumenter.
#define HK_HEARTBEAT_TIME(id, count) \
    template <> \
    struct HeartbeatCount<id> \
    { \
        static const hkUint32 s_heartbeatCount = count; \
    };

    /// Use this macro to specify the heartbeat counts for the heartbeat aware
    /// loop functions calls with the given id (that is, the beginLoop, endLoop
    /// and loopTick). This macro creates a specialization of the \ref
    /// HeartbeatLoopCount templated struct above.
    ///
    /// This macro is usually used in the auto-generated header files produced
    /// by the hkAsyncHeartbeatInstrumenter.
#define HK_HEARTBEAT_LOOP_TIMES(id, startCount, itCount) \
    template <> \
    struct HeartbeatLoopCount<id> \
    { \
        static const hkUint32 s_startCount = startCount; \
        static const hkUint32 s_iterationCount = itCount; \
    };
}

#include <Common/Base/Thread/Async/hkAsyncHeartbeat.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
