// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/System/hkBaseSystem.h>
#include <Common/Base/Config/hkConfigVersion.h>
#include <Common/Base/DebugUtil/MultiThreadCheck/hkMultiThreadCheck.h>
#include <Common/Base/System/Io/FileSystem/hkNativeFileSystem.h>
#include <Common/Base/Object/hkSingleton.h>

#include <Common/Base/System/Error/hkDefaultError.h>
#include <Common/Base/System/Log/hkLog.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>
#include <Common/Base/System/Io/Socket/hkSocket.h>
#include <Common/Base/System/Stopwatch/hkSystemTime.h>

#ifndef HK_DYNAMIC_DLL
    #include <Common/Base/System/hkBaseSystem.hxx>
#endif

#if defined(HK_PLATFORM_MAC) || defined(HK_PLATFORM_IOS)
    #include <sys/param.h>
    #include <sys/sysctl.h>
#endif

#if defined(HK_PLATFORM_LINUX) || defined(HK_PLATFORM_TIZEN)
    #include <unistd.h> 
#endif

#if defined(HK_PLATFORM_ANDROID)
    #include <Common/Base/System/Android/hkAndroidCpuInfo.h>
    #include <android/log.h>
    #include <unistd.h> // for sysconf 
#endif

#ifdef HK_DEBUG_SLOW
#   include <Common/Base/Fwd/hkcstdio.h>
    using namespace std;
#endif

HK_EXPORT_COMMON hkBool hkBaseSystemIsInitialized;
#if defined(HK_DEBUG)
    HK_EXPORT_COMMON hkBool hkBaseSystemInitVerbose = true;
#else
    HK_EXPORT_COMMON hkBool hkBaseSystemInitVerbose; /* = false; */
#endif

#ifndef HK_PLATFORM_CTR
#define PRINTF_FUNC printf
#else
#define PRINTF_FUNC nndbgDetailPrintf
#endif

#ifdef HK_DYNAMIC_DLL
extern HK_EXPORT_COMMON const char *HK_ANIMATION_KEYCODE;
extern HK_EXPORT_COMMON const char *HK_PHYSICS_2012_KEYCODE;
extern HK_EXPORT_COMMON const char *HK_PHYSICS_KEYCODE;
extern HK_EXPORT_COMMON const char *HK_BEHAVIOR_KEYCODE;
extern HK_EXPORT_COMMON const char *HK_CLOTH_KEYCODE;
extern HK_EXPORT_COMMON const char *HK_DESTRUCTION_2012_KEYCODE;
extern HK_EXPORT_COMMON const char *HK_DESTRUCTION_KEYCODE;
extern HK_EXPORT_COMMON const char *HK_AI_KEYCODE;
extern HK_EXPORT_COMMON const char *HK_FX_KEYCODE;
#else
extern const char HK_ANIMATION_KEYCODE[];
extern const char HK_PHYSICS_2012_KEYCODE[];
extern const char HK_PHYSICS_KEYCODE[];
extern const char HK_BEHAVIOR_KEYCODE[];
extern const char HK_CLOTH_KEYCODE[];
extern const char HK_DESTRUCTION_2012_KEYCODE[];
extern const char HK_DESTRUCTION_KEYCODE[];
extern const char HK_AI_KEYCODE[];
extern const char HK_FX_KEYCODE[];
#endif // HK_DYNAMIC_DLL

namespace hkKeyCode
{
    static hkResult calculateKeyCodeInfo(KeyCodeInfo* out, const char *keycode)
    {
        hkResult result = HK_FAILURE;

        if ( (keycode != nullptr) && (keycode[0] != '\0') )
        {
            out->keycode = keycode;

            const char* keycodePtr = keycode;

            // Skip to ':'
            while (*keycodePtr && (*keycodePtr++ != ':'))
            {
            }

            // Copy to and consume '.'
            {
                const char* expirationStart = keycodePtr;
                while (
                    *keycodePtr &&
                    (*keycodePtr != '.')
                    )
                {
                    keycodePtr++;
                }
                out->expirationDate = hkStringView(expirationStart, keycodePtr);
            }

            // Copy to and consume '.'
            keycodePtr++;
            if (*keycodePtr && out->expirationDate.getSize() == 10)
            {
                const char* productStart = keycodePtr;
                while (
                    *keycodePtr &&
                    (*keycodePtr != '.')
                    )
                {
                    keycodePtr++;
                }
                out->product = hkStringView(productStart, keycodePtr);

                keycodePtr++;
                if (*keycodePtr)
                {
                    out->identifier = hkStringView(keycodePtr);

                    if (out->identifier.beginsWith("Evaluation"))
                    {
                        out->keycodeType = KeycodeType::EVALUATION;
                        while (
                            *keycodePtr &&
                            (*keycodePtr != '.')
                            )
                        {
                            keycodePtr++;
                        }
                        out->identifier = hkStringView(keycodePtr);
                    }
                    else
                    {
                        out->keycodeType = KeycodeType::CLIENT;
                    }

                    // Remove the extra quotation at the end
                    if (out->identifier.endsWith("\""))
                        out->identifier = out->identifier.rtrim(1);

                    hkLong expiryDays = 0;

                    // Calculate the total number of days from 1970 of expiration date (e.g. 2011-04-04)
                    {
                        hkInt32 date[3] = { 0, 0, 0 };
                        hkInt32 dateSectionIndex = 0;

                        for (hkUint32 characterIndex = 0;
                            (characterIndex < 10) && (out->expirationDate[characterIndex] != '.');
                            ++characterIndex)
                        {
                            if (out->expirationDate[characterIndex] == '-')
                            {
                                dateSectionIndex++;
                                characterIndex++;
                            }
                            date[dateSectionIndex] = date[dateSectionIndex] * 10 + out->expirationDate[characterIndex] - '0';
                        }

                        date[0] -= 1970;

                        // Every 4th year is a leap year (except for every 100th (except for every 400th))
                        hkInt32 cumulative[] = { 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334 };
                        expiryDays = hkLong(date[0] * 365 + (date[0] / 4) * 1 + (date[0] / 100) * -1 + (date[0] / 400) * 1);
                        expiryDays += cumulative[date[1] - 1];
                        expiryDays += date[2] - 1;
                    }

                    const hkLong currentDays = hkLong(hkGetSystemTime() / (24 * 60 * 60));

                    out->daysUntilExpired = expiryDays - currentDays;

                    result = HK_SUCCESS;
                }
            }
        }

        return result;
    }

    hkInt32 HK_CALL getKeyCodeInfo(KeyCodeInfo* outKeyCodeInfo, hkInt32 maxInfo)
    {
        const char* const keycodes[] = {
            HK_PHYSICS_2012_KEYCODE,
            HK_PHYSICS_KEYCODE,
            HK_BEHAVIOR_KEYCODE,
            HK_DESTRUCTION_2012_KEYCODE,
            HK_DESTRUCTION_KEYCODE,
            HK_CLOTH_KEYCODE,
            HK_AI_KEYCODE,
            HK_FX_KEYCODE,
            HK_NULL
        };

        hkInt32 outKeyCodeInfoIndex = 0;

        for (auto* const keycode : keycodes)
        {
            if ( calculateKeyCodeInfo(&outKeyCodeInfo[outKeyCodeInfoIndex], keycode).isSuccess() )
            {
                outKeyCodeInfoIndex++;
            }
        }

        return outKeyCodeInfoIndex;
    }
}

static void HK_CALL showHavokBuild()
{
#if defined(HK_DEBUG_SLOW)
    if( hkBaseSystemInitVerbose )
    {
        PRINTF_FUNC("------------------------------------------------------------------\n");
        PRINTF_FUNC(" Havok Version %s\n", HAVOK_SDK_VERSION_NUM_STRING);
        PRINTF_FUNC(" Base system initialized.\n");
        PRINTF_FUNC("------------------------------------------------------------------\n");

        /// keys usually not set yet in hkBase init in DLL case
    #ifndef HK_DYNAMIC_DLL
        hkKeyCode::KeyCodeInfo keycodes[hkKeyCode::MAX_KEYCODES];
        hkInt32 numKeycodes = getKeyCodeInfo(keycodes, hkKeyCode::MAX_KEYCODES);

        PRINTF_FUNC("Havok License Keys:\n");
        for (hkInt32 keycodeIndex = 0; keycodeIndex < numKeycodes; ++keycodeIndex)
        {
            hkKeyCode::KeyCodeInfo &keycode = keycodes[keycodeIndex];
            PRINTF_FUNC( " %-16s : %s (expires %s) (%s)\n",
                hkStringBuf(keycode.product).cString(),
                keycode.keycode,
                hkStringBuf(keycode.expirationDate).cString(),
                hkStringBuf(keycode.identifier).cString());
        }

        {
            bool printedHeader = false;
            for (hkInt32 keycodeIndex = 0; keycodeIndex < numKeycodes; ++keycodeIndex)
            {
                hkKeyCode::KeyCodeInfo &keycode = keycodes[keycodeIndex];
                if ((keycode.keycodeType == hkKeyCode::EVALUATION && keycode.daysUntilExpired < 5) ||
                    (keycode.keycodeType == hkKeyCode::CLIENT && keycode.daysUntilExpired < 15))
                {
                    if(!printedHeader)
                    {
                        PRINTF_FUNC("Havok License Keys Expired or Expiring Soon:\n");
                        printedHeader = true;
                    }

                    if (keycode.daysUntilExpired < 0)
                    {
                        PRINTF_FUNC(" %-16s : Expired %d days ago\n",
                            hkStringBuf(keycode.product).cString(),
                            static_cast<hkInt32>( hkMath::abs(keycode.daysUntilExpired) ));
                    }
                    else
                    {
                        PRINTF_FUNC(" %-16s : Expires in less than %d days\n",
                            hkStringBuf(keycode.product).cString(),
                            static_cast<hkInt32>(keycode.daysUntilExpired));
                    }
                }
            }
        }

        PRINTF_FUNC("------------------------------------------------------------------\n");
    #endif
    }
#endif
}

namespace
{
    struct ErrorReport : public hkReferencedObject
    {
        HK_DECLARE_CLASS(ErrorReport, New, Singleton);
        ErrorReport(hkErrorReportFunction f, _In_opt_ void* o) : m_func(f), m_object(o) {}
        void print(_In_z_ const char* s) { if (m_func) (*m_func)(s, m_object); }
        hkErrorReportFunction m_func;
        void* m_object;
    };

    HK_SINGLETON_MANUAL_IMPLEMENTATION(ErrorReport, HK_NULL, hkSingletonUtil::quit<ErrorReport>);
}

void hkBaseSystem::error(_In_z_ const char* s, _Inout_opt_ void*)
{
    ErrorReport::getInstance().print(s);
}

#include <Common/Base/Config/hkProductFeatures.h>

hkResult hkBaseSystem::InitNode::init() const
{
    if (m_initFunction != HK_NULL)
    {
        return m_initFunction(m_arg);
    }
    return HK_SUCCESS;
}

hkResult hkBaseSystem::InitNode::quit() const
{
    if (m_quitFunction != HK_NULL)
    {
        return m_quitFunction(m_arg);
    }
    return HK_SUCCESS;
}


// We keep a list of explicitly registered entries separately from the
// ones which we have via static constructors. This list typically
// has initializers for DLL etc on it.
// Keeping these separate allows everything in common to be initialized
// before initializing the DLLs.
static hkBaseSystem::InitList s_initList("CommonInitNode");

void hkBaseSystem::registerInitList(_Inout_ hkBaseSystem::InitList* single)
{
    HK_ASSERT_NO_MSG(0x4415157f, single->m_next == HK_NULL);
    HK_ASSERT_NO_MSG(0xfe23f76, isInitialized() == false);

    // append to list
    if( hkBaseSystem::InitNode* p = &s_initList )
    {
        for(; p->m_next; p = p->m_next)
        {
        }
        p->m_next = single;
    }
}

void hkBaseSystem::unregisterInitList(_Inout_ hkBaseSystem::InitList* single)
{
    HK_ASSERT_NO_MSG(0x240409cb, isInitialized() == false);
    for(hkBaseSystem::InitNode* cur = &s_initList;
        cur;
        cur = cur->m_next)
    {
        if(cur->m_next == single)
        {
            cur->m_next = cur->m_next->m_next;
            cur->m_next = HK_NULL;
            return;
        }
    }

    HK_ASSERT_NO_MSG(0x7b4fdfec, 0); //not on list?!
}


hkResult HK_CALL hkBaseSystem::initNodes(_Inout_ hkBaseSystem::InitNode** addrHead)
{
    hkBaseSystem::InitNode** ptrToCur = addrHead;
    hkBaseSystem::InitNode* cur = *ptrToCur;

    hkArray<hkBaseSystem::InitNode*>::Temp again;

    while(cur)
    {
        if( cur->init().isSuccess() )
        {
            ptrToCur = &cur->m_next;
            cur = cur->m_next;
        }
        else
        {
            // If an init function returns HK_FAILURE it means
            // that the object was not ready to be constructed (perhaps
            // it depends on another entry?)
            // Remove it from the list and save it for later
            again.pushBack(cur);
            cur = cur->m_next;
            (*ptrToCur)->m_next = HK_NULL;
            *ptrToCur = cur;
        }
    }

    // Go through the deferred list.
    while(again.getSize())
    {
        int origSize = again.getSize();
        for(int i = origSize - 1; i >= 0; --i)
        {
            cur = again[i];
            HK_ASSERT_NO_MSG(0x491ec52a, cur->m_next == HK_NULL);

            if( cur->init().isSuccess() )
            {
                // succeeded, put it back on the global list.
                *ptrToCur = cur;
                ptrToCur = &cur->m_next;
                again.removeAt(i);
            }
        }
        HK_ASSERT(0x14db3060, again.getSize() < origSize, "No progress made during InitNode construction. Could be a dependency cycle or failed initialization.");
    }
    return HK_SUCCESS;
}

hkResult HK_CALL hkBaseSystem::quitNodes(_Inout_ hkBaseSystem::InitNode** head)
{
    hkArray<hkBaseSystem::InitNode*> toQuit;
    for(hkBaseSystem::InitNode* p = *head; p; p = p->m_next)
    {
        toQuit.pushBack(p);
    }
    for(int i = toQuit.getSize() - 1; i >= 0; i -= 1)
    {
        InitNode* c = toQuit[i];
        c->quit();
    }
    return HK_SUCCESS;
}

static void s_moveToFront(_Inout_ hkBaseSystem::InitNode* newHead)
{
    if(hkBaseSystem::InitNode::s_listHead == newHead)
    {
        return;
    }
    // find and unlink
    for(hkBaseSystem::InitNode* p = hkBaseSystem::InitNode::s_listHead; p; p = p->m_next)
    {
        if(p->m_next == newHead)
        {
            p->m_next = p->m_next->m_next;
            newHead->m_next = hkBaseSystem::InitNode::s_listHead;
            hkBaseSystem::InitNode::s_listHead = newHead;
            return;
        }
    }
}

#include <Common/Base/System/Stopwatch/hkSystemClock.h>

//  Initialize Havok's subsystems.
hkResult HK_CALL hkBaseSystem::init(_In_ hkMemoryRouter* memoryRouter, _In_opt_ hkErrorReportFunction errorReportFunction, _Inout_opt_ void* errorReportObject)
{
    if(hkBaseSystemIsInitialized==false)
    {
        hkSystemClock::getTicksPerSecond(); // workaround COM-3488
        initThread( memoryRouter );

        s_moveToFront(&hkLog::Detail::getGlobalInitNode());
        s_moveToFront(&hkError::singletonEntry);
        s_moveToFront(&hkFileSystem::singletonEntry);
        s_moveToFront(&ErrorReport::singletonEntry);

        ErrorReport::replaceInstance(new ErrorReport(errorReportFunction, errorReportObject));
        for(hkBaseSystem::InitNode* p = &s_initList; p; p = p->m_next)
        {
            p->init();
        }

        // hkProductFeatures.cxx must be included in order to register product features
        hkProductFeatures::initialize();

        showHavokBuild();
        HK_ON_DEBUG_MULTI_THREADING( hkMultiThreadCheck::staticInit(&memoryRouter->heap()));

        HK_CHECK_FLUSH_DENORMALS();

        hkBaseSystemIsInitialized = true;
    }

    return HK_SUCCESS;
}

#ifdef HK_PLATFORM_WIIU
#include <cafe/os.h>
#endif


hkResult hkBaseSystem::initThread(_In_ hkMemoryRouter* memoryRouter )
{
#ifndef HK_PLATFORM_ANDROID
    // Apparently Android doesn't initialize the thread-local variables to 0, so this might be non-null even the first time.
    if ( hkMemoryRouter::getInstancePtr() )
    {
        HK_ASSERT(0x1740f0d2, false, "hkBaseSystem::initThread() was called after thread was already initialized. Don't call this on the main thread." );
    }
#endif

    hkMemoryRouter::replaceInstance( memoryRouter );
    hkMonitorStream::init();
    return HK_SUCCESS;
}

hkResult hkBaseSystem::quitThread()
{
    if( hkMonitorStream* m = hkMonitorStream::getInstancePtr() )
    {
        m->quit();
    }
    if( hkMemoryRouter* a = hkMemoryRouter::getInstancePtr())
    {
        a->replaceInstance( HK_NULL );
    }
    return HK_SUCCESS;
}

//  Quit the subsystems. It is safe to call multiple times.
hkResult HK_CALL hkBaseSystem::quit()
{
    hkResult res = HK_SUCCESS;

    if(hkBaseSystemIsInitialized )
    {
        HK_ON_DEBUG_MULTI_THREADING( hkMultiThreadCheck::staticQuit() );

        InitNode* dummy = &s_initList;
        quitNodes(&dummy);

        // Be nice to the network and shut it down gracefully (if we used it and init'd it)
        if ( hkSocket::s_platformNetInitialized && hkSocket::s_platformNetQuit )
        {
            hkSocket::s_platformNetQuit();
            hkSocket::s_platformNetInitialized = false;
        }

        quitThread();
        hkBaseSystemIsInitialized = false;
    }
    return res;
}

hkBool HK_CALL hkBaseSystem::isInitialized()
{
    return hkBaseSystemIsInitialized;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
