// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM       : ALL
// PRODUCT        : COMMON
// VISIBILITY     : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Algorithm/Sort/hkSort.h>

/// Utility class to compute a windowed average
/// Also capable of computing total average


class hkWindowedAverage
{
public:

    HK_DECLARE_CLASS( hkWindowedAverage, New );

    /// Constructor
    hkWindowedAverage( int windowSize = 0 )
    {
        reset( windowSize );
    }
    hkWindowedAverage( hkReal timeWindow, hkReal timestep )
    {
        reset( ( int ) ( timeWindow / timestep ) );
    }

    /// Push a new value into the windowed average
    void pushNewValue( hkReal value )
    {
        int mod = m_numSamples % m_windowSize;

        m_numSamples++;

        if ( m_numSamples <= m_windowSize )
        {
            m_samples.setSize( m_numSamples );
        }

        m_samples[ mod ] = value;
        m_total += value;
        m_lastSample = value;
    }

    /// \return The last time pushed (or -1.0f if none pushed)
    hkReal getLastTime() const
    {
        if( m_numSamples == 0)
        {
            return -1.0f;
        }
        return m_lastSample;
    }

    /// \return The mean over the last min( getWindowSize(), getNumSamples() ) samples
    hkReal getWindowedMean() const
    {
        hkReal avg = 0.0f;
        const int size = m_samples.getSize();

        for( int i = 0; i < size; i++ )
        {
            avg += m_samples[i];
        }

        return avg / hkReal( size );
    }

    /// \return The median over the last min( getWindowSize(), getNumSamples() ) samples
    hkReal getWindowedMedian() const
    {
        const int size = m_samples.getSize();

        // This operation reorders m_samples. Due to this, no code should rely on the ordering of m_samples
        hkSort( ( hkReal* )m_samples.begin(), size );
        return 0.5f * ( m_samples[ size / 2 ] + m_samples[ ( size - 1 ) / 2 ] );
    }

    /// \return The standard deviation over the last min( getWindowSize(), getNumSamples() ) samples
    hkReal getWindowedStandardDeviation() const
    {
        const int size = m_samples.getSize();

        const hkReal mean = getWindowedMean();

        hkReal acc = 0.0f;

        for( int i = 0; i < size; i++ )
        {
            hkReal diff = ( m_samples[ i ] - mean );
            acc += diff * diff;
        }

        return hkMath::sqrt( 1.0f / hkReal( m_samples.getSize() - 1 ) * acc );
    }


    /// \return The max over the last min( getWindowSize(), getNumSamples() ) samples
    hkReal getWindowedMax() const
    {
        hkReal val = -HK_REAL_MAX;

        for ( int i = 0; i < m_samples.getSize(); i++ )
        {
            if ( m_samples[ i ] > val )
            {
                val = m_samples[ i ];
            }
        }

        return val;
    }

    hkReal getWindowedMin() const
    {
        hkReal val = HK_REAL_MAX;

        for ( int i = 0; i < m_samples.getSize(); i++ )
        {
            if ( m_samples[ i ] < val )
            {
                val = m_samples[ i ];
            }
        }

        return val;
    }

    /// \return The mean over ALL samples
    hkReal getTotalMean() const
    {
        return m_numSamples ? m_total / hkReal( m_numSamples ) : 0.0f;
    }

    /// \return The size of the sample window
    hkUint16 getWindowSize() const
    {
        return m_windowSize;
    }

    /// \return The total number of all samples taken
    int getNumSamples() const
    {
        return m_numSamples;
    }

    /// reset the windowed average, optionally providing a new window size
    void reset( int newWindowSize = -1 )
    {
        m_windowSize = ( newWindowSize == -1 ) ? m_windowSize : hkUint16( newWindowSize );
        m_numSamples = 0;
        m_total = 0.0f;
        m_samples.clear();
        m_samples.reserve( m_windowSize );
    }

private:

    hkArray< hkReal > m_samples;

    int m_numSamples; ///< Number of samples taken so far
    hkUint16 m_windowSize; ///< Size of the sample window
    hkReal m_total; ///< The total over ALL samples
    hkReal m_lastSample; ///< m_samples does not guarantee ordering, need to store last sample
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
