// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

class hkOstream;

    /// An object which can generate stack traces.
    /// Some platforms may also be able to associate addresses to
    /// function and source file information.
class HK_EXPORT_COMMON hkStackTracer
{
    public:

        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE_CLASS, hkStackTracer);

            /// Stores multiple call stacks in a tree to save space.
            /// Call stacks are inserted and return a TraceId which can be
            /// later used to retrieve the original call stack.
            /// For this to work optimally, the full call stack should be used
            /// so that identical prefixes can be merged and the tree does not
            /// become multi-rooted.
        class HK_EXPORT_COMMON CallTree
        {
            public:
                HK_DECLARE_PLACEMENT_ALLOCATOR();

                struct Node
                {
                    HK_DECLARE_CLASS(Node, New, Pod);
                    Node() : m_value(0), m_parent(-1), m_firstChild(-1), m_next(-1), m_usageCount(0) {}
                    hkUlong m_value;
                    int m_parent;
                    int m_firstChild;
                    int m_next;
                        /// The *external* usage count.
                    int m_usageCount;
                };

                typedef int TraceId;

                CallTree(_In_opt_ hkMemoryAllocator* allocator = HK_NULL) : m_allocator(allocator), m_rootNode(-1), m_firstFreeNode(-1) {}
                    /// Copy Ctor
                CallTree(const CallTree& rhs);

                    /// Assignment
                void operator=(const CallTree& rhs);

                ~CallTree();

                void init(_In_ hkMemoryAllocator* a) { m_allocator = a; }
                void quit();

                    /// Get a stack trace and insert it into the given tree.
                TraceId insertCallStack( hkStackTracer& tracer );

                    /// Add a call stack to the tree and get its id.
                TraceId insertCallStack(_In_reads_(numAddrs) const hkUlong* addrs, int numAddrs );

                    /// The identified callstack need no longer be stored.
                void releaseCallStack( TraceId id );

                    /// Retrieve a call stack from an id.
                int getCallStack( TraceId id, _Out_writes_(maxAddrs) hkUlong* addrs, int maxAddrs ) const;
                    /// Return the amount of addresses in the full call stack
                int getCallStackSize(TraceId id) const;

                    /// Get traces - the ids map into the arrays. The parents hold whats the parent for the current index. The parent is -1 if its the root
                void getTraces(hkArray<hkUlong>& addrs, hkArray<int>& parents) const;

                    /// Dump a trace from an id
                void dumpTrace(TraceId id, hkOstream& stream) const;

                    /// Read only access for traversal
                const Node& node(int i) const { return m_nodes[i]; }

                    /// Swap
                void swap(CallTree& rhs);

                    /// Get the nodes
                const hkArrayBase<Node>& getNodes() const { return m_nodes; }

                hkBool isEmpty() const;

            protected:
                    /// Get a free node or allocate one if none are free.
                int getFreeNode();

                    /// Check that the node is still required in the call tree.
                void removeNodeIfUnused( int id );

            protected:

                hkArrayBase<Node> m_nodes;
                hkMemoryAllocator* m_allocator;
                    /// The index of the root node or -1 if there is no root node yet.
                int m_rootNode;
                    /// Start of a stack of free nodes, linked by m_next links.
                int m_firstFreeNode;
        };

        hkStackTracer();

        virtual ~hkStackTracer();

        /// The type of the callback given to dumpStackTrace.
        typedef void (HK_CALL *printFunc)(const char*, void* context);

            /// Print the stack trace with pfunc.
            /// pfunc may be called multiple times.
            /// The output format is platform specific.
        void dumpStackTrace(_In_reads_(numtrace) const hkUlong* trace, int numtrace, printFunc pfunc, _Inout_opt_ void* context = HK_NULL) const;

            /// Write at most maxtrace stack entries into 'trace'.
            /// Return the number of entries written.
            /// origin is a platform specific pointer with data regarding where to start tracing. If none is given the calling function is used as starting point.
        int getStackTrace(_Out_writes_(maxtrace) hkUlong* trace, int maxtrace, int framesToSkip = 0, _Inout_opt_ void* origin = HK_NULL);

            /// If you dynamically load DLLs then you will need to refresh the symbols
            /// so that the stack tracer can see them. You can optionall restrict the load to
            /// just given dll
        void refreshSymbols(_In_opt_z_ const char* dllFile = HK_NULL);

            /// Print the module information with pfunc.
            /// pfunc may be called multiple times.
            /// The output format is platform specific.
        void getModuleInfo( printFunc pfunc, _Inout_opt_ void* context=HK_NULL ) const;

        /// Gets an opaque pointer to the singleton implementation.
        static void* getImplementation();

        /// Replace the singleton implementation with the provided pointer.
        static void replaceImplementation(_In_opt_ void* newInstance);

        /// Windows-specific, sets whether hkStackTracer initialization and cleanup must call SymInitialize.
        /// By default, SymInitialize will be called on the first call to one of the methods of
        /// hkStackTracer. setNeedsSymInitialize must be called with enabled = false *before* the first call
        /// to any of the hkStackTracer methods in order to prevent this behavior.
        static void setNeedsSymInitialize(bool enabled);
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
