// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

// Simple case - to add logging to a single cpp file
// ---------
// In the .cpp
// #define DEBUG_LOG_IDENTIFIER "name of the logger"
// #include <Common/Base/System/Log/hkLog.hxx>
//
// This creates a file-scope static hkLog::RegisteredOrigin variable s_debugLog.
//
// Logs are usually enabled by wildcard matching on the names. Thus, choose names
// with this in mind. For instance all serialization logs have "s11n" in them e.g. "s11n.TypeReader"

// Logging in several files
// ---------
// Log names need not be unique so if several logs have the same name, they should effectively be merged.
//
// If you really need to share the same object, do not define DEBUG_LOG_IDENTIFIER.
// Instead, assuming you have a variable myLogInstance of type hkLog::RegisteredOrigin,
// #define DEBUG_LOG_OBJECT myLogInstance
// #include <Common/Base/System/Log/hkLog.hxx>
// and logging macros will use myLogInstance.

#include <Common/Base/System/Log/hkLog.h>


#if defined(DEBUG_LOG_IDENTIFIER)
    #ifdef DEBUG_LOG_OBJECT
        #undef DEBUG_LOG_OBJECT
    #endif
    #ifdef HK_UNITY_BUILD
        #define DEBUG_LOG_OBJECT HK_PREPROCESSOR_JOIN_TOKEN(s_debugLog_, HK_UNITY_ID)
    #else
        #define DEBUG_LOG_OBJECT s_debugLog
    #endif
    #if defined(DEBUG_LOG_DEFAULT_LEVEL)
        static hkLog::RegisteredOrigin DEBUG_LOG_OBJECT(DEBUG_LOG_IDENTIFIER, hkLog::Level::DEBUG_LOG_DEFAULT_LEVEL);
    #else
        static hkLog::RegisteredOrigin DEBUG_LOG_OBJECT(DEBUG_LOG_IDENTIFIER);
    #endif
#endif

#if defined(DEBUG_LOG_OBJECT)
    // Macros beginning with Log_ (no hk prefix) are shortcuts for the common use case of
    // * using a predefined log object (DEBUG_LOG_OBJECT)

    #define Log_Error(...) hkLog_Error(DEBUG_LOG_OBJECT, __VA_ARGS__)
    #define Log_Warning(...) hkLog_Warning(DEBUG_LOG_OBJECT, __VA_ARGS__)
        /// See hkLog_Auto (lazy block scope)
    #define Log_Auto(...) hkLog_Auto(DEBUG_LOG_OBJECT, __VA_ARGS__)
        /// See hkLog_Scope (eager block scope)
    #define Log_Scope(LEVEL, ...) hkLog_Scope(DEBUG_LOG_OBJECT, LEVEL, __VA_ARGS__)
    #define Log_Info(...) hkLog_Info(DEBUG_LOG_OBJECT, __VA_ARGS__)
    #define Log_Dev(...) hkLog_Dev(DEBUG_LOG_OBJECT, __VA_ARGS__)
    #define Log_Debug(...) hkLog_Debug(DEBUG_LOG_OBJECT, __VA_ARGS__)

    // Macros beginning with DLOG_ are shortcuts for the common use case of
    // * using a predefined log object (DEBUG_LOG_OBJECT)
    // * outputting at hkLog::Level::Dev

    #if defined(HK_DEBUG_SLOW)
            /// Log a single message. First argument must be the format string.
            /// E.g. DLOG("{} happened", event)
        #define DLOG(...) \
            hkLog_Dev(DEBUG_LOG_OBJECT, __VA_ARGS__)

            /// See hkLog_Scope (eager block scope)
        #define DLOG_SCOPE(...) \
            hkLog_Scope(DEBUG_LOG_OBJECT, Dev, __VA_ARGS__)
            /// See hkLog_Auto (lazy block scope)
        #define DLOG_AUTO(...) \
            hkLog_Auto(DEBUG_LOG_OBJECT, __VA_ARGS__)
            /// Code only executed when logging macros are in effect (not when the log is enabled)
        #define DLOG_IF(...) __VA_ARGS__

        #define DLOG_ENABLED DEBUG_LOG_OBJECT.enabled(hkLog::Level::Dev)

            /// Use explicit BEGIN & END only when DLOG_SCOPE won't work. First argument must be the format string.
            /// E.g.:
            /// DLOG_BEGIN("Block {}", blockName);
            /// ...
            /// DLOG_END();
        #define DLOG_BEGIN(...) \
            hkLog_BlockBegin(DEBUG_LOG_OBJECT, Dev, __VA_ARGS__)
        #define DLOG_END() \
            hkLog_BlockEnd(DEBUG_LOG_OBJECT, Dev)
    #else

        #define DLOG(...) do { } while(0)
        #define DLOG_SCOPE(...)
        #define DLOG_AUTO(...)
        #define DLOG_IF(...)
        #define DLOG_ENABLED false
        #define DLOG_BEGIN(...)
        #define DLOG_END(...)

    #endif
#endif

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
