// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>

    /// Logging utilities.
namespace hkLog
{
    namespace Level
    {
        enum Enum
        {
            Disabled = 0, ///< Logging disabled.
            Error,  ///< Error message.
            Warning,  ///< A warning that might indicate a problem, but is not fatal when it is ignored.
            Info,  ///< Some general information about the current status.
            Dev,  ///< Even more detailed information, will be compiled out in release builds.
            Debug,  ///< Even more detailed information, will be compiled out in release builds.
            MaxLevel
        };
    };

    class RegisteredOrigin;
    class InstanceOrigin;
    class Output;
    class Message;
    class AutoMessage;

        /// This represents the system or object where the message originated.
    class HK_EXPORT_COMMON Origin
    {
        public:
            HK_DECLARE_PLACEMENT_ALLOCATOR();

            HK_ALWAYS_INLINE bool enabled(Level::Enum l) const { return m_level >= l; }
            HK_ALWAYS_INLINE void setLevel(Level::Enum l) { m_level = l; }
            HK_ALWAYS_INLINE Level::Enum getLevel() const { return m_level; }
            _Ret_maybenull_ Output* getOutput() const { return m_output; }
            _Ret_z_ const char* getName() const;

        protected:
            enum Flags { INVALID=0, TYPE_REG=1, TYPE_INST=2 };
            Origin(Flags flags, Level::Enum level = Level::Warning) : m_output(HK_NULL), m_level(level), m_flags( hkLosslessCast<hkUint16>(flags) ) {}
            Output* m_output;
            hkEnum<Level::Enum, hkUint8> m_level;
            hkFlags<Flags,hkUint16> m_flags;
    };

        /// The log message itself being transported through the log system.
    class HK_EXPORT_COMMON Message
    {
        public:
            HK_DECLARE_CLASS( Message, New );

            Message( const Origin& origin, Level::Enum level );
            Message& setText( _In_z_ const char* fmt ) { m_txt = fmt; return *this; }
            Message& setText( _In_z_ const char* fmt, hkVarArgs::VaTypes argTypes, ... );
            Message& setBegin() { m_scopeDelta=+1; return *this; }
            Message& setEnd() { m_scopeDelta=-1; return *this; }
            Message& setLocation( _In_z_ const char* filename, int line, int col ) { m_locFilename = filename; m_locLine = line; m_locColumn = col; return *this; }
            Message& setId(hkUint32 id) { m_id = id; return *this; }
            Message& setId(hkInt32 id) { m_id = hkUint32(id); return *this; }
            Message& setData( _In_opt_ hkReferencedObject* data ) { m_data = data; return *this; }

            const Origin& m_origin;
            Level::Enum m_level;
            hkUint32 m_id;
            hkStringPtr m_txt;
            hkRefPtr<hkReferencedObject> m_data;
            const char* m_locFilename;
            int m_locLine;
            int m_locColumn;
            int m_scopeDelta;
    };

        /// Utility class used by hkLog_xxx() macros to build-and-fire a Message.
        /// Automatically dispatch the message from its destructor.
    class HK_EXPORT_COMMON AutoMessage : public Message
    {
        public:
            HK_DECLARE_CLASS( AutoMessage, NoNew, NonCopyable );
            AutoMessage( const Origin& origin, Level::Enum level ) : Message( origin, level ) {}
            ~AutoMessage();
    };

        /// Consumer of messages. Outputs may be shared between several logs.
    class HK_EXPORT_COMMON Output : public hkReferencedObject
    {
        public:
            HK_DECLARE_CLASS(Output, New);

            enum Action
            {
                PropagateToNextOutput,
                StopMessagePropagation
            };

                ///
            Output(Level::Enum l) : m_level(l) {}

                ///
            virtual ~Output();

                /// Handle the given message. Returning StopMessagePropagation is for advanced use only.
            virtual Action put(const Message& message) = 0;

                /// Flush any output.
            virtual void flush();

                /// Get the max level which we handle.
            Level::Enum getLevel() const { return m_level; }

            void setLevel( Level::Enum level ) { m_level = level; }

        protected:
            Level::Enum m_level;
    };

    class HK_EXPORT_COMMON TextOutput : public Output
    {
        public:
            TextOutput(Level::Enum level = Level::Warning);

            virtual Output::Action put(const Message& message) HK_OVERRIDE;

            _Ret_maybenull_z_
            const char* getText(hkStringBuf& buf) const;

        protected:

            bool getMessageText( const Message& msg, hkStringBuf& buf ) const;

            struct Item { hkStringPtr m_txt; bool m_written; };
            hkArray<Item> m_items;
            int m_indent;
    };

    /// Print output to the error report function provided in the construction
    class HK_EXPORT_COMMON PrintOutput : public hkLog::Output
    {
        public:

            /// Func: the function to call with the formatted string
            /// arg: User defined argument to pass to func
            /// l: Warning levels to print
            PrintOutput(hkErrorReportFunction func, _In_opt_ void* arg, hkLog::Level::Enum l);

            virtual Output::Action put(const Message& message) HK_OVERRIDE;

        private:

            hkErrorReportFunction m_func;
            void* m_arg;
            int m_scopeDepth;
    };

    class HK_EXPORT_COMMON MultiOutput : public Output
    {
        public:
            HK_DECLARE_CLASS(MultiOutput, New, NonCopyable);
            MultiOutput();
            ~MultiOutput();
            virtual Output::Action put(const Message& message) HK_OVERRIDE;
            virtual void flush() HK_OVERRIDE;
            void addOutput(_Inout_ Output* o);
            void removeOutput(_Inout_ Output* o);
            void removeAllOutputs();
        protected:
            hkArray< hkRefPtr<Output> > m_outputs;
            hkCriticalSection m_lock;
    };

        /// An origin which registers itself globally.
        /// Each subsystem which uses logging registers itself on a global list during construction.
    class RegisteredOrigin : public Origin
    {
        public:
            HK_DECLARE_PLACEMENT_ALLOCATOR();
                /// Create a named and globally registered log.
                /// The log must last the lifetime of the program.
                /// The name is typically of the form "system.subsytem"
                /// Initially the log is disabled and not connected to any output.
            explicit RegisteredOrigin(_In_z_ const char* name, Level::Enum level = Level::Warning);
                /// Get the name.
            _Ret_z_ const char* getName() const { return m_name; }
                /// Get the next log.
            _Ret_maybenull_ RegisteredOrigin* getNext() { return m_next; }
                /// Access the output.
            MultiOutput& accessOutput() { return m_multi; }

        protected:
            MultiOutput m_multi;
            const char* m_name;
            RegisteredOrigin* m_next;
    };

        /// Logging origin which is an object instance.
    class InstanceOrigin : public Origin
    {
        public:
            InstanceOrigin(_Inout_opt_ RegisteredOrigin* p);
            _Ret_maybenull_ const RegisteredOrigin* getParent() const { return m_parent; }
        protected:
            RegisteredOrigin* m_parent;
    };

        /// Utilities for modifying log behavior for a given scope.
    namespace Raii
    {
        struct AutoScope;

            /// Add the given output for this object scope.
            /// Use this to add an output sink for a particular log or the global log.
        struct HK_EXPORT_COMMON AddOutput
        {
            MultiOutput& m_multi;
            Output* m_out;
            AddOutput(MultiOutput& m, _Inout_ Output* o) : m_multi(m), m_out(o) { m_multi.addOutput(m_out); }
            ~AddOutput() { m_multi.removeOutput(m_out); }
        };
            /// Set the level for this object scope.
        struct HK_EXPORT_COMMON SetLevel
        {
            Origin& m_origin;
            Level::Enum m_savedLevel;
            SetLevel(Origin& o, Level::Enum l) : m_origin(o), m_savedLevel(o.getLevel()) { m_origin.setLevel(l); }
            ~SetLevel() { m_origin.setLevel(m_savedLevel); }
        };
            /// Hook output for this thread.
            /// Unlike AddOutput, this effectively adds an output for all logs.
        class HK_EXPORT_COMMON ThreadHookOutput
        {
            public:
                ThreadHookOutput() : m_orig(HK_NULL) {}
                ThreadHookOutput(_Inout_ Output* o);
                ~ThreadHookOutput();
                void hook(_Inout_ Output* o);
            protected:
                Output* m_orig;
                AutoScope* m_autoScope;
        };

        struct CaptureOutput : public TextOutput
        {
            CaptureOutput() { m_hook.hook(this); }
            ThreadHookOutput m_hook;
            //Inherits char* TextOutput::getText();
        };

            /// Utility to end a block when it goes out of scope
        struct HK_EXPORT_COMMON BlockEnd
        {
            // Don't be tempted to add logging in the constructor
            // Doing so would force evaluation of the args, even if logging is disabled.
            inline BlockEnd(const Origin& c, Level::Enum l) : m_origin(c), m_level(l) { }
            inline ~BlockEnd()
            {
                if(m_origin.enabled(m_level))
                {
                    AutoMessage(m_origin, m_level).setEnd();
                }
            }
            const Origin& m_origin;
            Level::Enum m_level;
        };

            /// Utility to add some extra context which only gets printed if a message is fired
            /// while it is active.
        struct HK_EXPORT_COMMON AutoScope
        {
            static HK_THREAD_LOCAL(hkLog::Raii::AutoScope*) s_currentLogAuto;

                // Lightweight var for smaller code.
            struct LwVar
            {
                template<typename T>
                HK_ALWAYS_INLINE LwVar(_In_ const T* v) : p(v), t(hkReflect::getType<T>()) {}
                const void* p; const hkReflect::Type* t;
            };
            AutoScope(const Origin& origin, LwVar args, _In_z_ const char* file, int line);
            ~AutoScope();

            void show(_Inout_ Output* output);
            void markFired(Level::Enum l);

        protected:

            const Origin& m_origin;
            LwVar m_args;
            Level::Enum m_lowestFired;
            AutoScope* m_prev;
            const char* m_file;
            const int m_line;
        };
    }

        /// Flush all registered logs
    HK_EXPORT_COMMON void flush();
        /// Reset all registered logs
    HK_EXPORT_COMMON void reset();

        /// Utility method to output logs whose names contain pattern to the given filename.
    HK_EXPORT_COMMON _Ret_maybenull_ Output* connectToFile(_In_z_ const char* pattern, _In_z_ const char* filename, Level::Enum level);
        /// Utility method to output logs whose names contain pattern to the given output.
    HK_EXPORT_COMMON void  connectToOutput(_In_z_ const char* pattern, _Inout_ hkLog::Output* output, Level::Enum level);
        /// Disconnect the given output from the logs matching pattern.
    HK_EXPORT_COMMON void disconnect(_In_z_ const char* pattern, _Inout_ hkLog::Output* output);
        /// Enable or disable based on pattern.
    HK_EXPORT_COMMON void enableLogs(_In_z_ const char* pattern, Level::Enum e);
        /// Set the log level of the default origin from the base system.
    HK_EXPORT_COMMON void setBaseSystemLogLevel(Level::Enum level);

        /// Access the global output.
    HK_EXPORT_COMMON MultiOutput& HK_CALL accessGlobalOutput();

        /// An iterator over all the registered logs.
    class HK_EXPORT_COMMON OriginIterator
    {
        public:
            HK_DECLARE_CLASS(OriginIterator, New);
            OriginIterator() : m_headIdx(-1), m_origin(HK_NULL) {}
            bool advance();
            _Ret_maybenull_ RegisteredOrigin* current();
        protected:
            int m_headIdx;
            RegisteredOrigin* m_origin;
    };

    namespace Detail
    {
        HK_EXPORT_COMMON hkBaseSystem::InitNode& HK_CALL getGlobalInitNode();
            /// Each module (DLL/EXE) has its own list of entries.
        struct ModuleLocalList;

            /// Add a list of logs from a module (DLL) into a central location.
        HK_EXPORT_COMMON hkResult addModuleLocalList(_In_ ModuleLocalList* e);
            /// Undo addListHead.
        HK_EXPORT_COMMON hkResult removeModuleLocalList(_In_ ModuleLocalList* e);
    }
}

HK_REFLECT_ENUM( HK_EXPORT_COMMON, hkLog::Level::Enum );

    /// Log a message. Third argument must be the format string.
    /// E.g. hkLog_Info(origin, "{} happened", event)
    // It's tempting to just call the log methods directly on the log object but this would
    // force evaluation of the arguments. Macros make this lighter when the log is disabled.
#define hkLog_Log(ORIGIN, LEVEL, ...) \
    if( ORIGIN.enabled(LEVEL) ) hkLog::AutoMessage(ORIGIN, LEVEL).setText( HK_VARARGS1_WRAP((__VA_ARGS__)) ).setLocation(__FILE__,__LINE__,0)

#define hkLog_Error(ORIGIN, ...) hkLog_Log(ORIGIN, hkLog::Level::Error, __VA_ARGS__ )
#define hkLog_Warning(ORIGIN, ...) hkLog_Log(ORIGIN, hkLog::Level::Warning, __VA_ARGS__ )
#define hkLog_Info(ORIGIN, ...) hkLog_Log(ORIGIN, hkLog::Level::Info, __VA_ARGS__ )
#define hkLog_Dev(ORIGIN, ...) hkLog_Log(ORIGIN, hkLog::Level::Dev, __VA_ARGS__ )
#define hkLog_Debug(ORIGIN, ...) hkLog_Log(ORIGIN, hkLog::Level::Debug, __VA_ARGS__ )


    /// Push a delayed log scope onto the stack. The message will not be fired,
    /// or even formatted, until a non-auto message is fired within its scope.
    /// This is many times faster than hkLog_Scope if no messages are fired while this scope is active.
    /// Note that this shallow copies the arguments so any pointers need to remain valid for the scope.
#define hkLog_Auto(ORIGIN, ...) \
    hkDetail_Log_Auto(HK_PP_JOIN(logAutoScope,__COUNTER__), HK_PP_JOIN(logAutoArgs,__COUNTER__), ORIGIN, __VA_ARGS__)

    /// Begin a log block. Second argument must be the format string.
    /// E.g.:
    /// {
    ///     hkLog_Scope(origin, "Block {}", blockName);
    ///     ...
    /// }
    /// Contrast this with hkLog_Auto which only produces a scope if a message is fired while
    /// it is active.
#define hkLog_Scope(ORIGIN, LEVEL, ...) \
    hkLog::Raii::BlockEnd HK_PP_JOIN(obj,__LINE__)(ORIGIN,hkLog::Level::LEVEL); \
    hkLog_Log(ORIGIN, hkLog::Level::LEVEL, __VA_ARGS__).setBegin()

    // Push a thread-local autoscope.
#define hkDetail_Log_Auto(AUTO_SYMBOL, ARGS_SYMBOL, ORIGIN, ...) \
    auto ARGS_SYMBOL = hkTupleT::make<const char*>(__VA_ARGS__); \
    hkLog::Raii::AutoScope AUTO_SYMBOL(ORIGIN, &ARGS_SYMBOL, __FILE__, __LINE__)

#define hkLog_BlockBegin(ORIGIN, LEVEL, ...) \
    hkLog_Log(ORIGIN, hkLog::Level::LEVEL, __VA_ARGS__).setBegin()
#define hkLog_BlockEnd(ORIGIN, LEVEL) \
    if(ORIGIN.enabled(hkLog::Level::LEVEL)) hkLog::AutoMessage(ORIGIN, hkLog::Level::LEVEL).setEnd()

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
