// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/System/Log/hkLog.h>
#include <Common/Base/Object/hkSingleton.h>
#include <Common/Base/Container/StringMap/hkStorageStringMap.h>
#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>
#include <Common/Base/System/Log/hkLogModule.cxx> // for static lib build impl, and hkCommonAll
#include <Common/Base/Fwd/hkcstdarg.h>
// hkLog::Origin

 _Ret_z_ const char* hkLog::Origin::getName() const
 {
     const RegisteredOrigin* reg = m_flags.allAreSet(TYPE_REG)
         ? reinterpret_cast<const RegisteredOrigin*>(this)
         : reinterpret_cast<const InstanceOrigin*>(this)->getParent();
     return reg->getName();
 }

 hkLog::InstanceOrigin::InstanceOrigin(_Inout_opt_ RegisteredOrigin* p)
     : Origin(TYPE_INST)
     , m_parent(p)
 {
     if(p)
     {
         m_level = p->getLevel();
         m_output = p->getOutput();
     }
 }

// hkLog::Message

hkLog::Message::Message(const Origin& origin, Level::Enum level)
    : m_origin(origin)
    , m_level(level)
    , m_id(0)
    , m_locFilename(nullptr)
    , m_locLine(0)
    , m_locColumn(0)
    , m_scopeDelta(0)
{
}

hkLog::Message& hkLog::Message::setText(_In_z_ const char* fmt, hkVarArgs::VaTypes argTypes, ...)
{
    hkVarArgs::FixedArray<10> args;
    HK_VARARGS_UNPACK(args, argTypes);
    hkStringBuf sb; sb.formatV(fmt, args);
    m_txt = sb;
    return *this;
}


// hkLog::Output

hkLog::Output::~Output()
{
}

void hkLog::Output::flush()
{
}

hkLog::MultiOutput::MultiOutput()
    : Output(Level::Disabled), m_lock(0)
{
}

hkLog::MultiOutput::~MultiOutput()
{
}

hkLog::Output::Action hkLog::MultiOutput::put(const Message& message)
{
    HK_ASSERT_NO_MSG(0x6772bd5, m_level >= message.m_level);
    hkCriticalSectionLock lock(&m_lock);
    for(int i = m_outputs.getSize() - 1; i >= 0; --i)
    {
        if (m_outputs[i]->put(message) == Action::StopMessagePropagation)
        {
            return Action::StopMessagePropagation;
        }
    }
    return Action::PropagateToNextOutput;
}

void hkLog::MultiOutput::flush()
{
    for(int i = 0; i < m_outputs.getSize(); ++i)
    {
        m_outputs[i]->flush();
    }
}

void hkLog::MultiOutput::addOutput(_Inout_ Output* o)
{
    hkCriticalSectionLock lock(&m_lock);
    if(m_outputs.lastIndexOf(o) < 0)
    {
        m_outputs.pushBack(o);
        // level can only increase via an add
        m_level = hkMath::max2(m_level, o->getLevel());
    }
}

void hkLog::MultiOutput::removeOutput(_Inout_ Output* o)
{
    hkCriticalSectionLock lock(&m_lock);
    int index = m_outputs.lastIndexOf(o);
    if( index >= 0 )
    {
        m_outputs.removeAtAndCopy(index);
        if (o->getLevel() == m_level)
        {
            // level can only reduced via remove
            m_level = Level::Disabled;
            for (int i = 0; i < m_outputs.getSize(); ++i)
            {
                m_level = hkMath::max2(m_level, o->getLevel());
            }
        }
        if ( m_outputs.getSize() == 0 )
        {
            // Note: Because MultiOutputs are held by static RegisteredOrigin objects,
            // ensure that we fully cleanup our memory in the empty case to allow
            // systems to ensure this havok memory is completely deallocated
            // before modules with static objects is unloaded.
            m_outputs.clearAndDeallocate();
        }
    }
}

void hkLog::MultiOutput::removeAllOutputs() 
{
    m_outputs.clearAndDeallocate();
    m_level = Level::Disabled;
}

// hkLog

namespace
{
    HK_THREAD_LOCAL(hkLog::Output*) hkLog_threadLocalOutput;
    using namespace hkLog;

    static bool fireMessage(Message& msg, _Inout_opt_ Output* out, _Inout_opt_ Raii::AutoScope* aut)
    {
        if (out)
        {
            if (out->getLevel() >= msg.m_level)
            {
                if (aut)
                {
                    aut->show(out);
                }
                if (out->put(msg) == hkLog::Output::Action::StopMessagePropagation)
                {
                    return true;
                }
            }
        }
        return false;
    }
}

hkLog::AutoMessage::~AutoMessage()
{
    hkLog::Raii::AutoScope* logAuto = HK_THREAD_LOCAL_GET(hkLog::Raii::AutoScope::s_currentLogAuto);

    // Check for thread-local override
    if (fireMessage(*this, HK_THREAD_LOCAL_GET(hkLog_threadLocalOutput), logAuto))
    {
        // Message was captured
    }
    else
    {
        // Fire to the origin's own output
        fireMessage(*this, m_origin.getOutput(), logAuto);

        // fire to the global log
        fireMessage(*this, &accessGlobalOutput(), logAuto);
    }

    // Mark auto scopes used, but only for the current level.
    // Each output should get to see the auto scope once.
    // We'll check this level in the show() method next time.
    if (logAuto)
    {
        logAuto->markFired(m_level);
    }
}

hkLog::Raii::ThreadHookOutput::ThreadHookOutput(_Inout_ Output* o)
    : m_orig(HK_NULL)
{
    hook(o);
}

void hkLog::Raii::ThreadHookOutput::hook(_Inout_ Output* o)
{
    HK_ASSERT_NO_MSG(0x2a1d9522, m_orig == HK_NULL);
    m_orig = HK_THREAD_LOCAL_GET(hkLog_threadLocalOutput);
    HK_THREAD_LOCAL_SET(hkLog_threadLocalOutput, o);
    // Hooking the output also "restarts" the set of auto scopes
    m_autoScope = HK_THREAD_LOCAL_GET(AutoScope::s_currentLogAuto);
    HK_THREAD_LOCAL_SET(AutoScope::s_currentLogAuto, nullptr);
}

hkLog::Raii::ThreadHookOutput::~ThreadHookOutput()
{
    HK_THREAD_LOCAL_SET(hkLog_threadLocalOutput, m_orig);
    HK_THREAD_LOCAL_SET(AutoScope::s_currentLogAuto, m_autoScope);
}

HK_THREAD_LOCAL(hkLog::Raii::AutoScope*) hkLog::Raii::AutoScope::s_currentLogAuto;

hkLog::Raii::AutoScope::AutoScope(const Origin& origin, LwVar args, _In_z_ const char* file, int line)
    : m_origin(origin), m_args(args), m_file(file), m_line(line)
{
    m_lowestFired = Level::MaxLevel;
    m_prev = HK_THREAD_LOCAL_GET(s_currentLogAuto);
    HK_THREAD_LOCAL_SET(s_currentLogAuto, this);
}

hkLog::Raii::AutoScope::~AutoScope()
{
    if (m_lowestFired < Level::MaxLevel)
    {
        AutoMessage(m_origin, m_lowestFired).setEnd();
    }

    HK_THREAD_LOCAL_SET(s_currentLogAuto, m_prev);
}

void hkLog::Raii::AutoScope::show(_Inout_ Output* output)
{
    if (output->getLevel() >= m_lowestFired)
    {
        return;
    }
    if (m_prev)
    {
        m_prev->show(output);
    }

    Message message( m_origin, output->getLevel() );
    message.setBegin().setLocation( m_file, m_line, 0 );

    // Flatten the tuple of args into an array of vars for formatting.
    // Special case: the first field is the format string.
    {
        const char* fmt = *(const char**)m_args.p;
        using namespace hkReflect;
        hkReflect::Detail::AddrAndType args[10];
        hkReflect::Var var( m_args.p, m_args.t, nullptr );
        DeclIter<DataFieldDecl> iter( m_args.t );
        iter.advance(); // skip fmt string
        int count = 0;
        for ( count = 0; count < HK_COUNT_OF( args ) && iter.advance(); ++count )
        {
            Var v = var[iter.current()];
            args[count].m_addr = v.getAddress();
            args[count].m_type = v.getType();
            HK_ASSERT(0x6db557c, v.getType()->getImpl() == v.getImpl(), "No custom impls allowed");
        }
        hkStringBuf sb; sb.formatV( fmt, hkVarArgs::Vector( args, count ) );
        message.setText( sb );
    }

    output->put( message );
}


void hkLog::Raii::AutoScope::markFired(Level::Enum level)
{
    AutoScope* cur = this;
    while (cur && (level < cur->m_lowestFired))
    {
        cur->m_lowestFired = level;
        cur = cur->m_prev;
    }
}

hkLog::TextOutput::TextOutput(Level::Enum level)
    : Output(level), m_indent(0)
{
}

hkLog::Output::Action hkLog::TextOutput::put(const Message& msg)
{
    hkStringBuf sb;
    getMessageText( msg, sb );

    if(msg.m_scopeDelta >= 1)
    {
        Item& it = m_items.expandOne();
        it.m_txt = sb;
        it.m_written = false;
        m_indent += 1;
    }
    else if(msg.m_scopeDelta <= -1)
    {
        if(m_items.back().m_written)
        {
            Item& it = m_items.expandOne();
            it.m_txt = sb;
            it.m_written = true;
        }
        else
        {
            m_items.popBack();
        }
        m_indent -= 1;
    }
    else
    {
        for(int i = m_items.getSize() - 1; i >= 0 && m_items[i].m_written == false; --i)
        {
            m_items[i].m_written = true;
        }
        Item& it = m_items.expandOne();
        it.m_txt = sb;
        it.m_written = true;
    }
    return Action::PropagateToNextOutput;
}

_Ret_maybenull_z_
const char* hkLog::TextOutput::getText(hkStringBuf& buf) const
{
    HK_ASSERT_NO_MSG(0x4965983e, m_items.getSize() == 0 || m_items.back().m_written);
    buf.clear();
    for(int i = 0; i < m_items.getSize(); ++i)
    {
        buf.append(m_items[i].m_txt);
    }
    return buf.getLength() ? buf.cString() : HK_NULL;
}

bool hkLog::TextOutput::getMessageText( const Message& msg, hkStringBuf& buf ) const
{
    if( msg.m_txt )
    {
        const char s_spaces[] = "                                                                             ";
        int clampedIndent = hkMath::clamp( m_indent, 0, HK_COUNT_OF( s_spaces ) - 1 );
        const char* space = &s_spaces[0] + ( HK_COUNT_OF( s_spaces ) - 1 - clampedIndent );
        const char* origin = "";
        if( const char* n = msg.m_origin.getName() )
        {
            origin = n;
        }

        buf.printf( "%s[%s] %s", space, origin, msg.m_txt.cString() );
        if( msg.m_locFilename )
        {
            buf.appendPrintf( " (%s:%i)", msg.m_locFilename, msg.m_locLine );
        }

        buf.append( "\n" );
    }

    return msg.m_txt;
}

hkLog::PrintOutput::PrintOutput(hkErrorReportFunction func, _In_opt_ void* arg, hkLog::Level::Enum l)
    : Output(l), m_func(func), m_arg(arg), m_scopeDepth(0) {}

hkLog::Output::Action hkLog::PrintOutput::put(const hkLog::Message& msg)
{
    if (msg.m_level > m_level)
    {
        return Action::PropagateToNextOutput;
    }

    int curScopeDepth = hkAtomic::exchangeAdd(&m_scopeDepth, msg.m_scopeDelta);

    if (msg.m_txt)
    {
        hkStringBuf sb;
        const char s_spaces[] = "                                                                             ";
        int clampedIndent = hkMath::clamp(curScopeDepth, 0, HK_COUNT_OF(s_spaces) - 1);
        const char* space = &s_spaces[0] + (HK_COUNT_OF(s_spaces) - 1 - clampedIndent);
        const char* origin = "";
        if (const char* n = msg.m_origin.getName())
        {
            origin = n;
        }
        if (msg.m_id)
        {
            sb.printf("%s[%s] [0x%08x] %s", space, origin, msg.m_id, msg.m_txt.cString());
        }
        else
        {
            sb.printf("%s[%s] %s", space, origin, msg.m_txt.cString());
        }
        if (msg.m_locFilename)
        {
            sb.appendPrintf(" (%s:%i)", msg.m_locFilename, msg.m_locLine);
        }
        sb.append("\n");
        (*m_func)(sb, m_arg);
    }
    return Action::PropagateToNextOutput;
}

namespace hkLog
{
    class Registry : public hkReferencedObject
    {
        public:
            HK_DECLARE_CLASS(Registry, New, Singleton);

            class BaseSystemOutput : public PrintOutput
            {
            public:
                BaseSystemOutput( hkErrorReportFunction func, _In_opt_ void* arg, Level::Enum level )
                    : PrintOutput( func, arg, level )
                {
                }
                void setLevel( Level::Enum level )
                {
                    m_level = level;
                }
            };

            Registry();
            virtual ~Registry();
            void add(_Inout_ Detail::ModuleLocalList* e);
            void remove(_Inout_ Detail::ModuleLocalList* e);
            void flush();
            void reset();
            _Ret_maybenull_ Output* connectToFile(_In_z_ const char* pattern, _In_z_ const char* filename, Level::Enum level);
            void connectToOutput(_In_z_ const char* pattern, _Inout_ hkLog::Output* output, Level::Enum level);
            void disconnect(_In_z_ const char* pattern, _Inout_ Output* output);
            void enableLogs(_In_z_ const char* pattern, Level::Enum level);
            void setBaseSystemLogLevel(Level::Enum level);

            BaseSystemOutput m_baseSystemOutput;
            MultiOutput m_multi; // extra outputs can be added here. By default it just contains m_baseSystemOutput

            struct RegOutput;
            _Ret_maybenull_ RegOutput* reuseOrOpen(_In_z_ const char* fname);

            hkStorageStringMap<RegOutput*> m_outputFromFname;
            hkArray<Detail::ModuleLocalList*> m_heads;
    };

    struct Registry::RegOutput : public hkLog::Output
    {
        HK_DECLARE_CLASS(RegOutput, New);

        RegOutput()
            : Output(Level::Dev)
        {
        }

        static _Ret_maybenull_ RegOutput* open(_In_z_ const char* fname)
        {
            if( hkRefPtr<hkStreamWriter> w = hkFileSystem::getInstance().openWriter(fname) )
            {
                RegOutput* o = new RegOutput;
                o->m_writer = w;
                o->m_indent = 0;
                return o;
            }
            return HK_NULL;
        }

        virtual hkLog::Output::Action put(const hkLog::Message& msg) HK_OVERRIDE
        {
            if( msg.m_txt )
            {
                const int NAME_COLS = 0;
                const char* logName = msg.m_origin.getName();
                int len = logName ? hkMath::min2(NAME_COLS, hkString::strLen(logName)) : 0;
                m_writer->write(logName, len);
                const char s_spaces[] = "                                                                             ";
                m_writer->write(s_spaces, NAME_COLS-len);
                int clampedIndent = hkMath::clamp(m_indent, 0, HK_COUNT_OF(s_spaces)-1);
                m_writer->write(s_spaces, clampedIndent);
                m_writer->write(s_spaces, clampedIndent);

                if (msg.m_id)
                {
                    hkStringBuf strBuf;
                    strBuf.printf("[0x%08x] ", msg.m_id);
                    m_writer->write(strBuf.cString(), strBuf.getLength());
                }

                m_writer->write(msg.m_txt.cString(), msg.m_txt.getLength());
                m_writer->write("\n", 1);
                //flush();
            }
            m_indent += msg.m_scopeDelta;
            HK_ASSERT_NO_MSG(0x5cea88a5, m_indent>=0);
            return Action::PropagateToNextOutput;
        }
        virtual void flush() HK_OVERRIDE
        {
            m_writer->flush();
        }
        hkRefPtr<hkStreamWriter> m_writer;
        int m_indent;
    };
}


HK_SINGLETON_IMPLEMENTATION(hkLog::Registry);

hkLog::MultiOutput& HK_CALL hkLog::accessGlobalOutput()
{
    return Registry::singleton->m_multi;
}

hkBaseSystem::InitNode& hkLog::Detail::getGlobalInitNode()
{
    return Registry::singletonEntry;
}

bool hkLog::OriginIterator::advance()
{
    if(m_origin && m_origin->getNext())
    {
        m_origin = m_origin->getNext();
        return true;
    }
    else
    {
        Registry* reg = Registry::singleton;
        if(m_headIdx + 1 < reg->m_heads.getSize())
        {
            m_headIdx += 1;
            m_origin = reg->m_heads[m_headIdx]->entries;
            return true;
        }
    }
    return false;
}

_Ret_maybenull_
hkLog::RegisteredOrigin* hkLog::OriginIterator::current()
{
    return m_origin;
}

hkLog::Registry::Registry()
    : m_baseSystemOutput(&hkBaseSystem::error, HK_NULL, Level::Warning)
{
    add(&Detail::s_moduleLocalList); // common dll, and also static lib case
    m_multi.addOutput(&m_baseSystemOutput);
}

hkLog::Registry::~Registry()
{
    // hkLogs which are registered with hkLogReg have global static lifetimes.
    // However the Ouput lifetimes are linked to basesystem::init & quit.
    // Thus, we need to "reset" the loggers in the same way we'd have to clearAndDeallocate a global hkArray.
    reset();

    // Now release any outputs we have opened
    for( hkStorageStringMap<Output*>::Iterator it = m_outputFromFname.getIterator();
        m_outputFromFname.isValid(it); it = m_outputFromFname.getNext(it) )
    {
        if(Output* o = m_outputFromFname.getValue(it))
        {
            o->removeReference();
        }
    }
}

void hkLog::Registry::flush()
{
    for (int h=0; h < m_heads.getSize(); ++h)
    {
        for( RegisteredOrigin* cur = m_heads[h]->entries; cur; cur = cur->getNext() )
        {
            if( hkLog::Output* o = cur->getOutput() )
            {
                o->flush();
            }
        }
    }
}

void hkLog::Registry::reset()
{
    for (int h=0; h < m_heads.getSize(); ++h)
    {
        for( RegisteredOrigin* cur = m_heads[h]->entries; cur; cur = cur->getNext() )
        {
            cur->accessOutput().removeAllOutputs();
        }
    }
}

void hkLog::Registry::connectToOutput(_In_z_ const char* pattern, _Inout_ hkLog::Output* output, Level::Enum level)
{
    for (int h=0; h < m_heads.getSize(); ++h)
    {
        for( RegisteredOrigin* cur = m_heads[h]->entries; cur; cur = cur->getNext() )
        {
            if( hkString::strStr( cur->getName(), pattern) )
            {
                cur->accessOutput().addOutput( output );
                if(cur->getLevel() < level)
                {
                    cur->setLevel(level);
                }
            }
        }
    }
}

_Ret_maybenull_
hkLog::Output* hkLog::Registry::connectToFile(_In_z_ const char* pattern, _In_z_ const char* filename, Level::Enum level)
{
    if(hkLog::Output* o = reuseOrOpen(filename))
    {
        connectToOutput(pattern, o, level);
        return o;
    }
    return HK_NULL;
}

void hkLog::Registry::disconnect(_In_z_ const char* pattern, _Inout_ hkLog::Output* output)
{
    for(int h = 0; h < m_heads.getSize(); ++h)
    {
        for(RegisteredOrigin* cur = m_heads[h]->entries; cur; cur = cur->getNext())
        {
            if(hkString::strStr(cur->getName(), pattern))
            {
                cur->accessOutput().removeOutput(output);
            }
        }
    }
}

void hkLog::Registry::enableLogs(_In_z_ const char* pattern, Level::Enum level)
{
    for (int h=0; h < m_heads.getSize(); ++h)
    {
        for( RegisteredOrigin* cur = m_heads[h]->entries; cur; cur = cur->getNext() )
        {
            if( hkString::strStr( cur->getName(), pattern) )
            {
                cur->setLevel(level);
            }
        }
    }
}

void hkLog::Registry::setBaseSystemLogLevel(Level::Enum level)
{
    if ( m_baseSystemOutput.getLevel() != level )
    {
        m_multi.removeOutput( &m_baseSystemOutput );
        m_baseSystemOutput.setLevel( level );
        m_multi.addOutput( &m_baseSystemOutput );
    }
}

void hkLog::Registry::add(_Inout_ Detail::ModuleLocalList* e)
{
    if ( m_heads.indexOf(e) < 0)
    {
        m_heads.pushBack(e);
    }
}

void hkLog::Registry::remove(_Inout_ Detail::ModuleLocalList* e)
{
    int idx = m_heads.indexOf(e);
    if(idx >= 0)
    {
        for(RegisteredOrigin* cur = e->entries; cur; cur = cur->getNext())
        {
            cur->accessOutput().removeAllOutputs();
        }
        m_heads.removeAt(idx);
    }
}

_Ret_maybenull_
hkLog::Registry::RegOutput* hkLog::Registry::reuseOrOpen(_In_z_ const char* fname)
{
    RegOutput* ret = m_outputFromFname.getOrInsert(fname, HK_NULL);
    if( ret == HK_NULL )
    {
        if(RegOutput* o = RegOutput::open(fname) )
        {
            m_outputFromFname.insert(fname, o);
            ret = o;
        }
    }
    return ret;
}

// Forward to singleton implementations.
// Normally we would just expose the singleton. However it's just too handy to be
// able to call these static methods from the debugger. Most debuggers can execute
// static methods but cannot chain foo::getInstance().bar()
hkResult hkLog::Detail::addModuleLocalList(_In_ Detail::ModuleLocalList* e)
{
    if(Registry* reg = Registry::getInstancePtr())
    {
        reg->add(e);
        return HK_SUCCESS;
    }
    return HK_FAILURE;
}

hkResult hkLog::Detail::removeModuleLocalList(_In_ Detail::ModuleLocalList* e)
{
    if(Registry* reg = Registry::getInstancePtr())
    {
        reg->remove(e);
        return HK_SUCCESS;
    }
    return HK_FAILURE;
}

void hkLog::flush()
{
    Registry::getInstance().flush();
}

void hkLog::reset()
{
    Registry::getInstance().reset();
}

_Ret_maybenull_ hkLog::Output* hkLog::connectToFile(_In_z_ const char* pattern, _In_z_ const char* filename, Level::Enum level)
{
    return Registry::getInstance().connectToFile(pattern, filename, level);
}

void hkLog::connectToOutput(_In_z_ const char* pattern, _Inout_ hkLog::Output* output, Level::Enum level)
{
    Registry::getInstance().connectToOutput(pattern, output, level);
}

void hkLog::disconnect(_In_z_ const char* pattern, _Inout_ hkLog::Output* output)
{
    Registry::getInstance().disconnect(pattern, output);
}

void hkLog::enableLogs(_In_z_ const char* pattern, Level::Enum level)
{
    Registry::getInstance().enableLogs(pattern, level);
}

void hkLog::setBaseSystemLogLevel(Level::Enum level)
{
    Registry::getInstance().setBaseSystemLogLevel(level);
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
