// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/System/Io/Writer/Compressed/hkCompressedStreamWriter.h>
#include <Common/Base/Algorithm/Compression/hkCompression.h>

// GZIP compliant header
//  <name> <len>
//  ID1:    1   [0x1f]
//  ID2:    1   [0x8b]
//  CM:     1   [0x3b, our compression method identifier]
//  Flags:  1   [Default = 0]
//  MTIME:  4   [Modification time, in Unix format, default = 0, no time stamp is available]
//  XFL:    1   [Extra flags, default = 0]
//  OS:     1   [File system on which compression took place, default = 255, unknown]
//
hkCompressedStreamWriter::hkCompressedStreamWriter(_Inout_ hkStreamWriter* s, hkBool seekTellSupported, int bufSize)
    :   m_stream(s)
    , m_ok(true)
    , m_bufSize(bufSize)
    , m_uncompbufpos(0)
    , m_uncomplen(0)
    , m_seekTellSupported(seekTellSupported)
    , m_totalwritten(0)
    , m_crc(0)
{
    HK_ASSERT_NO_MSG(0x4325ab34, m_stream != HK_NULL);
    m_stream->addReference();
    m_uncompbufsize = bufSize - hkCompression::BLOCK_HEADER_SIZE;
    m_uncompbuf = hkMemHeapBlockAlloc<hkUchar>(m_uncompbufsize);

    // Write GZIP compliant header
    hkUchar header[10];
    header[0] = 0x1f; // ID1
    header[1] = 0x8b; // ID2
    header[2] = 0xb5; // CM
    header[3] = 0; // FLG
    *((hkUint32Be*)(&header[4])) = 0; // MTIME
    header[8] = 0; // XFL
    header[9] = 0xff; // OS
    m_stream->write(header, 10);
}

// GZIP compliant footer
//  <name> <len>
//  CRC32   4   [Cyclic Redundancy Check value of the uncompressed data]
//  ISIZE   4   [Size of the uncompressed input, modulo 2^32]
//
hkCompressedStreamWriter::~hkCompressedStreamWriter()
{
    if(m_uncompbufpos)
    {
        writeBlock();
    }
    // Write GZIP compliant footer
    hkUchar footer[8];
    *((hkUint32Be*)(&footer[0])) = m_crc;
    *((hkUint32Be*)(&footer[4])) = m_totalwritten;
    m_stream->write(footer, 8);
    m_stream->removeReference();
    hkMemHeapBlockFree(m_uncompbuf, m_uncompbufsize);
}

_Ret_range_(0, nbytes) int hkCompressedStreamWriter::write(_In_reads_bytes_(nbytes) const void* buffer, int nbytes)
{
    if (!m_ok)
    {
        return 0;
    }

    hkUchar* cbuf = (hkUchar*)buffer;
    int total_copied = 0;
    while (nbytes > 0)
    {
        int copy = hkMath::min2(m_uncompbufsize - m_uncompbufpos, nbytes);

        // To support seek/tell, keep growing buffer for uncompressed data, only compress when finished writing.
        if (m_seekTellSupported && copy < nbytes)
        {
            int old_size = m_uncompbufsize;
            m_uncompbufsize = hkMath::max2(hkLosslessCast<int>(1.5 * m_uncompbufsize), m_uncompbufsize + nbytes);
            m_bufSize = m_uncompbufsize + hkCompression::BLOCK_HEADER_SIZE;
            m_uncompbuf = hkMemHeapBufRealloc(m_uncompbuf, old_size, m_uncompbufsize);
            copy = nbytes;
        }

        hkString::memCpy(m_uncompbuf + m_uncompbufpos, cbuf, copy);

        m_uncompbufpos += copy;
        if (m_uncompbufpos > m_uncomplen)
        {
            m_uncomplen = m_uncompbufpos;
        }
        nbytes -= copy;
        cbuf += copy;
        total_copied += copy;

        if (!m_seekTellSupported && m_uncompbufpos == m_uncompbufsize)
        {
            writeBlock();
            if (!isOk())
            {
                return 0; 
            }
        }
    }
    return total_copied;
}

void hkCompressedStreamWriter::writeBlock()
{
    const void* in = m_uncompbuf;
    void* out_start = hkMemTempBlockAlloc<char>(m_bufSize);
    void* out_end = out_start;
    hkCompression::Result res = hkCompression::compress(in, m_uncompbufpos, out_end, m_bufSize, m_bufSize);
    HK_ASSERT_NO_MSG(0x4324992a, res == hkCompression::COMP_NEEDINPUT);
    m_stream->write(out_start, hkGetByteOffsetInt(out_start, out_end) );
    hkMemTempBlockFree(out_start, m_bufSize);
    m_totalwritten += m_uncompbufpos;
    m_crc = hkHash::appendCrc32(m_crc, m_uncompbuf, m_uncompbufpos);
    m_uncompbufpos = 0;
}

void hkCompressedStreamWriter::flush()
{
    writeBlock();
    m_stream->flush();
}

hkBool hkCompressedStreamWriter::isOk() const
{
    return m_ok && m_stream->isOk();
}

hkBool hkCompressedStreamWriter::seekTellSupported() const
{
    return m_seekTellSupported;
}

hkResult hkCompressedStreamWriter::seek(int offset, SeekWhence whence)
{
    if (!m_seekTellSupported)
    {
        HK_ASSERT_NO_MSG(0x2f2d9d5c, 0);
        return HK_FAILURE;
    }
    int pos = m_uncompbufpos;
    switch (whence)
    {
        case STREAM_SET:
            pos = offset;
            break;
        case STREAM_CUR:
            pos += offset;
            break;
        case STREAM_END:
            pos = m_uncomplen - offset;
            break;
    }
    if (pos >= 0 && pos <= m_uncomplen)
    {
        m_uncompbufpos = pos;
        return HK_SUCCESS;
    }
    // Invalid seek offset, file position is unaffected and error indicator for stream is set
    m_ok = false;
    return HK_FAILURE;
}

int hkCompressedStreamWriter::tell() const
{
    if (!m_seekTellSupported)
    {
        HK_ASSERT_NO_MSG(0x3e6eb54c, 0);
        return -1;
    }
    return m_uncompbufpos;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
