// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/System/Io/Writer/hkStreamWriter.h>

//IGNORE_EXTERN_REFLECTED
//HK_HAVOK_ASSEMBLY_EXCLUDE_FILE


    /// Class which handles large memory streams in memory by representing the stream as an
    /// array of sectors. Sectors can optionally be unloaded after they have been read by
    /// setting unloadSectorsAfterRead to true. If unloading sectors after read, all reading
    /// must be sequential.
class HK_EXPORT_COMMON hkMemoryTrack
{
    public:

        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE_CLASS, hkMemoryTrack );

            /// Ctor
        hkMemoryTrack( int numBytesPerSector = 512 );

        ~hkMemoryTrack();

            /// clear the stream
        void clear();

            /// append numBytes to the stream
        void write (_In_reads_bytes_(numBytes) const void* data, int numBytes );

            /// Unloads any sectors that have been read.
        void unloadReadSectors();

        /// Append other stream to this one, will free the memory used in the other track
        void appendByMove(_Inout_ hkMemoryTrack* other );

            /// Get the total number of bytes that have been written to the track.
        int getSize() const
        {
            return (m_sectors.getSize() - 1 + m_numSectorsUnloaded) * m_numBytesPerSector + m_numBytesLastSector;
        }

            /// Get the number of bytes that are available for reading.
        int getAvailable() const
        {
            return getSize() - m_numBytesRead;
        }

            /// Get the number of bytes that have been read from the track
        int getNumBytesRead() const
        {
            return m_numBytesRead;
        }

            /// Gets if the track has read all data written to it.
        bool hasReadAllData() const
        {
            return m_numBytesRead == getSize();
        }

            /// Read numBytes from the track.
        void read(_Out_writes_bytes_(numBytes) void* data, int numBytes );

            /// Seek to the start of the track
        void reset();

        //
        //  Member variables
        //
    public:
        int m_numBytesPerSector;                ///< The number of bytes per sector
        int m_numBytesLastSector;               ///< The number of bytes used in the last sector

        int m_numBytesRead;                     ///< The number of bytes read from the track
        int m_numSectorsUnloaded;               ///< The number of sectors that were unloaded after reading.

        hkArray<hkUint8*> m_sectors;    ///< The sectors
};


/// Writer which uses an hkMemoryTrack as its storage.
class HK_EXPORT_COMMON hkMemoryTrackStreamWriter : public hkStreamWriter
{
    public:
        HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE);
        enum TrackOwnership
        {
            TRACK_TAKE,
            TRACK_BORROW
        };


        /// Create an hkMemoryTrackStreamWriter which writes in to the memory track m_track.
        /// If o is TRACK_TAKE, this object owns the array and will destroy
        /// it in this objects destructor. If o is TRACK_BORROW, the track
        /// will not be deleted.
        hkMemoryTrackStreamWriter(_In_ hkMemoryTrack* track, TrackOwnership o)
            : m_track(track), m_ownerShip(o)
        {
        }


        virtual ~hkMemoryTrackStreamWriter()
        {
            if( m_ownerShip == TRACK_TAKE )
            {
                delete m_track;
            }
        }

        virtual void clear();

        virtual _Ret_range_(0, size) int write(_In_reads_bytes_(size) const void* mem, int size);

        virtual hkBool isOk() const
        {
            return true;
        }

        virtual hkBool seekTellSupported() const {  return false;   }
        virtual hkResult seek(int offset, SeekWhence whence){ return HK_FAILURE; }
        virtual int tell() const{   return m_track->getSize();  }

    protected:

        hkMemoryTrack* m_track;
        TrackOwnership m_ownerShip;
};



#include <Common/Base/System/Io/Reader/hkStreamReader.h>

/// Wraps a reader around a memory track.
class HK_EXPORT_COMMON hkMemoryTrackStreamReader : public hkStreamReader
{
    public:
    HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE);
        ///
        enum MemoryType
        {
            MEMORY_COPY,
            MEMORY_TAKE,
            MEMORY_INPLACE
        };

        /// Create a stream from the specified track.
        /// MEMORY_TAKE will take ownership of the memory (which was
        /// allocated with hkAllocate) and hkDeallocate it on destruction.
        /// MEMORY_INPLACE will use the memory in place
        /// and must exist for the lifetime of this object.
        hkMemoryTrackStreamReader(_In_ const hkMemoryTrack* track, MemoryType t, bool unloadSectorsAfterRead = false);

        ~hkMemoryTrackStreamReader();

        virtual _Ret_range_(0, nbytes) int read(_Out_writes_bytes_(nbytes) void* buf, int nbytes);

        virtual _Ret_range_(0, nbytes) int skip( int nbytes);

        virtual hkBool isOk() const { return m_track->getSize() > m_overflowOffset; }

    protected:

        const hkMemoryTrack* m_track;
        int m_overflowOffset;
        MemoryType m_memType; // owned or referenced
        bool m_unloadSectorsAfterRead;
};


/// Writer which uses an hkArray as its storage.
/// The written buffer area is from [0, hkArray.getSize()].
/// This class maintains a single null byte directly after
/// the buffer area so the buffer may be interpreted
/// as a c style string.
class HK_EXPORT_COMMON hkArrayStreamWriter : public hkStreamWriter
{
    public:
    HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE);
        enum ArrayOwnership
        {
            ARRAY_TAKE, ///< take ownership of a created array
            ARRAY_BORROW, ///< reference an existing array
            ARRAY_INTERNAL ///< create and manage an internal array
        };

        void nullTerminate()
        {
            m_arr->_reserve(m_allocator, m_arr->getSize()+1);
            m_arr->begin()[ m_arr->getSize() ] = 0;
        }

        _Ret_notnull_ void* getData() { return m_arr->begin(); }
        int getDataSize() const { return m_arr->getSize(); }

            /// Create with an internally managed array.
            /// Access the memory with getData() and getDataSize()
        hkArrayStreamWriter();

            /// Create an hkStreamWriter which writes in to the array *arr.
            /// The write position is initially set to the end of the array.
            /// If o is ARRAY_TAKE, this object owns the array and will destroy
            /// it in this objects destructor. If o is ARRAY_BORROW, the array
            /// will not be deleted.
            /// A baseOffset may be given so that seek and tell will consider the start of the array
            /// to be at the given offset.
        hkArrayStreamWriter(_In_ hkArray<char>* arr, ArrayOwnership o, hkLong baseOffset = 0)
            : m_arr(arr)
            , m_allocator(hkArray<char>::allocator_type().get(arr))
            , m_offset(arr->getSize())
            , m_baseOffset(baseOffset)
            , m_ownerShip(o)
        {
            nullTerminate();
        }

            /// Generic Constructor independent from the allocator used
        hkArrayStreamWriter(_In_ hkArrayBase<char>* arr, hkMemoryAllocator& allocator, ArrayOwnership o, hkLong baseOffset=0)
            : m_arr(arr), m_allocator(allocator), m_offset(arr->getSize()), m_baseOffset(baseOffset), m_ownerShip(o)
        {
            nullTerminate();
        }

        ~hkArrayStreamWriter();

        virtual void clear();

        virtual _Ret_range_(0, size) int write(_In_reads_bytes_(size) const void* mem, int size);

        virtual hkBool isOk() const
        {
            return true;
        }

        virtual hkBool seekTellSupported() const
        {
            return true;
        }

        virtual hkResult seek(int offset, SeekWhence whence);

        virtual int tell() const
        {
            return hkLosslessCast<int>(m_offset + m_baseOffset);
        }

    protected:

        hkArrayBase<char>* m_arr; // written chars always in area 0,getSize()
        hkMemoryAllocator& m_allocator; // allocator used for the array
        int m_offset; // invariant: m_offset <= m_arr.getSize()
        hkLong m_baseOffset;
        ArrayOwnership m_ownerShip;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
