// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/System/Io/Util/hkLoadUtil.h>
#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/Container/hkContainerAllocators.h>

#define DEBUG_LOG_IDENTIFIER "io.LoadUtil"
#include <Common/Base/System/Log/hkLog.hxx>

hkLoadUtil::hkLoadUtil(_In_opt_z_ const char* fname)
    : m_fileName(fname)
{
}

hkLoadUtil::hkLoadUtil(_Inout_ hkStreamReader* sr )
    : m_fileName(HK_NULL), m_reader(sr)
{
}

hkLoadUtil::hkLoadUtil( hkIstream& is )
    : m_fileName(HK_NULL), m_reader(is.getStreamReader())
{
}

hkLoadUtil::~hkLoadUtil()
{
    //out of line for refptr
}

// outputs

bool hkLoadUtil::toArray( hkArrayBase<char>& out, hkMemoryAllocator& mem )
{
    hkRefPtr<hkStreamReader> sr;
    hkInt64 oldOffset = -1;
    hkInt64 size = -1;
    if( m_fileName )
    {
        hkFileSystem::Entry entry;
        hkResult res = hkFileSystem::getInstance().stat(m_fileName, entry);
        if(res.isSuccess())
        {
            sr = entry.openReader();
            if (sr == HK_NULL)
            {
                return false;
            }
            size = (int)entry.getSize();
            out._reserve(mem, out.getSize() + int(size));
        }
        else if(res.isExactly(HK_E_NOT_IMPLEMENTED))
        {
            // Stat is not currently implemented by the server filesystem
            sr = hkFileSystem::getInstance().openReader(m_fileName);
            if( sr == HK_NULL )
            {
                Log_Warning("Failed to stat {}", m_fileName);
                return false;
            }
        }
        else
        {
            Log_Warning("Failed to stat {}", m_fileName);
            return false;
        }
    }
    else if( m_reader && m_reader->isOk() )
    {
        sr = m_reader;
        if( hkSeekableStreamReader* s = sr->isSeekTellSupported() )
        {
            oldOffset = s->tell();
            if ( s->seek(0, hkSeekableStreamReader::STREAM_END).isSuccess() )
            {
                int streamSize = s->tell();
                s->seek((int)oldOffset, hkSeekableStreamReader::STREAM_SET);
                out._reserve( mem, streamSize );
            }
        }
    }
    else
    {
        Log_Warning("No input given to hkLoadUtil");
        return false; // no source available
    }

    bool ret;
    while( true )
    {
        const int N = ( 1<<16 );
        char buf[N];
        int n = sr->read( buf, N );
        if( n > 0 )
        {
            const int prevSize = out.getSize();

            // Resize the array
            if ( out._trySetSize(mem, prevSize + n).isSuccess() )
            {
                hkMemUtil::memCpy( &out[prevSize], buf, n );
            }
            else
            {
                // Out of memory
                ret = false;
                Log_Warning("Input too large, out of memory ({})", prevSize+n);
                break;
            }

        }
        else if( n == 0 )
        {
            ret = true;
            break;
        }
        else // n<0
        {
            ret = false;
            break;
        }
    }

    if( oldOffset != -1 )
    {
        sr->isSeekTellSupported()->seek( int(oldOffset), hkSeekableStreamReader::STREAM_SET);
    }
    return ret;
}


bool hkLoadUtil::toArray( hkArray<char>::Temp& out )
{
    return toArray( out, hkContainerTempAllocator().get(&out) );
}


bool hkLoadUtil::toArray( hkArray<char>& out )
{
    return toArray( out, hkContainerHeapAllocator().get(&out) );
}

_Ret_maybenull_
void* hkLoadUtil::toAllocation( _Out_ int* sizeOut, hkMemoryAllocator& mem )
{
    hkArray<char>::Temp temp;
    if( toArray(temp) )
    {
        *sizeOut = temp.getSize();
        void* p = mem.blockAlloc( temp.getSize() );
        hkMemUtil::memCpy(p, temp.begin(), temp.getSize() );
        return p;
    }
    return HK_NULL;
}

bool hkLoadUtil::toString( hkStringBuf& buf )
{
    // Pop back null at the end of the string
    HK_ASSERT_NO_MSG(0x528933f, buf.getArray().back() == 0);
    buf.getArray().popBack();
    bool ret = toArray(buf.getArray());
    buf.getArray().pushBack(0);
    return ret;
}

bool hkLoadUtil::toString( hkStringPtr& buf )
{
    hkArray<char>::Temp temp;
    if( toArray(temp) )
    {
        buf.set( temp.begin(), temp.getSize() );
        return true;
    }
    return false;
}

int hkLoadUtil::toString( _Out_writes_bytes_(bufSize) char* buf, int bufSize )
{
    hkArray<char>::Temp temp;
    if( toArray(temp) )
    {
        int len = hkMath::min2( temp.getSize(), bufSize-1 );
        hkMemUtil::memCpy(buf, temp.begin(), len );
        buf[len] = 0;
        return len;
    }
    return 0;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
