// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL !REFLECT
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Types/hkEndian.h>
#include <Common/Base/Serialize/Detail/hkWriteBuffer.h>
#include <Common/Base/Serialize/Detail/hkReadBuffer.h>

namespace hkIo { class ReadBufferView; }

namespace hkSerialize
{
    typedef hkUint32 Ident;
    enum { SCOPE_BITS = 2, PACK_SHIFT = (sizeof(hkUint32) * 8) - SCOPE_BITS };

    inline hkInt32 pack(hkInt32 sm, hkLong big) { return (sm<<PACK_SHIFT) | hkLosslessCast<int>(big); }
    inline hkInt32 unpacks(hkInt32 packed) { return hkUint32(packed)>>PACK_SHIFT; }
    inline hkInt32 unpackb(hkInt32 packed) { return (hkUint32(-1)>>SCOPE_BITS) & packed; }

        /// Structured file format.
        /// Each section begins with an eight byte header followed by its data. Sections may be nested.
        /// The header is two four byte big endian integers. The top two bits encode the type of
        /// section and the bottom 30 bits are the size. Currently only 2 section types are supported ;
        /// "Branch" where the contents are themselves sections, and "Leaf" which has arbitrary content.
        /// The second value is a name or tag for the section. By convention, these are often packed ascii bytes.
    struct HffBase
    {
        enum ScopeType { BRANCH=0, LEAF=1 /*SPARE=2,SPARE=3*/ };
        enum IdentType { IDENT_NONE=0, IDENT_IO_ERROR=1 };
    };


        /// Utility to create an Hff file where the data can be traversed as it is written.
    class HffWriter : public HffBase
    {
        public:

            HffWriter(hkIo::WriteBuffer& buf) : m_buf(buf) {}
            ~HffWriter() { HK_ASSERT_NO_MSG(0x39ee3ab4, m_scopes.getSize() == 0); }

                /// Open a branch (contents must be branches/leaves)
            void openBranch(Ident id) { openScope(BRANCH, id); }
                /// Open a leaf (can only contain raw data)
            void openLeaf(Ident id) { openScope(LEAF, id); }
                /// Close the most recent branch or leaf.
            void close() { closeScope(); }

                /// Shortcut to openLeaf(), write & close() the leaf.
            void writeLeaf(Ident id, _In_reads_bytes_(len) const void* buf, hkInt32 len);

        protected:

            struct Scope
            {
                Scope(ScopeType s, hkLong size) : val(pack(s, hkLosslessCast<hkInt32>(size))) {}
                hkLong off() const { return unpackb(val); }
                ScopeType type() const { return (ScopeType)unpacks(val); }
                hkInt32Be val;
            };
            hkInplaceArray<Scope,16> m_scopes;

            void openScope(ScopeType s, Ident id);
            void closeScope();

            hkIo::WriteBuffer& m_buf;
    };


        /// Utility to create an Hff file where multiple leaves can be open at the same time.
        /// Contrast this to the HffWriter which can only write in a single pass.
    class HffBuilder : public HffBase
    {
        public:

            typedef int BranchId;

            virtual ~HffBuilder();

                /// Begin a stream.
            virtual BranchId openRoot(Ident id) { return openBranch(-1, id); }

                /// Begin a branch with the given parent.
            virtual BranchId openBranch(BranchId parent, Ident id) = 0;

                /// Begin writing a leaf node.
                /// All writebuffers may be used in any order until commit() is called.
            virtual _Ret_notnull_ hkIo::WriteBuffer* openLeaf(BranchId parent, Ident id) = 0;

                /// Finalize the write, any further use of buffer or this object is an error.
            virtual void close() = 0;
    };

        /// Creates a structured stream by buffering the contents
    class HffMemoryBuilder : public HffBuilder
    {
        public:

            HffMemoryBuilder() : m_closed(false) {}
            ~HffMemoryBuilder();

            virtual BranchId openBranch(BranchId parent, Ident id) HK_OVERRIDE;
            virtual _Ret_notnull_ hkIo::WriteBuffer* openLeaf(BranchId parent, Ident id) HK_OVERRIDE;
            virtual void close() HK_OVERRIDE;
            void writeTo(const hkIo::Detail::WriteBufferAdapter& sink);

            typedef void (*Callback)(Ident ident, _In_reads_bytes_(dataSize) const void* data, hkLong dataSize, _Inout_opt_ void* cbData);
            void walkOver(Callback cb, _Inout_opt_ void* cbData);

        protected:

            void reset();

            struct Buffer;
            struct Node
            {
                HK_DECLARE_CLASS(Node, New);
                Node(Ident i, BranchId p, Buffer* b) : id(i), parent(p), buf(b) {}
                Ident id;
                int parent;
                Buffer* buf;
            };
            hkArray<Node> m_nodes;
            bool m_closed;
    };


        /// Utility to parse Hff files.
        /// May be templated on either hkIo::ReadBuffer (streaming) or hkIo::ReadBufferView (in-memory)
    template<typename READER>
    class HffReaderBase : public HffBase
    {
        #if defined(HK_ARCH_ARM)
        HK_ALWAYS_INLINE hkUint32Be unalignedLoad32(_In_reads_bytes_(sizeof(hkUint32Be)) const void* ube)
        {
            hkUint32Be u;
            hkMemUtil::memCpyOne<hkUint32Be>(&u, reinterpret_cast<const hkUint32Be*>(ube));
            return u;
        }
        #else
        HK_ALWAYS_INLINE hkUint32Be unalignedLoad32(_In_reads_bytes_(sizeof(hkUint32Be)) const void* ube)
        {
            return *reinterpret_cast<const hkUint32Be*>(ube);
        }
        #endif

        public:

            template<typename T>
            HffReaderBase(T& t ) : m_buf(t), m_scopeEnd(hkTrait::NumericLimits<hkLong>::maxValue()) {}

            HffReaderBase() : m_scopeEnd(hkTrait::NumericLimits<hkLong>::maxValue()) {}

                /// It is an error
            ~HffReaderBase()
            {
                HK_ASSERT(0x4b9c0772, m_scopes.isEmpty(), "Scopes still active, call abort() if there is an error");
            }

                /// Try to enter a scope.
                /// Returns the ident of the entered scope.
                /// or IDENT_IO_ERROR on truncated input
                /// or IDENT_NONE if a scope was not entered.
                /// Failing to enter can be because this branch is exhausted or the ident does not match.
            HK_NEVER_INLINE Ident enter(Ident ident = IDENT_NONE)
            {
                if(m_buf.tell() < m_scopeEnd)
                {
                    const int req = sizeof(hkUint32Be) * 2;
                    hkLong n = m_buf.prefetch(req);
                    if(n < req)
                    {
                        return IDENT_IO_ERROR; //truncated input
                    }
                    Ident thisIdent = unalignedLoad32( m_buf.template peekAt<hkUint32Be>(4) );
                    if (thisIdent == IDENT_NONE)
                    {
                        m_buf.status().setUnexpected();
                        return IDENT_IO_ERROR; // 0 ident not allowed
                    }
                    else if( ident == IDENT_NONE || ident == thisIdent )
                    {
                        hkUint32 packed = unalignedLoad32( m_buf.template peekAt<hkUint32Be>(0) );
                        hkLong curOff = m_buf.tell();
                        hkLong newEnd = curOff + unpackb(packed);
                        if ( (hkUlong(newEnd) > hkUlong(m_scopeEnd)) || (unpackb(packed) < 8) )
                        {
                            m_buf.status().setUnexpected();
                            return IDENT_IO_ERROR;
                        }
                        ScopeState& s = m_scopes.expandOne();
                        s.off = hkLosslessCast<int>(curOff);
                        s.packed = packed;
                        s.ident = thisIdent;
                        s.savedEnd = m_scopeEnd;
                        m_scopeEnd = newEnd;
                        m_buf.skip(req);
                        return s.ident;
                    }
                }
                return IDENT_NONE; // Failed to enter. Either ident mismatch or end of scope
            }

                /// Leave the most recently entered scope.
                /// The contents must have been consumed.
            void leave(Ident id=IDENT_NONE)
            {
                HK_ASSERT_NO_MSG(0x70d6894, id== IDENT_NONE || id==m_scopes.back().ident);
                if (m_buf.tell() != (m_scopes.back().off + m_scopes.back().scopeSize()))
                {
                    m_buf.status().setUnexpected(); //Error id 0x57154d7c
                    return;
                }
                m_scopeEnd = m_scopes.back().savedEnd;
                m_scopes.popBack();
            }

                /// Size of this scope contents NOT INCLUDING the scope header.
            hkLong sizeContent() const
            {
                return m_scopes.back().contentSize();
            }

                /// Size of this scope INCLUDING the scope header.
            hkLong sizeScope() const
            {
                return m_scopes.back().scopeSize();
            }

                /// Get the current scopes identifier
            Ident curIdent() const
            {
                const ScopeState& s = m_scopes.back();
                return s.ident;
            }

                /// Skip over the currently entered scope.
            hkResult skipContent()
            {
                const ScopeState& s = m_scopes.back();
                HK_ASSERT_NO_MSG(0x6a3fa2f6, s.off + 8 == m_buf.tell());
                m_buf.skip(s.contentSize());
                return m_buf.status().isOk() ? HK_SUCCESS : HK_FAILURE;
            }

                /// Call this on error to avoid asserting in the destructor.
            void abort()
            {
                m_scopes.clear();
            }

        protected:

            struct ScopeState
            {
                Ident ident;
                hkInt32 packed; // scopetype + size
                hkInt32 off;
                hkLong savedEnd; //m_scopeEnd of prev scope

                hkInt32 scopeSize() const
                {
                    hkInt32 size = unpackb(packed);
                    HK_ASSERT_NO_MSG(0x770b92ff, size >= 8);
                    return hkLosslessCast<int>(size);
                }
                hkInt32 contentSize() const { return scopeSize() - 8; }
            };

            READER m_buf;
            hkLong m_scopeEnd;
            hkInplaceArray<ScopeState, 16> m_scopes;
    };


    class HffMemoryReader : public HffReaderBase<hkIo::ReadBufferView>
    {
        public:

            HffMemoryReader(_In_reads_bytes_(len) const void* buf, hkLong len)
            {
                m_buf = hkIo::ReadBufferView(buf, len);
                m_scopeEnd = len;
            }

                /// Get a view of the current scope and advance the read position.
            hkIo::ReadBufferView view()
            {
                const ScopeState& s = m_scopes.back();
                HK_ASSERT_NO_MSG(0x6a3fa2f6, s.off+8 == m_buf.tell());
                return hkIo::ReadBufferView(m_buf.access<void>(s.contentSize()), s.contentSize());
            }
    };


    class HffStreamReader : public HffReaderBase<hkIo::ReadBuffer&>
    {
        public:

            HffStreamReader(hkIo::ReadBuffer& rb) : HffReaderBase<hkIo::ReadBuffer&>(rb) {}

                ///
            hkLong read(_Out_writes_bytes_(n) void* buf, hkLong n) { return m_buf.read(buf, n); }

            hkIo::ReadBuffer& getReadBuffer() { return m_buf; }
    };

    namespace Detail
    {
        struct HffSection
        {
            Ident m_ident;
            hkUint32 m_offset;
            hkUint32 m_size;
            bool m_isLeaf;
        };
        HK_EXPORT_COMMON hkResult indexHff(hkArray<HffSection>& out, _In_reads_bytes_(len) const void* buf, hkLong len);
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
