// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

// this: #include <Common/Base/System/Io/Socket/hkSocket.h>


#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Base/System/Io/Writer/hkStreamWriter.h>

#include <Common/Base/System/Io/Socket/hkInetAddr.h>

class hkSocketImpl;
class hkStringBuf;
/// A simple platform independent socket.

class HK_EXPORT_COMMON hkSocket : public hkReferencedObject
{
    public:

        enum SOCKET_EVENTS
        {
            SOCKET_NOEVENTS = 0, // register for this only to get back to a blocking socket (default)
            SOCKET_CAN_READ = 1, // Data has arrived to read
            SOCKET_CAN_WRITE = 2, // Space is now available to write
            SOCKET_CONNECT = 4, // A connect has succeeded
            SOCKET_CLOSED = 8, // The socket has closed.
            SOCKET_ALLEVENTS = 0xff // register for all of the above
        };

    public:

        HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE);
        enum ERROR_CODES
        {
            SOCKET_NO_ERROR        = 0,
            SOCKET_WOULD_BLOCK     = 1,
            SOCKET_CONN_RESET      = 2,
            SOCKET_NOT_INITIALIZED = 3,
            SOCKET_ADDR_IN_USE     = 4,
            SOCKET_NET_DOWN        = 5
        };

        static hkSocket* (HK_CALL *create)();

            /// Return true if the connection is still live.
        virtual hkBool isOk() const = 0;

            /// Close the connection.
        virtual void close() = 0;

            /// Read at most nbytes into buf. Return number of bytes read.
        virtual int read(_Out_writes_bytes_(nbytes) void* buf, int nbytes) = 0;

            /// Write at most nbytes from buf. Return number of bytes written.
        virtual int write(_In_reads_bytes_(nbytes) const void* buf, int nbytes) = 0;

            /// Connect to host 'remote' at specified port.
        virtual hkResult connect(_In_z_ const char* remote, int port) = 0;

            /// Connect to host located at remoteAddr.
        virtual hkResult connect( hkInetAddr remoteAddr) = 0;

            /// Start listening for connections on the specified port.
        virtual hkResult listen(int port) = 0;

            /// Get notification of network events instead of having to poll for them.
        virtual hkResult bind(int port) = 0;

            /// Get the local address (IP:PORT) to which the socket is bound to
        virtual hkResult getInetAddr(hkInetAddr& inetAddr) = 0;

            /// Returns true if the socket can write, this is an indication the socket has fully connected.
        virtual bool canWrite() const { return true; }

            /// Returns true if the socket has data available to read.
        virtual bool canRead() const { return false; /*safer to assume can't than spin forever on blocking reads*/ }

            /// Check for new connections on a socket which we are listen()ing on.
            /// Returns NULL if there are no new connections. remoteAddr will contain the
            /// address of the remote endpoint to which the new connection was established.
        virtual _Ret_maybenull_ hkSocket* pollForNewClient(_Inout_opt_ hkInetAddr* remoteAddr = HK_NULL) = 0;

            /// Creates and configure the socket as a datagram (UDP) socket. Datagram sockets are
            /// connectionless and can be used for broadcasting. port is the portnumber of the socket
            /// to be bound to.
        virtual hkResult createDatagramSocket(int port, int listenAddress = 0){ return HK_FAILURE;}

            /// Sends data to the specified remote address. Can only be used for datagram (UDP) sockets.
        virtual int sendTo(_In_reads_bytes_(nbytes) const void* buf, int nbytes, const hkInetAddr& toAddress) { return 0; }

            /// Receives a datagram of a maximum size of nbytes. Any data received is stored in buf, fromAddress
            /// contains the address:port of the sender.
        virtual int receiveFrom(_Out_writes_bytes_(nbytes) void* buf, int nbytes, hkInetAddr &fromAddress) { return 0; }

            /// Get the error code of the last operation.
        virtual int getLastError() = 0;

            /// Get the socket address and port.
        virtual void getAddress( hkStringBuf& hostOut, int& portOut ) const;
            /// Set the socket to perform either blocking or non-blocking IO
        virtual hkResult setBlocking(hkBool blocking) = 0;

            /// Get a stream reader for this connection.
        hkStreamReader& getReader() { return m_reader; }

            /// Get a stream writer for this connection.
        hkStreamWriter& getWriter() { return m_writer; }

            /// This flag is used to ensure the network is only set up when a socket is created.
            /// Initially set to false this is check in the base hkSocket constructor and when false calls hkSocket::s_platformNetInit
        static hkBool s_platformNetInitialized;

            /// Pointers to default network initialization function.
            /// This is called during hkBaseSystem::init().
            /// Setting this to HK_NULL before hkBaseSystem::init() is called will bypass initialization.
        static void (HK_CALL *s_platformNetInit)(void);

            /// Pointers to default network shutdown function.
            /// This is called during hkBaseSystem::quit().
            /// Setting this to HK_NULL before hkBaseSystem::quit() is called will bypass shutdown.
        static void (HK_CALL *s_platformNetQuit)(void);

            /// Get the host's IP address or name
        static void (HK_CALL *s_platformGetAddressString)(hkStringBuf& addr);

    protected:

        hkSocket();

    private:

        class ReaderAdapter : public hkStreamReader
        {
            public:
                HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE);
                ReaderAdapter() : m_socket(HK_NULL) { }
                virtual _Ret_range_(0, nbytes) int read(_Out_writes_bytes_(nbytes) void* buf, int nbytes );
                virtual hkBool isOk() const;

                hkSocket* m_socket;
        };

        class WriterAdapter : public hkStreamWriter
        {
            public:
                HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE);
                WriterAdapter() : m_socket(HK_NULL) { }
                virtual _Ret_range_(0, nbytes) int write(_In_reads_bytes_(nbytes) const void* buf, int nbytes );
                virtual hkBool isOk() const;

                hkSocket* m_socket;
        };

        ReaderAdapter m_reader;
        WriterAdapter m_writer;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
