// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/System/Io/Socket/hkSocket.h>
#include <Common/Base/Container/String/hkStringBuf.h>

namespace
{
    static void HK_CALL dummyGetAddressString(hkStringBuf& addr)
    {
        addr = "unknown";
    }
}

#if defined(HK_PLATFORM_PSP)

// PSP(R) (PlayStation(R)Portable) not currently supported
    #include <Common/Base/System/Io/Platform/Psp/hkPspSocket.h>
    static _Ret_notnull_ hkSocket* HK_CALL hkSocketCreate()
    {
        return new hkPspSocket();
    }

    void (HK_CALL *hkSocket::s_platformNetInit)() = hkPspNetworkInit;
    void (HK_CALL *hkSocket::s_platformNetQuit)() = hkPspNetworkQuit;
    void (HK_CALL *hkSocket::s_platformGetAddressString)(hkStringBuf&) = dummyGetAddressString;

#elif defined(HK_PLATFORM_PSVITA)

// PlayStation(R)Vita socket implementation
    #include <Common/Base/System/Io/Platform/PsVita/hkPsVitaSocket.h>
    static _Ret_notnull_ hkSocket* HK_CALL hkSocketCreate()
    {
        return new hkPsVitaSocket();
    }

    void (HK_CALL *hkSocket::s_platformNetInit)() = hkPsVitaNetworkInit;
    void (HK_CALL *hkSocket::s_platformNetQuit)() = hkPsVitaNetworkQuit;
    void (HK_CALL *hkSocket::s_platformGetAddressString)(hkStringBuf&) = hkPsVitaGetAddressString;

#elif defined(HK_PLATFORM_WIIU)

    #if defined(HK_USE_DEPRECATED_HIO_CONNECTION)

// Deprecated Wii behavior using HIO (TCP/IP is recommended)
    #include <Common/Base/System/Io/Platform/WiiU/hkWiiuSocket.h>
    static _Ret_notnull_ hkSocket* HK_CALL hkSocketCreate()
    {
        return new hkWiiuSocket();
    }

    void (HK_CALL *hkSocket::s_platformNetInit)() = hkWiiuNetworkInit;
    void (HK_CALL *hkSocket::s_platformNetQuit)() = hkWiiuNetworkQuit;
    void (HK_CALL *hkSocket::s_platformGetAddressString)(hkStringBuf&) = dummyGetAddressString;

    #else // !defined(HK_DEPRECATED_HIO_CONNECTION)

// Default WiiU behavior using TCP/IP
    #include <Common/Base/System/Io/Platform/Bsd/hkBsdSocket.h> 
    static _Ret_notnull_ hkSocket* HK_CALL hkSocketCreate()
    {
        return new hkBsdSocket();
    }

    void (HK_CALL *hkSocket::s_platformNetInit)() = hkBsdNetworkInit;
    void (HK_CALL *hkSocket::s_platformNetQuit)() = hkBsdNetworkQuit;
    void (HK_CALL *hkSocket::s_platformGetAddressString)(hkStringBuf&) = hkBsdGetAddressString;
    #endif // !defined(HK_DEPRECATED_HIO_CONNECTION)

#elif 0 // defined(HK_PLATFORM_WINRT) : Let WINRT use WinSock, assumes UWP

    // Win RT
    #include <Common/Base/System/Io/Platform/WinRT/hkWinRTSocket.h>
    static _Ret_notnull_ hkSocket* HK_CALL hkSocketCreate()
    {
        return new hkWinRTSocket();
    }

    void (HK_CALL *hkSocket::s_platformNetInit)() = HK_NULL;
    void (HK_CALL *hkSocket::s_platformNetQuit)() = HK_NULL;
    void (HK_CALL *hkSocket::s_platformGetAddressString)(hkStringBuf&) = dummyGetAddressString;


#elif defined(HK_PLATFORM_NACL)

// Google Native Client, none for now
    static _Ret_null_ hkSocket* HK_CALL hkSocketCreate()
    {
        HK_WARN(0x1f65b352, "No socket implementation for this platform");
        return HK_NULL;
    }

    void (HK_CALL *hkSocket::s_platformNetInit)() = HK_NULL;
    void (HK_CALL *hkSocket::s_platformNetQuit)() = HK_NULL;
    void (HK_CALL *hkSocket::s_platformGetAddressString)(hkStringBuf&) = dummyGetAddressString;

#elif defined(HK_PLATFORM_PS4)

// PlayStation(R)4 socket implementation
    #include <Common/Base/System/Io/Platform/Ps4/hkPs4Socket.h>
    static _Ret_notnull_ hkSocket* HK_CALL hkSocketCreate()
    {
        return new hkPs4Socket();
    }

    void (HK_CALL *hkSocket::s_platformNetInit)() = hkPs4NetworkInit;
    void (HK_CALL *hkSocket::s_platformNetQuit)() = hkPs4NetworkQuit;
    void (HK_CALL *hkSocket::s_platformGetAddressString)(hkStringBuf&) = dummyGetAddressString;

#else

// General BSD socket support (LINUX, Windows, Mac, Xbox etc)
    #include <Common/Base/System/Io/Platform/Bsd/hkBsdSocket.h> 
    static _Ret_notnull_ hkSocket* HK_CALL hkSocketCreate()
    {
        return new hkBsdSocket();
    }

    void (HK_CALL *hkSocket::s_platformNetInit)() = hkBsdNetworkInit;
    void (HK_CALL *hkSocket::s_platformNetQuit)() = hkBsdNetworkQuit;
    void (HK_CALL *hkSocket::s_platformGetAddressString)(hkStringBuf&) = hkBsdGetAddressString;

#endif

hkBool hkSocket::s_platformNetInitialized = false;

hkSocket* (HK_CALL *hkSocket::create)() = hkSocketCreate;

hkSocket::hkSocket()
{
    m_reader.m_socket = this;
    m_writer.m_socket = this;

    if (s_platformNetInitialized == false && s_platformNetInit)
    {
        s_platformNetInit();
        s_platformNetInitialized = true;
    }
}

_Ret_range_(0, nbytes) int hkSocket::ReaderAdapter::read(_Out_writes_bytes_(nbytes) void* buf, int nbytes)
{
    char* cbuf = static_cast<char*>(buf);
    int size = 0;
    while( size < nbytes )
    {
        int r = m_socket->read(cbuf+size, nbytes-size);
        size += r;
        if( r == 0 )
        {
            return size;
        }
    }
    return nbytes;
}

hkBool hkSocket::ReaderAdapter::isOk() const
{
    return m_socket->isOk();
}

_Ret_range_(0, nbytes) int hkSocket::WriterAdapter::write(_In_reads_bytes_(nbytes) const void* buf, int nbytes)
{
    const char* cbuf = static_cast<const char*>(buf);
    int size = 0;
    while( size < nbytes )
    {
        int w = m_socket->write(cbuf+size, nbytes-size);
        size += w;
        if( w == 0 )
        {
            return size;
        }
    }
    return nbytes;

}

hkBool hkSocket::WriterAdapter::isOk() const
{
    return m_socket->isOk();
}

void hkSocket::getAddress( hkStringBuf& hostOut, int& portOut ) const
{
    s_platformGetAddressString(hostOut);
    portOut = -1;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
