// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/System/Io/Socket/hkInetAddr.h>
#include <Common/Base/System/Stopwatch/hkStopwatch.h>

class hkIArchive;
class hkOArchive;
class hkSocket;

    /// Provides a discovery service for client/server applications.
class hkNetLobby : public hkReferencedObject
{
    public:

        HK_DECLARE_CLASS(hkNetLobby, New);

        enum Status
        {
            STATUS_INITIALIZED, ///<
            STATUS_DISCOVERY_IN_PROGRESS, ///< Broadcast packet sent, awaiting replies.
            STATUS_DISCOVERY_FINISHED, ///<
            STATUS_ADVERTISING_SERVICE, ///< Listening and will respond to broadcast queries
        };

        enum
        {
            DEFAULT_DISCOVERY_PORT = 28000
        };

        struct StatusInfo
        {
            hkEnum<Status, int> m_status;
            hkReal m_secondsInThisState;
        };


            /// Container class to store session information. This is used by
            /// servers to advertise sessions on a LAN, and by clients when
            /// searching for available sessions.
        class SessionInfo
        {
            public:

                HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE, SessionInfo);

                enum { MAX_INFO_SIZE = 1024 };

                HK_EXPORT_COMMON SessionInfo();

                HK_EXPORT_COMMON SessionInfo(hkInetAddr& serverAddr, int serverID, _When_(infoSize != 0, _In_reads_bytes_(infoSize)) const char* sessionInfo = HK_NULL, _In_range_(0, MAX_INFO_SIZE) hkUint32 infoSize = 0);

                    /// The IP:PORT at which the session is hosted
                hkInetAddr m_serverAddr;

                    /// A unique server ID. This is used to distinguish between different
                    /// server applications that may advertise their sessions on the same LAN
                int m_serverID;

                    /// Additional session information, optional
                char m_sessionInfo[MAX_INFO_SIZE];

                    /// Size of the additional session information
                hkUint32 m_infoSize;
        };

            ///
        HK_EXPORT_COMMON hkNetLobby();

            ///
        HK_EXPORT_COMMON ~hkNetLobby();

            /// Advertises a session as specified in sessionInfo.
            /// discoveryPort specifies the UDP port number to be used for advertising
            /// the session. While advertising a session, call step() in regular intervals
            /// to process any network traffic.
        HK_EXPORT_COMMON hkResult advertiseSession(const SessionInfo& sessionInfo, int discoveryPort = DEFAULT_DISCOVERY_PORT);
        HK_EXPORT_COMMON hkResult advertiseSession(const SessionInfo& sessionInfo, const hkInetAddr& addr);

            /// Clear current results and find any available sessions. serverID specifies the server
            /// ID that we are interested in.  Sessions are only found if a server advertises sessions
            /// with a matching server ID.  discoveryPort specifies the UDP port number to be used.
            /// This function returns immediately, you must call step() to discover servers.
            /// step() will continue to find servers until discoveryTimeout has elapsed (or some error occurs)
            /// at which point the status becomes STATUS_DISCOVERY_FINISHED.
            /// To accumulate results, process the current results and call findSessions again.
        HK_EXPORT_COMMON hkResult findSessions(int serverID, hkUint16 discoveryPort = DEFAULT_DISCOVERY_PORT, hkReal discoveryTimeoutMs = 1000);
        HK_EXPORT_COMMON hkResult findSessions(int serverID, const hkInetAddr& addr, hkReal discoveryTimeoutMs = 1000);

            /// Advances the service. This handles any network traffic and processes incoming messages. Call this
            /// function repeatedly when advertising or searching for sessions.
        HK_EXPORT_COMMON void step();

            /// Get the number of available results.
        HK_EXPORT_COMMON int getNumResults() const;

            /// Clears the list of results from a previous discovery
        HK_EXPORT_COMMON void clearResults();

            /// Get a specific session info from the available result set.
            /// index specifies the element in the result set and has to be < getNumResults()
        HK_EXPORT_COMMON void getSessionInfo(SessionInfo &info, int index);

            /// Get the current status of the lobby.
        HK_EXPORT_COMMON Status getStatus() const;

            /// Get the current status of the lobby.
        HK_EXPORT_COMMON StatusInfo getStatusInfo() const;

    protected:

            /// Get the index of the session info for the particular addr
        int indexOfSessionInfo(const hkInetAddr& addr) const;

            /// Stores a session info in the result list
        void addSessionInfo(SessionInfo info);

            /// Poll for new servers through the client advertise socket.
        void pollClientAdvertiseSocket();

            /// The session info to be used when advertising a session
        SessionInfo m_sessionInfo;

            /// List of available results for recent findSessions
        hkArray<SessionInfo> m_results;

            /// The UDP socket to be used for discovering
        hkSocket* m_discoverySocket;

            /// A timeout for discovery, if this is zero or negative the net lobby will never timeout
            /// but instead re-broadcast discovery requests at each step.
        hkReal m_discoveryTimeoutMs;

            /// The UDP socket to be used for advertising
        hkSocket* m_advertiseSocket;

            /// Id representing the current group of servers that are of interest
        int m_serverID;

        struct StatusInternal
        {
            public:

                HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(0, StatusInternal);

                StatusInternal()
                {
                    m_status = STATUS_INITIALIZED;
                    m_stopwatch.start();
                }

                void setStatus( Status s )
                {
                    m_status = s;
                    m_stopwatch.reset();
                    m_stopwatch.start();
                }
                operator Status() const { return m_status; }
                hkReal getNumSeconds() const { return m_stopwatch.getElapsedSeconds(); }

            protected:

                hkStopwatch m_stopwatch; ///< Time we've been in this state
                Status m_status; ///< Current status
        };

        StatusInternal m_status;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
