// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/System/Io/Socket/hkNetLobby.h>

#include <Common/Base/System/Io/Socket/hkSocket.h>
#include <Common/Base/System/Io/OArchive/hkOArchive.h>
#include <Common/Base/System/Io/IArchive/hkIArchive.h>
#include <Common/Base/Serialize/StructuredBinaryStream/hkStructuredBinaryStream.h>

#define DEBUG_LOG_IDENTIFIER "common.base.network.netlobby"
#include <Common/Base/System/Log/hkLog.hxx>


namespace
{
    enum
    {
        LOBBY_MAX_MESSAGE_SIZE = 1 + sizeof(hkNetLobby::SessionInfo), // int8(type) + body
    };

    struct SessionInfoSerializer;

    struct WriteStream : public hkStructuredBinaryStreamWriter<hkStructuredBinaryStream::FixedWriterStream, SessionInfoSerializer>
    {
        WriteStream() : hkStructuredBinaryStreamWriter<hkStructuredBinaryStream::FixedWriterStream, SessionInfoSerializer>(m_stream) {}
        hkStructuredBinaryStream::InplaceWriterStream<LOBBY_MAX_MESSAGE_SIZE> m_stream;

        _Ret_notnull_ const void* begin() const { return m_stream.m_buffer; }
        int getSize() const { return m_stream.getUsed(); }
    };
    struct ReadStream : public hkStructuredBinaryStreamReader<SessionInfoSerializer>
    {
        ReadStream(_In_bytecount_(size) void* p, int size)
            : hkStructuredBinaryStreamReader<SessionInfoSerializer>(m_stream)
            , m_stream(p,size){}
        hkStructuredBinaryStream::ReaderStream m_stream;
    };

    enum
    {
            // int8 id; int32 serverIdRequested;
        MSG_DISCOVERY_BROADCAST = 1,
            // int8 id; sessionInfo
        MSG_DISCOVERY_REPLY = 2,
            // int8 id; int32 serverId; sessionInfo
        MSG_ADVERTISE_BROADCAST = 3,
    };

    struct SessionInfoSerializer
    {
        template< typename Writer >
        static void save(const hkNetLobby::SessionInfo& si, Writer& os)
        {
            os.writeData(si.m_serverAddr.m_ipAddress);
            os.writeData(si.m_serverAddr.m_port);
            os.writeData(si.m_serverID);
            os.writeData(si.m_infoSize);
            os.writeData(si.m_sessionInfo, si.m_infoSize);
        }
        template< typename Reader >
        static void load(hkNetLobby::SessionInfo& si, Reader& os)
        {
            os.readData(si.m_serverAddr.m_ipAddress);
            os.readData(si.m_serverAddr.m_port);
            os.readData(si.m_serverID);
            os.readData(si.m_infoSize);
            os.readData(si.m_sessionInfo, si.m_infoSize);
        }
    };
}

hkNetLobby::SessionInfo::SessionInfo():
m_serverID(0),
m_infoSize(0)
{
}

hkNetLobby::SessionInfo::SessionInfo(hkInetAddr& serverAddr, int serverID, _When_(infoSize != 0, _In_reads_bytes_(infoSize)) const char* sessionInfo, _In_range_(0, MAX_INFO_SIZE) hkUint32 infoSize):
m_serverAddr(serverAddr),
m_serverID(serverID),
m_infoSize(infoSize)
{
    if (infoSize > 0)
    {
        hkString::memCpy(m_sessionInfo, sessionInfo, infoSize);
    }
}

hkNetLobby::hkNetLobby():
m_discoverySocket(HK_NULL),
m_discoveryTimeoutMs(-1),
m_advertiseSocket(HK_NULL),
m_serverID(-1)
{
}

hkNetLobby::~hkNetLobby()
{
    if (m_discoverySocket)
    {
        delete m_discoverySocket;
        m_discoverySocket = HK_NULL;
    }
    if (m_advertiseSocket)
    {
        delete m_advertiseSocket;
        m_advertiseSocket = HK_NULL;
    }
}

hkResult hkNetLobby::advertiseSession(const SessionInfo& sessionInfo, int discoveryPort)
{
    return advertiseSession(sessionInfo, hkInetAddr(0, (hkUint16)discoveryPort));
}

hkResult hkNetLobby::advertiseSession(const SessionInfo& sessionInfo, const hkInetAddr& addr)
{
    hkResult result = HK_SUCCESS;

    
    m_sessionInfo = sessionInfo;
    if (m_status == STATUS_INITIALIZED || m_status == STATUS_DISCOVERY_FINISHED)
    {
        if (!m_advertiseSocket)
        {
            m_advertiseSocket = hkSocket::create();
            result = m_advertiseSocket->createDatagramSocket(addr.m_port, addr.m_ipAddress);
            if ( result.isFailure())
            {
                HK_WARN(0xfebadd01, "Failed to create datagram socket");
            }
        }
        m_status.setStatus(STATUS_ADVERTISING_SERVICE);
    }

    // Send out an initial broadcast so that existing clients can pick
    // up on this server without having to refresh at high frequency
    WriteStream wrs;
    wrs.writeData(hkUint8(MSG_ADVERTISE_BROADCAST));
    wrs.writeData(m_sessionInfo.m_serverID);
    wrs.writeData(m_sessionInfo);

    hkInetAddr broadcastAddr = hkInetAddr::getBroadcastAddr(hkUint16(addr.m_port));
    m_advertiseSocket->sendTo(wrs.begin(), wrs.getSize(), broadcastAddr);

    return result;
}

hkResult hkNetLobby::findSessions(int serverID, hkUint16 discoveryPort, hkReal discoveryTimeoutSec)
{
    return findSessions(serverID, hkInetAddr(0, discoveryPort), discoveryTimeoutSec);
}

hkResult hkNetLobby::findSessions(int serverID, const hkInetAddr& addr, hkReal discoveryTimeoutMs)
{
    hkResult discoveryResult = HK_SUCCESS;
    m_results.clear();
    m_serverID = serverID;
    if (!m_discoverySocket)
    {
        m_discoverySocket = hkSocket::create();
        m_discoveryTimeoutMs = discoveryTimeoutMs;
        discoveryResult = m_discoverySocket->createDatagramSocket(0, addr.m_ipAddress);
        if (discoveryResult.isFailure())
        {
            HK_WARN(0xfebadd02, "Failed to create datagram socket");
        }
    }

    hkResult advertiseResult = HK_SUCCESS;
    // also create a socket to listen for potential broadcast messages from newly created servers
    if (!m_advertiseSocket)
    {
        m_advertiseSocket = hkSocket::create();
        advertiseResult = m_advertiseSocket->createDatagramSocket(addr.m_port, addr.m_ipAddress);
        if (advertiseResult.isFailure())
        {
            HK_WARN(0xfebadd03, "Failed to create datagram socket");
        }
    }

    hkInetAddr broadcastAddr = hkInetAddr::getBroadcastAddr(hkUint16(addr.m_port));
    WriteStream wrs;
    wrs.writeData( hkUint8( MSG_DISCOVERY_BROADCAST ) );
    wrs.writeData( m_serverID );

    m_discoverySocket->sendTo( wrs.begin(), wrs.getSize(), broadcastAddr );
    m_status.setStatus(STATUS_DISCOVERY_IN_PROGRESS);

    return ( discoveryResult.isFailure() ) ? discoveryResult : advertiseResult;
}

void hkNetLobby::step()
{
    if (m_status == STATUS_DISCOVERY_IN_PROGRESS)
    {
        hkBool32 timeoutIsInstant = ( m_discoveryTimeoutMs <= 0 );
        hkBool32 timeoutElapsed = ( ( m_status.getNumSeconds() * 1000 ) >= m_discoveryTimeoutMs );

        if ( timeoutIsInstant || !timeoutElapsed )
        {
            char buf[LOBBY_MAX_MESSAGE_SIZE];
            hkString::memSet(buf, 0, sizeof(buf));
            hkInetAddr sender;
            int rcv = m_discoverySocket->receiveFrom(buf, sizeof(buf)-1, sender);
            if (rcv > 0)
            {
                if (buf[0] == MSG_DISCOVERY_REPLY)
                {
                    SessionInfo sessionInfo;

                    ReadStream rds(buf+1, rcv-1);
                    rds.readData(sessionInfo);
                    // In case the IP address in the session info is 0 (ie. it wasn't
                    // specified by the advertiser), replace it with the IP obtained
                    // from the socket.
                    if (sessionInfo.m_serverAddr.m_ipAddress == 0)
                    {
                        sessionInfo.m_serverAddr.m_ipAddress = sender.m_ipAddress;
                    }
                    char buf2[hkInetAddr::MAX_STRING_LEN];
                    Log_Info( "Found server at {}", sessionInfo.m_serverAddr.toString(buf2) );
                    addSessionInfo(sessionInfo);
                }
            }

            pollClientAdvertiseSocket();
        }

        if ( timeoutIsInstant || timeoutElapsed )
        {
            m_status.setStatus( STATUS_DISCOVERY_FINISHED );
            delete m_discoverySocket;
            m_discoverySocket = HK_NULL;
        }
    }

    else if (m_status == STATUS_ADVERTISING_SERVICE)
    {
        char buf[LOBBY_MAX_MESSAGE_SIZE];
        hkInetAddr sender;
        int rcvd = m_advertiseSocket->receiveFrom(buf, LOBBY_MAX_MESSAGE_SIZE, sender);
        if (rcvd > 0)
        {
            if (buf[0] == MSG_DISCOVERY_BROADCAST)
            {
                ReadStream rds(buf+1, rcvd-1);
                int serverID; rds.readData(serverID);
                if (m_sessionInfo.m_serverID == serverID)
                {
                    char buf2[hkInetAddr::MAX_STRING_LEN];
                    Log_Info( "Received discovery broadcast from {}", sender.toString( buf2 ) );

                    WriteStream wrs;
                    wrs.writeData( hkUint8( MSG_DISCOVERY_REPLY ) );
                    wrs.writeData( m_sessionInfo );
                    m_advertiseSocket->sendTo( wrs.begin(), wrs.getSize(), sender );
                }
            }
        }
    }

    // We are finished searching, but new servers may advertise their session when they startup
    else
    {
        pollClientAdvertiseSocket();
    }
}

int hkNetLobby::getNumResults() const
{
    return m_results.getSize();
}

void hkNetLobby::clearResults()
{
    m_results.clear();
}

void hkNetLobby::getSessionInfo(SessionInfo &info, int index)
{
    info = m_results[index];
}

hkNetLobby::Status hkNetLobby::getStatus() const
{
    return m_status;
}

hkNetLobby::StatusInfo hkNetLobby::getStatusInfo() const
{
    StatusInfo info;
    info.m_status = m_status;
    info.m_secondsInThisState = m_status.getNumSeconds();
    return info;
}

int hkNetLobby::indexOfSessionInfo( const hkInetAddr& addr ) const
{
    for ( int i = 0; i < m_results.getSize(); i++ )
    {
        const SessionInfo& info = m_results[i];
        if ( info.m_serverAddr == addr )
        {
            return i;
        }
    }
    return -1;
}

void hkNetLobby::addSessionInfo( SessionInfo info )
{
    int idx = indexOfSessionInfo( info.m_serverAddr );
    if ( idx != -1 )
    {
        m_results[idx] = info;
    }
    else
    {
        m_results.pushBack(info);
    }
}

void hkNetLobby::pollClientAdvertiseSocket()
{
    if (m_advertiseSocket)
    {
        char buf[LOBBY_MAX_MESSAGE_SIZE];
        hkInetAddr sender;
        int rcvd = m_advertiseSocket->receiveFrom(buf, LOBBY_MAX_MESSAGE_SIZE, sender);
        if (rcvd > 0)
        {
            if ( buf[0] == MSG_ADVERTISE_BROADCAST )
            {
                ReadStream rds(buf + 1, rcvd - 1);
                int serverID; rds.readData(serverID);
                if (m_serverID == serverID)
                {
                    char buf2[hkInetAddr::MAX_STRING_LEN];
                    Log_Info(
                            "Received advertise broadcast from {}",
                        sender.toString(buf2) );

                    SessionInfo sessionInfo;
                    rds.readData(sessionInfo);
                    if (sessionInfo.m_serverAddr.m_ipAddress == 0)
                    {
                        sessionInfo.m_serverAddr.m_ipAddress = sender.m_ipAddress;
                    }

                    addSessionInfo( sessionInfo );
                }
            }
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
