// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

class hkBufferedStreamReader;
class hkSeekableStreamReader;
class hkInplaceStreamReader;

/// A generic interface to reading a stream of bytes.
/// The reader may optionally support mark/rewind or may be
/// wrapped as a child stream of an hkBufferedStreamReader.
/// Derived classes need only override isOk() and read(). Usually
/// readers which do not support buffering are wrapped in an
/// hkBufferedStreamReader.
class HK_EXPORT_COMMON hkStreamReader : public hkReferencedObject
{
    public:

        HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_STREAM);

        hkStreamReader() {}
            /// Return false after we have tried to read past the end of file or some other error.
            /// Otherwise return true.
        virtual hkBool isOk() const = 0;

            /// Read 'nbytes' into the buffer 'buf'.
            /// Return the number of characters read or zero for end of file.
        virtual _Ret_range_(0, nbytes) int read(_In_reads_bytes_(nbytes) void* buf, int nbytes) = 0;

            /// Skip nbytes bytes of input.
            /// Return the actual number of bytes skipped.
            /// The default implementation uses read().
        virtual _Ret_range_(0, nbytes) int skip(int nbytes);

            /// Read at most n characters without changing the stream position.
            /// Returns the number of characters peeked.
        virtual _Ret_range_(0, nbytes) int peek(_Out_writes_bytes_(nbytes) void* buf, int nbytes);

            /// Return a seekable stream if seeking is supported on this stream. By default not supported.
        virtual _Ret_maybenull_ hkSeekableStreamReader* isSeekTellSupported() { return HK_NULL; }

            /// Return non-null if the stream is inplace (i.e. entirely contained in memory)
        virtual hkInplaceStreamReader* isInplace() { return HK_NULL; }
};

/// A stream reader which supports seek and tell.
class HK_EXPORT_COMMON hkSeekableStreamReader : public hkStreamReader
{
    public:

        HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_STREAM);

        virtual _Ret_maybenull_ hkSeekableStreamReader* isSeekTellSupported() { return this; }

            /// Parameter for seek method.
        enum SeekWhence { STREAM_SET=0, STREAM_CUR=1, STREAM_END=2 };

            /// Seek to offset from whence.
        virtual hkResult seek(int offset, SeekWhence whence) = 0;

            /// Get the current file offset if supported or -1 on error.
        virtual int tell() const = 0;
};

/// A stream which exists entirely in memory. It allows noncopying reads.
class HK_EXPORT_COMMON hkInplaceStreamReader : public hkSeekableStreamReader
{
    public:
        HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_STREAM);

            //
        virtual _Ret_maybenull_ hkInplaceStreamReader* isInplace() { return this; }

        virtual hkLong available() = 0;

            /// Access at most "len" bytes at the current position.
            /// Any subsequent non-const calls invalidate the returned view.
        virtual hkArrayView<void> peekInto(hkUlong nbytes) const = 0;
};

/// A stream reader that has a HANDLE which can be waited on:
class hkWaitableStreamReader : public hkStreamReader
{
public:
    virtual _Ret_maybenull_ void* getWaitHandle() { return nullptr; }
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
