// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64 DURANGO OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/System/Io/Platform/Win32/hkWin32FileSystem.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/Container/String/hkUtf8.h>
#include <Common/Base/System/Io/Platform/Stdio/hkStdioStreamReader.h>
#include <Common/Base/System/hkBaseSystem.h>
#include <Common/Base/Memory/System/hkMemorySystem.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>
#include <Common/Base/Thread/Thread/hkThread.h>
#include <Common/Base/Types/hkScopedPtr.h>
#include <Common/Base/Types/Uuid/hkUuid.h>

#include <Common/Base/System/Io/Platform/Win32/hkWin32StreamWriter.h>

#define HK_WIN32_FIND_DATA WIN32_FIND_DATAW
#define HK_WIDEN(X) hkUtf8::WideFromUtf8(X).cString()
#define HK_NARROW(X) hkUtf8::Utf8FromWide(X).cString()
#define HK_STRCMP(VAR,CONST_STRING) wcscmp(VAR, L ## CONST_STRING)
#include <windows.h>

#define DEBUG_LOG_IDENTIFIER "common.base.filesystem"
#include <Common/Base/System/Log/hkLog.hxx>

static void s_convertFileNameToNative(hkStringBuf& filename)
{
    filename.replace('/', '\\');
}

#include <io.h>

//Keep me to compute the #100ns since the epoch
//  SYSTEMTIME st = { 1970,1,0,1,   0,0,0,0 };
//  FILETIME ft;
//  SystemTimeToFileTime(&st, &ft);
//  hkInt64 delta = s_combineHiLoDwords( ft.dwHighDateTime, ft.dwLowDateTime );
//  delta == 116444736000000000UI64 * 100ns from win32 epoch to linux epoch

#define HK_TIMESTAMP_NSEC100_TO_UNIX_EPOCH  116444736000000000UI64

static hkUint64 s_combineHiLoDwords(DWORD high, DWORD low)
{
    return low + (hkUint64(high)<<32);
}

static hkUint64 s_convertWindowsFiletimeToUnixTime(DWORD high, DWORD low)
{
    hkUint64 filetime = low + (hkUint64(high)<<32);
    return (filetime - HK_TIMESTAMP_NSEC100_TO_UNIX_EPOCH) * 100;
}

// Populate an entry from a finddata. Return false on failure (e.g. name==".")
static void s_entryFromFindData(hkFileSystem::Entry& e, _Inout_ hkFileSystem* fs, _In_z_ const char* top, const HK_WIN32_FIND_DATA& f)
{
    e.setAll
    (
        fs,
        hkStringBuf(top).pathAppend( HK_NARROW( f.cFileName ) ),
        f.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY ? hkFileSystem::Entry::F_ISDIR : hkFileSystem::Entry::F_ISFILE,
        s_convertWindowsFiletimeToUnixTime( f.ftLastWriteTime.dwHighDateTime, f.ftLastWriteTime.dwLowDateTime ),
        s_combineHiLoDwords( f.nFileSizeHigh, f.nFileSizeLow )
    );
}

hkRefNew<hkStreamReader> hkWin32FileSystem::_openReader(_In_z_ const char* name, OpenFlags flags)
{
    hkStringBuf sb = name; s_convertFileNameToNative(sb);
    // Stdio stream reader is already buffered so we don't need to buffer again
    return _handleFlags( hkStdioStreamReader::open(sb), OpenFlags(flags&~OPEN_BUFFERED) );
}

hkRefNew<hkStreamWriter> hkWin32FileSystem::_openWriter(_In_z_ const char* name, OpenFlags flags)
{
    hkStringBuf sb = name; s_convertFileNameToNative(sb);
    int dwCreationDisposition = (flags & OPEN_TRUNCATE) ? CREATE_ALWAYS : OPEN_ALWAYS;
    const bool shared = !(flags & OPEN_EXCLUSIVE_ACCESS);
    return _handleFlags( hkWin32StreamWriter::open(sb, dwCreationDisposition, shared), flags );
}

hkResult hkWin32FileSystem::remove(_In_z_ const char* path)
{
    hkStringBuf sb = path; s_convertFileNameToNative(sb);
    hkUtf8::WideFromUtf8 widePath(sb.cString());

    DWORD fileAttributes = GetFileAttributesW(widePath);
    if (fileAttributes & FILE_ATTRIBUTE_READONLY)
    {
        fileAttributes &= ~FILE_ATTRIBUTE_READONLY;
        SetFileAttributesW(widePath, fileAttributes);
    }

    if (fileAttributes & FILE_ATTRIBUTE_DIRECTORY)
    {
        if (!RemoveDirectory(widePath))
        {
            return GetLastError() == ERROR_FILE_NOT_FOUND ? HK_E_FILE_NOT_FOUND : HK_FAILURE;
        }
        return HK_SUCCESS;
    }
    else
    {
        return DeleteFileW(widePath) ? HK_SUCCESS : HK_FAILURE;
    }
}

hkResult hkWin32FileSystem::mkdir(_In_z_ const char* path, CreateFlag flag)
{
    hkStringBuf sb = path; s_convertFileNameToNative(sb);
    if (!CreateDirectoryW(HK_WIDEN(sb), HK_NULL))
    {
        if (GetLastError() == ERROR_ALREADY_EXISTS)
        {
            return flag == ERROR_IF_EXISTS ? HK_E_ALREADY_EXISTS : HK_SUCCESS;
        }
        return HK_FAILURE;
    }
    return HK_SUCCESS;
}

hkResult hkWin32FileSystem::moveFile(_In_z_ const char* sourceFile, _In_z_ const char* destPath)
{
    hkStringBuf sb = sourceFile; s_convertFileNameToNative(sb);
    hkStringBuf db = destPath;   s_convertFileNameToNative(db);

    const hkUtf8::WideFromUtf8 wideSource( sb );
    const hkUtf8::WideFromUtf8 wideDest( db );

    // Try to move file; fail if the move is across a different volume.
    // Do not use the MOVEFILE_COPY_ALLOWED flag because this makes MoveFileEx
    // succeed when the file is copied but the original one not deleted.
    const BOOL result = MoveFileExW( wideSource, wideDest, 0 );
    if ( result != 0 )
    {
        return HK_SUCCESS;
    }
    else if ( result != ERROR_NOT_SAME_DEVICE )
    {
        return HK_FAILURE;
    }

#if defined(HK_PLATFORM_DURANGO)
    return HK_FAILURE;
#else
    // Try to copy + delete across a different volume.
    if ( CopyFileW( wideSource, wideDest, /* fail if exists */ TRUE ) == 0 )
    {
        return HK_FAILURE;
    }
    return ( DeleteFileW( wideSource ) != 0 ? HK_SUCCESS : HK_FAILURE);
#endif
}

hkResult hkWin32FileSystem::getTempDir(hkStringBuf& tempDirOut)
{
#if defined(HK_PLATFORM_DURANGO)
    return HK_FAILURE;
#else
    wchar_t tempPath[MAX_PATH];
    const int pathLength = GetTempPathW(MAX_PATH, tempPath);

    if (pathLength == 0)
    {
        return HK_FAILURE;
    }

    tempDirOut.set(HK_NARROW(tempPath)).pathNormalize();

    return HK_SUCCESS;
#endif
}

hkResult hkWin32FileSystem::createTempSubDir(hkStringBuf& tempDirOut)
{
    if (getTempDir(tempDirOut).isFailure())
    {
        return HK_FAILURE;
    }

    tempDirOut.appendFormat("{}", hkUuid::random());

    return hkFileSystem::getInstance().mkdir(tempDirOut, hkFileSystem::ERROR_IF_EXISTS);
}

hkResult hkWin32FileSystem::copy(_In_z_ const char* srcPath, _In_z_ const char* dstPath)
{
#if !defined( HK_PLATFORM_WINRT ) && !defined(HK_PLATFORM_DURANGO )
    hkStringBuf nativeSrcPath = srcPath; s_convertFileNameToNative(nativeSrcPath);
    hkStringBuf nativeDstPath = dstPath; s_convertFileNameToNative(nativeDstPath);

    DWORD fileAttributes = GetFileAttributesW(HK_WIDEN(nativeDstPath));
    if (fileAttributes & FILE_ATTRIBUTE_READONLY)
    {
        fileAttributes &= ~FILE_ATTRIBUTE_READONLY;
        SetFileAttributesW(HK_WIDEN(nativeDstPath), fileAttributes);
    }

    BOOL result = CopyFileW(HK_WIDEN(nativeSrcPath), HK_WIDEN(nativeDstPath), FALSE);
    return (result == TRUE) ? HK_SUCCESS : HK_FAILURE;
#else
    return hkFileSystem::copy(srcPath, dstPath);
#endif
}


hkResult hkWin32FileSystem::stat(_In_z_ const char* path, Entry& entryOut )
{
    hkStringBuf sb = path; s_convertFileNameToNative(sb);
    HK_ASSERT(0x129e4884, hkString::strChr(path,'*')==0, "Use an iterator for wildcards" );
    HK_WIN32_FIND_DATA findData;
    // FindFirstFileW cannot handle a path that ends with a slash, so we have to remove it.
    while (sb.endsWith("\\"))
    {
        sb.chompEnd(1);
    }

    // FindFirstFile cannot handle drive roots at all, so these must be checked differently
    if (sb.getLength() == 2 && isalpha(sb[0]) && sb[1] == ':')
    {
        sb.append("\\");
        DWORD dwAttributes = GetFileAttributesW(HK_WIDEN(sb));
        if (dwAttributes == INVALID_FILE_ATTRIBUTES)
        {
            return HK_FAILURE;
        }
        entryOut.setAll(this, sb, hkFileSystem::Entry::F_ISDIR, hkFileSystem::TimeStamp(0), 0);
        return HK_SUCCESS;
    }

    HANDLE h = FindFirstFileW( HK_WIDEN(sb), &findData);
    if( h != INVALID_HANDLE_VALUE )
    {
        entryOut.setAll
        (
            this,
            sb,
            findData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY ? hkFileSystem::Entry::F_ISDIR : hkFileSystem::Entry::F_ISFILE,
            s_convertWindowsFiletimeToUnixTime( findData.ftLastWriteTime.dwHighDateTime, findData.ftLastWriteTime.dwLowDateTime ),
            s_combineHiLoDwords( findData.nFileSizeHigh, findData.nFileSizeLow )
        );
        FindClose(h);
        return HK_SUCCESS;
    }
    return HK_FAILURE;
}

_Ret_maybenull_z_
const char* hkWin32FileSystem::getOperatingSystemPath(_In_z_ const char* pathIn, hkStringBuf& pathOut, OSPathFlags flags )
{
    Entry entry;
    if (stat(pathIn, entry).isSuccess())
    {
        if (flags & OS_PATH_ABSOLUTE)
        {
            TCHAR buf[MAX_PATH];
            if (GetFullPathNameW(HK_WIDEN(entry.getPath()), MAX_PATH, buf, HK_NULL))
            {
                pathOut = HK_NARROW(buf);
                return pathOut.cString();
            }
            else
            {
                return HK_NULL;
            }
        }
        pathOut = entry.getPath();
        return pathOut.cString();
    }
    return HK_NULL;
}

namespace
{
    struct Win32Impl : public hkFileSystem::Iterator::Impl
    {
        Win32Impl(_Inout_ hkFileSystem* fs, _In_z_ const char* top, _In_z_ const char* wildcard)
            : m_fs(fs)
            , m_handle(0)
            , m_top(top)
            , m_wildcard(wildcard)
        {
            HK_ASSERT(0x1e0bb0cd, hkString::strChr(m_top,'*') == HK_NULL, "Path part cannot contain a *");
            HK_ASSERT(0x47f9de01, wildcard==HK_NULL || hkString::strChr(wildcard,'*'), "Wildcard must be null or contain a *" );
        }

        virtual bool advance(hkFileSystem::Entry& e)
        {
            if( m_handle == 0 ) // first time through
            {
                HK_WIN32_FIND_DATA findData;
                hkStringBuf pattern( m_top );
                pattern.pathAppend( m_wildcard ? m_wildcard : "*");
                s_convertFileNameToNative(pattern);

                m_handle = FindFirstFileW( HK_WIDEN(pattern), &findData );
                if( m_handle == INVALID_HANDLE_VALUE )
                {
                    return false;
                }
                if( hkFileSystem::Iterator::nameAcceptable( HK_NARROW(findData.cFileName), HK_NULL) )
                {
                    s_entryFromFindData(e, m_fs, m_top, findData);
                    return true;
                }
            }
            if( m_handle != INVALID_HANDLE_VALUE )
            {
                HK_WIN32_FIND_DATA findData;
                while( FindNextFileW( m_handle, &findData) )
                {
                    if( hkFileSystem::Iterator::nameAcceptable( HK_NARROW(findData.cFileName), HK_NULL ) )
                    {
                        s_entryFromFindData(e, m_fs, m_top, findData);
                        return true;
                    }
                }
                // we ran out of entries
                FindClose( m_handle );
                m_handle = INVALID_HANDLE_VALUE;
            }
            return false;
        }

        ~Win32Impl()
        {
            if( m_handle != 0 && m_handle != INVALID_HANDLE_VALUE )
            {
                FindClose( m_handle );
            }
        }

        hkFileSystem* m_fs;
        hkStringPtr m_top;
        hkStringPtr m_wildcard;
        HANDLE m_handle;
    };
}

// One of these exists for every watch we create in this FS
class Win32FileSystemWatcher : public hkFileSystem::Watcher
{
public:
    HK_DECLARE_CLASS(Win32FileSystemWatcher, New);

    enum
    {
        FILE_MODIFICATION_COMPLETED_TIMEOUT = 250
    };

    void* m_handle;
    hkWin32FileSystem::WatcherImpl* m_watcher;
    hkStringPtr m_rootDirectory;

    bool m_stoppedWatching;

    // This should be as large as possible, as data is lost if the buffer overflows, but less than 64kB or it will fail on some network drives
    enum { BUFFER_SIZE = 60416, OVERLAPPED_SIZE = 64 };
    HK_ALIGN16(char) m_overlappedBuffer[OVERLAPPED_SIZE];
    char m_buffer[BUFFER_SIZE];

    virtual hkResult getCompletedChanges(hkArray<Change>& changesOut) HK_OVERRIDE;

    void updatePendingCompletionQueue()
    {
        FILETIME currentFileTime;
        GetSystemTimeAsFileTime(&currentFileTime);

        // This seems questionable but it is what MSDN recommends
        const hkInt64 diffInTicks =
            reinterpret_cast<LARGE_INTEGER*>(&currentFileTime)->QuadPart -
            reinterpret_cast<LARGE_INTEGER*>(&m_lastInProgressModification)->QuadPart;
        const hkInt64 diffInMillis = diffInTicks / 10000;

        if(diffInMillis > FILE_MODIFICATION_COMPLETED_TIMEOUT)
        {
            
            if(m_inProgressChanges.getSize())
            {
                for(int i=0; i < m_inProgressChanges.getSize(); i++)
                {
                    hkFileSystem::Watcher::Change& oldChange = m_inProgressChanges[i];
                    if(oldChange.m_change == hkFileSystem::Watcher::Change::_Rename_Old_Name)
                    {
                        for(int scanForNewNameIndex = i + 1; scanForNewNameIndex < m_inProgressChanges.getSize(); scanForNewNameIndex++)
                        {
                            hkFileSystem::Watcher::Change& newChange = m_inProgressChanges[scanForNewNameIndex];
                            if(newChange.m_change == hkFileSystem::Watcher::Change::_Rename_New_Name)
                            {
                                m_completedChanges.pushBack(newChange);
                                m_completedChanges.back().m_oldFileName = oldChange.m_fileName;
                                m_completedChanges.back().m_change = hkFileSystem::Watcher::Change::Renamed;

                                m_inProgressChanges.removeAtAndCopy(i);
                                m_inProgressChanges.removeAtAndCopy(scanForNewNameIndex - 1); // We have removed one from before it!
                                break;
                            }
                        }
                    }
                    else if(oldChange.m_change == hkFileSystem::Watcher::Change::_Rename_New_Name)
                    {
                        // New name without an old, this is probably an error but just leave it here for now
                    }
                    else
                    {
                        m_completedChanges.pushBack(oldChange);
                        m_inProgressChanges.removeAtAndCopy(i);
                    }
                }
            }
        }
    }

    void updateCompletionQueueTime()
    {
        FILETIME currentFileTime;
        GetSystemTimeAsFileTime(&currentFileTime);
        hkString::memCpy(&m_lastInProgressModification, &currentFileTime, hkSizeOf(FILETIME));
    }

    virtual hkResult stopWatching() HK_OVERRIDE;

    // Only read or written from inside hkWin32FileSystem::WatcherImpl::m_criticalSection
    hkArray<hkFileSystem::Watcher::Change> m_inProgressChanges;
    hkArray<hkFileSystem::Watcher::Change> m_completedChanges;
    FILETIME m_lastInProgressModification;
};

#if !defined(HK_PLATFORM_DURANGO)
// This is the common watcher impl, shared by every actual watcher. Created and destroyed on-demand
class hkWin32FileSystem::WatcherImpl : public hkReferencedObject
{
public:
    HK_DECLARE_CLASS(WatcherImpl, New);

    // Callback structure, we just add a member to find ourself
    struct OverLapped : _OVERLAPPED
    {
        Win32FileSystemWatcher* m_dir;
    };

    WatcherImpl() : m_numOutstandingRequests(0), m_terminationRequested(false)
    {
        m_criticalSection = new hkCriticalSection( 4000 );
        m_thread.startThread( threadStartProc, this, "FileWatcherThread", 4096 );
    }

    hkRefNew<hkFileSystem::Watcher> addTrackedDirectory(_In_z_ const char* dir)
    {
        HANDLE hDir = ::CreateFileW(
            hkUtf8::WideFromUtf8(dir),           // pointer to the file name
            FILE_LIST_DIRECTORY,    // access (read/write) mode
            FILE_SHARE_READ         // share mode
            | FILE_SHARE_WRITE
            | FILE_SHARE_DELETE,
            NULL, // security descriptor
            OPEN_EXISTING,         // how to create
            FILE_FLAG_BACKUP_SEMANTICS // file attributes
            | FILE_FLAG_OVERLAPPED,
            NULL);                 // file with attributes to copy

        HK_ASSERT_NO_MSG(0x5a6ed002, hDir != INVALID_HANDLE_VALUE);

        if (hDir == INVALID_HANDLE_VALUE)
        {
            Log_Error("Failed to create file watcher for {}.", dir);
            return HK_NULL;
        }

        hkScopedPtr<Win32FileSystemWatcher> newDirectory(new Win32FileSystemWatcher);

        newDirectory->m_handle = hDir;
        newDirectory->m_watcher = this;
        newDirectory->m_rootDirectory = dir;
        newDirectory->m_stoppedWatching = false;

        //printf("Queue APC add dir: %s\n", dir);

        if (QueueUserAPC(addDirectoryProc, m_thread.getHandle(), (ULONG_PTR)newDirectory.get()) == 0)
        {
            Log_Error("Failed to add file watcher for {} with error {}.", dir, GetLastError());
            return HK_NULL;
        }

        m_watchedDirectories.pushBack(newDirectory.steal());
        return m_watchedDirectories.back().val();
    }

    virtual ~WatcherImpl()
    {
        void* hThread = m_thread.getHandle();
        if ( !m_watchedDirectories.isEmpty() )
        {
            for (int i =0; i < m_watchedDirectories.getSize(); i++ )
            {
                Win32FileSystemWatcher* dir = m_watchedDirectories[i];
                dir->addReference(); // Add a reference since the pointer is captured as a ULONG below.
                QueueUserAPC(terminateProc, hThread, (ULONG_PTR)dir);
            }
        }
        else
        {
            QueueUserAPC(terminateProc2, hThread, (ULONG_PTR)this);
        }
        ::WaitForSingleObjectEx(hThread, 10000, true);

        delete m_criticalSection;
    }

    // Called by QueueUserAPC to start orderly shutdown.
    static void CALLBACK terminateProc( ULONG_PTR arg)
    {
        Win32FileSystemWatcher* dir = (Win32FileSystemWatcher*)arg;
        dir->m_watcher->requestTermination(dir);
        dir->removeReference();
    }

    // Called by QueueUserAPC to start orderly shutdown.
    static void CALLBACK terminateProc2( ULONG_PTR arg)
    {
        WatcherImpl* self = (WatcherImpl*)arg;
        self->requestTermination(HK_NULL);
    }

    static void cancleIoAndCloseHandle(_Inout_opt_ Win32FileSystemWatcher* dir)
    {
        if (dir && dir->m_handle)
        {
            ::CancelIo(dir->m_handle);
            ::CloseHandle(dir->m_handle);
            dir->m_handle = HK_NULL;
        }
    }

    // Called by QueueUserAPC to stop watching a given directory.
    static void CALLBACK stopWatching(ULONG_PTR arg)
    {
        cancleIoAndCloseHandle((Win32FileSystemWatcher*)arg);
    }

    static _Ret_notnull_ void* HK_CALL threadStartProc(_Inout_ void* arg)
    {
        WatcherImpl* self = (WatcherImpl*)arg;
        self->run();
        return 0;
    }

    static void CALLBACK addDirectoryProc(ULONG_PTR arg)
    {
        Win32FileSystemWatcher* dir = (Win32FileSystemWatcher*)arg;
        dir->m_watcher->addDirectoryWatch( dir );
    }

    void requestTermination(_Inout_ Win32FileSystemWatcher* dir)
    {
        cancleIoAndCloseHandle(dir);
        m_terminationRequested = true;
    }

    void run()
    {
        hkMemoryRouter* memoryRouter = reinterpret_cast<hkMemoryRouter*>(new char[sizeof(hkMemoryRouter)]);
        new (memoryRouter) hkMemoryRouter();

        hkMemorySystem::getInstance().threadInit( *memoryRouter, "hkFileSystemWatcherWin32" );
        hkBaseSystem::initThread( memoryRouter );


        while (m_numOutstandingRequests || !m_terminationRequested)
        {
            //printf("Sleeping, %d requests\n", m_numOutstandingRequests);
            /*DWORD rc =*/ ::SleepEx(INFINITE, true);
            //printf("Awake, %d requests%s\n", m_numOutstandingRequests, (m_terminationRequested ? " + TERMINATE" : ""));
        }
        hkBaseSystem::quitThread();
        hkMemorySystem::getInstance().threadQuit(*memoryRouter);
    }

    void addDirectoryWatch(_Inout_ Win32FileSystemWatcher* dir )
    {
        HANDLE dirH = dir->m_handle;

        // Initialize the _OVERLAPPED structure
        OverLapped* overlapped = (OverLapped*)dir->m_overlappedBuffer;
        hkString::memSet4( overlapped, 0, sizeof(OverLapped)>>2);
        overlapped->m_dir = dir;
        overlapped->hEvent = dir;

        hkAtomic::exchangeAdd32( &m_numOutstandingRequests, 1 );

        BOOL watchSubtree = true;
        DWORD bytes = 0;
        //printf("Reading changes: %s\n", dir->m_rootDirectory.cString());
        BOOL result = ReadDirectoryChangesW( dirH, dir->m_buffer, dir->BUFFER_SIZE, watchSubtree,
            FILE_NOTIFY_CHANGE_LAST_WRITE | FILE_NOTIFY_CHANGE_FILE_NAME | FILE_NOTIFY_CHANGE_DIR_NAME,
            &bytes, overlapped,  &NotificationCompletion);

        //printf("Return: %s\n", result ? "OK" : "FAIL");

        if ( !result )
        {
            hkAtomic::exchangeAdd32( &m_numOutstandingRequests, (hkUint32)-1 );
            DWORD lastError = GetLastError();
            lastError = lastError;
            return;
        }
    }

    static void CALLBACK NotificationCompletion(
        DWORD errorCode,                                // completion code
        DWORD numberOfBytesTransfered,                  // number of bytes transferred
        LPOVERLAPPED lpOverlapped)                      // I/O information buffer
    {
        OverLapped* mo = (OverLapped*) lpOverlapped;
        Win32FileSystemWatcher* dir = mo->m_dir;
        WatcherImpl* self = dir->m_watcher;

        //printf("Notification completion: %s (%d/%d)\n", dir->m_rootDirectory.cString(), errorCode, numberOfBytesTransfered);

        if (errorCode == ERROR_OPERATION_ABORTED)
        {
            // If the Win32FileSystemWatcher doesn't have a handle, it means that this watcher was removed via stopWatching method,
            // which added a reference to ensure the watcher exists up to here.
            if (dir->m_stoppedWatching)
            {
                dir->removeReference();
            }
            hkAtomic::exchangeAdd32(&self->m_numOutstandingRequests, (hkUint32)-1 );
            return;
        }
        // This might mean overflow? Not sure.
        if(!numberOfBytesTransfered)
        {
            hkAtomic::exchangeAdd32(&self->m_numOutstandingRequests, (hkUint32)-1);
            return;
        }


        {
            hkCriticalSectionLock lock(self->m_criticalSection);

            dir->updatePendingCompletionQueue();
            // Only update stored time if we have changed the queue
            dir->updateCompletionQueueTime();

            // process results
            {
                char* pBase = dir->m_buffer;

                for (;;)
                {
                    FILE_NOTIFY_INFORMATION& fni = (FILE_NOTIFY_INFORMATION&)*pBase;

                    char fileName[1024];
                    int fileNameLength = fni.FileNameLength/sizeof(wchar_t);
                    hkUtf8::utf8FromWide( fileName, 1023, fni.FileName, fileNameLength );

                    hkStringBuf fullPath; fullPath.appendJoin( dir->m_rootDirectory, "/", fileName );

                    //printf("Change to file %s\n", fullPath.cString());

                    //      If it could be a short filename, expand it.
                    //          LPCWSTR wszFilename = PathFindFileNameW(wstrFilename);
                    //          int len = lstrlenW(wszFilename);
                    //          // The maximum length of an 8.3 filename is twelve, including the dot.
                    //          if (len <= 12 && wcschr(wszFilename, L'~'))
                    //          {
                    //              // Convert to the long filename form. Unfortunately, this
                    //              // does not work for deletions, so it's an imperfect fix.
                    //              wchar_t wbuf[MAX_PATH];
                    //              if (::GetLongPathName(wstrFilename, wbuf, _countof (wbuf)) > 0)
                    //                  wstrFilename = wbuf;
                    //          }

                    hkFileSystem::Watcher::Change& newChange = dir->m_inProgressChanges.expandOne();
                    newChange.m_fileName = fullPath;

                    newChange.m_change = hkFileSystem::Watcher::Change::Unknown;
                    switch(fni.Action)
                    {
                    case FILE_ACTION_ADDED: newChange.m_change = hkFileSystem::Watcher::Change::Added; break;
                    case FILE_ACTION_REMOVED: newChange.m_change = hkFileSystem::Watcher::Change::Removed; break;
                    case FILE_ACTION_MODIFIED: newChange.m_change = hkFileSystem::Watcher::Change::Modified; break;
                        
                    case FILE_ACTION_RENAMED_OLD_NAME: newChange.m_change = hkFileSystem::Watcher::Change::_Rename_Old_Name; break;
                    case FILE_ACTION_RENAMED_NEW_NAME: newChange.m_change = hkFileSystem::Watcher::Change::_Rename_New_Name; break;
                    }

                    if (!fni.NextEntryOffset)
                        break;
                    pBase += fni.NextEntryOffset;
                };
            }
        }

        // re-add this watch to the queue
        self->addDirectoryWatch( dir );

        // Request completed
        hkAtomic::exchangeAdd32(&self->m_numOutstandingRequests, (hkUint32)-1 );
    }

    hkResult updateCompletedChanges(_Inout_ Win32FileSystemWatcher* dir, hkArray<hkFileSystem::Watcher::Change>& changesOut)
    {
        hkCriticalSectionLock lock(m_criticalSection);
        dir->updatePendingCompletionQueue();
        changesOut.append(dir->m_completedChanges);
        dir->m_completedChanges.clear();

        return HK_SUCCESS;
    }

    hkThread m_thread;  // my thread
    hkUint32 m_numOutstandingRequests;
    hkBool  m_terminationRequested;
    hkCriticalSection* m_criticalSection;
    hkArray<hkRefPtr<Win32FileSystemWatcher> > m_watchedDirectories;
};

HK_COMPILE_TIME_ASSERT( sizeof ( hkWin32FileSystem::WatcherImpl::OverLapped ) <= Win32FileSystemWatcher::OVERLAPPED_SIZE );

hkResult Win32FileSystemWatcher::getCompletedChanges(hkArray<Change>& changesOut)
{
    hkResult res = m_watcher->updateCompletedChanges(this, changesOut);
    if(res.isSuccess())
    {
        combineMatchingChanges(changesOut);
    }
    return res;
}

hkResult Win32FileSystemWatcher::stopWatching()
{
    if (!m_stoppedWatching)
    {
        // Watcher needs to persist until NotificationCompletion gets fired by ReadDirectoryChangesW, but the watched directories
        // list might have last reference to it. This reference will be removed on ReadDirectoryChangesW method.
        this->addReference();
        this->m_stoppedWatching = true;
        int watchedDirectoryIndex = m_watcher->m_watchedDirectories.indexOf(this);
        m_watcher->m_watchedDirectories.removeAtAndCopy(watchedDirectoryIndex);

        QueueUserAPC(hkWin32FileSystem::WatcherImpl::stopWatching, m_watcher->m_thread.getHandle(), (ULONG_PTR)this);
    }

    return HK_SUCCESS;
}

hkRefNew<hkFileSystem::Watcher> hkWin32FileSystem::createWatcher(_In_z_ const char* topDirectory)
{
    if(!m_watcherImpl)
    {
        m_watcherImpl = hkRefNew<hkWin32FileSystem::WatcherImpl>(new hkWin32FileSystem::WatcherImpl);
    }

    hkStringBuf sb = topDirectory; s_convertFileNameToNative(sb);
    return m_watcherImpl->addTrackedDirectory(sb.cString());
}

#else

hkRefNew<hkFileSystem::Watcher> hkWin32FileSystem::createWatcher(_In_z_ const char* topDirectory)
{
    return HK_NULL;
}

#endif // !defined(HK_PLATFORM_DURANGO)

hkRefNew<hkFileSystem::Iterator::Impl> hkWin32FileSystem::createIterator(_In_z_ const char* top, _In_z_ const char* wildcard )
{
    return new Win32Impl(this, top, wildcard);
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
