// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : NX32 NX64
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/System/Io/Platform/Nx/hkNxStreamWriter.h>

#include <nn/fs.h>

hkNxStreamWriter* hkNxStreamWriter::open(const char* name, hkFileSystem::OpenFlags flags)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, name, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
    if (result.IsSuccess() && (flags & hkFileSystem::OPEN_TRUNCATE))
    {
        nn::fs::SetFileSize(handle, 0);
    }
    else if (nn::fs::ResultPathNotFound::Includes(result))
    {
        result = nn::fs::CreateFile(name, 0);
        if (result.IsSuccess())
        {
            result = nn::fs::OpenFile(&handle, name, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
        }
    }
    else if (nn::fs::ResultTargetLocked::Includes(result))
    {
        HK_ERROR(0x4d4394b3, "The target file is locked.");
    }
    else if (nn::fs::ResultAllocationMemoryFailed::Includes(result))
    {
        HK_ERROR(0x701a3ed3, "Insufficient memory.");
    }
    if (result.IsSuccess())
    {
        return new hkNxStreamWriter(handle, true);
    }
    return HK_NULL;
}

hkNxStreamWriter::hkNxStreamWriter(nn::fs::FileHandle handle, hkBool shouldClose)
    : m_handle(handle), m_shouldClose(shouldClose), m_offset(0)
{
}

void hkNxStreamWriter::close()
{
    if( (m_handle.handle != HK_NULL) && m_shouldClose)
    {
        flush();
        nn::fs::CloseFile(m_handle);
        m_handle.handle = HK_NULL;
    }
}

hkNxStreamWriter::~hkNxStreamWriter()
{
    close();
}

int hkNxStreamWriter::write( const void* buf, int nbytes)
{
    if (m_handle.handle != HK_NULL)
    {
        nn::fs::WriteOption option{ 0 };  // auto flush disabled
        nn::Result result = nn::fs::WriteFile(m_handle, m_offset, buf, nbytes, option);
        if (result.IsFailure())
        {
            if (nn::fs::ResultUsableSpaceNotEnough::Includes(result))
            {
                HK_ERROR(0x3297994a, "Not enough space to write file");
            }
            else
            {
                HK_ERROR(0x1a64d54b, "Writing to file failed");
            }
            close();
        }
        else
        {
            m_offset += nbytes;
        }
        return nbytes;
    }
    return 0;
}

void hkNxStreamWriter::flush()
{
    if (m_handle.handle != HK_NULL)
    {
        nn::fs::FlushFile(m_handle);  // result currently ignored
    }
}

hkBool hkNxStreamWriter::isOk() const
{
    return m_handle.handle != HK_NULL;
}

hkBool hkNxStreamWriter::seekTellSupported() const
{
    return true;
}

hkResult hkNxStreamWriter::seek( int offset, SeekWhence whence)
{
    // flush and get filesize
    nn::fs::FlushFile(m_handle);
    int64_t fileSize;
    if (nn::fs::GetFileSize(&fileSize, m_handle).IsFailure())
        fileSize = 0;

    if (m_handle.handle == HK_NULL)
    {
        return HK_FAILURE;
    }
    if (whence == STREAM_SET)
    {
        m_offset = offset;
    }
    else if (whence == STREAM_END)
    {
        m_offset = fileSize + offset;
    }
    else
    {
        m_offset += offset;
    }
    return HK_SUCCESS;
}

int hkNxStreamWriter::tell() const
{
    return (int) m_offset;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
