// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64 LINUX32 LINUX64 MAC IOS ANDROID WIIU DURANGO UWP OSINTERNAL NX32 NX64
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/System/Io/Platform/Bsd/hkBsdSocket.h>
#include <Common/Base/Container/String/hkStringBuf.h>

#if ( defined(HK_PLATFORM_DURANGO) || defined(HK_PLATFORM_WIN32) ) && !defined(HK_PLATFORM_NX)
#   define _WINSOCK_DEPRECATED_NO_WARNINGS 
#   include <winsock2.h> 
#   include <Ws2tcpip.h>
#   pragma comment(lib,"Ws2_32.lib")

    typedef int socklen_t;
    HK_COMPILE_TIME_ASSERT( sizeof(hkBsdSocket::socket_t) == sizeof(SOCKET) );

#elif defined(HK_PLATFORM_WIIU)
#   include <cafe.h>
#   include <cafe/network.h>
#   include <nn/ac/ac_Api.h>
#   define INVALID_SOCKET (-1)

#elif defined(HK_PLATFORM_NX)
#   include <nn/socket.h>
#   include <nn/nifm/nifm_ApiIpAddress.h>
#   define closesocket nn::socket::Close
#   define socketconnect nn::socket::Connect
#   define socketlisten nn::socket::Listen
#   define select nn::socket::Select
#   define accept nn::socket::Accept
#   define recv nn::socket::Recv
#   define recvfrom nn::socket::RecvFrom
#   define send nn::socket::Send
#   define sendto nn::socket::SendTo
#   define htons nn::socket::InetHtons
#   define getaddrinfo nn::socket::GetAddrInfo
#   define freeaddrinfo nn::socket::FreeAddrInfo
#   define getsockname nn::socket::GetSockName
#   define ntohs nn::socket::InetNtohs
#   define inet_ntop nn::socket::InetNtop
#   define setsockopt nn::socket::SetSockOpt

#   define fcntl nn::socket::Fcntl
#   if !defined(HK_PLATFORM_NX_WIN32) && !defined(HK_PLATFORM_NX_X64)
#       define INVALID_SOCKET (-1)
#       define SOCKET_ERROR (-1)
#   else
#       undef close
#       undef read
#       undef write
#       undef connect
#       undef listen
#   endif

#else
// Normal BSD socket:
#   include <sys/types.h>
#   include <sys/time.h>
#   include <sys/socket.h>
#   include <sys/ioctl.h>
#   include <netinet/in.h>
#   include <netinet/tcp.h>
#   include <arpa/inet.h>
#   include <unistd.h>
#   include <netdb.h>
#   include <Common/Base/Fwd/hkcstring.h>
#   define closesocket close
#   define INVALID_SOCKET (-1)
#   define SOCKET_ERROR (-1)
#   if defined(HK_PLATFORM_LINUX) || defined(HK_PLATFORM_ANDROID) || defined(HK_PLATFORM_PS4) || defined(HK_PLATFORM_MAC) || defined(HK_PLATFORM_IOS)
    // For non-blocking sockets
#       include <fcntl.h>
#       include <errno.h>
#   endif
#endif

#define DEBUG_LOG_IDENTIFIER "common.base.network.bsdsocket"
#include <Common/Base/System/Log/hkLog.hxx>


static hkBool g_defaultPlatformInitOnce = false;
void HK_CALL hkBsdNetworkInit()
{
    if( !g_defaultPlatformInitOnce )
    {
#       if defined(HK_PLATFORM_WIN32)
        {
            // initialize win sockets
            const int WSAVERSION = 0x202; // winsock version
            WSADATA wsaData;
            if(WSAStartup(WSAVERSION,&wsaData) == SOCKET_ERROR)
            {
                HK_ERROR(0x321825f8, "(Windows)WSAStartup failed with error!");
            }
        }
#       endif

#       if defined(HK_PLATFORM_WIIU)
        nn::ac::Initialize();
        SOInit();
#       endif

        g_defaultPlatformInitOnce = true;
    }
}

void HK_CALL hkBsdNetworkQuit()
{

}

hkBsdSocket::hkBsdSocket(socket_t s)
    : m_socket(s)
{
    if ( m_socket == INVALID_SOCKET )
    {
        createSocket();
    }
}

hkBool hkBsdSocket::isOk() const
{
    return m_socket != INVALID_SOCKET;
}

void hkBsdSocket::close()
{
    if(m_socket != INVALID_SOCKET)
    {
#if defined(HK_PLATFORM_WIIU)
        ::socketclose(m_socket);
#else
        ::closesocket(m_socket);
#endif
        m_socket = INVALID_SOCKET;
    }
}

#if defined(HK_PLATFORM_WIN32) && !defined(HK_PLATFORM_DURANGO) && !defined(HK_PLATFORM_WINRT)
namespace
{
    // From VersionHelpers.h in the Windows 8.1 Kit
    HK_INLINE bool IsWindowsVersionOrGreater(WORD wMajorVersion, WORD wMinorVersion, WORD wServicePackMajor)
    {
        OSVERSIONINFOEXW osvi = { sizeof(osvi), 0, 0, 0, 0, {0}, 0, 0 };
        DWORDLONG        const dwlConditionMask = VerSetConditionMask(
            VerSetConditionMask(
            VerSetConditionMask(
            0, VER_MAJORVERSION, VER_GREATER_EQUAL),
            VER_MINORVERSION, VER_GREATER_EQUAL),
            VER_SERVICEPACKMAJOR, VER_GREATER_EQUAL);

        osvi.dwMajorVersion = wMajorVersion;
        osvi.dwMinorVersion = wMinorVersion;
        osvi.wServicePackMajor = wServicePackMajor;

        return VerifyVersionInfoW(&osvi, VER_MAJORVERSION | VER_MINORVERSION | VER_SERVICEPACKMAJOR, dwlConditionMask) != FALSE;
    }

    HK_INLINE bool IsWindows7SP1OrGreater()
    {
        // This may not be defined for e.g. MSVC2008
#   ifndef _WIN32_WINNT_WIN7
#       define _WIN32_WINNT_WIN7                   0x0601
#   endif
        return IsWindowsVersionOrGreater(HIBYTE(_WIN32_WINNT_WIN7), LOBYTE(_WIN32_WINNT_WIN7), 1);
    }
}

#if !defined(WSA_FLAG_NO_HANDLE_INHERIT)
#define WSA_FLAG_NO_HANDLE_INHERIT 0x80
#endif

#endif

hkResult hkBsdSocket::createSocket()
{
    close();

#if defined(HK_PLATFORM_WIN32) && !defined(HK_PLATFORM_DURANGO) && !defined(HK_PLATFORM_WINRT)
    bool noInheritSupported = IsWindows7SP1OrGreater();
    m_socket = WSASocket(AF_INET, SOCK_STREAM, 0, NULL, 0, noInheritSupported ? WSA_FLAG_NO_HANDLE_INHERIT : 0);
#elif defined(HK_PLATFORM_WIIU)
    m_socket = static_cast<socket_t>( ::socket(AF_INET, SOCK_STREAM, IPPROTO_TCP) );
#elif defined(HK_PLATFORM_NX)
    m_socket = static_cast<socket_t>( nn::socket::Socket(AF_INET, SOCK_STREAM, IPPROTO_TCP));
#else
    m_socket = static_cast<socket_t>( ::socket(AF_INET, SOCK_STREAM, 0) );
#endif

    if(m_socket == INVALID_SOCKET)
    {
        HK_WARN(0x3b98e883, "Error creating socket!");
        return HK_FAILURE;
    }

#if defined(HK_PLATFORM_WIN32) && !defined(HK_PLATFORM_DURANGO) && !defined(HK_PLATFORM_WINRT)
    // If WSA_FLAG_NO_HANDLE_INHERIT is not supported, manually change the handle information. This is subject to a race condition if
    // a new process is created before the handle has been changed, but it's the best we can do.
    if(!noInheritSupported)
    {
        SetHandleInformation(reinterpret_cast<HANDLE>(m_socket), HANDLE_FLAG_INHERIT, 0);
    }
#endif

    return HK_SUCCESS;
}

hkBsdSocket::~hkBsdSocket()
{
    close();
}

int hkBsdSocket::read(_Out_writes_bytes_(nbytes) void* buf, int nbytes)
{
    if(m_socket != INVALID_SOCKET && nbytes > 0)
    {
        int n = ::recv(m_socket, static_cast<char*>(buf), nbytes, 0);

        if (n <= 0 || n == SOCKET_ERROR)
        {
#ifdef HK_PLATFORM_WIN32 // might be non blocking
            int werror = WSAGetLastError();
            if (werror == WSAEWOULDBLOCK)
                return 0; // don't close, as async
#elif defined(HK_PLATFORM_LINUX) || defined(HK_PLATFORM_MAC) || defined(HK_PLATFORM_ANDROID) || defined(HK_PLATFORM_IOS)
            if(errno == EAGAIN)
            {
                return 0;
            }
#endif
                // have to remove the warning as this warning allocates memory which breaks the SPU simulator
            //HK_WARN(0x4bb09a0f, "Read fail! Was the receiving end of socket closed?");
            close();
        }
        else
            return n;
    }
    return 0;
}

int hkBsdSocket::write(_In_reads_bytes_(nbytes) const void* buf, int nbytes)
{
    if ( ( m_socket != INVALID_SOCKET ) && ( nbytes > 0 ) )
    {
        const int numSent = ::send(m_socket, static_cast<const char*>( buf ), nbytes, 0);

        if ( ( numSent <= 0 ) || ( numSent == SOCKET_ERROR ) )
        {
#ifdef HK_PLATFORM_WIN32 // might be non blocking
            const int wsaError = WSAGetLastError();
            if ( wsaError == WSAEWOULDBLOCK )
            {
                return 0; // don't close, is async
            }
#elif defined(HK_PLATFORM_LINUX) || defined(HK_PLATFORM_MAC) || defined(HK_PLATFORM_ANDROID) || defined(HK_PLATFORM_IOS)
            if(errno == EAGAIN)
            {
                return 0;
            }

            // We have to disable the warn as it breaks the SPU simulator
            //HK_WARN(0x4cb4c0c7, "Socket send fail! Was the receiving end of socket closed?");
            //Log_Error( "FAILED TO WRITE ON SOCKET {} ERROR {}", m_socket, wsaError ); // bad idea to send anything on write fail, assume higher up code will do it.
            // as in the case of say the console viewer, the reports try to write to socket and do a report, to the socket, etc until crash
#endif
            close();
        }
        else
        {
            return numSent;
        }
    }
    return 0;
}

static hkBool HK_CALL hkIsDigit(int c)
{
    return c >= '0' && c <= '9';
}

hkResult hkBsdSocket::connect(_In_z_ const char* servername, int portNumber)
{
    // find the address of the server
    struct sockaddr_in server;
    {
        hkString::memSet(&server,0,sizeof(server));
        server.sin_family = AF_INET;
        server.sin_port = htons((unsigned short)portNumber);

        if(hkIsDigit(servername[0]))
        {
            //server.sin_addr.s_addr = inet_addr(servername);
#if defined(HK_PLATFORM_WIIU)
            SOInetPtoN(AF_INET, servername, &server.sin_addr.s_addr);
#elif defined(HK_PLATFORM_NX)
            nn::socket::InetAton(servername, &server.sin_addr);
#else
#   if !defined(NTDDI_VERSION) || (NTDDI_VERSION > 0x05020200)
            // Anything not XP
            struct in_addr addr;
#       if defined(HK_PLATFORM_WIN32)
            // Windows version of inet_pton
            if (InetPtonA(AF_INET, servername, &addr) != 1)
#       else
            if (inet_pton(AF_INET, servername, &addr) != 1)
#       endif
            {
                return HK_FAILURE;
            }
            server.sin_addr.s_addr = addr.s_addr;
#   else
            // Windows XP
            server.sin_addr.s_addr = ::inet_addr(servername);
#   endif
#endif
        }
        else
        {
#           if defined(HK_PLATFORM_WIN32) || defined(HK_PLATFORM_LINUX) || defined(HK_PLATFORM_MAC) || defined(HK_PLATFORM_IOS) || defined(HK_PLATFORM_ANDROID) || defined(HK_PLATFORM_NX)
                struct addrinfo hints;
                struct addrinfo *servinfo;

                memset(&hints, 0, sizeof hints); // make sure the struct is empty
                hints.ai_family = AF_INET;     // don't care IPv4 or IPv6
                hints.ai_socktype = SOCK_STREAM; // TCP stream sockets

                hkStringBuf portName; portName.printf("%d", portNumber);

                if ( getaddrinfo( servername, portName, &hints, &servinfo ) == 0 )
                {
                    for (struct addrinfo* thisInfo = servinfo; thisInfo != NULL; thisInfo = thisInfo->ai_next)
                    {
                        if (thisInfo->ai_family == AF_INET)
                        {
                            struct sockaddr_in *ipv4 = (struct sockaddr_in *)thisInfo->ai_addr;

                            hkString::memCpy(&(server.sin_addr), &(ipv4->sin_addr), sizeof(in_addr));
                        }
                    }

                    freeaddrinfo( servinfo );
                }
                else
                {
                    HK_WARN(0x1f2dd0e8, "Invalid server address!");
                    return HK_FAILURE;
                }
#           endif
        }
    }

    if( m_socket == INVALID_SOCKET )
    {
        if (createSocket().isFailure() )
        {
            return HK_FAILURE;
        }
    }
#ifdef HK_PLATFORM_NX
    if( ::socketconnect( m_socket, ( struct sockaddr* )&server, sizeof(server)) < 0 )
#else
    //Log_Info( "~~~~~~~~~~~~~~~~~ SOCKET {} ~~~ CONNECTING TO {} ~~~ PORT {}", m_socket, server.sin_addr.s_addr, portNumber );
    if ( ::connect( m_socket, ( struct sockaddr* )&server, sizeof( server ) ) < 0 )
#endif
    {
#ifdef HK_PLATFORM_WIN32
        // may be an aync socket
        if (WSAGetLastError() == WSAEWOULDBLOCK)
        {
            return HK_SUCCESS;
        }
#endif

        HK_WARN(0x46d25e96, "Cannot connect to server!");
        close();
        return HK_FAILURE;
    }
    return HK_SUCCESS;
}


hkResult hkBsdSocket::connect( hkInetAddr remoteAddr )
{
    struct sockaddr_in server;
    {
        hkString::memSet(&server,0,sizeof(server));
        server.sin_family = AF_INET;
        server.sin_port = htons(remoteAddr.m_port);
        server.sin_addr.s_addr = remoteAddr.m_ipAddress;
    }

    //Log_Info( "~~~~~~~~~~~~~~~~~ SOCKET {} ~~~ CONNECTING TO {} ~~~ PORT {}", m_socket, server.sin_addr.s_addr, remoteAddr.m_port );
    //struct sockaddr_in sin;
    //socklen_t len = sizeof(sin);
    //if (getsockname(m_socket, (struct sockaddr *)&sin, &len) == -1)
    //{
    //  perror("getsockname");
    //}
    //else
    //{
    //  Log_Info( "Before connect socket port number {}", ntohs(sin.sin_port) );
    //}
#if defined(HK_PLATFORM_NX)
    if (::socketconnect(m_socket, (struct sockaddr*)&server, sizeof(server)) < 0)
#else
    if(::connect(m_socket, (struct sockaddr*)&server, sizeof(server)) < 0)
#endif
    {
#ifdef HK_PLATFORM_WIN32
        // may be an async socket
        const int wsaError = WSAGetLastError();
        if ( wsaError == WSAEWOULDBLOCK )
        {
            return HK_SUCCESS;
        }
        else
        {
            Log_Error( "Socket connection WSA error code #{:x}", wsaError );
        }
#endif

        HK_WARN(0x34d25e96, "Cannot connect to server!");
        close();
        return HK_FAILURE;
    }

    //if (getsockname(m_socket, (struct sockaddr *)&sin, &len) == -1)
    //{
    //  perror("getsockname");
    //}
    //else
    //{
    //  Log_Info( "After connect socket port number {}", ntohs(sin.sin_port) );
    //}

    return HK_SUCCESS;
}


hkResult hkBsdSocket::listen(int port)
{
    if( createSocket().isFailure())
    {
        return HK_FAILURE;
    }

    // bind to specified port
    struct sockaddr_in local;
    local.sin_family = AF_INET;
    local.sin_addr.s_addr = INADDR_ANY;
    local.sin_port = htons((unsigned short)port);
    //Log_Info( "~~~~~~~~~~~~~~~~~ SOCKET {} ~~~ BINDING TO {} ~~~ PORT {}", m_socket, local.sin_addr.s_addr, port );

#if defined(HK_PLATFORM_NX)
    if (nn::socket::Bind(m_socket, (struct sockaddr*)&local, sizeof(local)) == SOCKET_ERROR)
#else
    if( ::bind(m_socket,(struct sockaddr*)&local,sizeof(local) ) == SOCKET_ERROR )
#endif
    {
        HK_WARN(0x661cf90d, "Error binding to socket!");
        close();
        return HK_FAILURE;
    }

    // put the server socket into a listening state
#ifdef HK_PLATFORM_NX
    if( ::socketlisten(m_socket,4) == SOCKET_ERROR )
#else
    if( ::listen(m_socket,4) == SOCKET_ERROR )
#endif
    {
        HK_WARN(0x14e1a0f9, "Error listening to socket!");
        close();
        return HK_FAILURE;
    }

    // try to report which set of addresses we are listening on
    hkStringBuf addrString;
    int portDummy;
    getAddress( addrString, portDummy );
    Log_Info( "Listening on host [{}] port {}", addrString, port );
    return HK_SUCCESS;
}

void hkBsdSocket::getAddress( hkStringBuf& hostOut, int& portOut ) const
{
    s_platformGetAddressString( hostOut );

    sockaddr_in sa;
    socklen_t sa_len = sizeof(sa);

    if ( getsockname( m_socket, (sockaddr*)&sa, &sa_len ) != -1 )
    {
        portOut = ntohs( sa.sin_port );
    }
}

bool hkBsdSocket::canWrite() const
{
    if ( m_socket != INVALID_SOCKET )
    {
        fd_set writeFds;
        FD_ZERO( &writeFds );
        FD_SET( m_socket, &writeFds );

        int maxFd = ( int ) ( m_socket + 1 );
        timeval t = { 0, 0 };   // no wait time -- i.e. non blocking select
        int numHits = ::select( maxFd, HK_NULL, &writeFds, HK_NULL, &t );

        return numHits > 0;
    }
    return false;
}

bool hkBsdSocket::canRead() const
{
    if (m_socket != INVALID_SOCKET)
    {
        fd_set readFds;
        FD_ZERO(&readFds);
        FD_SET(m_socket, &readFds);

        int maxFd = (int)(m_socket + 1);
        timeval t = {0, 0}; // no wait time -- i.e. non blocking select
        int numHits = ::select(maxFd, &readFds, HK_NULL, HK_NULL, &t);

        return numHits > 0;
    }
    return false;
}

_Ret_maybenull_ hkSocket* hkBsdSocket::pollForNewClient(_Inout_opt_ hkInetAddr* remoteAddr)
{
    //xx HK_ASSERT( 0x73993156, m_socket != INVALID_SOCKET, "kBsdSocket::pollForNewClient has no socket!");

    // poll the listener socket for new client sockets
    if( m_socket != INVALID_SOCKET )
    {
        fd_set readFds;
        FD_ZERO(&readFds);
        FD_SET(m_socket, &readFds);

        fd_set exceptFds;
        FD_ZERO(&exceptFds);
        FD_SET(m_socket, &exceptFds);


        // check if there is any client trying to connect

        int maxFd = (int)(m_socket + 1);
        timeval t = {0, 0}; // no wait time -- i.e. non blocking select
        int numHits = select(maxFd, &readFds, HK_NULL, &exceptFds, &t);

        if( (numHits > 0) && FD_ISSET(m_socket, &readFds) )
        {
            struct sockaddr_in from;
            socklen_t fromlen = sizeof(from);
            socket_t s = static_cast<socket_t>( ::accept(m_socket, (struct sockaddr*)&from, &fromlen));

            if (remoteAddr)
            {
                remoteAddr->m_ipAddress = from.sin_addr.s_addr;
                remoteAddr->m_port = ntohs(from.sin_port);
            }
#if !defined(NTDDI_VERSION) || (NTDDI_VERSION > 0x05020200)
            // Anything not Windows XP
#   if defined(HK_PLATFORM_WIIU) || defined(HK_PLATFORM_NX)
            char strBuffer[65];
#   else
            char strBuffer[INET6_ADDRSTRLEN];
#   endif
            if ( inet_ntop( AF_INET, &from.sin_addr, strBuffer, sizeof( strBuffer ) ) )
            {
                const unsigned short port = ntohs( from.sin_port );
                Log_Info( "Socket got connection from [{}:{}]", strBuffer, port );
            }
#else
            // Windows XP
            {
                // work around GCC/Clang statement expression incompatible with Log_xxx() macro expansion
                const char* const server = inet_ntoa( from.sin_addr );
                const unsigned short port = ntohs( from.sin_port );
                Log_Info( "Socket got connection from [{}:{}]", server, port );
            }
#endif

            if ( s == INVALID_SOCKET )
            {
                HK_WARN(0x774fad25, "Error accepting a connection!");
            }
            else
            {
                // Add the current connection to the servers list
                unsigned int optval = 1;
                ::setsockopt( s, IPPROTO_TCP, TCP_NODELAY, (char *)&optval, sizeof( unsigned int ) );

                return new hkBsdSocket(s);
            }
        }
        else if(numHits == SOCKET_ERROR)
        {
            HK_WARN(0x3fe16171, "select() error");
        }
    }

    return HK_NULL;
}


hkResult hkBsdSocket::setBlocking(hkBool blocking)
{
#if defined(HK_PLATFORM_WIN32)
    u_long iMode = blocking ? 0 : 1;

    int err = ioctlsocket(m_socket, FIONBIO, &iMode);
    return err == 0 ? HK_SUCCESS : HK_FAILURE;
#elif defined(HK_PLATFORM_LINUX) || defined(HK_PLATFORM_ANDROID) || defined(HK_PLATFORM_PS4) || defined(HK_PLATFORM_MAC) || defined(HK_PLATFORM_IOS) || defined(HK_PLATFORM_NX)
    int flags = fcntl(m_socket, F_GETFL, 0);
    int ret;
    if( blocking )
    {
        ret = fcntl(m_socket, F_SETFL, flags & ~O_NONBLOCK);
    }
    else
    {
        ret = fcntl(m_socket, F_SETFL, flags | O_NONBLOCK);
    }

    return ret == SOCKET_ERROR ? HK_FAILURE : HK_SUCCESS;
#else
    return blocking ? HK_SUCCESS : HK_FAILURE;
#endif
}

void hkBsdGetAddressString( hkStringBuf& addrString )
{
    addrString = "unknown";

#if defined(HK_PLATFORM_WIN32) || defined (HK_PLATFORM_LINUX) || defined(HK_PLATFORM_MAC) || defined(HK_PLATFORM_IOS) || defined(HK_PLATFORM_ANDROID)
    char nb[128];
    if ( ::gethostname(nb, 128) >= 0 ) // not IPv6 compliant though, some Linux system by balk at this
    {
        addrString = nb;
        hkStringBuf localName = nb;

#if defined(HK_PLATFORM_IOS)
        localName += ".local";
#endif

        hostent* details = ::gethostbyname(localName.cString());

        if (details)
        {
            in_addr** ipAddress = (in_addr**) details->h_addr_list;
            while (*ipAddress != HK_NULL)
            {
                const char* ipAddr = inet_ntoa(**(ipAddress++));

                addrString += " [";
                addrString += ipAddr;
                addrString += "]";
            }
        }
    }
#elif defined (HK_PLATFORM_NX)
    in_addr outIpAddress = { 0 };
    nn::Result res = nn::nifm::GetCurrentPrimaryIpAddress(&outIpAddress);
    if (res.IsSuccess())
    {
        addrString += nn::socket::InetNtoa(outIpAddress);
    }
    else
    {
        addrString = "unknown";
    }
#endif
}
hkResult hkBsdSocket::createDatagramSocket(int port, int listenAddress)
{
    close();
#if defined(HK_PLATFORM_NX)
    m_socket = static_cast<socket_t>(nn::socket::Socket(AF_INET, SOCK_DGRAM, 0));
#else
    m_socket = static_cast<socket_t>( ::socket(AF_INET, SOCK_DGRAM, 0) );
#endif
    if(m_socket == INVALID_SOCKET)
    {
        HK_WARN(0x3b98e884, "Error creating socket!");
        return HK_FAILURE;
    }

    // bind to specified port
    struct sockaddr_in local;
    local.sin_family = AF_INET;
    local.sin_addr.s_addr = listenAddress;
    local.sin_port = htons((unsigned short)port);

    int res = 0;

    unsigned int optval = 1;
    res = ::setsockopt( m_socket, SOL_SOCKET, SO_BROADCAST, (char *)&optval, sizeof( unsigned int ) );

    if ( res != 0 )
    {
        HK_WARN(0x3b98e886, "Error setting socket option SO_BROADCAST, error code " << getLastError() << "\n");
        return HK_FAILURE;
    }
    union
    {
        int reuseAddress;
        char data[1];
    } option;
    option.reuseAddress = 1;
    setsockopt( m_socket, SOL_SOCKET, SO_REUSEADDR, &option.data[0], sizeof( option ) );

    setBlocking(false);
    //Log_Info( "~~~~~~~~~~~~~~~~~ SOCKET {} ~~~ BINDING TO {} ~~~ PORT {}", m_socket, local.sin_addr.s_addr, port );
#if defined(HK_PLATFORM_NX)
    if (  nn::socket::Bind( m_socket, (struct sockaddr*)&local, sizeof(local) ) == SOCKET_ERROR )
#else
    if( ::bind(m_socket,(struct sockaddr*)&local,sizeof(local) ) == SOCKET_ERROR )
#endif
    {
        HK_WARN(0x661cf90d, "Error binding to socket, error code " << getLastError() << "\n");
        close();
        return HK_FAILURE;
    }

    return HK_SUCCESS;

}

int hkBsdSocket::sendTo(_In_reads_bytes_(nbytes) const void* buf, int nbytes, const hkInetAddr& toAddress)
{
    struct sockaddr_in remoteAddr;
    {
        hkString::memSet(&remoteAddr,0,sizeof(remoteAddr));
        remoteAddr.sin_family = AF_INET;
        remoteAddr.sin_port = htons( (unsigned short)toAddress.m_port );
        remoteAddr.sin_addr.s_addr = toAddress.m_ipAddress;
    }

    int bytesSent = sendto( m_socket, (const char*)buf, nbytes, 0, ( struct sockaddr* )&remoteAddr, sizeof( remoteAddr ) );

    if ( bytesSent == SOCKET_ERROR )
    {
        HK_WARN( 0x661cf90d, "sendTo failed!" );
    }

    return bytesSent;
}

int hkBsdSocket::receiveFrom(_Out_writes_bytes_(nbytes) void* buf, int nbytes, hkInetAddr &fromAddress)
{
    struct sockaddr_in senderAddr;
    socklen_t len = sizeof(senderAddr);
    int bytesReceived = ::recvfrom( m_socket, (char*)buf, nbytes, 0, (sockaddr *)&senderAddr, &len );
    fromAddress.m_ipAddress = senderAddr.sin_addr.s_addr;
    fromAddress.m_port = ntohs(senderAddr.sin_port);
    return bytesReceived;
}

int hkBsdSocket::getLastError()
{
    #if defined(HK_PLATFORM_WIN32)
        int errcode = WSAGetLastError();
        switch (errcode)
        {
            case NO_ERROR:
                return SOCKET_NO_ERROR;
                break;
            case WSAEWOULDBLOCK:
                return SOCKET_WOULD_BLOCK;
                break;
            case WSAECONNRESET:
                return SOCKET_CONN_RESET;
                break;
            case WSANOTINITIALISED:
                return SOCKET_NOT_INITIALIZED;
                break;
            case WSAEADDRINUSE:
                return SOCKET_ADDR_IN_USE;
                break;
            case WSAENETDOWN:
                return SOCKET_NET_DOWN;
                break;
            default:
                return SOCKET_NET_DOWN;
        }
    #else
        return 0;
    #endif
}

hkResult hkBsdSocket::bind( int port )
{
    // bind to specified port
    struct sockaddr_in local;
    local.sin_family = AF_INET;
    local.sin_addr.s_addr = INADDR_ANY;
    local.sin_port = htons( (unsigned short)port );

    union
    {
        int reuseAddress;
        char data[1];
    } option;
    option.reuseAddress = 1;
    setsockopt( m_socket, SOL_SOCKET, SO_REUSEADDR, &option.data[0], sizeof( option ) );

#ifdef HK_PLATFORM_WIN32
    {
        const int wsaError = WSAGetLastError();
        if ( wsaError )
        {
            Log_Error( "Failed to setsockopt(), WSA error code #{:x}", wsaError );
        }
    }
    BOOL bReUse = FALSE;
    setsockopt( m_socket, SOL_SOCKET, SO_EXCLUSIVEADDRUSE, (const char *)&bReUse, sizeof( bReUse ) );
#endif

    //Log_Info( "~~~~~~~~~~~~~~~~~ SOCKET {} ~~~ BINDING TO {} ~~~ PORT {}", m_socket, local.sin_addr.s_addr, port );
#if defined(HK_PLATFORM_NX)
    if (nn::socket::Bind(m_socket, (struct sockaddr*)&local, sizeof(local)) == SOCKET_ERROR)
#else
    if( ::bind(m_socket,(struct sockaddr*)&local,sizeof(local) ) == SOCKET_ERROR )
#endif
    {
#ifdef HK_PLATFORM_WIN32
        {
            const int wsaError = WSAGetLastError();
            Log_Error( "Failed to bind(), WSA error code #{:x}", wsaError );
        }
#endif

        HK_WARN(0x661cf9d0, "Error binding to socket!");
        // We dont close socket on a bind fail. That way can still be used for connect()
        return HK_FAILURE;
    }
    return HK_SUCCESS;

}

hkResult hkBsdSocket::getInetAddr( hkInetAddr& inetAddr )
{
    struct sockaddr_in addr;
    socklen_t addrLen = sizeof(addr);
    int res = ::getsockname( m_socket, ( struct sockaddr* )&addr, &addrLen );

    if (res == SOCKET_ERROR)
    {
        HK_WARN(0x146cf90d, "Error getting socket address!");
        return HK_FAILURE;
    }
    inetAddr.m_ipAddress = addr.sin_addr.s_addr;
    inetAddr.m_port = ntohs( addr.sin_port );

    return HK_SUCCESS;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
