// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 LINUX32 LINUX64 X64 MAC ANDROID METRO UWP OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Fwd/hkcstdio.h>
#include <Common/Base/Fwd/hkcstdarg.h>
#include <Common/Base/Config/hkConfigVersion.h>

#include <Common/Base/System/Io/OptionParser/hkOptionParser.h>

#ifndef HK_OPTION_PARSER_CONSOLE_WIDTH
# define HK_OPTION_PARSER_CONSOLE_WIDTH 80
#endif

const static int s_consoleWidth = HK_OPTION_PARSER_CONSOLE_WIDTH;
const static int s_maxErrorMessageLength = 256;
const static int s_maxDefaultTextLength = 512;

#undef HK_OPTION_PARSER_CONSOLE_WIDTH

// A specific assert is being defined here because the option parser code may be executed before the base system and memory manager have been set up.
// HK_ASSERT_NO_MSG relies on creating an object (using said memory manager) which means that we cannot use it here.
#ifdef HK_DEBUG_SLOW
    #define OPTION_PARSER_ASSERT(x, s) {if (!(x)) {HK_BREAKPOINT(0); outputAssertMessage(s);}}
    #define OPTION_PARSER_ASSERT_RETURN(x, s) {if (!(x)) {HK_BREAKPOINT(0); outputAssertMessage(s); return false;}}
#else
    #define OPTION_PARSER_ASSERT(x, s) {}
    #define OPTION_PARSER_ASSERT_RETURN(x, s) {}
#endif

namespace
{
    void outputAssertMessage(_Printf_format_string_ const char* format, ...);

    enum OptionFlagInfoFlags
    {
        FLAGINFO_LONG   = 1,    // --flag
        FLAGINFO_EQUALS = 2,    // --flag=VALUE | -f=VALUE
        FLAGINFO_CONCAT = 4     // --flagVALUE | -fVALUE
    };

    struct OptionFlagInfo
    {
        const char* m_name;
        int m_nameLength;
        const char* m_value;
        int m_valueLength;
        int m_flags;
    };

    void nullIfEmpty(_Inout_opt_z_ const char*& str)
    {
        if (str != HK_NULL && hkString::strLen(str) == 0)
        {
            str = HK_NULL;
        }
    }

    bool isDigitChar(char c)
    {
        return (c >= '0' && c <= '9');
    }

    bool isLiteralChar(char c)
    {
        return (c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z') || (c >= '0' && c <= '9');
    }

    bool isValidFlag(_In_z_ const char* flag)
    {
        int i = 0;

        while (flag[i] != '\0')
        {
            if (!isLiteralChar(flag[i]) && flag[i] != '-')
            {
                return false;
            }
            i ++;
        }
        return true;
    }

    bool isInteger(_In_z_ const char* value, int valueLength, bool allowNegative = true)
    {
        int offset = 0;

        if (value[0] == '-')
        {
            if (allowNegative)
            {
                offset = 1;
            }
            else
            {
                return false;
            }
        }

        bool valid = true;
        bool hex = false;

        if (value[offset] == '0')
        {
            if ((value[1 + offset] == 'x') && valueLength >= 3)
            {
                hex = true;
            }
            else
            {
                // Allow a single 0
                if (valueLength != 1)
                {
                    valid = false;
                }
            }
        }

        if (valid)
        {
            for (int i = offset; i < valueLength; i++)
            {
                // Can skip 'x' if hexadecimal
                if (hex && i == 1 + offset)
                {
                    continue;
                }

                // Standard digit test
                if (!isDigitChar(value[i]))
                {
                    valid = false;
                }

                // Check for hex
                if (hex && i > 1 + offset)
                {
                    if ((value[i] >= 'A' && value[i] <= 'F') || (value[i] >= 'a' && value[i] <= 'f'))
                    {
                        valid = true;
                    }
                }
            }
        }

        return valid;
    }

    void printTextBlock(_In_z_ const char* text, const int indent = 0, const bool absolute = false)
    {
        OPTION_PARSER_ASSERT(indent >= 0, "Start position must not be negative");

        const int width = s_consoleWidth - indent;

        const int textLength = hkString::strLen(text);
        int startIndex = 0;

        #ifdef HK_DEBUG_SLOW
        for (int i = 0; i < textLength; i++)
        {
            OPTION_PARSER_ASSERT(text[i] != '\t', "Text blocks cannot contain tab characters");
        }
        #endif

        while (startIndex < textLength)
        {
            // Print left whitespace
            if ((indent != 0) && (absolute || startIndex > 0))
            {
                printf("%*s", indent, "");
            }

            char line[s_consoleWidth + 1];
            const int lineLength = hkString::snPrintf(line, s_consoleWidth, s_consoleWidth, "%.*s", width, &text[startIndex]);

            int endIndex = startIndex + lineLength;

            // First check for user end lines
            {
                bool endOfLineFound = false;
                for (int i = startIndex; i < endIndex; i++)
                {
                    if (text[i] == '\n')
                    {
                        printf("%.*s\n", i - startIndex, &text[startIndex]);
                        startIndex = i + 1;
                        endOfLineFound = true;
                        break;
                    }
                }

                if (endOfLineFound)
                {
                    continue;
                }
            }


            bool skipEndOfLineSpace = false;

            // Check if we need to break a word at the end of the line
            if (lineLength == width && line[width - 1] != ' ')
            {
                // Don't break words, find the ending position of the last word
                int backtrace = 1;

                while ((line[lineLength - backtrace] != ' ') && (lineLength != backtrace))
                {
                    backtrace++;
                }

                // If word does not take up whole line
                if (lineLength != backtrace)
                {
                    endIndex -= backtrace - 1; // -1 for not including the space on the next line
                }
            }
            // If we have a space right at the end of the line, we need to replace that with a new line character
            else if (endIndex < textLength)
            {
                // Remove end of line space (i.e. don't print "hello world /n", instead print "hello world/n")
                skipEndOfLineSpace = true;
            }

            printf("%.*s\n", endIndex - startIndex - (skipEndOfLineSpace ? 1 : 0), &text[startIndex]);
            startIndex = endIndex;
        }
    }

    void outputAssertMessage(_Printf_format_string_ const char* format, ...)
    {
        va_list va;
        va_start(va, format);
        char errorMessage[s_maxErrorMessageLength];
        hkString::vsnPrintf(errorMessage, s_maxErrorMessageLength, format, va);
        va_end(va);
        printf("ASSERT FAILED: ");
        printTextBlock(errorMessage, 15);
    }

    void outputErrorMessage(_Inout_ hkOptionParser* parser, _Printf_format_string_ const char* format, ...)
    {
        va_list va;
        va_start(va, format);
        char errorMessage[s_maxErrorMessageLength];
        hkString::vsnPrintf(errorMessage, s_maxErrorMessageLength, format, va);
        va_end(va);
        printf("ERROR: ");
        printTextBlock(errorMessage, 7);
        printf("\n");
    }

    _Ret_z_ const char* getFlagName(_In_ const OptionFlagInfo* flagInfo, _In_ const hkOptionParser::Option* option)
    {
        return ((flagInfo->m_flags & FLAGINFO_LONG) ? option->m_longFlag : option->m_shortFlag);
    }

    // Parse a single argument and validate it as a flag
    hkResult parseFlag(_In_z_ const char* argv, OptionFlagInfo* flagInfo, bool acceptShortFlags)
    {
        // Clear flagInfo struct
        flagInfo->m_name        = HK_NULL;
        flagInfo->m_nameLength  = 0;
        flagInfo->m_value       = HK_NULL;
        flagInfo->m_valueLength = 0;
        flagInfo->m_flags       = 0;

        // All flags must start with a '-'
        if (argv[0] == '-')
        {
            flagInfo->m_name = argv + 1;

            if (argv[1] == '-')
            {
                // Flag is a long flag "--flag" as opposed to a short flag "-f"
                flagInfo->m_name++;
                flagInfo->m_flags |= FLAGINFO_LONG;
            }
            else if (!acceptShortFlags)
            {
                return HK_FAILURE;
            }

            // Loop over characters
            while (true)
            {
                if (flagInfo->m_name[flagInfo->m_nameLength] == '=')
                {
                    // Flag has a value
                    flagInfo->m_flags |= FLAGINFO_EQUALS;
                    break;
                }
                else if (flagInfo->m_name[flagInfo->m_nameLength] == '\0')
                {
                    // Flag does not have a value
                    break;
                }

                flagInfo->m_nameLength++;
            }

            // Assumes any flag which has a digit at the end is of type '--flagVALUE' and does not use '=' is a concatenated flag
            if (!(flagInfo->m_flags & FLAGINFO_EQUALS) &&
                flagInfo->m_nameLength > (flagInfo->m_flags & FLAGINFO_LONG ? 2 : 1) &&
                isDigitChar(flagInfo->m_name[flagInfo->m_nameLength - 1]))
            {
                flagInfo->m_flags |= FLAGINFO_CONCAT;
            }

            // if the flag name length is at least 1 character
            if (flagInfo->m_nameLength > 0)
            {
                if (flagInfo->m_flags & FLAGINFO_EQUALS)
                {
                    // Parse value
                    flagInfo->m_value = flagInfo->m_name + flagInfo->m_nameLength + 1;

                    while (flagInfo->m_value[flagInfo->m_valueLength] != '\0')
                    {
                        flagInfo->m_valueLength++;
                    }
                }
                else if (flagInfo->m_flags & FLAGINFO_CONCAT)
                {
                    // Alter name length and value for a counter flag --flagVALUE
                    while (isDigitChar(flagInfo->m_name[flagInfo->m_nameLength - 1]))
                    {
                        flagInfo->m_nameLength --;
                        flagInfo->m_valueLength ++;
                    }

                    flagInfo->m_value = (flagInfo->m_name + flagInfo->m_nameLength);
                }
                return HK_SUCCESS;
            }
        }

        return HK_FAILURE;
    }

    // Search our stored options for the option in question
    hkResult findOption(_In_reads_z_(nameLength) const char* name, int nameLength, int flags, _Inout_updates_(optionCount) hkOptionParser::Option* options, int optionCount, _In_opt_ hkOptionParser::Option* option = HK_NULL, int skipIndex = -1)
    {
        bool found = false;

        for (int i = 0; i < optionCount; i ++)
        {
            // Used when verifying options to make sure that there are no duplicates
            if (i == skipIndex)
            {
                continue;
            }

            hkOptionParser::Option* currentOption = (options + i);

            if (flags & FLAGINFO_LONG)
            {
                // Check that the length of both matches, this is not necessary for short flags as they are always 1
                if (nameLength == hkString::strLen(currentOption->m_longFlag))
                {
                    found = (hkString::strNcasecmp(name, currentOption->m_longFlag, nameLength) == 0);
                }
            }
            else if (currentOption->m_shortFlag)
            {
                found = (hkString::strNcasecmp(name, currentOption->m_shortFlag, nameLength) == 0);
            }

            if (found)
            {
                if (option != HK_NULL)
                {
                    option->m_shortFlag = currentOption->m_shortFlag;
                    option->m_longFlag  = currentOption->m_longFlag;
                    option->m_help      = currentOption->m_help;
                    option->m_type      = currentOption->m_type;
                    option->m_value     = currentOption->m_value;
                    option->m_default   = currentOption->m_default;
                }
                return HK_SUCCESS;
            }
        }

        return HK_FAILURE;
    }

    // Parses the string value to set the option's value in its native format
    hkResult setOptionValue(_Inout_ hkOptionParser::Option* option, _In_reads_z_(valueLength) const char* value, int valueLength = -1)
    {
        if (valueLength == -1)
        {
            valueLength = hkString::strLen(value);
        }

        switch (option->m_type)
        {
        case hkOptionParser::OPTION_BOOL:
            {
                if (valueLength == 0 || (valueLength == 1 && value[0] == '1'))
                {
                    option->m_value->b = true;
                }
                else if (valueLength == 1 && value[0] == '0')
                {
                    option->m_value->b = false;
                }
                else
                {
                    return HK_FAILURE;
                }
                break;
            }
        case hkOptionParser::OPTION_COUNTER:
            {
                if (valueLength == 0)
                {
                    option->m_value->i ++;
                }
                else if (isInteger(value, valueLength, false))
                {
                    option->m_value->i += hkString::atoi(value);
                }
                else
                {
                    return HK_FAILURE;
                }
            }
            break;
        case hkOptionParser::OPTION_INT:
            {
                if (valueLength > 0)
                {
                    if (isInteger(value, valueLength))
                    {
                        option->m_value->i = hkString::atoi(value);
                    }
                    else
                    {
                        return HK_FAILURE;
                    }
                }
                else
                {
                    return HK_FAILURE;
                }
            }
            break;
        case hkOptionParser::OPTION_STRING:
            {
                option->m_value->s = value;
            }
            break;
        case hkOptionParser::OPTION_MULTI_STRING:
            {
                HK_ASSERT_NO_MSG(0x7e3c723d, option->m_value->ms.m_optionCount < hkOptionParser::MultiOptionValue::NUM_MULTI_ARGS);
                option->m_value->ms.m_options[option->m_value->ms.m_optionCount++] = value;
            }
            break;
        }

        return HK_SUCCESS;
    }

    // Reset all option's values to their defaults
    void defaultOptions(_Inout_updates_(optionCount) hkOptionParser::Option* options, int optionCount)
    {
        for (int i = 0; i < optionCount; i++)
        {
            hkOptionParser::Option* currentOption = (options + i);

            switch (currentOption->m_type)
            {
            case hkOptionParser::OPTION_BOOL:
                currentOption->m_value->b = currentOption->m_default.b;
                break;
            case hkOptionParser::OPTION_COUNTER:
                currentOption->m_value->u = currentOption->m_default.u;
                break;
            case hkOptionParser::OPTION_INT:
                currentOption->m_value->i = currentOption->m_default.i;
                break;
            case hkOptionParser::OPTION_STRING:
                currentOption->m_value->s = currentOption->m_default.s;
                break;
            case hkOptionParser::OPTION_MULTI_STRING:
                for(int j=0; j < hkOptionParser::MultiOptionValue::NUM_MULTI_ARGS; j++)
                {
                    currentOption->m_value->ms.m_options[j] = HK_NULL;
                    currentOption->m_value->ms.m_optionCount = 0;
                }
                break;
            }
        }
    }
}

hkOptionParser::hkOptionParser(_In_z_ const char* title, _In_z_ const char* desc) :
    m_options(HK_NULL),
    m_optionCount(0),
    m_arguments(HK_NULL),
    m_argumentCount(0),
    m_title(title),
    m_desc(desc),
    m_argumentsName(HK_NULL),
    m_argumentsHelp(HK_NULL),
    m_argumentsType(ARGUMENTS_ZERO_OR_MORE)
{}

hkOptionParser::Option::Option(_In_z_ const char* shortFlag, _In_z_ const char* longFlag, _In_z_ const char* help, _Inout_ const char** value, _In_opt_z_ const char* defaultValue) :
    m_shortFlag(shortFlag),
    m_longFlag(longFlag),
    m_help(help),
    m_type(hkOptionParser::OPTION_STRING)
{
    m_value = static_cast<hkOptionParser::OptionValue*>((void*)value);
    m_default.s = defaultValue;
}

hkOptionParser::Option::Option(_In_z_ const char* shortFlag, _In_z_ const char* longFlag, _In_z_ const char* help, _Inout_ bool* value, bool defaultValue) :
    m_shortFlag(shortFlag),
    m_longFlag(longFlag),
    m_help(help),
    m_type(hkOptionParser::OPTION_BOOL)
{
    m_value = static_cast<hkOptionParser::OptionValue*>((void*)value);
    m_default.b = defaultValue;
}

hkOptionParser::Option::Option(_In_z_ const char* shortFlag, _In_z_ const char* longFlag, _In_z_ const char* help, _Inout_ int* value, int defaultValue) :
    m_shortFlag(shortFlag),
    m_longFlag(longFlag),
    m_help(help),
    m_type(hkOptionParser::OPTION_INT)
{
    m_value = static_cast<hkOptionParser::OptionValue*>((void*)value);
    m_default.i = defaultValue;
}

hkOptionParser::Option::Option(_In_z_ const char* shortFlag, _In_z_ const char* longFlag, _In_z_ const char* help, _Inout_ unsigned int* value, unsigned int defaultValue) :
    m_shortFlag(shortFlag),
    m_longFlag(longFlag),
    m_help(help),
    m_type(hkOptionParser::OPTION_COUNTER)
{
    m_value = static_cast<hkOptionParser::OptionValue*>((void*)value);
    m_default.u = defaultValue;
}

hkOptionParser::Option::Option(_In_z_ const char* shortFlag, _In_z_ const char* longFlag, _In_z_ const char* help, _Inout_ MultiOptionValue* value) :
    m_shortFlag(shortFlag),
    m_longFlag(longFlag),
    m_help(help),
    m_type(hkOptionParser::OPTION_MULTI_STRING)
{
    m_value = reinterpret_cast<hkOptionParser::OptionValue*>(value);
    for(int i=0; i < hkOptionParser::MultiOptionValue::NUM_MULTI_ARGS; i++)
    {
        m_default.ms.m_options[i] = HK_NULL;
    }
    m_default.ms.m_optionCount = 0;
}


bool hkOptionParser::setOptions(_Inout_updates_(count) Option* options, int count)
{
    for (int i = 0; i < count; i++)
    {
        Option* option = &options[i];

        // Change the empty string to HK_NULL
        nullIfEmpty(option->m_shortFlag);
        nullIfEmpty(option->m_longFlag);
        nullIfEmpty(option->m_help);

        // Null checks
        OPTION_PARSER_ASSERT_RETURN(option->m_longFlag != HK_NULL, "Long flag must be specified\n");
        OPTION_PARSER_ASSERT_RETURN(option->m_help != HK_NULL, "Help text must beprovided\n");
        OPTION_PARSER_ASSERT_RETURN(option->m_value != HK_NULL, "Address to store option value must be provided\n");

        // Check option flags are valid
        OPTION_PARSER_ASSERT_RETURN(option->m_shortFlag == HK_NULL || hkString::strLen(option->m_shortFlag) == 1, "Length of short flag must be exactly 1\n");
        OPTION_PARSER_ASSERT_RETURN(option->m_shortFlag == HK_NULL || isLiteralChar(option->m_shortFlag[0]), "Short flag not valid\n");
        OPTION_PARSER_ASSERT_RETURN(hkString::strLen(option->m_longFlag) >= 2, "Length of long flag must be at least 2\n");
        OPTION_PARSER_ASSERT_RETURN(isValidFlag(option->m_longFlag), "Long flag not valid, must be alphanumeric\n");

        // Check if the flag of counters ends in a number (this is not allowed)
        OPTION_PARSER_ASSERT_RETURN(option->m_type != OPTION_COUNTER || (!isDigitChar(option->m_longFlag[hkString::strLen(option->m_longFlag) - 1])), "Flags for counters cannot end in a digit");
        OPTION_PARSER_ASSERT_RETURN(option->m_type != OPTION_COUNTER || option->m_shortFlag == HK_NULL || (!isDigitChar(option->m_shortFlag[0])), "Flags for counters cannot end in a digit");

        // Check if the help option isn't manually defined
        OPTION_PARSER_ASSERT_RETURN(option->m_shortFlag == HK_NULL || hkString::strNcasecmp(option->m_shortFlag, "h", 1) != 0, "The -h flag is already built in, and cannot be manually defined");
        OPTION_PARSER_ASSERT_RETURN(!(hkString::strNcasecmp(option->m_longFlag, "help", 4) == 0 && hkString::strLen(option->m_longFlag) == 4), "The -help flag is already built in, and cannot be manually defined");

        // Check if option flags are not already in use
        OPTION_PARSER_ASSERT_RETURN(!(option->m_shortFlag && (findOption(option->m_shortFlag, hkString::strLen(option->m_shortFlag), 0, options, count, HK_NULL, i).isSuccess())), "Duplicate short flag definition\n");
        OPTION_PARSER_ASSERT_RETURN(!(option->m_longFlag && (findOption(option->m_longFlag, hkString::strLen(option->m_longFlag), FLAGINFO_LONG, options, count, HK_NULL, i).isSuccess())), "Duplicate long flag definition\n");
    }

    m_options = options;
    m_optionCount = count;

    return true;
}

void hkOptionParser::setArguments(_In_z_ const char* name, _In_z_ const char* help, ArgumentsType argumentsType, _Inout_updates_all_(count) const char** arguments, int count)
{
    // Change the empty string to HK_NULL
    nullIfEmpty(name);
    nullIfEmpty(help);

    OPTION_PARSER_ASSERT(name != HK_NULL, "Name of the arguments must be specified\n");

    // Store argument buffer
    m_arguments = arguments;
    m_argumentCount = count;

    // Store argument information
    m_argumentsName = name;
    m_argumentsHelp = help;
    m_argumentsType = argumentsType;

    // null arguments
    for (int i = 0; i < count; i ++)
    {
        arguments[i] = HK_NULL;
    }
}

hkOptionParser::ParseResult hkOptionParser::parse(int argc, _In_reads_(argc) const char** argv, bool acceptShortFlags)
{
    bool doubleDash = false;
    bool counterDefaultOverride = false;
    int argumentIndex = 0;

    // Make sure options are at there default values
    defaultOptions(m_options, m_optionCount);

    if(m_optionCount == 0)
        doubleDash = true;

    // For each argument passed
    for (int i = 1; i < argc; i ++)
    {
        if (!doubleDash && hkString::strCmp(argv[i], "--") == 0)
        {
            // Stop processing if a '--' is found
            doubleDash = true;
            continue;
        }

        OptionFlagInfo flagInfo;
        Option option;

        // Check if option flag syntax is valid, and is in the list of accepted options
        if (!doubleDash && (parseFlag(argv[i], &flagInfo, acceptShortFlags).isSuccess()))
        {
            if (findOption(flagInfo.m_name, flagInfo.m_nameLength, flagInfo.m_flags & FLAGINFO_LONG, m_options, m_optionCount, &option).isSuccess())
            {
                const char* flagValue   = HK_NULL;
                int flagValueLength     = 0;

                // Get value
                if (flagInfo.m_flags & FLAGINFO_EQUALS)
                {
                    if (flagInfo.m_valueLength > 0 || option.m_type == OPTION_STRING) // Can be the empty string
                    {
                        flagValue = flagInfo.m_value;
                        flagValueLength = flagInfo.m_valueLength;
                    }
                    else if (option.m_type == OPTION_INT || option.m_type == OPTION_COUNTER || option.m_type == OPTION_BOOL)
                    {
                        outputErrorMessage(this, "Option '%s' requires a value", getFlagName(&flagInfo, &option));
                        return PARSE_FAILURE;
                    }
                }
                else if (flagInfo.m_flags & FLAGINFO_CONCAT)
                {
                    if (option.m_type == OPTION_COUNTER)
                    {
                        flagValue = flagInfo.m_value;
                        flagValueLength = flagInfo.m_valueLength;
                    }
                    else
                    {
                        const char* flagName = getFlagName(&flagInfo, &option);
                        outputErrorMessage(this, "Option '%s' does not support %sVALUE syntax. Use %s=VALUE instead", flagName, flagName, flagName);
                        return PARSE_FAILURE;
                    }
                }
                else
                {
                    // Use the next positional argument as the value for integers and strings
                    if (option.m_type == OPTION_INT || option.m_type == OPTION_STRING || option.m_type == OPTION_MULTI_STRING)
                    {
                        i++;

                        if (i < argc)
                        {
                            flagValue = argv[i];
                            flagValueLength = hkString::strLen(flagValue);
                        }
                        else
                        {
                            outputErrorMessage(this, "Option '%s' requires a value", getFlagName(&flagInfo, &option));
                            return PARSE_FAILURE;
                        }
                    }
                }

                // Since counters increment a value and don't override it, we have to manually erase the default value if a counter is specified
                if (option.m_type == OPTION_COUNTER && (!counterDefaultOverride || flagInfo.m_flags & FLAGINFO_EQUALS))
                {
                    option.m_value->u = 0u;
                    counterDefaultOverride = true;
                }

                // Check value and parse
                if (setOptionValue(&option, flagValue, flagValueLength).isFailure())
                {
                    outputErrorMessage(this, "Invalid value '%s' found for option '%s'", flagValue, getFlagName(&flagInfo, &option));
                    return PARSE_FAILURE;
                }
            }
            else
            {
                // Check for the help flag
                if ( (hkString::strNcasecmp(flagInfo.m_name, "h", 1) == 0 && flagInfo.m_nameLength == 1) ||
                     (hkString::strNcasecmp(flagInfo.m_name, "help", 4) == 0 && flagInfo.m_nameLength == 4) )
                {
                    // Help was sought
                    usage();
                    return PARSE_HELP;
                }
                else
                {
                    // Throw an error
                    outputErrorMessage(this, "Cannot process '%s', unknown flag", flagInfo.m_name);
                    return PARSE_FAILURE;
                }
            }
        }
        else
        {
            // Is a positional argument
            if (argumentIndex < m_argumentCount)
            {
                m_arguments[argumentIndex++] = argv[i];
            }
            else
            {
                outputErrorMessage(this, "The maximum number of positional arguments (%d) has been exceeded", m_argumentCount);
                return PARSE_FAILURE;
            }
        }
    }

    // Check number of arguments
    if (m_argumentsType == ARGUMENTS_ONE &&  argumentIndex != 1)
    {
        outputErrorMessage(this, "Incorrect number of positional arguments; Should be 1, found %d", argumentIndex);
        return PARSE_FAILURE;
    }
    else if (m_argumentsType == ARGUMENTS_ONE_OR_MORE &&  argumentIndex == 0)
    {
        outputErrorMessage(this, "Incorrect number of positional arguments; Should be 1 or more, found 0");
        return PARSE_FAILURE;
    }

    return PARSE_SUCCESS;
}

void hkOptionParser::usage(_In_opt_z_ const char* extra)
{
    // Extra information
    if (extra)
    {
        printf("\nMESSAGE: ");
        printTextBlock(extra, 9);
        printf("\n");
    }

    // Program details
    printTextBlock(m_title);
    printf("Built with %s\n\n", HAVOK_SDK_VERSION_STRING);
    printTextBlock(m_desc);

    {
        // Usage string
        int count = printf("\nUsage: ");

        // Usage options
        for (int i = 0; i < m_optionCount; i++)
        {
            const char* typeString("");

            switch (m_options[i].m_type)
            {
            case OPTION_COUNTER:
                typeString = "=COUNTER";
                break;
            case OPTION_BOOL:
                typeString = "=BOOL";
                break;
            case OPTION_INT:
                typeString = "=INT";
                break;
            case OPTION_STRING:
                typeString = "=STR";
                break;
            case OPTION_MULTI_STRING:
                break;
            }

            const char* optionStart("[--");
            const char* optionEnd("] ");

            int optionLength = hkString::strLen(optionStart) + hkString::strLen(m_options[i].m_longFlag) + hkString::strLen(typeString) + hkString::strLen(optionEnd);

            if (optionLength >= s_consoleWidth - count)
            {
                printf("\n%*s", 7, "");
                count = 7;
            }

            count += printf("%s%s%s%s", optionStart, m_options[i].m_longFlag, typeString, optionEnd);
        }

        // Usage positional argument
        if (m_arguments)
        {
            switch (m_argumentsType)
            {
            case ARGUMENTS_ONE:
                if (count + hkString::strLen(m_argumentsName) > s_consoleWidth)
                {
                    printf("\n%*s", 7, "");
                }
                printf("%s", m_argumentsName);
                break;
            case ARGUMENTS_ONE_OR_MORE:
                if (count + 8 + hkString::strLen(m_argumentsName) + hkString::strLen(m_argumentsName) > s_consoleWidth)
                {
                    printf("\n%*s", 7, "");
                }
                printf("%s [%s ...]", m_argumentsName, m_argumentsName);
                break;
            case ARGUMENTS_ZERO_OR_MORE:
                if (count + 6 + hkString::strLen(m_argumentsName) > s_consoleWidth)
                {
                    printf("\n%*s", 7, "");
                }
                printf("[%s ...]", m_argumentsName);
                break;
            }
        }
    }

    // Positional arguments
    if (m_argumentsName)
    {
        printf("\n\nPositional arguments:\n\n");
        int posArgWidth = printf("    %s    ", m_argumentsName);
        printTextBlock(m_argumentsHelp, posArgWidth);
    }

    // Optional arguments
    printf("\nOptional arguments:\n\n");

    // Get longest short and long flag
    int longestLongFlag = 0;

    for (int i = 0; i < m_optionCount; i ++)
    {
        int longFlagLength = hkString::strLen(m_options[i].m_longFlag);

        if (longestLongFlag < longFlagLength)
        {
            longestLongFlag = longFlagLength;
        }
    }

    // Print help option
    {
        int c = printf("    -h    --help");
        c += printf("%s%*s ", "       ", longestLongFlag - 4, "");
        printTextBlock("show this help message and exit", c);
    }

    // Print options
    for (int i = 0; i < m_optionCount; i ++)
    {
        int count = 0;

        if (m_options[i].m_shortFlag)
        {
            count += printf("    -%s    ", m_options[i].m_shortFlag);
        }
        else
        {
            count += printf("          ");
        }

        int num = 0;

        count += num = printf("--%s", m_options[i].m_longFlag);

        const char* typeString("");

        switch (m_options[i].m_type)
        {
        case OPTION_INT:
            typeString = "[=INT]  ";
            break;
        case OPTION_STRING:
            typeString = "[=STR]  ";
            break;
        case OPTION_BOOL:
            typeString = "[=BOOL] ";
            break;
        case OPTION_COUNTER:
            typeString = "[=NUM]  ";
            break;
        case OPTION_MULTI_STRING:
            break;
        }

        count += printf("%s%*s ", typeString, longestLongFlag - num + 2, "");

        printTextBlock(m_options[i].m_help, count);

        {
            char defaultText[s_maxDefaultTextLength];

            switch (m_options[i].m_type)
            {

            case OPTION_INT:
                if (m_options[i].m_default.i != 0)
                {
                    hkString::snPrintf(defaultText, s_maxDefaultTextLength, s_maxDefaultTextLength, "(default: %d)", m_options[i].m_default.i);
                    printTextBlock(defaultText, count, true);
                }
                break;
            case OPTION_STRING:
                if (m_options[i].m_default.s != HK_NULL)
                {
                    hkString::snPrintf(defaultText, s_maxDefaultTextLength, s_maxDefaultTextLength, "(default: %s)", m_options[i].m_default.s);
                    printTextBlock(defaultText, count, true);
                }
                break;
            case OPTION_BOOL:
                if (m_options[i].m_default.b != false)
                {
                    hkString::snPrintf(defaultText, s_maxDefaultTextLength, s_maxDefaultTextLength, "(default: %s)", m_options[i].m_default.b ? "1" : "0");
                    printTextBlock(defaultText, count, true);
                }
                break;
            case OPTION_COUNTER:
                if (m_options[i].m_default.u != 0)
                {
                    hkString::snPrintf(defaultText, s_maxDefaultTextLength, s_maxDefaultTextLength, "(default: %u)", m_options[i].m_default.u);
                    printTextBlock(defaultText, count, true);
                }
                break;
            case OPTION_MULTI_STRING:
                break;
            }
        }
    }

    // Print type information
    printf("\nType Information:\n\n");
    printf("    BOOL    ");
    printTextBlock("1 for True, 0 for False", 12);
    printf("    INT     ");
    printTextBlock("Signed integer value between -268435455 and 268435455. Also supports hexadecimal input (-0xFFFFFFF to 0xFFFFFFF)", 12);
    printf("    STR     ");
    printTextBlock("String of characters. Surround strings containing spaces with double quotes", 12);
    printf("    NUM     ");
    printTextBlock("Same as INT, but cannot be negative", 12);
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
