// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0


#include <Common/Base/hkBase.h>
#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>
#include <Common/Base/System/Io/OArchive/hkOArchive.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>
#include <Common/Base/System/Io/Writer/Buffered/hkBufferedStreamWriter.h>

hkOArchive::hkOArchive(_In_ hkStreamWriter* sw, bool bs)
    :   m_writer(sw), m_byteSwap(bs)
{
}

hkOArchive::hkOArchive(_In_z_ const char* filename, bool bs)
    : m_byteSwap(bs)
{
    m_writer = hkFileSystem::getInstance().openWriter(filename);
}

hkOArchive::hkOArchive(_Out_writes_bytes_(memSize) void* mem, int memSize, bool bs)
    : m_byteSwap(bs)
{
    m_writer.setAndDontIncrementRefCount( new hkBufferedStreamWriter(mem, memSize, false) );
}

hkOArchive::hkOArchive(hkArray<char>& arr, bool bs)
    : m_byteSwap(bs)
{
    m_writer.setAndDontIncrementRefCount( new hkArrayStreamWriter(&arr, hkArrayStreamWriter::ARRAY_BORROW) );
}

hkOArchive::~hkOArchive()
{
}

//
// Singles
//

void hkOArchive::write8(hkChar c)
{
    m_writer->write(&c, 1);
}

void hkOArchive::write8u(hkUchar u)
{
    m_writer->write(&u, 1);
}

void hkOArchive::write16(hkInt16 i)
{
    writeArrayGeneric( &i, 2, 1);
}

void hkOArchive::write16u(hkUint16 u)
{
    writeArrayGeneric( &u, 2, 1);
}

void hkOArchive::write32(hkInt32 i)
{
    writeArrayGeneric( &i, 4, 1);
}

void hkOArchive::write32u(hkUint32 u)
{
    writeArrayGeneric( &u, 4, 1);
}

void hkOArchive::write64(hkInt64 i)
{
    writeArrayGeneric( &i, 8, 1);
}

void hkOArchive::write64u(hkUint64 u)
{
    writeArrayGeneric( &u, 8, 1);
}

void hkOArchive::writeFloat32(hkFloat32 f)
{
    writeArrayGeneric( &f, 4, 1);
}

void hkOArchive::writeDouble64(hkDouble64 d)
{
    writeArrayGeneric( &d, 8, 1);
}

void hkOArchive::writeString(_In_z_ const char* s )
{
    while ( *s != '\0' )
    {
        write8( *s );
        s++;
    }
    write8( '\0' );
}

//
// Arrays
//

void hkOArchive::writeArray8(_In_reads_(nelem) const hkInt8* array, int nelem)
{
    writeArrayGeneric(array, 1, nelem);
}

void hkOArchive::writeArray8u(_In_reads_(nelem) const hkUint8* array, int nelem)
{
    writeArrayGeneric(array, 1, nelem);
}

void hkOArchive::writeArray16(_In_reads_(nelem) const hkInt16* array, int nelem)
{
    writeArrayGeneric(array, 2, nelem);
}

void hkOArchive::writeArray16u(_In_reads_(nelem) const hkUint16* array, int nelem)
{
    writeArrayGeneric(array, 2, nelem);
}

void hkOArchive::writeArray32(_In_reads_(nelem) const hkInt32* array, int nelem)
{
    writeArrayGeneric(array, 4, nelem);
}

void hkOArchive::writeArray32u(_In_reads_(nelem) const hkUint32* array, int nelem)
{
    writeArrayGeneric(array, 4, nelem);
}

void hkOArchive::writeArray64(_In_reads_(nelem) const hkInt64* array, int nelem)
{
    writeArrayGeneric(array, 8, nelem);
}

void hkOArchive::writeArray64u(_In_reads_(nelem) const hkUint64* array, int nelem)
{
    writeArrayGeneric(array, 8, nelem);
}

void hkOArchive::writeArrayFloat32(_In_reads_(nelem) const hkFloat32* array, int nelem)
{
    writeArrayGeneric(array, 4, nelem);
}

void hkOArchive::writeArrayFloat32(_In_reads_(nelem) const hkDouble64* array, int nelem)
{
    for (int i =0; i < nelem; i++)
    {
        writeFloat32( hkFloat32(array[i]) );
    }
}


void hkOArchive::writeArrayDouble64(_In_reads_(nelem) const hkDouble64* array, int nelem)
{
    writeArrayGeneric(array, 8, nelem);
}

//
// Master
//

static HK_INLINE void localbyteswap(char& a, char& b)
{
    char t = a;
    a = b;
    b = t;
}

void hkOArchive::writeArrayGeneric(_In_reads_bytes_(elemSize * arraySize) const void* ptr, int elemSize, int arraySize)
{
    if( m_byteSwap == false)
    {
        m_writer->write(ptr, arraySize * elemSize);
    }
    else
    {
        const int BUFSIZE = 512;
        char buf[BUFSIZE];
        const char* src = static_cast<const char*>(ptr);
        int bytesLeft = elemSize * arraySize;

        int chunkBytes = BUFSIZE;
        int chunkElems = BUFSIZE / elemSize;
        HK_ASSERT_NO_MSG(0x14a08fb1,  BUFSIZE % elemSize == 0);

        int leftoverBytes = bytesLeft % BUFSIZE;
        int leftoverElems = leftoverBytes / elemSize;
        HK_ASSERT_NO_MSG(0x7405771d,  leftoverBytes % elemSize == 0);

        while( bytesLeft > 0 )
        {
            if( bytesLeft < BUFSIZE )
            {
                chunkBytes = leftoverBytes;
                chunkElems = leftoverElems;
            }
            hkString::memCpy( buf, src, chunkBytes );

            switch( elemSize )
            {
                case 1:
                {
                    break;
                }
                case 2:
                {
                    char* bufp = buf;
                    for(int i = 0; i < chunkElems; ++i)
                    {
                        localbyteswap(bufp[0], bufp[1]);
                        bufp += 2;
                    }
                    break;
                }
                case 4:
                {
                    char* bufp = buf;
                    for(int i = 0; i < chunkElems; ++i)
                    {
                        localbyteswap(bufp[0], bufp[3]);
                        localbyteswap(bufp[1], bufp[2]);
                        bufp += 4;
                    }
                    break;
                }
                case 8:
                {
                    char* bufp = buf;
                    for(int i = 0; i < chunkElems; ++i)
                    {
                        localbyteswap(bufp[0], bufp[7]);
                        localbyteswap(bufp[1], bufp[6]);
                        localbyteswap(bufp[2], bufp[5]);
                        localbyteswap(bufp[3], bufp[4]);
                        bufp += 8;
                    }
                    break;
                }
                default:
                {

                    HK_ASSERTV(0x2cbaf98f, 0, "elemsize {} not handled.\n"
                        "elemsize must be a power of two and no greater than 8 (64 bits)", elemSize);
                }
            }

            m_writer->write(buf, chunkBytes);
            bytesLeft -= chunkBytes;
            src += chunkBytes;
        }
    }
}

//
// Misc
//

int hkOArchive::writeRaw(_In_reads_bytes_(nbytes) const void* buf, int nbytes)
{
    return m_writer->write(buf, nbytes);
}

void hkOArchive::setByteSwap(hkBool on)
{
    m_byteSwap = on;
}

hkBool hkOArchive::getByteSwap() const
{
    return m_byteSwap;
}

hkBool hkOArchive::isOk() const
{
    return m_writer->isOk();
}

_Ret_maybenull_ hkStreamWriter* hkOArchive::getStreamWriter()
{
    return m_writer;
}

void hkOArchive::setStreamWriter(_In_ hkStreamWriter* newWriter)
{
    m_writer = newWriter;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
