// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/System/Io/Reader/hkStreamReader.h>

/// Endian-aware binary formatted data reader.
/// The data may optionally be byteswapped as it is read.
/// The default is to read in little endian format.
class HK_EXPORT_COMMON hkIArchive : public hkReferencedObject
{
    public:

        HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_STREAM);

            /// Constructs a new hkIArchive with the specified hkStreamReader.
            /// Note: hkIArchive is not safe to use with non-blocking streams.
        hkIArchive(_In_ hkStreamReader* sb, bool byteswap=HK_ENDIAN_BIG);

            /// Uses the hkFileSystem to get an hkStreamReader for filename.
        hkIArchive(_In_z_ const char* filename, bool byteswap=HK_ENDIAN_BIG);

            /// Uses the 'mem' inplace. 'mem' should be valid for the lifetime of this object.
        hkIArchive(_In_reads_bytes_(memSize) const void* mem, int memSize, bool byteswap=HK_ENDIAN_BIG);

            /// Destructor.
        virtual ~hkIArchive();

            //
            // Read single elements.
            //

            /// Reads 8 bits.
            /// Value is undefined in the case of error / eof. Use isOk() to validate.
        HK_INLINE hkChar read8();

            /// Reads 8 bits.
            /// Value is undefined in the case of error / eof. Use isOk() to validate.
        HK_INLINE hkUchar read8u();

            /// Reads 16 bits.
            /// Value is undefined in the case of error / eof. Use isOk() to validate.
        HK_INLINE hkInt16 read16();

            /// Reads 16 bits.
            /// Value is undefined in the case of error / eof. Use isOk() to validate.
        HK_INLINE hkUint16 read16u();

            /// Reads 32 bits.
            /// Value is undefined in the case of error / eof. Use isOk() to validate.
        HK_INLINE hkInt32 read32();

            /// Reads 32 bits.
            /// Value is undefined in the case of error / eof. Use isOk() to validate.
        HK_INLINE hkUint32 read32u();

            /// Reads 64 bits.
            /// Value is undefined in the case of error / eof. Use isOk() to validate.
        HK_INLINE hkInt64 read64();

            /// Reads 64 bits.
            /// Value is undefined in the case of error / eof. Use isOk() to validate.
        HK_INLINE hkUint64 read64u();

            /// Reads 32 bit floats.
            /// Value is undefined in the case of error / eof. Use isOk() to validate.
        HK_INLINE hkFloat32 readFloat32();

            /// Reads 64 bit doubles.
            /// Value is undefined in the case of error / eof. Use isOk() to validate.
        HK_INLINE hkDouble64 readDouble64();

            /// Reads a string into buf until bufSize is reached or a null-terminator is encountered.
            /// Returns number of bytes read (including null-terminator).
            /// If bufSize is reached, the results will not be null-terminated.
        int readString(_Out_writes_z_(bufSize) char* buf, int bufSize);

            /// Reads a string into the string buf and returns it's length.
        int readString(hkStringBuf& buf);

            //
            // Read array elements.
            //

            /// Reads array of 8 byte data.
        HK_INLINE void readArray8(_Out_writes_(nelem) hkInt8* buf, int nelem);

            /// Reads array of 8 byte data.
        HK_INLINE void readArray8u(_Out_writes_(nelem) hkUint8* buf, int nelem);

            /// Reads array of 16 byte data.
        HK_INLINE void readArray16(_Out_writes_(nelem) hkInt16* buf, int nelem);

            /// Reads array of 16 byte data.
        HK_INLINE void readArray16u(_Out_writes_(nelem) hkUint16* buf, int nelem);

            /// Reads array of 32 byte data.
        HK_INLINE void readArray32(_Out_writes_(nelem) hkInt32* buf, int nelem);

            /// Reads array of 32 byte data.
        HK_INLINE void readArray32u(_Out_writes_(nelem) hkUint32* buf, int nelem);

            /// Reads array of 64 byte data.
        HK_INLINE void readArray64(_Out_writes_(nelem) hkInt64* buf, int nelem);

            /// Reads array of 64 byte data.
        HK_INLINE void readArray64u(_Out_writes_(nelem) hkUint64* buf, int nelem);

            /// Reads array of 32 byte float data.
        HK_INLINE void readArrayFloat32(_Out_writes_(nelem) hkFloat32* buf, int nelem);

            /// Reads array of 32 byte float data and converts it into double
        void readArrayFloat32(_Out_writes_(nelem) hkDouble64* buf, int nelem);

            /// Reads array of 64 byte float data.
        HK_INLINE void readArrayDouble32u(_Out_writes_(nelem) hkDouble64* buf, int nelem);

            /// Reads array of sizeelem byte data.
        void readArrayGeneric(_Out_writes_bytes_(nelem * sizeelem) void* buf, int sizeelem, int nelem);

            //
            // Other.
            //

            /// Reads raw data.
        int readRaw(_Out_writes_bytes_(nbytes) void* buf, int nbytes);

            /// Set byteswapping.
        HK_INLINE void setByteSwap(hkBool on);

            /// Set byteswapping.
        HK_INLINE hkBool getByteSwap() const;

            /// Returns the current error status of the stream.
            /// Notice that end-of-file will only be detected after an unsuccessful read
        hkBool isOk() const;

            /// Returns the underlying streambuf used by this hkIstream
        _Ret_maybenull_ hkStreamReader* getStreamReader();

            /// Sets the underlying hkStreamReader for this hkIstream
        void setStreamReader(_In_ hkStreamReader* newReader);

    protected:

            /// The underlying stream reader.
        hkRefPtr<hkStreamReader> m_streamReader;

            /// Should we byteswap.
        hkBool m_byteSwap;
};

typedef hkIArchive hkIfArchive;

#include <Common/Base/System/Io/IArchive/hkIArchive.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
