// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT    : COMMON
// VISIBILITY : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

// this: #include <Common/Base/System/Io/FileSystem/hkFileSystem.h>

#include <Common/Base/Container/String/hkStringBuf.h>

class hkStreamReader;
class hkStreamWriter;

    /// Interface for hkStreamReader/hkStreamWriter creation and filesystem browsing.
    /// When streams and archives are given a resource to open by name, they
    /// ask the FileSystem to open it for them. The user may wish to
    /// replace the default factory with one which reads from packed files
    /// or searches in several locations for instance.
    /// \note Most methods which take a filename string as a parameter pass it
    /// to the filesystem singleton to actually open a stream. To handle I/O errors
    /// gracefully you will generally have to implement this interface or bypass it
    /// completely by using methods which accept streams or in memory buffers directly.
class HK_EXPORT_COMMON hkFileSystem : public hkReferencedObject
{
    public:

        HK_DECLARE_CLASS(hkFileSystem, New);
        HK_DECLARE_CLASS_Singleton(hkFileSystem);

        enum AccessMode
        {
            ACCESS_READ = 1,
            ACCESS_WRITE = 2,
        };

        enum OpenFlags
        {
                /// The file is buffered. This makes peek() available and takes care of any platform-specific
                /// alignment and block size requirements.
            OPEN_BUFFERED = 1,
                /// Truncate the file if it exists. Without this flag, the content (if any) is not
                /// truncated. In either case, the initial file offset will always be zero, so a seek is
                /// required to append data.
            OPEN_TRUNCATE = 2,
                /// Default read mode
            OPEN_DEFAULT_READ = OPEN_BUFFERED,
                /// Default write mode
            OPEN_DEFAULT_WRITE = OPEN_BUFFERED | OPEN_TRUNCATE,
                /// Exclusive file access mode. Prevents other processes from opening a file
                /// if they request delete, read, or write access.
            OPEN_EXCLUSIVE_ACCESS = 4,
                /// Exclusive write mode.
            OPEN_EXCLUSIVE_WRITE = OPEN_EXCLUSIVE_ACCESS | OPEN_BUFFERED | OPEN_TRUNCATE
        };

        enum CreateFlag
        {
            ALLOW_EXISTING = 1,
            ERROR_IF_EXISTS = 2
        };

            /// Platform independent timestamp that can be used to reference points in time.
            /// Stores timestamp in nanoseconds that have passed since
            /// 1970-01-01T00:00:00.000 UTC (Unix time in nanoseconds).
            /// Can represent valid ranges in (1970-01-01T00:00:00.000 UTC, 2554-07-21T23:34:33 UTC].
        struct HK_EXPORT_COMMON TimeStamp
        {
            public:

                enum { TIMESTAMP_UNAVAILABLE=0 };

                TimeStamp() : m_time(TIMESTAMP_UNAVAILABLE) {}
                TimeStamp(hkUint64 nsSinceEpoch) : m_time(nsSinceEpoch) {}

                HK_INLINE hkBool32 operator==(const TimeStamp& rhs) const { return m_time == rhs.m_time; }
                HK_INLINE hkBool32 operator!=( const TimeStamp& rhs ) const { return m_time != rhs.m_time; }
                HK_INLINE hkBool32 operator<(const TimeStamp& rhs) const { return m_time < rhs.m_time; }
                HK_INLINE hkBool32 operator>(const TimeStamp& rhs) const { return m_time > rhs.m_time; }
                HK_INLINE hkBool32 operator>=(const TimeStamp& rhs) const { return m_time >= rhs.m_time; }
                HK_INLINE hkBool32 operator<=( const TimeStamp& rhs ) const { return m_time <= rhs.m_time; }

                    /// Sets the timestamp to the given Unix time in nanoseconds.
                void set(hkUint64 nsSinceEpoch);

                    /// Returns the nanoseconds passed since Unix epoch.
                hkUint64 get() const;

                    /// Returns whether the stored time is valid.
                    /// Invalid values are returned by file systems that do not support timestamps.
                hkBool isValid() const;

            private:

                hkUint64 m_time;
        };

            /// Represents a file or directory in a filesystem.
        struct HK_EXPORT_COMMON Entry
        {
            public:

                HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE, hkFileSystem::Entry);

                Entry() : m_fs(0), m_mtime(), m_size(-1), m_flags(0) {}

                enum FlagValues
                {
                    F_ISFILE = 1, ///< Regular file
                    F_ISDIR = 2, ///< Folder type
                    F_ISUNKNOWN = 4, ///< Unrecognised type (e.g. link, device)
                };

                    ///
                typedef hkFlags<FlagValues, hkUint32> Flags;

                hkBool32 isDir() const { return m_flags.get(F_ISDIR); }
                hkBool32 isFile() const { return m_flags.get(F_ISFILE); }
                hkFileSystem* getFileSystem() const { return m_fs; }
                _Ret_z_ const char* getPath() const { return m_path; }
                _Ret_z_ const char* getName() const;
                TimeStamp getMtime() const { return m_mtime; }
                hkInt64 getSize() const { return m_size; }
                Flags getFlags() const { return m_flags; }

                void setPath(_Inout_ hkFileSystem* fs, _In_z_ const char* path);
                void setMtime(TimeStamp mt) { m_mtime = mt; }
                void setSize(hkInt64 s) { m_size = s; }
                void setFlags(Flags f) { m_flags = f; }
                void setAll(_In_ hkFileSystem* fs, _In_z_ const char* fullPath, Flags flags, TimeStamp mt, hkInt64 sz);

                hkRefNew<hkStreamReader> openReader(OpenFlags flags=OPEN_DEFAULT_READ) const;
                hkRefNew<hkStreamWriter> openWriter(OpenFlags flags=OPEN_DEFAULT_WRITE) const;

            private:

                hkFileSystem* m_fs; ///< The filesystem this entry belongs to.
                hkStringPtr m_path; ///< Path to the entry including the name.
                TimeStamp m_mtime;  ///< Last modified time.
                hkInt64 m_size;     ///< Number of bytes in this entry or -1 if unknown.
                Flags m_flags;      ///< Flags (isdir, isfile etc)
        };


            /// Returns a stream reader for file 'name' or null if unable.
            /// Invokes _openReader, followed by any registered open reader callbacks. Should not be overridden.
        hkRefNew<hkStreamReader> openReader( _In_z_ const char* name, OpenFlags flags=OPEN_DEFAULT_READ );

            /// Returns a stream writer for file 'name' or null if unable.
            /// Invokes _openWriter, followed by any registered open writer callbacks. Should not be overridden.
        hkRefNew<hkStreamWriter> openWriter( _In_z_ const char* name, OpenFlags flags=OPEN_DEFAULT_WRITE );

            /// Remove the given path.
            /// Can fail with error code HK_E_FILE_NOT_FOUND if the given path doesn't exist.
        virtual hkResult remove(_In_z_ const char* path) { return HK_E_NOT_IMPLEMENTED; }

            /// Move the given file to another file (rename) or directory.
            /// The source must be an existing file.
            /// The destination must be either a non-existing file, or an existing directory.
            /// In that later case, the source file name is appended to the existing directory,
            /// and the resulting path must be that of a non-existing file.
            /// Ex: move( "/path/to/a", "/newpath/to/newa" ) : /path/to/a -> /newpath/to/newa
            ///     move( "/path/to/a", "/newpath/to/" )     : /path/to/a -> /newpath/to/a
            /// The implementation should handle cross-filesystem move, possibly using copy + delete.
        virtual hkResult moveFile(_In_z_ const char* sourceFile, _In_z_ const char* destPath) { return HK_E_NOT_IMPLEMENTED; }

            /// Create a folder with the given name.
            /// Can fail with error code HK_E_ALREADY_EXISTS if the given path already exists.
        virtual hkResult mkdir(_In_z_ const char* path, CreateFlag flag) { return HK_E_NOT_IMPLEMENTED; }

            /// Creates the directory at the given path, as well any
            /// intermediate directory if they don't exist yet. The default
            /// implementation uses \ref stat to determine if a directory
            /// already exists and \ref mkdir to create the directories.
            /// Can fail with error code HK_E_ALREADY_EXISTS if the given path already exists.
        virtual hkResult mkdirRecursive(_In_z_ const char* path, CreateFlag flag);

            /// Copies file from source path to destination path
            /// (always overwrites existing file)
        virtual hkResult copy(_In_z_ const char* srcPath, _In_z_ const char* dstPath);

            /// Look up a single named path, returns true if an entry exists.
        virtual hkResult stat(_In_z_ const char* path, Entry& entryOut ) = 0;


        /// Open reader/writer callbacks, per the signatures below, can be registered (and unregistered) with a file system instance.
        /// Registered callbacks are invoked by the openReader/openWriter functions, in ascending order of priority, after a stream reader/writer has been opened.
        /// They may return a new/modified stream reader/writer.

            /// Open reader callback
        typedef hkResult(*OpenReaderCb)(_Inout_ hkRefPtr<hkStreamReader>& orig, _In_z_ const char* path, OpenFlags flags, _In_opt_ void* cbData);

            /// Open writer callback
        typedef hkResult(*OpenWriterCb)(_Inout_ hkRefPtr<hkStreamWriter>& orig, _In_z_ const char* path, OpenFlags flags, _In_opt_ void* cbData);

            /// Register an open reader callback. A callback function can be registered more than once with different user data.
            /// Returns true if a new callback was registered, false if an existing callback was overwritten.
        bool registerOpenReaderCb(_In_ OpenReaderCb callback, _In_opt_ void* cbData=nullptr, int priority=0);

            /// Register an open writer callback. A callback function can be registered more than once with different user data.
            /// Returns true if a new callback was registered, false if an existing callback was overwritten.
        bool registerOpenWriterCb(_In_ OpenWriterCb callback, _In_opt_ void* cbData=nullptr, int priority=0);

            /// If callback with associated user data is registered, unregister it and return HK_SUCCESS, else return HK_FAILURE.
        hkResult unregisterOpenReaderCb(_In_ OpenReaderCb callback, _In_opt_ void* cbData=nullptr);

            /// If callback with associated user data is registered, unregister it and return HK_SUCCESS, else return HK_FAILURE.
        hkResult unregisterOpenWriterCb(_In_ OpenWriterCb callback, _In_opt_ void* cbData=nullptr);


        enum OSPathFlags
        {
            OS_PATH_NONE = 0,
                /// Get an absolute path
            OS_PATH_ABSOLUTE = 1,
        };

        virtual _Ret_z_ const char* getOperatingSystemPath(_In_z_ const char* pathIn, hkStringBuf& pathOut, OSPathFlags flags=OS_PATH_NONE ) = 0;

            /// Interface to iterate over the entries in a filesystem tree.
        class HK_EXPORT_COMMON Iterator
        {
            public:

                HK_DECLARE_PLACEMENT_ALLOCATOR();

                    /// Construct an iterator at the given folder name.
                Iterator(_In_ hkFileSystem* fs, _In_z_ const char* folder, _In_opt_z_ const char* wildcard = HK_NULL);

                    /// Attempt to advance to the next entry.
                    /// Return true if there is an entry or return false if there are no more entries.
                bool advance();

                    /// Access the current entry. This is only valid after a call to advance() returns true.
                const Entry& current() const { return m_entry; }

                    /// Recurse into the given path when the current iterator is exhausted.
                void recurseInto(const char* path)
                {
                    m_todo.pushBack(path);
                }

                    /// Return true if the name matches the wildcard and is not "." nor "..".
                    /// A null wildcard is considered to match everything.
                static bool nameAcceptable(_In_z_ const char* name, _In_opt_z_ const char* wildcard);

                    // Internal interface for filesystem
                struct Impl : public hkReferencedObject
                {
                    HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_STREAM);
                    virtual bool advance(Entry& e) = 0;
                };

            private:

                hkRefPtr<hkFileSystem> m_fs;
                const char* m_wildcard;
                hkRefPtr<Impl> m_impl;
                Entry m_entry;
                hkArray<hkStringPtr> m_todo;
        };

        
        class HK_EXPORT_COMMON Watcher : public hkReferencedObject
        {
        public:
            struct Change
            {
                enum ChangeType
                {
                    Added,
                    Removed,
                    Modified,
                    Renamed,
                    Unknown,

                    // Used internally, these are combined into Rename
                    _Rename_Old_Name,
                    _Rename_New_Name,
                };

                hkEnum<ChangeType, hkUint8> m_change;
                hkStringPtr m_oldFileName; // Only for rename
                hkStringPtr m_fileName;

                hkUlong hash() const;
            };

            // Check for changes. Cost depends on implementation but should be low if no changes have occurred
            virtual hkResult getCompletedChanges(hkArray<Change>& changesOut) = 0;
            virtual ~Watcher() {}

            // Default implementation is that it can't stop watching.
            virtual hkResult stopWatching() { return HK_FAILURE; }
        protected:
            static void combineMatchingChanges(hkArray<Change>& changesOut);
        };

            // Creating a watcher may be an expensive operation, depending on platform it may need to start threads
        virtual hkRefNew<Watcher> createWatcher(_In_z_ const char* topDirectory);

            // Internal function used by Iterator. Use the Iterator class instead.
        virtual hkRefNew<Iterator::Impl> createIterator(_In_z_ const char* top, _In_z_ const char* wildcard) = 0;

    public:

        struct HK_EXPORT_COMMON DirectoryListing
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE, hkFileSystem::DirectoryListing);

            DirectoryListing()
                : m_fs(HK_NULL), m_top(HK_NULL)
            {
            }

            DirectoryListing(_In_opt_ const void* alloc, _In_opt_z_ const char* top = HK_NULL)
                : m_fs(HK_NULL), m_top(top)
            {
                m_entries.reserve(32);
            }

            void clear()
            {
                m_entries.clear();
            }

            void addEntry(const Entry& ent)
            {
                HK_ASSERT_NO_MSG(0x76231193, m_fs != HK_NULL);
                m_entries.pushBack(ent);
            }

            void addDirectory(_In_z_ const char* name)
            {
                HK_ASSERT_NO_MSG(0x5d400f6f, m_fs != HK_NULL);
                m_entries.expandOne().setAll( m_fs, hkStringBuf(m_top.cString()).pathAppend(name), Entry::F_ISDIR, TimeStamp(), 0);
            }

                /// Adds a copy of name to the file names list.
                /// Also stores the last modified time stamp (default 0).
            void addFile(_In_z_ const char* name, TimeStamp timeModified, hkInt64 size=-1)
            {
                HK_ASSERT_NO_MSG(0x7a9590f9, m_fs != HK_NULL);
                m_entries.expandOne().setAll(m_fs, hkStringBuf(m_top.cString()).pathAppend(name), Entry::F_ISFILE, timeModified, size);
            }

            const hkArrayBase<Entry>& getEntries() const
            {
                return m_entries;
            }

            void setFs(_In_ hkFileSystem* fs) { m_fs = fs; }

            hkBool isEmpty() const { return m_entries.isEmpty(); }

        private:

            hkArray<Entry> m_entries;
            hkFileSystem* m_fs;
            hkStringPtr m_top;
        }; // DirectoryListing

            /// Deprecated: use Iterator instead.
            /// List all the directories and files in the "path" directory, returns HK_FAILURE if the path is not valid.
        hkResult listDirectory(_In_z_ const char* basePath, DirectoryListing& listingOut);

    protected:

        /// Returns a stream reader for file 'name' or null if unable, must be defined by each file system type.
        /// Invoked by the openReader function declared earlier.
    virtual hkRefNew<hkStreamReader> _openReader(_In_z_ const char* name, OpenFlags flags = OPEN_DEFAULT_READ)=0;

        /// Returns a stream reader for file 'name' or null if unable, must be defined by each file system type.
        /// Invoked by the openWriter function declared earlier.
    virtual hkRefNew<hkStreamWriter> _openWriter(_In_z_ const char* name, OpenFlags flags = OPEN_DEFAULT_WRITE)=0;

        _Ret_maybenull_ hkStreamReader* _handleFlags(_Inout_opt_ hkStreamReader* sr, OpenFlags flags);
        _Ret_maybenull_ hkStreamWriter* _handleFlags(_Inout_opt_ hkStreamWriter* sw, OpenFlags flags);

    template<typename T>
    class PriorityComparison;

    template<typename T>
    struct CbInfo
    {
        CbInfo(T cb, int priority, _In_opt_ void* cbData) : m_cb( cb ), m_priority( priority ), m_data( cbData ) {}
        T m_cb;
        int m_priority;
        void* m_data;
    };

    hkArray<CbInfo<OpenReaderCb>> m_postOpenReaderCbs;
    hkArray<CbInfo<OpenWriterCb>> m_postOpenWriterCbs;
};

HK_REFLECT_ENUM(HK_EXPORT_COMMON, hkFileSystem::Watcher::Change::ChangeType);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
