// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/System/Io/Reader/Buffered/hkBufferedStreamReader.h>
#include <Common/Base/System/Io/Writer/Buffered/hkBufferedStreamWriter.h>
#include <Common/Base/Container/StringMap/hkStringMap.h>
#include <Common/Base/Container/Set/hkSet.h>
#include <Common/Base/Object/hkSingleton.h>
#include <Common/Base/System/Io/Util/hkCopyUtil.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>

void hkFileSystem::TimeStamp::set(hkUint64 nsSinceEpoch)
{
    m_time = nsSinceEpoch;
}

hkUint64 hkFileSystem::TimeStamp::get() const
{
    return m_time;
}

hkBool hkFileSystem::TimeStamp::isValid() const
{
  return m_time != hkFileSystem::TimeStamp::TIMESTAMP_UNAVAILABLE;
}



hkRefNew<hkStreamReader> hkFileSystem::Entry::openReader(OpenFlags flags) const
{
    return m_fs->openReader( m_path, flags );
}

hkRefNew<hkStreamWriter> hkFileSystem::Entry::openWriter(OpenFlags flags) const
{
    return m_fs->openWriter( m_path, flags );
}

hkRefNew<hkStreamReader> hkFileSystem::openReader(_In_z_ const char* name, OpenFlags flags)
{
    if (hkRefPtr<hkStreamReader> sr = _openReader( name, flags ))
    {
        for (int i = 0; i < m_postOpenReaderCbs.getSize(); i++)
        {
            if ((m_postOpenReaderCbs[i].m_cb(sr, name, flags, m_postOpenReaderCbs[i].m_data )).isFailure())
            {
                return nullptr;
            }
        }
        sr->addReference();
        return sr.val();
    }
    return nullptr;
}

hkRefNew<hkStreamWriter> hkFileSystem::openWriter(_In_z_ const char* name, OpenFlags flags)
{
    if (hkRefPtr<hkStreamWriter> sw = _openWriter( name, flags ))
    {
        for (int i = 0; i < m_postOpenWriterCbs.getSize(); i++)
        {
            if ((m_postOpenWriterCbs[i].m_cb(sw, name, flags, m_postOpenWriterCbs[i].m_data )).isFailure())
            {
                return nullptr;
            }
        }
        sw->addReference();
        return sw.val();
    }
    return nullptr;
}

hkResult hkFileSystem::mkdirRecursive(_In_z_ const char* path, CreateFlag flag)
{
    // Normalize the path, so that it will be strictly the individuals
    // directories, separated by a single /.
    hkStringBuf pathNormalized = path;
    pathNormalized.pathNormalize();

    // Start with the full path, and keep removing the last directory, to get
    // paths to the intermediate directories. If an intermediate directory
    // doesn't exist yet, we add it to the todo list. As soon as we find our
    // first directory which does exist, we know that all the higher ones must
    // exist as well, so we're done.
    hkStringBuf sb(pathNormalized);
    sb.pathDirname();

    hkArray<hkStringPtr>::Temp todo;
    hkFileSystem::Entry e;
    while(sb.getLength() && stat(sb, e).isFailure())
    {
        todo.expandOne() = sb.cString();
        sb.pathDirname();
    }

    // Create the individual directories in the opposite order they were added
    // to the todo list.
    for(int i = todo.getSize() - 1; i >= 0; --i)
    {
        HK_RETURN_IF_FAILED(mkdir(todo[i], flag));
    }

    return mkdir(pathNormalized, flag);
}

    /// Default (slow) implementation of copy function
hkResult hkFileSystem::copy(_In_z_ const char* srcPath, _In_z_ const char* dstPath)
{
    Entry srcEntry;
    HK_RETURN_IF_FAILED(stat(srcPath, srcEntry));

    hkStringBuf dstWithoutFilename = dstPath;
    int lastSlash = dstWithoutFilename.lastIndexOf("/");
    if (lastSlash >= 0)
    {
        dstWithoutFilename.chompEnd(dstWithoutFilename.getLength() - lastSlash);
    }

    Entry dstEntry;
    HK_RETURN_IF_FAILED(stat(dstWithoutFilename, dstEntry));

    hkStringBuf dstResolved = dstEntry.getPath();

    if (lastSlash >= 0)
    {
        dstResolved.append(dstPath + lastSlash);
    }

    /// write(read()) default implementation
    {
        hkRefPtr<hkStreamReader> reader = openReader(srcEntry.getPath());
        if (!reader || !reader->isOk())
        {
            return HK_FAILURE;
        }

        hkRefPtr<hkStreamWriter> writer = openWriter(dstResolved);
        if (!writer || !writer->isOk())
        {
            return HK_FAILURE;
        }

        if (!hkIo::copyStream(reader, writer))
        {
            return HK_FAILURE;
        }
    }

    return HK_SUCCESS;
}

/// For sorting m_postOpenReaderCbs and m_postOpenWriterCbs arrays
template<typename T>
class hkFileSystem::PriorityComparison
{
public:
    HK_INLINE bool operator() (const hkFileSystem::CbInfo<T>& a, const hkFileSystem::CbInfo<T>& b)
    {
        return a.m_priority < b.m_priority;
    }
};

bool hkFileSystem::registerOpenReaderCb(_In_ OpenReaderCb callback, _In_opt_ void* cbData, int priority)
{
    for (auto cbInfo: m_postOpenReaderCbs)
    {
        // Cannot register same callback with same user args twice, but can overwrite priority
        if (cbInfo.m_cb == callback && cbInfo.m_data == cbData)
        {
            cbInfo.m_priority = priority;
            return false;
        }
    }
    CbInfo<OpenReaderCb> cb(callback, priority, cbData);
    m_postOpenReaderCbs.pushBack(cb);
    // Keep m_postOpenReaderCbs sorted on priority
    PriorityComparison<OpenReaderCb> cmp;
    hkAlgorithm::insertionSort<CbInfo<OpenReaderCb>, PriorityComparison<OpenReaderCb>>(m_postOpenReaderCbs.begin(), m_postOpenReaderCbs.getSize(), cmp);
    return true;
}

bool hkFileSystem::registerOpenWriterCb(_In_ OpenWriterCb callback, _In_opt_ void* cbData, int priority)
{
    for (auto cbInfo: m_postOpenWriterCbs)
    {
        // Cannot register same callback with same user args twice, but can overwrite priority
        if (cbInfo.m_cb == callback && cbInfo.m_data == cbData)
        {
            cbInfo.m_priority = priority;
            return false;
        }
    }
    CbInfo<OpenWriterCb> cb(callback, priority, cbData);
    m_postOpenWriterCbs.pushBack(cb);
    // Keep m_postOpenWriterCbs sorted on priority
    PriorityComparison<OpenWriterCb> cmp;
    hkAlgorithm::insertionSort<CbInfo<OpenWriterCb>, PriorityComparison<OpenWriterCb>>(m_postOpenWriterCbs.begin(), m_postOpenWriterCbs.getSize(), cmp);
    return true;
}

hkResult hkFileSystem::unregisterOpenReaderCb(_In_ OpenReaderCb callback, _In_opt_ void* cbData)
{
    for (int i = 0; i < m_postOpenReaderCbs.getSize(); i++)
    {
        if (m_postOpenReaderCbs[i].m_cb == callback && m_postOpenReaderCbs[i].m_data == cbData)
        {
            m_postOpenReaderCbs.removeAtAndCopy(i);
            return HK_SUCCESS;
        }
    }
    return HK_FAILURE;
}

hkResult hkFileSystem::unregisterOpenWriterCb(_In_ OpenWriterCb callback, _In_opt_ void* cbData)
{
    for (int i = 0; i < m_postOpenWriterCbs.getSize(); i++)
    {
        if (m_postOpenWriterCbs[i].m_cb == callback && m_postOpenWriterCbs[i].m_data == cbData)
        {
            m_postOpenWriterCbs.removeAtAndCopy(i);
            return HK_SUCCESS;
        }
    }
    return HK_FAILURE;
}

void hkFileSystem::Entry::setPath(_Inout_ hkFileSystem* fs, _In_z_ const char* path)
{
    hkStringBuf normalizedPath(path);
    normalizedPath.pathNormalize();

    HK_ASSERT_NO_MSG(0x8692bcc, hkString::endsWith(normalizedPath, "/")==false);
    m_fs = fs;
    m_path = normalizedPath;
}

_Ret_z_ const char* hkFileSystem::Entry::getName() const
{
    HK_ASSERT_NO_MSG(0x73b2865a, m_path.endsWith("/") == hkFalse32);
    if( const char* name = hkString::strRchr(m_path, '/') )
    {
        do
        {
            ++name;
        } while( *name=='/' );
        // maybe it was like "foo/bar/"
        if( *name )
        {
            return name;
        }
    }
    // "/" not found, just return the head
    return m_path;
}

void hkFileSystem::Entry::setAll(_In_ hkFileSystem* fs, _In_z_ const char* fullPath, Flags flags, TimeStamp mt, hkInt64 sz)
{
    setPath(fs, fullPath);
    m_flags = flags;
    m_mtime = mt;
    m_size = sz;
}

hkResult hkFileSystem::listDirectory(_In_z_ const char* basePath, DirectoryListing& listingOut)
{
    hkFileSystem::Iterator iter(this, basePath);
    listingOut.setFs(this);
    while( iter.advance() )
    {
        listingOut.addEntry( iter.current() );
    }
    return HK_SUCCESS;
}

hkFileSystem::Iterator::Iterator(_In_ hkFileSystem* fs, _In_z_ const char* top, _In_opt_z_ const char* wildcard)
    : m_fs(fs), m_wildcard(wildcard)
{
    HK_ASSERT_NO_MSG(0x2b9fb1b7, wildcard==HK_NULL || hkString::strChr(wildcard, '/')==HK_NULL );
    m_todo.pushBack( top );
}

bool hkFileSystem::Iterator::advance()
{
    while( 1 )
    {
        if( m_impl != HK_NULL )
        {
            if( m_impl->advance(m_entry) )
            {
                return true;
            }
            m_impl = HK_NULL;
        }
        if( m_todo.getSize() )
        {
            // try again with the next item
            m_impl = m_fs->createIterator( m_todo.back(), m_wildcard );
            m_todo.popBack();
        }
        else
        {
            // end of this iteration and no more todos, we're done
            return false;
        }
    }
}

bool hkFileSystem::Iterator::nameAcceptable(_In_z_ const char* name, _In_opt_z_ const char* wildcard)
{
    // nonempty, not . nor ..
    if( name[0]==0 ) return false;
    if( name[0]=='.' && name[1]==0 ) return false;
    if( name[0]=='.' && name[1]=='.' && name[2]==0 ) return false;

    if( wildcard == HK_NULL ) return true;

    HK_ASSERT_NO_MSG(0x61cee828, hkString::strChr(wildcard, '?') == HK_NULL );
    HK_ASSERT(0x374596bd, hkString::strChr(name, '*') == HK_NULL, "name cannot contain wildcards" );
    HK_ASSERT(0x268fada4, wildcard[0] == '*', "only simple patterns currently supported (*.xxx)" );
    HK_ASSERT(0x7f34a731, hkString::strChr(wildcard+1, '*')==HK_NULL, "only simple patterns currently supported (*.xxx)" );

    return hkString::endsWith(name, wildcard+1);
}

_Ret_maybenull_ hkStreamReader* hkFileSystem::_handleFlags(_Inout_opt_ hkStreamReader* sr, OpenFlags flags)
{
    HK_ASSERT_NO_MSG(0x5b74fb14, (flags & ~OPEN_BUFFERED) == 0 );
    if( sr )
    {
        if( flags & OPEN_BUFFERED )
        {
            hkBufferedStreamReader* buf = new hkBufferedStreamReader(sr);
            sr->removeReference();
            sr = buf;
        }
    }
    return sr;
}

_Ret_maybenull_ hkStreamWriter* hkFileSystem::_handleFlags(_Inout_opt_ hkStreamWriter* sw, OpenFlags flags)
{
    if( sw )
    {
        if( flags & OPEN_BUFFERED )
        {
            hkBufferedStreamWriter* buf = new hkBufferedStreamWriter(sw);
            sw->removeReference();
            sw = buf;
        }
    }
    return sw;
}

hkRefNew<hkFileSystem::Watcher> hkFileSystem::createWatcher(_In_z_ const char* topDirectory)
{
    HK_ASSERT_NO_MSG(0x1f919885, 0); // Default is not supported
    return HK_NULL;
}

hkUlong hkFileSystem::Watcher::Change::hash() const
{
    hkUlong hashValue = hkStringMapOperations().hash(hkUlong(m_fileName.cString()));
    hashValue ^= hkUlong(m_change);
    return hashValue;
}

void hkFileSystem::Watcher::combineMatchingChanges(hkArray<hkFileSystem::Watcher::Change>& changesOut)
{
    hkSet<hkUlong> processedChanges;
    for(int i=0; i < changesOut.getSize(); i++)
    {
        const hkUlong changeHash = changesOut[i].hash();
        hkResult resOut;
        if(!processedChanges.tryInsert(changeHash, resOut))
        {
            changesOut.removeAtAndCopy(i);
            i--;
        }
    }
}

#include <Common/Base/System/Io/FileSystem/hkNativeFileSystem.h>
HK_SINGLETON_SUBCLASS_IMPLEMENTATION(hkFileSystem, hkNativeFileSystem);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
