// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>

    /// Virtual filesystem as a combination of sub-filesystems.
    /// The current implementation is rather simpleminded. The requested
    /// operation is tried on each mount which has a matching prefix.
    /// An empty prefix is considered to always match so will often be
    /// used as a final fallback.
    ///
    /// Each mount with matching prefix is tried in turn with the most
    /// recently mounted tried before older mounts.
    ///
    /// A typical setup for a mobile device might have several mounts:
    /// a readonly mount into the application bundle, then a read-write
    /// mount into an sdcard location. And finally a catch-all mount to
    /// handle absolute paths.
    /// 1. ("Resources", "/path/to/app/bundle/Resources", writable=False)
    /// 2. ("Resources", "/mnt/sdcard/Havok/Resources", writable=True)
    /// 3. ("", "", writable=False)
    ///
    /// A typical setup for a workstation will almost certainly have a
    /// pass-through mount in order to support absolute paths
    /// 1. ("", "", writable=True)
    ///
    /// There is no support for whiteouts so if a file exists in more than
    /// one mount, it will still be present after removing the first one.
    /// Similarly when iterating, a file with the same name but different
    /// contents may be returned.
class HK_EXPORT_COMMON hkUnionFileSystem : public hkFileSystem
{
    public:

        HK_DECLARE_CLASS(hkUnionFileSystem, New);

        virtual hkResult remove(_In_z_ const char* path) HK_OVERRIDE;

        virtual hkResult moveFile(_In_z_ const char* sourceFile, _In_z_ const char* destPath) HK_OVERRIDE;

        virtual hkResult mkdir(_In_z_ const char* path, CreateFlag flag) HK_OVERRIDE;

        virtual hkResult copy(_In_z_ const char* srcPath, _In_z_ const char* dstPath) HK_OVERRIDE;

        virtual hkResult stat(_In_z_ const char* path, Entry& entryOut ) HK_OVERRIDE;

        virtual _Ret_maybenull_ hkFileSystem* resolvePath(_In_z_ const char* pathIn, hkStringBuf& pathOut);

        virtual _Ret_maybenull_z_ const char* getOperatingSystemPath(_In_z_ const char* pathIn, hkStringBuf& pathOut, OSPathFlags flags) HK_OVERRIDE;

        _Ret_maybenull_z_ const char* srcPathFromDstPath(_In_z_ const char* dstPath, hkStringBuf& pathOut);

            ///
        struct HK_EXPORT_COMMON Mount
        {
            HK_DECLARE_PLACEMENT_ALLOCATOR();

            hkRefPtr<hkFileSystem> m_fs;
            hkStringPtr m_srcPath;
            hkStringPtr m_dstPath;
            hkBool m_writable;
        };

            /// Mount a filesystem so that accesses to the srcPath hierarchy map to accesses
            /// in the underlying filesystem's dstPath hierarchy.
        void mount(_Inout_ hkFileSystem* fs, _In_z_ const char* srcPath, _In_z_ const char* dstPath, hkBool writable);

            /// Unmount if exists
            // if dstPath is NULL then compare only source path
        void unmount(_In_z_ const char* srcPath, _In_opt_z_ const char* dstPath = HK_NULL);

        virtual hkRefNew<Iterator::Impl> createIterator(_In_z_ const char* top, _In_z_ const char* wildcard ) HK_OVERRIDE;

        virtual hkRefNew<Watcher> createWatcher(_In_z_ const char* topDirectory) HK_OVERRIDE;

        const hkArray<Mount>& getMounts() const;

    protected:

        virtual hkRefNew<hkStreamReader> _openReader(_In_z_ const char* name, OpenFlags flags) HK_OVERRIDE;
        virtual hkRefNew<hkStreamWriter> _openWriter(_In_z_ const char* name, OpenFlags flags) HK_OVERRIDE;

        hkArray<Mount> m_mounts;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
