// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/System/Io/FileSystem/Union/hkUnionFileSystem.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/System/Io/Util/hkCopyUtil.h>

#define DEBUG_LOG_IDENTIFIER "Filesystem.Union"
#include <Common/Base/System/Log/hkLog.hxx>

namespace
{
    struct MountIterator
    {
        typedef hkUnionFileSystem::Mount Mount;

        MountIterator(hkArray<Mount>& mounts, _In_z_ const char* pathIn)
            : m_mounts(mounts), m_idx(m_mounts.getSize()), m_pathIn(pathIn)
        {
            if(m_pathIn)
            {
                
                
                while( hkString::beginsWith(m_pathIn, "./") )
                {
                    m_pathIn += 2;
                }
            }
        }

        bool advance()
        {
            if( m_idx > 0 )
            {
                m_idx -= 1;
                return true;
            }
            return false;
        }

        Mount& current()
        {
            return m_mounts[m_idx];
        }

        _Ret_z_ const char* path()
        {
            const Mount& m = m_mounts[m_idx];

            // Src path is empty.
            if (m.m_srcPath.getLength() == 0)
            {
                // This was mounted at the root
                m_pathOut = m.m_dstPath;
                m_pathOut.pathAppend(m_pathIn);
            }

            // Src path is equal to in-path.
            else if (m.m_srcPath.compareTo(m_pathIn) == 0)
            {
                m_pathOut = m.m_dstPath;
            }

            // Src path is a path prefix of in-path.
            else
            {
                // We need to make sure that we do not replace parts of a filename by ensuring a trailing slash.
                hkStringBuf prefixPath = m.m_srcPath.cString();
                if (!prefixPath.endsWith("/"))
                {
                    prefixPath.append("/");
                }

                if (hkString::beginsWith(m_pathIn, prefixPath.cString()))
                {
                    m_pathOut = m.m_dstPath;
                    m_pathOut.pathAppend(m_pathIn + prefixPath.getLength());
                }
                else
                {
                    // We were not able to resolve the path.
                    m_pathOut = "";
                }
            }

            return m_pathOut;
        }

        hkArray<Mount>& m_mounts;
        int m_idx;
        const char* m_pathIn;
        hkStringBuf m_pathOut;
    };
}

_Ret_maybenull_ hkFileSystem* hkUnionFileSystem::resolvePath(_In_z_ const char* pathIn, hkStringBuf& pathOut)
{
    MountIterator it(m_mounts, pathIn);
    while( it.advance() )
    {
        const Mount& m = it.current();
        Entry entry;
        if( m.m_fs->stat(it.path(), entry).isSuccess())
        {
            pathOut = it.path();
            return m.m_fs;
        }
    }
    return HK_NULL;
}

const hkArray<hkUnionFileSystem::Mount>& hkUnionFileSystem::getMounts() const
{
    return m_mounts;
}

_Ret_maybenull_z_ const char* hkUnionFileSystem::getOperatingSystemPath(_In_z_ const char* pathIn, hkStringBuf& pathOut, OSPathFlags flags)
{
    MountIterator it(m_mounts, pathIn);
    while( it.advance() )
    {
        const Mount& m = it.current();
        Entry entry;
        if( const char* res = m.m_fs->getOperatingSystemPath(it.path(), pathOut, flags) )
        {
            return res;
        }
    }
    return HK_NULL;
}


hkRefNew<hkStreamReader> hkUnionFileSystem::_openReader(_In_z_ const char* name, OpenFlags flags)
{
    DLOG_SCOPE("UnionFS.openReader {}", name);
    MountIterator it(m_mounts, name);
    while( it.advance() )
    {
        const Mount& m = it.current();
        if( hkStreamReader* r = m.m_fs->openReader( it.path(), flags ).stealOwnership() )
        {
            DLOG("Success ({})", it.path());
            return r;
        }
    }
    DLOG("Failed");
    return HK_NULL;
}

// Open a file writer, creating the file directory structure as we go along
static hkRefNew<hkStreamWriter> openWriterCreatingDirectories(hkArray<hkUnionFileSystem::Mount>& mounts, _In_z_ const char* name, hkFileSystem::OpenFlags flags)
{
    MountIterator it(mounts, name);

    while( it.advance() )
    {
        const hkUnionFileSystem::Mount& m = it.current();
        if( m.m_writable )
        {
            // Make each directory subpart
            hkStringBuf directoryBits(it.path());
            directoryBits.pathNormalize();
            directoryBits.pathDirname();
            hkArray<const char*>::Temp parts;
            int totalNumParts = directoryBits.split('/', parts);
            for(int numParts=1; numParts<=totalNumParts; numParts++)
            {
                hkStringBuf currentPathPart(m.m_dstPath);
                for(int i=0;i<numParts;i++)
                {
                    currentPathPart.pathAppend(parts[i]);
                }
                m.m_fs->mkdir(currentPathPart, hkFileSystem::ALLOW_EXISTING);
            }

            if( hkStreamWriter* w = m.m_fs->openWriter( it.path(), flags ).stealOwnership() )
            {
                DLOG("Success ({})", it.path());
                return w;
            }
        }
    }

    DLOG("Failed");
    return HK_NULL;
}

hkRefNew<hkStreamWriter> hkUnionFileSystem::_openWriter(_In_z_ const char* name, OpenFlags flags)
{
    DLOG_SCOPE("UnionFS.openWriter {}", name);
    MountIterator it(m_mounts, name);
    bool dirExistsInReadableFS = false;

    while( it.advance() )
    {
        const Mount& m = it.current();
        if( m.m_writable )
        {
            if( hkStreamWriter* w = m.m_fs->openWriter( it.path(), flags ).stealOwnership() )
            {
                DLOG("Success ({})", it.path());
                return w;
            }
        }
        else if(!dirExistsInReadableFS)
        {
            hkStringBuf pathDirName(it.path());
            pathDirName.pathDirname();

            hkFileSystem::Entry entry;
            dirExistsInReadableFS = m.m_fs->stat(pathDirName, entry).isSuccess() && entry.isDir();
        }
    }
    if(dirExistsInReadableFS)
    {
        // If the directory exists in a read-only file system, create a
        // copy of the directory structure in the writeable file system
        return openWriterCreatingDirectories(m_mounts, name, flags);
    }
    DLOG("Failed");
    return HK_NULL;
}

hkResult hkUnionFileSystem::remove(_In_z_ const char* path)
{
    DLOG_SCOPE("UnionFS.remove {}", path);
    MountIterator it(m_mounts, path);
    while( it.advance() )
    {
        const Mount& m = it.current();
        if( m.m_writable )
        {
            if(m.m_fs->remove( it.path() ).isSuccess())
            {
                DLOG("Success {}", it.path());
                return HK_SUCCESS;
            }
        }
    }
    DLOG("Failed");
    return HK_FAILURE;
}

hkResult hkUnionFileSystem::moveFile(_In_z_ const char* sourceFile, _In_z_ const char* destPath)
{
    DLOG_SCOPE("UnionFS.moveFile {} {}", sourceFile, destPath);

    if ( hkString::isNullOrEmpty( sourceFile ) || hkString::isNullOrEmpty( destPath ) )
    {
        DLOG( "Failed" );
        return HK_FAILURE;
    }

    // Resolve the source file, which must exists and be a file.
    Entry sourceEntry;
    {
        MountIterator it( m_mounts, sourceFile );
        while ( it.advance() )
        {
            const Mount& mount = it.current();

            // Loop over writable mount points only
            if ( !mount.m_writable )
            {
                continue;
            }

            if ( mount.m_fs->stat( it.path(), sourceEntry ).isSuccess() )
            {
                if ( !sourceEntry.isFile() )
                {
                    DLOG( "Failed" );
                    return HK_FAILURE;
                }
                break;
            }
        }
    }
    if ( !sourceEntry.isFile() )
    {
        DLOG( "Failed" );
        return HK_FAILURE;
    }

    // Get file name out of source path
    hkStringBuf fileName( sourceEntry.getPath() );
    fileName.pathBasename();

    // Get the source file name
    hkStringBuf destFile;
    hkFileSystem* destFileSystem = nullptr;
    {
        MountIterator it( m_mounts, destPath );
        while ( it.advance() )
        {
            const Mount& mount = it.current();

            // Loop over writable mount points only
            if ( !mount.m_writable )
            {
                continue;
            }

            Entry destDirEntry;
            if ( mount.m_fs->stat( it.path(), destDirEntry ).isSuccess() )
            {
                // If item exists, it must be a directory
                if ( destDirEntry.isDir() )
                {
                    // If path exists and is a directory, append the source filename.
                    // In that case the resulting file must *not* exist.
                    destFile.pathAppend( destDirEntry.getPath(), fileName );
                    hkStringBuf dummy;
                    if ( resolvePath( destFile, dummy ) )
                    {
                        destFile.clear(); // mark as not found
                    }
                }

                // If the item exists, whether or not it was valid, we're done
                destFileSystem = mount.m_fs;
                break;
            }
            else
            {
                // If item does not exists, it must be a file, and its directory must exist
                destFile = it.path();
                if ( destFile.isEmpty() )
                {
                    // Item does not match the mount point; ignore it
                    continue;
                }

                // Get the directory containing the file
                hkStringBuf destDir( destFile );
                destDir.pathDirname();

                // If the directory is empty, this means the destination path is simply a relative filename.
                // Otherwise the directory must exists, and the file will be contained in it.
                if ( destDir.isEmpty() || ( mount.m_fs->stat( destDir.cString(), destDirEntry ).isSuccess() ) )
                {
                    destFileSystem = mount.m_fs;
                    break;
                }

                // If the directory also does not exist, simply continue the search
                destFile.clear(); // mark as not found
            }
        }

        if ( destFile.isEmpty() )
        {
            DLOG( "Failed" );
            return HK_FAILURE;
        }
    }

    if ( destFileSystem == sourceEntry.getFileSystem() )
    {
        const hkResult copyResult = sourceEntry.getFileSystem()->moveFile( sourceEntry.getPath(), destFile );
        if ( copyResult.isFailure() )
        {
            DLOG( "Failed" );
            return copyResult;
        }
    }
    else
    {
        hkRefPtr<hkStreamReader> reader = sourceEntry.openReader();
        if ( !reader || !reader->isOk() )
        {
            DLOG( "Failed" );
            return HK_FAILURE;
        }

        hkRefPtr<hkStreamWriter> writer = destFileSystem->openWriter( destFile );
        if ( !writer || !writer->isOk() )
        {
            DLOG( "Failed" );
            return HK_FAILURE;
        }

        if ( !hkIo::copyStream( reader, writer ) )
        {
            DLOG( "Failed" );
            return HK_FAILURE;
        }

        if ( sourceEntry.getFileSystem()->remove( sourceEntry.getPath() ).isSuccess() )
        {
            destFileSystem->remove( destFile ); // clean up
            DLOG( "Failed" );
            return HK_FAILURE;
        }
    }

    DLOG( "Success {} {}", sourceEntry.getPath(), destFile );
    return HK_SUCCESS;
}

hkResult hkUnionFileSystem::mkdir(_In_z_ const char* path, CreateFlag flag)
{
    DLOG_SCOPE("UnionFS.mkdir {}", path);
    MountIterator it(m_mounts, path);
    while( it.advance() )
    {
        const Mount& m = it.current();
        if( m.m_writable )
        {
            hkResult r = m.m_fs->mkdir(it.path(), flag);
            if (r.isSuccess())
            {
                DLOG("Success {}", it.path());
                return HK_SUCCESS;
            }
            if (r.isExactly(HK_E_ALREADY_EXISTS))
            {
                DLOG("Path exists {}", it.path());
                return HK_E_ALREADY_EXISTS;
            }
        }
    }
    DLOG("Failed");
    return HK_FAILURE;
}

hkResult hkUnionFileSystem::copy(_In_z_ const char* srcPath, _In_z_ const char* dstPath)
{
    DLOG_SCOPE("UnionFS.copy {} {}", srcPath, dstPath);

    hkStringBuf srcResolved;
    hkFileSystem *srcFileSystem = resolvePath(srcPath, srcResolved);

    if (!srcFileSystem)
    {
        DLOG("Failed");
        return HK_FAILURE;
    }

    hkStringBuf dstWithoutFilename = hkStringBuf(dstPath).pathDirname();

    hkStringBuf dstResolved;
    hkFileSystem *dstFileSystem = resolvePath(dstWithoutFilename, dstResolved);

    if (!dstFileSystem)
    {
        DLOG("Failed");
        return HK_FAILURE;
    }

    hkStringBuf dstFilename = hkStringBuf(dstPath).pathBasename();

    if (dstFilename.getLength()>0)
    {
        dstResolved.pathAppend(dstFilename);
    }

    if (srcFileSystem == dstFileSystem)
    {
        hkResult copyResult = srcFileSystem->copy(srcResolved, dstResolved);
        if (copyResult.isFailure())
        {
            DLOG("Failed");
            return copyResult;
        }
    }
    else
    {
        hkRefPtr<hkStreamReader> reader = srcFileSystem->openReader(srcResolved);
        if (!reader || !reader->isOk())
        {
            DLOG("Failed");
            return HK_FAILURE;
        }

        hkRefPtr<hkStreamWriter> writer = dstFileSystem->openWriter(dstResolved);
        if (!writer || !writer->isOk())
        {
            DLOG("Failed");
            return HK_FAILURE;
        }

        if (!hkIo::copyStream(reader, writer))
        {
            DLOG("Failed");
            return HK_FAILURE;
        }
    }

    DLOG("Success {} {}", srcPath, dstPath);
    return HK_SUCCESS;
}


// We blindly assume that the path we were returned by the MountIterator begins with the m.m_dstPath. This can be false for a few reasons:
// * Relative paths outside the mount
// * Mixing \ and /
// * Case sensitivity
// Rather than patching up every case we should probably get a proper path container that knows about all of these things

hkResult hkUnionFileSystem::stat(_In_z_ const char* path, Entry& entryOut )
{
    MountIterator it(m_mounts, path);
    while( it.advance() )
    {
        const Mount& m = it.current();
        if( m.m_fs->stat( it.path(), entryOut ).isSuccess() )
        {
            // reverse map the filename
            hkStringBuf sb = m.m_srcPath.cString();
            if (!(hkString::strLen(entryOut.getPath()) >= hkString::strLen(m.m_dstPath))) // COM-2843 check but not a fix
            {
                continue;
            }
            sb.pathAppend( entryOut.getPath() + hkString::strLen(m.m_dstPath) );
            entryOut.setPath(this, sb);
            return HK_SUCCESS;
        }
    }
    return HK_FAILURE;
}

void hkUnionFileSystem::mount(_Inout_ hkFileSystem* fs, _In_z_ const char* srcPath, _In_z_ const char* dstPath, hkBool writable)
{
    DLOG_SCOPE("UnionFS.mount ({}) {} -> {} ", writable ? "RW" : "RO", srcPath, dstPath);
    unmount(srcPath, dstPath);
    Mount& m = m_mounts.expandOne();
    m.m_fs = fs;
    m.m_srcPath = hkStringBuf(srcPath).pathNormalize();
    m.m_dstPath = hkStringBuf(dstPath).pathNormalize();
    m.m_writable = writable;
}

void hkUnionFileSystem::unmount(_In_z_ const char* srcPath, _In_opt_z_ const char* dstPath)
{
    int idx = 0;
    hkArray<Mount>::iterator it = m_mounts.begin();
    while (it != m_mounts.end())
    {
        if (it->m_srcPath.compareTo(srcPath) == 0 && (!dstPath || (it->m_dstPath.compareTo(dstPath) == 0)))
        {
            m_mounts.removeAtAndCopy(idx);
            it = m_mounts.begin() + idx;
        }
        else
        {
            ++it;
            ++idx;
        }
    }
}

namespace
{
    struct UnionIterator : public hkFileSystem::Iterator::Impl
    {
        UnionIterator(_Inout_ hkUnionFileSystem* fs, hkArray<hkUnionFileSystem::Mount>& mounts, _In_z_ const char* top, _In_z_ const char* wildcard)
            : m_fs(fs)
            , m_mounts(mounts)
            , m_top(top)
            , m_wildcard(wildcard)
            , m_next(HK_NULL)
            , m_mountIndex(mounts.getSize())
        {
        }

        virtual bool advance(hkFileSystem::Entry& e)
        {
            while(1)
            {
                if( m_next )
                {
                    if( m_next->advance(e) )
                    {
                        const hkUnionFileSystem::Mount& m = m_mounts[m_mountIndex];
                        HK_ASSERT_NO_MSG(0x33819923, hkString::beginsWith(e.getPath(), m.m_dstPath));
                        // reverse map the filename
                        hkStringBuf sb = m.m_srcPath.cString();
                        sb.pathAppend( e.getPath() + hkString::strLen(m.m_dstPath) );
                        e.setPath(m_fs, sb);
                        return true;
                    }
                    m_next = HK_NULL;
                }
                m_mountIndex -= 1;
                if( m_mountIndex >= 0 )
                {
                    const hkUnionFileSystem::Mount& m = m_mounts[m_mountIndex];
                    if( hkString::beginsWith(m_top, m.m_srcPath) )
                    {
                        hkStringBuf sb = m.m_dstPath.cString();
                        sb.pathAppend( m_top + hkString::strLen(m.m_srcPath) );
                        m_next = m.m_fs->createIterator(sb, m_wildcard);
                    }
                }
                else
                {
                    return false;
                }
            }
        }

        hkUnionFileSystem* m_fs;
        hkArray<hkUnionFileSystem::Mount>& m_mounts;
        hkStringPtr m_top;
        hkStringPtr m_wildcard;
        hkRefPtr<hkFileSystem::Iterator::Impl> m_next;
        int m_mountIndex;
    };

    // This is only used when the watch point is above any of the mounted file systems (i.e. in the union)
    // If we are watching only in a sub-system, just leave that there
    struct UnionWatcher : public hkFileSystem::Watcher
    {
    public:
        HK_DECLARE_CLASS(UnionWatcher, New);

        UnionWatcher(_Inout_ hkUnionFileSystem* parent, const hkArrayBase< hkRefPtr<Watcher> >& watchers) : m_parent(parent)
        {
            for(int i=0; i < watchers.getSize(); i++)
            {
                m_watchers.expandOne() = watchers[i];
            }
        }

        virtual hkResult getCompletedChanges(hkArray<Change>& changesOut) HK_OVERRIDE
        {
            for(int i=0; i < m_watchers.getSize(); i++)
            {
                if(m_watchers[i] && !m_watchers[i]->getCompletedChanges(changesOut).isSuccess())
                {
                    return HK_FAILURE;
                }
            }

            for(int i = 0; i < changesOut.getSize(); ++i)
            {
                hkStringBuf buf;
                changesOut[i].m_fileName = m_parent->srcPathFromDstPath(changesOut[i].m_fileName, buf);
                buf.clear();

                if(changesOut[i].m_oldFileName.cString())
                {
                    changesOut[i].m_oldFileName = m_parent->srcPathFromDstPath(changesOut[i].m_oldFileName, buf);
                }
            }

            return HK_SUCCESS;
        }

        virtual hkResult stopWatching() HK_OVERRIDE
        {
            hkResult result = HK_SUCCESS;

            for (int i = 0; i < m_watchers.getSize(); i++)
            {
                if (m_watchers[i]->stopWatching().isFailure())
                {
                    result = HK_FAILURE;
                }
            }

            return result;
        }

        hkRefPtr<hkUnionFileSystem> m_parent;
        hkArray<hkRefPtr<Watcher> > m_watchers;
    };
}

hkRefNew<hkFileSystem::Iterator::Impl> hkUnionFileSystem::createIterator(_In_z_ const char* top, _In_z_ const char* wildcard)
{
    return new UnionIterator( this, m_mounts, top, wildcard);
}

hkRefNew<hkFileSystem::Watcher> hkUnionFileSystem::createWatcher(_In_z_ const char* topDirectory)
{
    hkArray< hkRefPtr<hkFileSystem::Watcher> > watchers;

    MountIterator it(m_mounts, topDirectory);
    while( it.advance() )
    {
        const Mount& m = it.current();
        if( hkString::beginsWith(topDirectory, m.m_srcPath) )
        {
            hkStringBuf nativePath;
            if(resolvePath(topDirectory, nativePath))
            {
                hkRefPtr<hkFileSystem::Watcher> w = m.m_fs->createWatcher(nativePath.cString());
                if(w)  // can be null with fileserver on connect closed
                    watchers.pushBack(w);
            }
        }
    }

    // Only one child file system matched, so just use its child watcher
    if(watchers.getSize() == 1)
    {
        hkFileSystem::Watcher* watcher = watchers[0];
        watcher->addReference(); // Add a ref since we're returning a hkRefNew.
        return watcher;
    }
    else if (watchers.getSize() > 0)
    {
        return new UnionWatcher(this, watchers);
    }
    return HK_NULL;
}

_Ret_maybenull_z_ const char* hkUnionFileSystem::srcPathFromDstPath(_In_z_ const char* dstPath, hkStringBuf& pathOut)
{
    hkStringBuf normDstPath(dstPath);
    normDstPath.pathNormalize();

    MountIterator it(m_mounts, normDstPath);
    while(it.advance())
    {
        const Mount& m = it.current();
        Entry entry;
        if(hkString::beginsWithCase(normDstPath, m.m_dstPath))
        {
            pathOut = normDstPath;
            pathOut.replace(m.m_dstPath, m.m_srcPath);
            return pathOut;
        }
    }
    return HK_NULL;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
