// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/System/Io/FileSystem/Server/hkServerFileSystemThread.h>

#include <Common/Base/System/Io/Socket/hkSocket.h>
#include <Common/Base/System/Io/OArchive/hkOArchive.h>
#include <Common/Base/System/Io/IArchive/hkIArchive.h>
#include <Common/Base/Serialize/Detail/hkWriteBuffer.h>
#include <Common/Base/Serialize/Detail/hkReadBuffer.h>

#include <Common/Base/System/Io/FileSystem/Server/hkServerFileSystem.h>
#include <Common/Base/System/Io/FileSystem/Server/hkFileServerStreamReader.h>
#include <Common/Base/System/Io/FileSystem/Server/hkFileServerStreamWriter.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>

#define DEBUG_LOG_IDENTIFIER "io.RemoteFileSystemServer"
#include <Common/Base/System/Log/hkLog.hxx>

#ifdef HK_PLATFORM_WIN32
#include <Common/Base/Fwd/hkwindows.h>
#endif

static void HK_CALL dummyProgressFunc(_In_opt_z_ const char* filename, int progress, _In_opt_z_ const char* info)
{
}

hkFileServer::ThreadData::ThreadData()
{
    m_dataLock = HK_NULL;
    m_requestShutdown = false;
    m_socket = HK_NULL;
    m_progressFunc = &dummyProgressFunc;
    m_nonBlocking = false;
}


static int findFileRead(hkArray<hkFileServer::OpenReadFiles>& a, hkUint32 id)
{
    for (int i=0; i < a.getSize(); ++i)
    {
        if (a[i].id == id)
            return i;
    }
    return -1;
}

static int findFileWrite(hkArray<hkFileServer::OpenWriteFiles>& a, hkUint32 id)
{
    for (int i=0; i < a.getSize(); ++i)
    {
        if (a[i].id == id)
            return i;
    }
    return -1;
}

namespace
{
    struct WatcherAndId
    {
        hkUint32 m_id;
        hkRefPtr<hkFileSystem::Watcher> m_watcher;
    };
}

/*static*/ hkFileServer::ReturnCode hkFileServer::serve(_Inout_ ThreadData* threadData)
{
    HK_ASSERT_NO_MSG(0x56cb6888, threadData->m_socket);
    HK_ASSERT_NO_MSG(0x56cb6889, threadData->m_dataLock);
    if (threadData->m_fileSystem == HK_NULL)
    {
        DLOG("Using filesystem singleton for fileserver");
        threadData->m_fileSystem = hkFileSystem::getInstancePtr();
    }
    hkSocket* s = threadData->m_socket;
    hkArray<OpenReadFiles> curReads;
    hkArray<OpenWriteFiles> curWrites;
    hkUint32 curFileId = 0;

    hkArray<WatcherAndId> watchers;
    hkUint32 curWatcherId = 0;

    hkArray<char> tempBuffer;
    hkIArchive in(&s->getReader());
    hkIo::WriteBuffer out(&s->getWriter());

    ReturnCode ret = RETURN_SOCKET_ERROR;

    while (s->isOk() && !threadData->m_requestShutdown && ret == RETURN_SOCKET_ERROR)
    {
#ifdef HK_PLATFORM_WIN32
        if (threadData->m_nonBlocking && !s->canRead())
        {
            Sleep(1); //xx change to proper async sockets etc
            continue;// loop, and thus check m_requestShutdown again
        }
#endif
        DLOG_IF(hkUint32 packetSize =) in.read32u();
        hkUint32 cmd = in.read32u();
        if (!s->isOk())
        {
            ret = RETURN_SOCKET_ERROR;
            break; // dead
        }

        if (0)
        {
            DLOG("Handling command {} [packetSize: {}]", cmd, packetSize);
        }

        switch (cmd)
        {
        case hkServerFileSystem::SHUT_DOWN:
            {
                // Exited cleanly
                ret = RETURN_SOCKET_CLOSED;
                break;
            }
        case hkServerFileSystem::FILE_READ:
            {
                threadData->m_dataLock->enter();

                const hkUint32 mode = in.read32u();
                hkUint32 fileNameLen = in.read32u();
                hkArray<char> nameBuf; nameBuf.setSize(fileNameLen + 1); // has null term
                char* name = nameBuf.begin();
                in.readRaw( name, fileNameLen + 1 );

                hkStringBuf fullPath( threadData->m_rootDir, name );
                hkRefPtr<hkStreamReader> sr = threadData->m_fileSystem->openReader(fullPath.cString(), hkFileSystem::OpenFlags(mode | hkFileSystem::OPEN_BUFFERED));
                if ( !sr ) // try just path as is.. ? (may have absolute path in it, like say from drag n drop)
                {
                    sr = threadData->m_fileSystem->openReader(name, hkFileSystem::OpenFlags(mode | hkFileSystem::OPEN_BUFFERED));
                }

                if ( s->isOk() )
                {
                    if ( sr && sr->isSeekTellSupported() )
                    {
                        OpenReadFiles& r = curReads.expandOne();
                        r.id = ++curFileId;
                        r.s = sr->isSeekTellSupported();
                        r.filename = name;
                        r.bytesReadTotal = 0;

                        DLOG("File open (read): [{}][id:{}] mode:{}", r.filename.cString(), r.id, mode);
                        threadData->m_progressFunc( name, 0, HK_NULL );

                        out.write32u(2 * sizeof(int));
                        out.write32u(hkServerFileSystem::ACK);
                        out.write32u(r.id);

                    }
                    else
                    {
                        DLOG("File open (read) FAILED: [{}]", name);
                        out.write32u(1 * sizeof(int));
                        out.write32u(hkServerFileSystem::NOT_FOUND);
                    }
                }

                out.flush();
                threadData->m_dataLock->leave();
            }

            break;

        case hkServerFileSystem::FILE_WRITE:
            {
                threadData->m_dataLock->enter();

                const hkUint32 mode = in.read32u();
                hkUint32 fileNameLen = in.read32u();

                hkArray<char> nameBuf; nameBuf.setSize(fileNameLen + 1); // has null term
                char* name = nameBuf.begin();

                in.readRaw( name, fileNameLen + 1 );

                hkStringBuf fullPath( threadData->m_rootDir, name );
                hkRefPtr<hkStreamWriter> sw = threadData->m_fileSystem->openWriter(fullPath.cString(), hkFileSystem::OpenFlags(mode | hkFileSystem::OPEN_BUFFERED));
                if ( !sw ) // try just path as is.. ? (may have absolute path in it, like say from drag n drop)
                {
                    sw = threadData->m_fileSystem->openWriter(name, hkFileSystem::OpenFlags(mode | hkFileSystem::OPEN_BUFFERED));
                }

                if ( s->isOk() )
                {
                    if ( sw && sw->isOk() )
                    {
                        OpenWriteFiles& w = curWrites.expandOne();
                        w.id = ++curFileId;
                        w.s = hk::move(sw);
                        w.filename = name;
                        w.bytesWrittenTotal = 0;

                        DLOG("File open (write): [{}][id:{}] mode {}", w.filename.cString(), w.id, mode);
                        threadData->m_progressFunc( name, 0, HK_NULL );

                        out.write32u(2 * sizeof(int));
                        out.write32u(hkServerFileSystem::ACK);
                        out.write32u(w.id);
                    }
                    else
                    {
                        DLOG("File open (write) FAILED: [{}]", name);
                        out.write32u(1 * sizeof(int));
                        out.write32u(hkServerFileSystem::NOT_FOUND);
                    }
                }

                out.flush();
                threadData->m_dataLock->leave();
            }
            break;

        case hkServerFileSystem::DIR_LIST:
            {
                threadData->m_dataLock->enter();

                hkUint32 pathLen = in.read32u();

                hkArray<char> nameBuf; nameBuf.setSize(pathLen + 1); // has null term
                char* pathName = nameBuf.begin();

                in.readRaw( pathName, pathLen + 1 );
                hkStringBuf fullPath( threadData->m_rootDir, pathName );

                if (s->isOk())
                {
                    hkFileSystem::DirectoryListing listing;
                    hkResult res = threadData->m_fileSystem->listDirectory( fullPath, listing );
                    if (res.isSuccess())
                    {
                        hkArray<char> sendBuffer(1024);
                        sendBuffer.setSize(12); // packetsize, cmd, numentries
                        const hkArrayBase< hkFileSystem::Entry >& e = listing.getEntries();
                        for (int li=0; li < e.getSize(); ++li)
                        {
                            hkUint32 strNameLen = hkString::strLen( e[li].getName() );

                            hkUint32 paddedNameLen = strNameLen + 1 /*null*/ + 1 /* isDir */ + 4 /* len int */ + 8 /* file size */;
                            paddedNameLen = HK_NEXT_MULTIPLE_OF( 4, paddedNameLen );

                            hkUint32* nameLenPtr = (hkUint32*)( sendBuffer.expandBy(paddedNameLen) );
                            *nameLenPtr = paddedNameLen - 2 - 4 - 8; // null assumed and isDir not part of name etc

                            hkInt64* fileSizePtr = (hkInt64*) (nameLenPtr + 1);
                            hkInt64 fileSize = e[li].getSize();
                            hkString::memCpy( fileSizePtr, &fileSize, sizeof(hkInt64) );

                            hkUint8* isDirPtr = (hkUint8*)( fileSizePtr + 1);
                            *isDirPtr = e[li].isDir()? 1 : 0;

                            char* name = (char*)( isDirPtr + 1 );
                            hkString::memCpy(name, e[li].getName(), strNameLen + 1);
                        }

                        hkUint32& packetSize2 = *(hkUint32*)sendBuffer.begin();
                        hkUint32& cmd2 = *(hkUint32*)(sendBuffer.begin() + 4);
                        hkUint32& numEntries = *(hkUint32*)(sendBuffer.begin() + 8);
                        packetSize2 = sendBuffer.getSize() - 4;
                        cmd2 = hkServerFileSystem::SEND_DIR_LIST;
                        numEntries = e.getSize();
                        out.writeRaw(sendBuffer.begin(), sendBuffer.getSize());
                        DLOG("Path dir list sent for: [{}]", pathName);
                    }
                    else if ( s->isOk() )
                    {
                        DLOG("Path dir list FAILED: [{}]", pathName);
                        out.write32u(1 * sizeof(int));
                        out.write32u(hkServerFileSystem::NOT_FOUND);
                    }
                }
                out.flush();
                threadData->m_dataLock->leave();
            }
            break;

        case hkServerFileSystem::REMOVE:
            {
                threadData->m_dataLock->enter();

                hkUint32 fileNameLen = in.read32u();
                hkArray<char> nameBuf; nameBuf.setSize(fileNameLen + 1); // has null term
                char* name = nameBuf.begin();

                in.readRaw( name, fileNameLen + 1 );

                hkStringBuf fullPath( threadData->m_rootDir, name );

                if ( s->isOk() )
                {
                    hkResult res = threadData->m_fileSystem->remove(fullPath);

                    if ( res.isSuccess())
                    {
                        DLOG("File removed: [{}]", fullPath.cString());
                        threadData->m_progressFunc( name, 0, HK_NULL );

                        out.write32u(1 * sizeof(int));
                        out.write32u(hkServerFileSystem::ACK);
                    }
                    else
                    {
                        DLOG("File remove FAILED: [{}]", name);
                        out.write32u(1 * sizeof(int));
                        out.write32u(hkServerFileSystem::NOT_FOUND);
                    }
                }

                out.flush();
                threadData->m_dataLock->leave();
            }
            break;

        case hkServerFileSystem::MKDIR:
            {
                threadData->m_dataLock->enter();

                hkUint32 fileNameLen = in.read32u();
                hkArray<char> nameBuf; nameBuf.setSize(fileNameLen + 1); // has null term
                char* name = nameBuf.begin();

                in.readRaw( name, fileNameLen + 1 );

                hkStringBuf fullPath( threadData->m_rootDir, name );

                if (s->isOk())
                {
                    hkResult r = threadData->m_fileSystem->mkdir(fullPath, hkFileSystem::ALLOW_EXISTING);
                    if (r.isSuccess())
                    {
                        DLOG(("Directory created/exists: [{}]"), fullPath.cString());
                        threadData->m_progressFunc(name, 0, HK_NULL);

                        out.write32u(1 * sizeof(int));
                        out.write32u(hkServerFileSystem::ACK);
                    }
                    else
                    {
                        DLOG("Mkdir FAILED: [{}]", name);
                        out.write32u(1 * sizeof(int));
                        out.write32u(hkServerFileSystem::NOT_FOUND);
                    }
                }

                out.flush();
                threadData->m_dataLock->leave();
            }
            break;

            //
            // Reader
            //

        case hkFileServerStreamReader::READ_CHUNK:
            {
                hkUint32 fileId = in.read32u();
                hkUint32 nbytes = in.read32u();
                int idx = findFileRead( curReads, fileId );
                OpenReadFiles* w = idx >= 0? &curReads[idx] : HK_NULL;

                // don't need to check streamreader is OK here - if it's at EOF, we send something for that.
                if (w && s->isOk())
                {
                    int bytesRead = 0;
                    if (tempBuffer.getSize() < (int)nbytes)
                        tempBuffer.setSize(nbytes);

                    if(w->s->isOk())
                    {
                        bytesRead = w->s->read(tempBuffer.begin(), nbytes);
                    }

                    if (bytesRead > 0)
                    {
                        out.write32u(3 * sizeof(int) + bytesRead );
                        out.write32u(hkFileServerStreamReader::SEND_CHUNK);
                        out.write32u(w->id);
                        out.write32u(bytesRead);
                        out.writeRaw(tempBuffer.begin(), bytesRead);

                        w->bytesReadTotal += bytesRead;
                        threadData->m_progressFunc( w->filename.cString(), w->bytesReadTotal / 1024, HK_NULL );
                    }
                    else
                    {
                        out.write32u(2 * sizeof(int) );
                        out.write32u(hkFileServerStreamReader::EOF_OR_ERROR);
                        out.write32u(w->id);
                        threadData->m_progressFunc( w->filename.cString(), 100, HK_NULL );
                    }
                    out.flush();
                }
                else if (!w)
                {
                    DLOG("Requested read on file that is not open [id: {}]", fileId);
                }

            }
            break;

        case hkFileServerStreamReader::SEEK:
            {
                hkUint32 fileId = in.read32u();
                hkInt32 offset = in.read32();
                hkUint32 whence = in.read32u();
                int idx = findFileRead( curReads, fileId );
                OpenReadFiles* w = idx >= 0? &curReads[idx] : HK_NULL;

                // don't need to check streamreader is OK here - we can still seek if it's at EOF
                hkResult res = HK_FAILURE;
                if (w && s->isOk())
                {
                    res = w->s->seek(offset, (hkSeekableStreamReader::SeekWhence)whence);
                }

                out.write32u(1);
                hkBool seekOk = (res.isSuccess());
                out.write8(seekOk);
                out.flush();
                break;
            }

        case hkFileServerStreamReader::TELL:
            {
                hkUint32 fileId = in.read32u();
                if ( s->isOk() )
                {
                    int idx = findFileRead( curReads, fileId );
                    OpenReadFiles* w = idx >= 0 ? &curReads[idx] : HK_NULL;
                    if ( w )
                    {
                        int tellVal = w->s->tell();
                        out.write32u( 3 * sizeof( int ) );
                        out.write32u( hkFileServerStreamReader::SEND_TELL );
                        out.write32u( w->id );
                        out.write32u( tellVal );
                    }
                    else
                    {
                        out.write32u( 2 * sizeof( int ) );
                        out.write32u( hkFileServerStreamReader::EOF_OR_ERROR );
                        out.write32u( fileId );
                    }
                    out.flush();
                }
                break;
            }

        case hkFileServerStreamReader::IS_OK:
            {
                hkUint32 fileId = in.read32u();
                int idx = findFileRead( curReads, fileId );
                OpenReadFiles* w = idx >= 0? &curReads[idx] : HK_NULL;

                hkBool readerIsOk = (w && w->s->isOk()) ? true : false;
                out.write32u(1);
                out.write8((char) readerIsOk);
                out.flush();
            }
            break;

        case hkFileServerStreamWriter::WRITE_CHUNK:
            {
                hkUint32 fileId = in.read32u();
                hkUint32 nbytes = in.read32u();
                int idx = findFileWrite( curWrites, fileId );
                OpenWriteFiles* w = idx >= 0? &curWrites[idx] : HK_NULL;
                if (w && w->s->isOk() && s->isOk() )
                {
                    if (tempBuffer.getSize() < (int)nbytes)
                        tempBuffer.setSize(nbytes);

                    int bytesToBeWritten = in.readRaw(tempBuffer.begin(), nbytes);
                    if (s->isOk() && (bytesToBeWritten == (int)nbytes))
                    {
                        int bytesActuallyWritten = w->s->write( tempBuffer.begin(), bytesToBeWritten );

                        w->bytesWrittenTotal += bytesActuallyWritten;
                        out.write32u(3 * sizeof(int) );
                        out.write32u(hkFileServerStreamWriter::WROTE);
                        out.write32u(w->id);
                        out.write32u(bytesActuallyWritten);
                        threadData->m_progressFunc( w->filename.cString(), w->bytesWrittenTotal / 1024, HK_NULL );
                    }
                    else if ( s->isOk()  )
                    {
                        out.write32u(2 * sizeof(int) );
                        out.write32u(hkFileServerStreamWriter::IN_ERROR);
                        out.write32u(w->id);
                        threadData->m_progressFunc( w->filename.cString(), 100, HK_NULL );
                    }
                    out.flush();
                }
                else if (!w)
                {
                    DLOG("Requested read on file that is not open [id: {}]", fileId);
                }

            }
            break;

        case hkFileServerStreamWriter::SEEK:
            {
                hkUint32 fileId = in.read32u();
                hkInt32 offset = in.read32();
                hkUint32 whence = in.read32u();
                int idx = findFileWrite( curWrites, fileId );
                OpenWriteFiles* w = idx >= 0? &curWrites[idx] : HK_NULL;
                if (w && w->s->isOk() && s->isOk() )
                {
                    w->s->seek(offset, (hkStreamWriter::SeekWhence)whence);
                    // it does not expect an ack etc from seek
                }

                break;
            }

        case hkFileServerStreamWriter::TELL:
            {
                hkUint32 fileId = in.read32u();
                int idx = findFileWrite( curWrites, fileId );
                OpenWriteFiles* w = idx >= 0? &curWrites[idx] : HK_NULL;
                if (w && w->s->isOk() && s->isOk())
                {
                    int tellVal = w->s->tell();
                    out.write32u(3 * sizeof(int) );
                    out.write32u(hkFileServerStreamWriter::SEND_TELL);
                    out.write32u(w->id);
                    out.write32u(tellVal);
                    out.flush();
                }

                break;
            }

        case hkFileServerStreamWriter::FLUSH:
            {
                hkUint32 fileId = in.read32u();
                int idx = findFileWrite( curWrites, fileId );
                OpenWriteFiles* w = idx >= 0? &curWrites[idx] : HK_NULL;
                if (w && w->s->isOk() && s->isOk() )
                {
                    w->s->flush();
                    // it does not expect an ack etc from flush
                }

                break;
            }

        case hkFileServerStreamReader::CLOSE:
            {
                hkUint32 fileId = in.read32u();
                int idx = findFileRead( curReads, fileId );
                if (idx >= 0)
                {
                    hkStringBuf info;
                    if (curReads[idx].bytesReadTotal > 1023*1024)
                        info.printf("%.2fMiB read", (float)curReads[idx].bytesReadTotal / (1024.f*1024.f));
                    else if (curReads[idx].bytesReadTotal > 1023)
                        info.printf("%.2fKiB read", (float)curReads[idx].bytesReadTotal / 1024.f );
                    else
                        info.printf("%dB read", curReads[idx].bytesReadTotal );
                    threadData->m_progressFunc( curReads[idx].filename.cString(), 0, info.cString()  );

                    curReads.removeAt(idx);
                    DLOG("Closed (read) [id: {}]", fileId);
                }
                else
                {
                    DLOG("Close requested on file that is not open [id: {}]", fileId);
                }
            }
            break;

        case hkFileServerStreamWriter::CLOSE:
            {
                hkUint32 fileId = in.read32u();
                int idx = findFileWrite( curWrites, fileId );
                if (idx >= 0)
                {
                    hkStringBuf info;
                    if (curWrites[idx].bytesWrittenTotal > 1023*1024)
                        info.printf("%.2fMiB written", (float)curWrites[idx].bytesWrittenTotal / (1024.f*1024.f));
                    else if (curWrites[idx].bytesWrittenTotal > 1023)
                        info.printf("%.2fKiB written", (float)curWrites[idx].bytesWrittenTotal / 1024.f );
                    else
                        info.printf("%dB written", curWrites[idx].bytesWrittenTotal );
                    threadData->m_progressFunc( curWrites[idx].filename.cString(), 0, info.cString()  );

                    curWrites.removeAt(idx);

                    DLOG("Closed (write) [id: {}]", fileId);
                }
                else
                {
                    DLOG("Close requested on file that is not open [id: {}]", fileId);
                }
            }
            break;

        case hkServerFileSystem::FILE_STAT:
            {
                threadData->m_dataLock->enter();

                hkUint32 fileNameLen = in.read32u();
                hkArray<char> nameBuf; nameBuf.setSize(fileNameLen + 1); // has null term
                char* name = nameBuf.begin();

                in.readRaw( name, fileNameLen + 1 );

                hkStringBuf fullPath( threadData->m_rootDir, name );

                if ( s->isOk() )
                {
                    hkFileSystem::Entry entry;
                    hkResult res = threadData->m_fileSystem->stat(fullPath, entry);

                    if ( res.isSuccess() )
                    {
                        DLOG("stat success: [{}]", fullPath.cString());
                        threadData->m_progressFunc( name, 0, HK_NULL );

                        out.write32u(2 * sizeof(hkUint32) + 2 * sizeof(hkUint64));
                        out.write32u(hkServerFileSystem::SEND_STAT);
                        out.write64i(entry.getSize());
                        out.write64u(entry.getMtime().get());
                        out.write32u(entry.getFlags().get());
                    }
                    else
                    {
                        DLOG("stat FAILED: [{}]", name);
                        out.write32u(1 * sizeof(int));
                        out.write32u(hkServerFileSystem::NOT_FOUND);
                    }
                    out.flush();
                }

                threadData->m_dataLock->leave();
            }

            break;
        case hkServerFileSystem::WATCH:
            {
                threadData->m_dataLock->enter();

                hkUint32 dirLen = in.read32u();
                hkArray<char> dirBuf; dirBuf.setSize(dirLen + 1); // has null term
                in.readRaw( dirBuf.begin(), dirLen + 1 );

                if ( s->isOk() )
                {
                    WatcherAndId& watcher = watchers.expandOne();

                    hkStringBuf fullPath(threadData->m_rootDir, dirBuf.begin());
                    if (fullPath.cString())
                    {
                        watcher.m_watcher = threadData->m_fileSystem->createWatcher(fullPath.cString());
                        watcher.m_id = ++curWatcherId;

                        DLOG("created watcher {}: [{}]", watcher.m_id, dirBuf.begin());

                        out.write32u(2 * sizeof(hkUint32));
                        out.write32u(hkServerFileSystem::ACK);
                        out.write32u(watcher.m_id);
                    }
                    out.flush();
                }

                threadData->m_dataLock->leave();
            }

            break;
        case hkServerFileSystem::UNWATCH:
            {
                threadData->m_dataLock->enter();

                hkUint32 id = in.read32u();
                hkUint32 res = hkServerFileSystem::NOT_FOUND;
                for (int i = 0; i < watchers.getSize(); ++i)
                {
                    if (watchers[i].m_id == id)
                    {
                        DLOG("removing watcher {}", id);

                        watchers.removeAt(i);
                        res = hkServerFileSystem::ACK;
                        break;
                    }
                }
                out.write32u(res);
                out.flush();
                threadData->m_dataLock->leave();
            }

            break;
        case hkServerFileSystem::CHANGES:
            {
                threadData->m_dataLock->enter();

                hkUint32 id = in.read32u();
                for (int w = 0; w < watchers.getSize(); ++w)
                {
                    WatcherAndId& watcher = watchers[w];
                    if (watcher.m_id == id)
                    {

                        hkArray<hkServerFileSystem::Watcher::Change> changes;
                        
                        /*hkResult res =*/ watcher.m_watcher->getCompletedChanges(changes);
                        hkUint32 size = 1;
                        for (int c = 0; c < changes.getSize(); ++c)
                        {
                            size += 1;
                            size += changes[c].m_fileName.getLength() + 1 - threadData->m_rootDir.getLength(); //XXX So many assumptions here
                            if (changes[c].m_change == hkFileSystem::Watcher::Change::Renamed)
                            {
                                size += changes[c].m_oldFileName.getLength() + 1 - threadData->m_rootDir.getLength(); //XXX So many assumptions here
                            }
                        }
                        out.write32u(size);
                        out.write32u(hkServerFileSystem::SEND_CHANGES);
                        out.write32u(changes.getSize());
                        for (int c = 0; c < changes.getSize(); ++c)
                        {
                            out.write8((hkUchar)changes[c].m_change);

                            hkUint32 expectedNameStart = threadData->m_rootDir.getLength(); //XXX So many assumptions here

                            hkUint32 newSize = changes[c].m_fileName.getLength() - expectedNameStart;
                            out.write32u(newSize);
                            out.writeRaw(changes[c].m_fileName.cString() + expectedNameStart, newSize + 1);

                            if (changes[c].m_change == hkFileSystem::Watcher::Change::Renamed)
                            {
                                hkUint32 oldSize = changes[c].m_oldFileName.getLength() - expectedNameStart;
                                out.write32u(oldSize);
                                out.writeRaw(changes[c].m_oldFileName.cString() + expectedNameStart, oldSize + 1);
                            }

                        }
                        break;
                    }
                }
                out.flush();
                threadData->m_dataLock->leave();
            }

            break;

        } // switch

    } // while (isOK and ! requestShutdown)

    if (ret == RETURN_SOCKET_ERROR && threadData->m_requestShutdown)
    {
        ret = RETURN_SERVER_SHUTDOWN;
    }

    return ret;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
