// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/System/Io/FileSystem/Server/hkServerFileSystem.h>
#include <Common/Base/System/Io/Socket/hkSocket.h>
#if defined(HK_PLATFORM_WIN32) && !defined(HK_PLATFORM_WINRT)
#include <Common/Base/System/Io/Platform/Bsd/hkBsdSocket.h>
#endif

#include <Common/Base/System/Io/OArchive/hkOArchive.h>
#include <Common/Base/System/Io/IArchive/hkIArchive.h>

#include <Common/Base/System/Io/FileSystem/Server/hkFileServerStreamReader.h>
#include <Common/Base/System/Io/FileSystem/Server/hkFileServerStreamWriter.h>
#include <Common/Base/System/Io/Reader/Buffered/hkBufferedStreamReader.h>
#include <Common/Base/System/Io/Writer/Buffered/hkBufferedStreamWriter.h>
#include <Common/Base/Serialize/Detail/hkWriteBuffer.h>
#include <Common/Base/Serialize/Detail/hkReadBuffer.h>

#define DEBUG_LOG_DEFAULT_LEVEL Info
#define DEBUG_LOG_IDENTIFIER "io.RemoteFileSystemClient"
#include <Common/Base/System/Log/hkLog.hxx>

#if defined(HK_PLATFORM_DURANGO)
    enum { HK_SERVER_FILE_SYSTEM_DEFAULT_PORT=58002 };
#else
    enum { HK_SERVER_FILE_SYSTEM_DEFAULT_PORT=26002 };
#endif

int hkServerFileSystem::getDefaultPort()
{
    return HK_SERVER_FILE_SYSTEM_DEFAULT_PORT;
}

//
// Kind of backwards at the moment so that we can choose from the PC side
// what and when to serve to multiple clients from the same PC
// So in effect the hkServerFileSystem is really a client that
// can only accept from the server, rather than knowing the server ip etc
// and just connecting direct.
//

hkServerFileSystem::hkServerFileSystem(int port)
: m_mode(DEFAULT)
, m_connectionPort(port ? port : HK_SERVER_FILE_SYSTEM_DEFAULT_PORT)
, m_listenSocket(HK_NULL)
, m_connectSocket(HK_NULL)
, m_connectionLock(1000)
{
    Log_Debug("Server started");
}

hkServerFileSystem::~hkServerFileSystem()
{
    shutdown();
}

void hkServerFileSystem::closeConnection()
{
    if (m_connectSocket)
    {
        // Tell the server that we're done.
        if (m_connectSocket->isOk())
        {
            hkOArchive out(&m_connectSocket->getWriter());
            out.write32u( 1 );
            out.write32u( SHUT_DOWN );
        }

        if (m_listenSocket != m_connectSocket)
        {
            m_connectSocket->close();
        }
        m_connectSocket->removeReference();
        m_connectSocket = HK_NULL;
    }
}

void hkServerFileSystem::shutdown()
{
    closeConnection();

    if (m_listenSocket)
    {
        m_listenSocket->close();
        m_listenSocket->removeReference();
        m_listenSocket = HK_NULL;
    }
}

bool hkServerFileSystem::tryForConnection()
{
    if (!m_listenSocket)
    {
        m_listenSocket = hkSocket::create();

        HK_ASSERT_NO_MSG(0x285ff1ce, m_listenSocket);

        if ( (m_mode & WAIT_FOR_CONNECT_TO_SERVER) == 0)
        {
            m_listenSocket->listen(m_connectionPort);
            Log_Info("Virtual file system created and will poll on port {}", m_connectionPort);
        }
    }

    if ( (m_mode & WAIT_FOR_CONNECT_TO_SERVER) == 0)
    {
        m_connectSocket = m_listenSocket->pollForNewClient();
    }
    else
    {
        const char* addr = m_serverAddress.getLength() ? m_serverAddress.cString() : "localhost";
        if (m_listenSocket->connect(addr, m_connectionPort).isSuccess())
        {
            Log_Info("Virtual file system connected to {}:{}", addr, m_connectionPort);
            m_connectSocket = m_listenSocket;
            m_connectSocket->addReference();
        }
    }

#if defined(HK_PLATFORM_WIN32) && !defined(HK_PLATFORM_WINRT)
    if (m_connectSocket)
    {
        ((hkBsdSocket*)m_connectSocket)->setBlocking(true);
    }
#endif

    return m_connectSocket != HK_NULL;
}

bool hkServerFileSystem::waitForConnection()
{
    while ( !tryForConnection() ) {  /* should have some sort of force quit.. */ }
    return m_connectSocket != HK_NULL;
}

void hkServerFileSystem::setMode( Mode m )
{
    m_mode = m;
}

// {
//     int PACKET_SIZE; // num bytes following this int
//     int CMD;
//     ..
//  }

// File Read open
//  {
//     uint PACKET_SIZE; // num bytes following this int
//     uint FILE_READ;
//     uint filenamelen;
//     char filename, (null term so can use as is on read)
//  }

hkRefNew<hkStreamReader> hkServerFileSystem::_openReader(_In_z_ const char* name, OpenFlags mode)
{
    Log_Scope(Dev, "openReader '{}'", name);
    if (m_mode & VIRTUAL_READ)
    {
        if (m_connectSocket && !m_connectSocket->isOk())
        {
            closeConnection();
        }

        if (!m_connectSocket)
        {
            if (m_mode & (WAIT_FOR_CONNECT | WAIT_FOR_CONNECT_TO_SERVER))
            {
                waitForConnection();
            }
            else // single listen
            {
                tryForConnection();
            }
        }

        hkCriticalSectionLock socketLock(&m_connectionLock);
        if (m_connectSocket && m_connectSocket->isOk())
        {
            int fileNameLen = hkString::strLen(name);
            hkOArchive out(&m_connectSocket->getWriter());
            out.write32u( fileNameLen + 1 + (3 * sizeof(int)) );
            out.write32u( FILE_READ );
            out.write32u( mode );
            out.write32u( fileNameLen );
            out.writeRaw( name, fileNameLen + 1 );

            // see if it has it
            hkIArchive in(&m_connectSocket->getReader());
            if (m_connectSocket->isOk())
            {
                int packetSize = in.read32u();
                int cmd = in.read32u();
                while (m_connectSocket->isOk())
                {
                    if (cmd == ACK)
                    {
                        int id = in.read32u();
                        Log_Dev("id '{}'", id);
                        hkFileServerStreamReader* fs = new hkFileServerStreamReader( m_connectSocket, id, &m_connectionLock );
                        hkStreamReader* b = new hkBufferedStreamReader(fs, 16*1024); // 4K is default size, so want something a little larger if we are going to put up with net latency in between reads
                        fs->removeReference();
                        return b;
                    }
                    else if (cmd == NOT_FOUND)
                    {
                        Log_Dev("failed (not found on server)");
                        break;
                    }
                    else
                    {
                        Log_Warning("got crazy stuff from file read socket..: {}", cmd ).setId(0xdafa3);
                        if (packetSize < 0x100000)
                        {
                            char* ignoreData = hkAllocate<char>( packetSize, HK_MEMORY_CLASS_BASE);
                            in.readRaw(ignoreData, packetSize - sizeof(int));
                            hkDeallocate<char>( ignoreData);
                        }
                        //loop
                    }
                }
            }
        }
    }

    return HK_NULL;
}

hkRefNew<hkStreamWriter> hkServerFileSystem::_openWriter(_In_z_ const char* name, OpenFlags mode )
{
    Log_Scope(Dev, "openWriter '{}'", name);
    if (m_mode & VIRTUAL_WRITE)
    {
        if (m_connectSocket && !m_connectSocket->isOk())
        {
            closeConnection();
        }

        if (!m_connectSocket)
        {
            if (m_mode & (WAIT_FOR_CONNECT | WAIT_FOR_CONNECT_TO_SERVER))
            {
                waitForConnection();
            }
            else // single listen
            {
                tryForConnection();
            }
        }

        hkCriticalSectionLock socketLock(&m_connectionLock);
        if (m_connectSocket && m_connectSocket->isOk())
        {
            int fileNameLen = hkString::strLen(name);
            hkOArchive out(&m_connectSocket->getWriter());
            out.write32u( fileNameLen + 1 + (3* sizeof(int)) );
            out.write32u( FILE_WRITE );
            out.write32u( mode );
            out.write32u( fileNameLen );
            out.writeRaw( name, fileNameLen + 1 );

            // see if it has it
            hkIArchive in(&m_connectSocket->getReader());
            if (m_connectSocket->isOk())
            {
                int packetSize = in.read32u();
                int cmd = in.read32u();
                while (m_connectSocket->isOk())
                {
                    if (cmd == ACK)
                    {
                        int id = in.read32u();
                        Log_Dev("id '{}'", id);
                        hkFileServerStreamWriter* fw = new hkFileServerStreamWriter( m_connectSocket, id, &m_connectionLock );
                        hkStreamWriter* b = new hkBufferedStreamWriter(fw);
                        fw->removeReference();
                        return b;
                    }
                    else if (cmd == NOT_FOUND)
                    {
                        Log_Dev("file not found");
                        break;
                    }
                    else
                    {
                        Log_Warning("got crazy stuff from file read socket..: {}", cmd).setId(0x32534);
                        if (packetSize < 0x100000)
                        {
                            char* ignoreData = hkAllocate<char>( packetSize, HK_MEMORY_CLASS_BASE);
                            in.readRaw(ignoreData, packetSize - sizeof(int));
                            hkDeallocate<char>(ignoreData);
                        }
                        //loop
                    }
                }
            }
        }
    }

    return HK_NULL;
}

hkResult hkServerFileSystem::stat(_In_z_ const char* name, hkFileSystem::Entry& entryOut )
{
    //Log_Scope(Info, "stat '{}'", name);
    if (m_mode & VIRTUAL_READ)
    {
        if (m_connectSocket && !m_connectSocket->isOk())
        {
            closeConnection();
        }

        if (!m_connectSocket)
        {
            if (m_mode & (WAIT_FOR_CONNECT | WAIT_FOR_CONNECT_TO_SERVER))
            {
                waitForConnection();
            }
            else // single listen
            {
                tryForConnection();
            }
        }

        hkCriticalSectionLock socketLock(&m_connectionLock);
        if (m_connectSocket && m_connectSocket->isOk())
        {
            {
                int fileNameLen = hkString::strLen(name);

                hkIo::WriteBuffer out(&m_connectSocket->getWriter());
                out.write32u(fileNameLen + 1 + (2 * sizeof(int)));
                out.write32u(FILE_STAT);
                out.write32u(fileNameLen);
                out.writeRaw(name, fileNameLen + 1);
            }

            // see if it has it
            if (m_connectSocket->isOk())
            {
                hkIo::ReadBuffer in(&m_connectSocket->getReader());
                int packetSize = in.read32u();
                if (in.prefetch(packetSize) < packetSize)
                {
                    Log_Debug("short read");
                    return HK_FAILURE;
                }
                int cmd = in.read32u();
                while (m_connectSocket->isOk())
                {
                    if (cmd == SEND_STAT)
                    {
                        hkInt64 size = in.read64i();
                        hkUint64 timestamp = in.read64u();
                        hkFileSystem::Entry::Flags flags = in.read32u();

                        entryOut.setPath(this, name);
                        entryOut.setSize(size);
                        entryOut.setMtime(timestamp);
                        entryOut.setFlags(flags);
                        return HK_SUCCESS;
                    }
                    else if (cmd == NOT_FOUND)
                    {
                        Log_Debug("not found");
                        break;
                    }
                    else
                    {
                        Log_Warning("got crazy stuff from file read socket..: {}", cmd).setId(0x32534);
                        if (packetSize < 0x100000)
                        {
                            in.skip(packetSize - sizeof(int));
                        }
                        //loop
                    }
                }
            }
        }
    }
    return HK_FAILURE;
}

_Ret_z_ const char* hkServerFileSystem::getOperatingSystemPath(_In_z_ const char* pathIn, hkStringBuf& pathOut, OSPathFlags flags )
{
    pathOut = m_localPathPrefix; // XX getOperatingSystemPath is typically used for say dll loading. we don't know if packaged etc or not, so orig path is best guess along with a user set prefix as could be from any mount etc
    pathOut.pathAppend(pathIn);
    return pathOut.cString();
}

namespace
{
    struct ServerIterator : public hkFileSystem::Iterator::Impl
    {
        ServerIterator(_In_opt_z_ const char* dirPath = HK_NULL, _In_opt_z_ const char* wildcard = HK_NULL)
        : m_idx(-1)
        , m_listing(HK_NULL, dirPath)
        , m_wildcard(wildcard)
        {}

        virtual bool advance(hkFileSystem::Entry& e)
        {
            m_idx++;
            for ( ; m_idx < m_listing.getEntries().getSize(); m_idx++)
            {
                e = m_listing.getEntries()[m_idx];
                if( hkFileSystem::Iterator::nameAcceptable(e.getName(), m_wildcard) )
                {
                    return true;
                }
            }

            return false;
        }

        int m_idx;
        hkFileSystem::DirectoryListing m_listing;
        hkStringPtr m_wildcard;
    };
}

hkRefNew<hkFileSystem::Iterator::Impl> hkServerFileSystem::createIterator(_In_z_ const char* basePath, _In_z_ const char* wildcard)
{
    Log_Scope(Dev,"createIterator '{}'", basePath);
    if (m_connectSocket && !m_connectSocket->isOk())
    {
        closeConnection();
    }

    if (!m_connectSocket)
    {
        if (m_mode & (WAIT_FOR_CONNECT | WAIT_FOR_CONNECT_TO_SERVER))
        {
            waitForConnection();
        }
        else // single listen
        {
            tryForConnection();
        }
    }

    ServerIterator* iter = new ServerIterator(basePath, wildcard);
    DirectoryListing& listingOut = iter->m_listing;
    listingOut.setFs(this);

    hkCriticalSectionLock socketLock(&m_connectionLock);
    if (m_connectSocket && m_connectSocket->isOk())
    {
        int pathLen = hkString::strLen(basePath);
        hkOArchive out(&m_connectSocket->getWriter());
        out.write32u( pathLen + 1 + (2* sizeof(int)) );
        out.write32u( DIR_LIST );
        out.write32u( pathLen );
        out.writeRaw( basePath, pathLen + 1 );

        // see if it has it
        hkIArchive in(&m_connectSocket->getReader());
        if (m_connectSocket->isOk())
        {
            int packetSize = in.read32u();
            int cmd = in.read32u();
            while (m_connectSocket->isOk())
            {
                if (cmd == SEND_DIR_LIST)
                {
                    int numEntries = in.read32u();
                    
                    //Log_Info("Found path [" << basePath << "] on server, listing " << numEntries << " items");
                    listingOut.clear();
                    hkArray<char> nameBuffer;
                    for (int ne=0; ne < numEntries; ++ne)
                    {
                        int nameLen = in.read32u();
                        hkInt64 fileSize = in.read64();
                        bool isDir = in.read8u() > 0;
                        if (nameBuffer.getSize() <= nameLen  )
                        {
                            nameBuffer.setSize(nameLen + 1);
                        }
                        in.readRaw(nameBuffer.begin(), nameLen + 1); // incl null term
                        if (m_connectSocket->isOk())
                        {
                            if (isDir)
                            {
                                listingOut.addDirectory( nameBuffer.begin() );
                            }
                            else
                            {
                                listingOut.addFile( nameBuffer.begin(), TimeStamp()  /* time */, fileSize );
                            }
                        }
                        else
                        {
                            break;
                        }
                    }

                    if (m_connectSocket->isOk())
                    {
                        return static_cast<hkFileSystem::Iterator::Impl*>(iter);
                    }
                    else
                    {
                        break; // try local..?
                    }
                }
                else if (cmd == NOT_FOUND)
                {
                    Log_Debug("not found");
                    break;
                }
                else
                {
                    Log_Warning("got crazy stuff from listDirectory socket..: ", cmd).setId(0xdafa3);
                    if (packetSize < 0x100000)
                    {
                        char* ignoreData = hkAllocate<char>( packetSize, HK_MEMORY_CLASS_BASE);
                        in.readRaw(ignoreData, packetSize - sizeof(int));
                        hkDeallocate<char>( ignoreData);
                    }
                    //loop
                }
            }
        }
    }
    return static_cast<hkFileSystem::Iterator::Impl*>(iter);
}

namespace
{

class ServerWatcher : public hkFileSystem::Watcher
{
public:
    ServerWatcher(hkServerFileSystem& fs, hkUint32 id) : m_serverFs(fs), m_id(id) {}

    ~ServerWatcher()
    {
        m_serverFs.removeWatcher(m_id);
    }

    virtual hkResult getCompletedChanges(hkArray<Change>& changesOut) HK_OVERRIDE
    {
        return m_serverFs.getCompletedChanges(m_id, changesOut);
    }

private:
    hkServerFileSystem& m_serverFs;
    hkUint32 m_id;
};

}

hkRefNew<hkFileSystem::Watcher> hkServerFileSystem::createWatcher(_In_z_ const char* topDirectory )
{
    Log_Scope(Info, "createWatcher '{}'", topDirectory);
    if (m_connectSocket && !m_connectSocket->isOk())
    {
        closeConnection();
    }

    if (!m_connectSocket)
    {
        if (m_mode & (WAIT_FOR_CONNECT | WAIT_FOR_CONNECT_TO_SERVER))
        {
            waitForConnection();
        }
        else // single listen
        {
            tryForConnection();
        }
    }

    hkCriticalSectionLock socketLock(&m_connectionLock);
    int dirLen = hkString::strLen(topDirectory);
    hkOArchive out(&m_connectSocket->getWriter());
    out.write32u( dirLen + 1 + 2 * sizeof(hkUint32) );
    out.write32u( WATCH );
    out.write32u( dirLen );
    out.writeRaw( topDirectory, dirLen + 1 );

    hkIArchive in(&m_connectSocket->getReader());
    if (m_connectSocket->isOk())
    {
        int packetSize = in.read32u();
        int cmd = in.read32u();
        while (m_connectSocket->isOk())
        {
            if (cmd == ACK )
            {
                hkUint32 id = in.read32u();
                Log_Debug("id '{}'", id);
                return new ServerWatcher(*this, id);
            }
            else
            {
                Log_Warning("got crazy stuff from file write socket..: ", cmd ).setId(0x32534);
                if (packetSize < 0x100000)
                {
                    char* ignoreData = hkAllocate<char>( packetSize, HK_MEMORY_CLASS_BASE);
                    in.readRaw(ignoreData, packetSize - sizeof(int));
                    hkDeallocate<char>(ignoreData);
                }
                //loop
            }
        }
    }
    return HK_NULL;
}

void hkServerFileSystem::removeWatcher( hkUint32 id )
{
    if (m_connectSocket && !m_connectSocket->isOk())
    {
        closeConnection();
    }

    if (!m_connectSocket)
    {
        if (m_mode & (WAIT_FOR_CONNECT | WAIT_FOR_CONNECT_TO_SERVER))
        {
            waitForConnection();
        }
        else // single listen
        {
            tryForConnection();
        }
    }

    hkCriticalSectionLock socketLock(&m_connectionLock);

    hkOArchive out(&m_connectSocket->getWriter());
    out.write32u( 2 * sizeof(hkUint32) );
    out.write32u( UNWATCH );
    out.write32u( id );

    hkIArchive in(&m_connectSocket->getReader());
    if (m_connectSocket->isOk())
    {
        int packetSize = in.read32u();
        int cmd = in.read32u();
        while (m_connectSocket->isOk())
        {
            if (cmd == ACK )
            {
                return;
            }
            else if (cmd == NOT_FOUND )
            {
                Log_Warning("removeWatcher unable to remove remote watcher with id '{}'", id).setId(0x32534);
                return;
            }
            else
            {
                Log_Warning("removeWatcher got crazy stuff from file write socket..: ", cmd ).setId(0x32534);
                if (packetSize < 0x100000)
                {
                    char* ignoreData = hkAllocate<char>( packetSize, HK_MEMORY_CLASS_BASE);
                    in.readRaw(ignoreData, packetSize - sizeof(int));
                    hkDeallocate<char>(ignoreData);
                }
                //loop
            }
        }
    }
}

hkResult hkServerFileSystem::getCompletedChanges(hkUint32 id, hkArray<Watcher::Change>& changesOut)
{
    if (m_connectSocket && !m_connectSocket->isOk())
    {
        closeConnection();
    }

    if (!m_connectSocket)
    {
        if (m_mode & (WAIT_FOR_CONNECT | WAIT_FOR_CONNECT_TO_SERVER))
        {
            waitForConnection();
        }
        else // single listen
        {
            tryForConnection();
        }
    }

    hkCriticalSectionLock socketLock(&m_connectionLock);
    hkOArchive out(&m_connectSocket->getWriter());
    out.write32u( 2 * sizeof(hkUint32) );
    out.write32u( CHANGES );
    out.write32u( id );

    hkIArchive in(&m_connectSocket->getReader());
    if (m_connectSocket->isOk())
    {
        int packetSize = in.read32u();
        int cmd = in.read32u();
        while (m_connectSocket->isOk())
        {
            if (cmd == SEND_CHANGES)
            {
                hkUint32 numChanges = in.read32u();
                hkArray<char> buffer;
                for (hkUint32 c = 0; c < numChanges; ++c)
                {
                    Watcher::Change& change = changesOut.expandOne();
                    change.m_change = (Watcher::Change::ChangeType)in.read8u();

                    int newSize = in.read32u();
                    if (buffer.getSize() <= newSize)
                    {
                        buffer.setSize(newSize + 1);
                    }
                    in.readRaw(buffer.begin(), newSize + 1); // incl null term
                    change.m_fileName = buffer.begin();

                    if (change.m_change == Watcher::Change::Renamed)
                    {
                        int oldSize = in.read32u();
                        if (buffer.getSize() <= oldSize)
                        {
                            buffer.setSize(oldSize + 1);
                        }
                        in.readRaw(buffer.begin(), oldSize + 1); // incl null term
                        change.m_oldFileName = buffer.begin();
                    }
                }
                return HK_SUCCESS;
            }
            if (cmd == NOT_FOUND)
            {
                Log_Warning("getCompletedChanges Unable to get changes from remote watcher with id '{}'", id ).setId(0xabbaa606);
                return HK_FAILURE;
            }
            else
            {
                Log_Warning("getCompletedChanges Got crazy stuff from file write socket..: '{}'", cmd).setId(0x32534);
                if (packetSize < 0x100000)
                {
                    char* ignoreData = hkAllocate<char>( packetSize, HK_MEMORY_CLASS_BASE);
                    in.readRaw(ignoreData, packetSize - sizeof(int));
                    hkDeallocate<char>(ignoreData);
                }
                //loop
            }
        }
    }
    return HK_FAILURE;
}

void hkServerFileSystem::setServerAddress(_In_z_ const char* addr)
{
    m_serverAddress = addr;
}

hkResult hkServerFileSystem::remove(_In_z_ const char* name )
{
    return removeOrMkdir(name, REMOVE);
}

hkResult hkServerFileSystem::mkdir(_In_z_ const char* path, CreateFlag /*flag*/)
{
    return removeOrMkdir(path, MKDIR);
}

// These are implemented almost identically, so don't duplicate code
hkResult hkServerFileSystem::removeOrMkdir(_In_z_ const char* path, OutCommands command )
{
    HK_ASSERT_NO_MSG(0x3ac2f0dc, command == MKDIR || command == REMOVE);

    if (m_mode & VIRTUAL_READ)
    {
        if (m_connectSocket && !m_connectSocket->isOk())
        {
            closeConnection();
        }

        if (!m_connectSocket)
        {
            if (m_mode & (WAIT_FOR_CONNECT | WAIT_FOR_CONNECT_TO_SERVER))
            {
                waitForConnection();
            }
            else // single listen
            {
                tryForConnection();
            }
        }

        hkCriticalSectionLock socketLock(&m_connectionLock);
        if (m_connectSocket && m_connectSocket->isOk())
        {
            int fileNameLen = hkString::strLen(path);
            hkOArchive out(&m_connectSocket->getWriter());
            out.write32u( fileNameLen + 1 + (3 * sizeof(int)) );
            out.write32u( command );
            out.write32u( fileNameLen );
            out.writeRaw( path, fileNameLen + 1 );

            // see if it has succeeded
            hkIArchive in(&m_connectSocket->getReader());
            if (m_connectSocket->isOk())
            {
                /*int packetSize =*/ in.read32u();
                int cmd = in.read32u();
                if (cmd == ACK)
                {
                    return HK_SUCCESS;
                }
            }
        }
    }

    return HK_FAILURE;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
