// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/System/Io/FileSystem/Cache/hkCacheFileSystem.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>

#define DEBUG_LOG_IDENTIFIER "io.CacheFileSystem"
#include <Common/Base/System/Log/hkLog.hxx>

#if defined(HK_PLATFORM_WINRT) || (_WIN32_WINNT >= 0x0600 )
#include <Common/Base/Fwd/hkwindows.h>
#include <Common/Base/Container/String/hkUtf8.h>
#include <Common/Base/System/Platform/hkWinUtil8.h>
#endif

hkCacheFileSystem::hkCacheFileSystem(_In_ hkFileSystem* source, _In_ hkFileSystem* cache, _In_opt_z_ const char* cachePrefix)
    : m_source(source), m_cache(cache), m_cachePrefix(HK_NULL)
{
    if (cachePrefix)
    {
        hkStringBuf sb = cachePrefix;
        sb.pathNormalize();
        sb.append("/");
        m_cachePrefix = sb;
    }
}

hkCacheFileSystem::~hkCacheFileSystem()
{
}

// We need to specially handle absolute paths etc so that
// we don't generate invalid paths for the cached copy.
static void s_getCachedName(_In_z_ const char* in, hkStringBuf& out, _In_opt_z_ const char* cachePrefix)
{
    out = in;
    out.pathNormalize();
    if (out.startsWith("//"))
    {
        out.chompStart(2);
        out.prepend("$smb$"); // smb network path
    }
    else if (out.getLength() >= 2 && out[1] == ':')
    {
        out.replace(':', '_');
        out.prepend("$abs$"); // absolute path
    }
    else
    {
        out.prepend("$rel$"); // relative path
    }
    if( cachePrefix )
    {
        out.prepend(cachePrefix);
    }
}


hkRefNew<hkStreamReader> hkCacheFileSystem::_openReader(_In_z_ const char* sourceName, OpenFlags mode)
{
    HK_TIME_CODE_BLOCK("hkCacheFileSystem::openReader", HK_NULL);

    hkStringBuf cacheName;
    s_getCachedName(sourceName, cacheName, m_cachePrefix);
    Log_Scope(Info, "openReader name='{}', cache='{}'", sourceName, cacheName.cString());
    // try cache first
    {
        Entry cacheEntry;
        if (m_cache->stat(cacheName, cacheEntry).isSuccess())
        {
            Log_Info("exists in cache mtime={}", cacheEntry.getMtime().get());
            Entry cacheSource;
            if (m_source == HK_NULL)
            {
                return cacheEntry.openReader(mode); // CACHE HIT
            }
            else if (m_source->stat(sourceName, cacheSource).isSuccess())
            {
                Log_Info("exists in source mtime={}", cacheSource.getMtime().get());
                if (cacheEntry.getMtime() >= cacheSource.getMtime())
                {
                    Log_Info("using cached");
                    
                    return cacheEntry.openReader(mode); // CACHE HIT
                }
                Log_Info("cache out of date");
            }
            else // we have a local copy, out of date or but missing on server
            {
                Log_Info("removing stale file");
                m_cache->remove(cacheName);
                return HK_NULL;
            }
        }
        // else not in cache
    }

    // fallback to source
    if (m_source)
    {
        if( hkRefPtr<hkStreamReader> sourceReader = m_source->openReader(sourceName, mode) )
        {
            // create cache parent dirs
            {
                hkStringBuf sb(cacheName); sb.pathDirname();
                hkArray<hkStringPtr> todo;
                hkFileSystem::Entry e;
                while( sb.getLength() && m_cache->stat(sb, e).isFailure() )
                {
                    todo.expandOne() = sb.cString();
                    sb.pathDirname();
                }
                if( todo.getSize() )
                {
                    Log_Info("creating folder structure");
                    for (int i = todo.getSize() - 1; i >= 0; --i)
                    {
                        if(m_cache->mkdir(todo[i], ALLOW_EXISTING).isFailure())
                        {
                            HK_ASSERT_NO_MSG(0x5425c8c, 0);
                            return HK_NULL;
                        }
                    }
                }
            }

            // save a copy
            {
                Log_Info("caching file locally");
                if ( hkRefPtr<hkStreamWriter> cacheWriter = m_cache->openWriter(cacheName) )
                {
                    char buf[4096];
                    while (int r = sourceReader->read(buf, sizeof(buf)))
                    {
                        int w = cacheWriter->write(buf, r);
                        if (r != w)
                        {
                            Log_Warning("Error in copying file from {} to {}", sourceName, cacheName);
                            return HK_NULL;
                        }
                    }

                    cacheWriter = nullptr;

                    // change mtime of the cache file to the original one (even if server time is not the same as local)
                    //
                    // todo: make servers handshake, take server time and unique hash, every server should have unique hash by which we will mark every cache file.
                    // it will be unique cache for every server/user/version (or make global crc check)
#if defined(HK_PLATFORM_WINRT) || (_WIN32_WINNT >= 0x0600 )
                    {
                        hkUint64 HK_TIMESTAMP_NSEC100_TO_UNIX_EPOCH = 116444736000000000UI64;
                        hkStringBuf srcOSpath;
                        hkFileSystem::Entry srcEntry;
                        hkResult srcStatResult = m_source->stat(sourceName, srcEntry);
                        TimeStamp srcLastWriteTime = srcEntry.getMtime();
                        if (srcStatResult.isSuccess())
                        {
                            hkStringBuf cacheOSpath;
                            if (m_cache->getOperatingSystemPath(cacheName, cacheOSpath, OS_PATH_ABSOLUTE) != HK_NULL)
                            {
                                hkUtf8::WideFromUtf8 cacheUnicodePath(cacheOSpath);
                                HANDLE hCacheFile = CreateFile2(cacheUnicodePath.cString(), GENERIC_WRITE, FILE_SHARE_WRITE, OPEN_EXISTING, NULL);
                                if (hCacheFile != INVALID_HANDLE_VALUE)
                                {
                                    FILETIME ftWrite;

                                    // we keep entry's mtime in unix timestamp format, so we need to convert it to windows time before updating file's last-write-time
                                    //
                                    // convert and store source time to ftWrite
                                    {
                                        hkUint64 srcTime = srcLastWriteTime.get();
                                        srcTime /= 100;
                                        srcTime += HK_TIMESTAMP_NSEC100_TO_UNIX_EPOCH;
                                        ftWrite.dwLowDateTime = (srcTime & 0xFFFFFFFFUI64);
                                        ftWrite.dwHighDateTime = (srcTime >> 32);
                                    }

                                    if (!SetFileTime(hCacheFile, HK_NULL, HK_NULL, &ftWrite))
                                    {
                                        hkStringBuf temp;
                                        Log_Warning("Can't set last-write-time to cache file {}, error message is {}", cacheOSpath.cString(), hkWinUtil::formatHResult(GetLastError(), temp));
                                    }

                                    CloseHandle(hCacheFile);
                                }
                                else
                                {
                                    hkStringBuf temp;
                                    Log_Warning("Can't open cache file for writing {}, error message is {}", cacheOSpath.cString(), hkWinUtil::formatHResult(GetLastError(), temp));
                                }
                            }
                        }
                    }
#endif
                }
            }

            // return the copy
            return m_cache->openReader(cacheName, mode);
        }
    }
    return HK_NULL;
}

hkRefNew<hkStreamWriter> hkCacheFileSystem::_openWriter(_In_z_ const char* name, OpenFlags mode )
{
    return m_source->openWriter(name, mode);
}

hkResult hkCacheFileSystem::stat(_In_z_ const char* name, hkFileSystem::Entry& entryOut )
{
    HK_TIME_CODE_BLOCK("hkCacheFileSystem::stat", HK_NULL);
    hkResult res = m_source->stat(name, entryOut);
    if (res.isSuccess())
    {
        entryOut.setPath(this, entryOut.getPath());
    }
    return res;
}

_Ret_z_ const char* hkCacheFileSystem::getOperatingSystemPath(_In_z_ const char* pathIn, hkStringBuf& pathOut, OSPathFlags flags)
{
    return m_source->getOperatingSystemPath(pathIn, pathOut, flags);
}


namespace
{
        // We don't pass the parent iterator directly since that would put us out of the loop.
    struct CacheIterator : public hkFileSystem::Iterator::Impl
    {
        HK_DECLARE_CLASS(CacheIterator, New);
        CacheIterator(_In_ hkFileSystem* fs, _In_ hkFileSystem::Iterator::Impl* p) : m_fs(fs), m_parent(p) {}

        bool advance(hkFileSystem::Entry& e) HK_OVERRIDE
        {
            bool b = m_parent->advance(e);
            if(b)
            {
                e.setPath(m_fs, e.getPath());
            }
            return b;
        }

        hkFileSystem* m_fs;
        hkRefPtr<hkFileSystem::Iterator::Impl> m_parent;
    };
}

hkRefNew<hkFileSystem::Iterator::Impl> hkCacheFileSystem::createIterator(_In_z_ const char* basePath, _In_z_ const char* wildcard)
{
    if( hkRefNew<Iterator::Impl> srcImpl = m_source->createIterator(basePath, wildcard))
    {
        return new CacheIterator(this, srcImpl.stealOwnership());
    }
    return HK_NULL;
}

hkRefNew<hkFileSystem::Watcher> hkCacheFileSystem::createWatcher(_In_z_ const char* topDirectory )
{
    return m_source->createWatcher(topDirectory);
}

hkResult hkCacheFileSystem::remove(_In_z_ const char* name )
{
    return m_source->remove(name);
}

hkResult hkCacheFileSystem::mkdir(_In_z_ const char* path, CreateFlag flag)
{
    return m_source->mkdir(path, flag);
}

hkResult hkCacheFileSystem::copy(_In_z_ const char* srcPath, _In_z_ const char* dstPath)
{
    return m_source->copy(srcPath, dstPath);
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
