// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : NX32 NX64
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <nn/nifm.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/socket.h>
#include <nn/fs.h>

#include <cstdlib>
#include <nn/nifm/nifm_ApiIpAddress.h>

namespace
{
    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;
    uintptr_t g_heapAddress;
#if defined(HK_ARCH_ARM_64)
    const size_t g_mallocMemorySize = 2000 * 1024 * 1024;
#else
    const size_t g_mallocMemorySize = 800 * 1024 * 1024;
#endif
    bool __platform_initialized = false;
}

namespace
{
    void* Allocate(size_t size)
    {
        return std::malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        std::free(p);
    }
} // namespace

#if !defined(HK_PLATFORM_NX_WIN32) && !defined(HK_PLATFORM_NX_X64)
extern "C" void nninitStartup()
{
    nn::os::MemoryInfo memInfo;
    nn::os::QueryMemoryInfo(&memInfo);

    size_t size = static_cast<size_t>(memInfo.totalAvailableMemorySize) - memInfo.totalUsedMemorySize;
    size = nn::util::align_down(size , nn::os::MemoryBlockUnitSize);
    nn::Result result = nn::os::SetMemoryHeapSize(size);
    if (!result.IsSuccess())
    {
        NN_LOG("nn::os::SetMemoryHeapSize() failed with err code: %d\n", result.GetDescription());
    }

    result = nn::os::AllocateMemoryBlock( &g_heapAddress, g_mallocMemorySize );
    if (!result.IsSuccess())
    {
        NN_LOG("nn::os::AllocateMemoryBlock() failed with err code: %d\n", result.GetDescription());
    }
    nn::init::InitializeAllocator( reinterpret_cast<void*>(g_heapAddress), g_mallocMemorySize );
}
#endif

void PlatformFileSystemInitNx()
{
    nn::fs::SetAllocator(Allocate, Deallocate);

    // Mount filesystem on host PC (allow paths like "C:\foo\bar")
    //nn::fs::MountHostRoot();

    // Mount the working directory as "host:"
    {
#if !defined(HK_PLATFORM_NX_WIN32) && !defined(HK_PLATFORM_NX_X64)
        char* pPath = nn::os::GetHostArgv()[0];
        char* pDir = strrchr(pPath, '\\');
        *(++pDir) = '\0';
#else
        const char* pPath = ".";
#endif

        nn::Result result = nn::fs::MountHost("host", pPath);
        if (!result.IsSuccess())
        {
            NN_LOG("nn::fs::MountHost(\"host\", pPath) failed with err code: %d\n", result.GetDescription());
        }
    }
}

void PlatformFileSystemQuitNx()
{
    nn::fs::Unmount( "host" );
    //nn::fs::UnmountHostRoot();
}

void PlatformAddDefaultMounts(hkFileSystem* basefs, hkUnionFileSystem* ufs)
{
    ufs->mount(basefs, "", "host:/", true);
}

void PlatformNetworkInitNx()
{
    nn::Result result = nn::Result();
    result = nn::nifm::Initialize();
    if( result.IsFailure() )
    {
        NN_LOG("\nError: nn::nifm::Initialize() failed (error %d).\n", result.GetDescription());
    }
    nn::nifm::SubmitNetworkRequest();

    while( nn::nifm::IsNetworkRequestOnHold() )
    {
        NN_LOG("Waiting for network interface availability.\n");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    if( !nn::nifm::IsNetworkAvailable() )
    {
        NN_LOG("\nError: The network is not available.\n");
        nn::nifm::CancelNetworkRequest();
    }

    result = nn::socket::Initialize(g_SocketConfigWithMemory);
    if( result.IsFailure() )
    {
        NN_LOG("\nError: nn::socket::Initialize() failed (error %d).\n\n", result.GetDescription());
        nn::nifm::CancelNetworkRequest();
        return;
    }

    in_addr outIpAddress = { 0 };
    nn::nifm::GetCurrentPrimaryIpAddress(&outIpAddress);
    NN_LOG("Server: listening for incoming messages at: %s\n",
            nn::socket::InetNtoa(outIpAddress));
}

void PlatformInit()
{
    if (__platform_initialized)
    {
        return;
    }
    __platform_initialized = true;
#if defined(HK_ARCH_ARM_64)
    hkUint64 result;
    asm volatile("mrs %[result], FPCR" :[result] "=r" (result));
    result |= (1 << 24);
    asm volatile("msr FPCR, %[src]" : : [src] "r" (result));
#elif defined(HK_ARCH_ARM)
    int result;
    asm volatile("vmrs %[result], FPSCR" :[result] "=r" (result));
    result |= (1 << 24);
    asm volatile("vmsr FPSCR, %[src]" : : [src] "r" (result));
#elif defined(HK_COMPILER_HAS_INTRINSICS_IA32)
    _MM_SET_FLUSH_ZERO_MODE( _MM_FLUSH_ZERO_ON );
#endif

    PlatformNetworkInitNx();
    PlatformFileSystemInitNx();
}

void PlatformQuit()
{
    if (!__platform_initialized)
    {
        return;
    }
    __platform_initialized = false;

    PlatformFileSystemQuitNx();

#if !defined(HK_PLATFORM_NX_WIN32) && !defined(HK_PLATFORM_NX_X64)
    nn::nifm::CancelNetworkRequest();

    nn::os::FreeMemoryBlock(g_heapAddress, g_mallocMemorySize);
#endif
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
