// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

// this: #include <Common/Base/System/Hardware/hkHardwareInfo.h>

/// Run-time hardware information.
/// This is a strict superset of compile-time information.
class HK_EXPORT_COMMON hkHardwareInfo : public hkReferencedObject
{
    public:

        /// Supported features.
        enum CpuFeature
        {
            CMPXCHG8,
            CMPXCHG16B,
            CMOV,
            CLFLUSH,
            PCLMULDQ,
            MOVBE,

            /// Most relevant intrinsics (not a complete list):
            /// - int _mm_popcnt__u32 (unsigned int v)
            /// - __int64 _mm_popcnt__u64 (unsigned __int64 v)
            POPCNT,

            LZCNT,
            F16C,
            FMA,
            AES,
            MMX,
            AVX,
            AVX2,
            SSE,
            SSE2,
            SSE3,

            /// Most relevant intrinsics (not a complete list):
            /// __m128i _mm_shuffle_epi8 (__m128i a, __m128i b)
            /// __m128i _mm_hadd_epi32 (__m128i a, __m128i b)
            SSSE3,

            SSE41,
            SSE42,
            FP16C,
            BMI1,
            BMI2,

            /// Dummy feature, always set.
            DUMMY,

            NUM_CPU_FEATURES
        };

    public:

        HK_DECLARE_CLASS(hkHardwareInfo, New, Singleton);

        /// Constructor.
        hkHardwareInfo();

        /// Return true if the feature set is supported, false otherwise.
        template <CpuFeature F0, CpuFeature F1 = DUMMY, CpuFeature F2 = DUMMY, CpuFeature F3 = DUMMY>
        struct HasFeature // This needs to be a class template, for VS2012 support
        {
            static HK_INLINE bool hasFeatureSet()
            {
                static const hkUint32 featureSet = hkUint32( ( 1 << F0 ) | ( 1 << F1 ) | ( 1 << F2 ) | ( 1 << F3 ) );
                static const bool value = ( hkHardwareInfo::getInstance().m_cpuFeatures & featureSet ) == featureSet;
                return value;
            }
        };

        /// Return true if the feature is supported, false otherwise.
        template <CpuFeature FEATURE>
        static HK_INLINE bool hasFeature()
        {
            return HasFeature<FEATURE>::hasFeatureSet();
        }

        /// Return true if the feature set is supported, false otherwise.
        static HK_INLINE bool hasFeatureSet( CpuFeature f0, CpuFeature f1 = DUMMY, CpuFeature f2 = DUMMY, CpuFeature f3 = DUMMY)
        {
            const hkUint32 featureSet = hkUint32( (1 << f0) | (1 << f1) | (1 << f2) | (1 << f3) );
            return (hkHardwareInfo::getInstance().m_cpuFeatures & featureSet) == featureSet;
        }

        /// Return the number of hardware threads.
        static HK_INLINE int getNumHardwareThreads()
        {
            return hkHardwareInfo::getInstance().m_numHardwareThreads;
        }

        /// Calculate the number of hardware threads.
        /// This method does not require the singleton to be initialized.
        static int calcNumHardwareThreads(_Out_opt_ int* m_numThreadsPerCpuOut = HK_NULL);

        /// Get a human-recognizable name for the hardware's architecture.
        static const char* getArchitectureName();

        /// Get a human-recognizable name for the hardware's SIMD support.
        static const char* getSimdSupportString();

    public:

        int         m_numHardwareThreads;    ///< Number of hardware Cpus 
        int         m_numThreadsPerCpu;      ///< set to 2  if hyperthreading is enabled
        hkUint32    m_cpuFeatures;           ///< Feature set.
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
