// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/System/Hardware/hkHardwareInfo.h>
#include <Common/Base/System/Platform/hkPlatformInfo.h>
#include <Common/Base/System/hkBaseSystem.h>
#include <Common/Base/Thread/Atomic/hkAtomicPrimitives.h>
#include <Common/Base/Object/hkSingleton.h>

#if defined(HK_PLATFORM_WIN32)
    #include <Common/Base/Fwd/hkwindows.h>
    typedef BOOL (__stdcall *tGetLogicalProcessorInformation)(OUT PSYSTEM_LOGICAL_PROCESSOR_INFORMATION Buffer,IN PDWORD ReturnLength);
#endif

#if defined(HK_PLATFORM_MAC) || defined(HK_PLATFORM_IOS)
    #include <sys/param.h>
    #include <sys/sysctl.h>
#endif

#if defined(HK_PLATFORM_LINUX) || defined(HK_PLATFORM_TIZEN)
    #include <unistd.h> 
#endif

#if defined(HK_PLATFORM_ANDROID)
    #include <Common/Base/System/Android/hkAndroidCpuInfo.h>
    #include <android/log.h>
    #include <unistd.h> // for sysconf 
#endif

#if defined(HK_PLATFORM_NX)
#   include <nn/os/os_ThreadApi.h>
#endif

HK_SINGLETON_IMPLEMENTATION(hkHardwareInfo);

// cpuid implementation
#if ( defined(HK_ARCH_IA32) || defined(HK_ARCH_X64) ) && !defined(HK_ARCH_ARM_64)
    #if defined(HK_PLATFORM_WIN32) || defined(HK_PLATFORM_NX)
    int __get_cpuid(int level, _Out_ unsigned* eax, _Out_ unsigned* ebx, _Out_ unsigned* ecx, _Out_ unsigned* edx)
        {
            int registers[4] = {0,0,0,0}; // EAX, EBX, ECX, EDX.
            __cpuid(registers, level);
            *eax = registers[0];
            *ebx = registers[1];
            *ecx = registers[2];
            *edx = registers[3];
            return registers[0];
        }
    #elif defined(HK_PLATFORM_LINUX) || defined(HK_PLATFORM_PS4) || defined(HK_PLATFORM_ANDROID) || defined(HK_PLATFORM_MAC)
        #include <cpuid.h>
    #else
        #error fixme
    #endif
#endif


hkHardwareInfo::hkHardwareInfo()
{
    HK_COMPILE_TIME_ASSERT( NUM_CPU_FEATURES <= (sizeof(m_cpuFeatures) * 8) );

    //
    // Gather information's about CPU features.
    //

    m_cpuFeatures = hkUint32(1 << DUMMY);
    m_numHardwareThreads = calcNumHardwareThreads( &m_numThreadsPerCpu );

    // Refs for Intel:
    // https://www.microbe.cz/docs/CPUID.pdf
    // https://www.intel.com/content/dam/www/public/us/en/documents/manuals/64-ia-32-architectures-software-developer-vol-2a-manual.pdf
    // http://msdn.microsoft.com/en-us/library/hskdteyh(v=vs.90).aspx
    // http://en.wikipedia.org/wiki/CPUID
    //
    // Refs for AMD:
    // http://amd-dev.wpengine.netdna-cdn.com/wordpress/media/2012/10/25481.pdf

#if ( defined(HK_ARCH_IA32) || defined(HK_ARCH_X64) ) && defined(HK_SSE_VERSION) && (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED) && !defined(HK_ARCH_ARM_64)

    unsigned eax,ebx,ecx,edx;

    #define SET_FEATURE_BIT( _register_ , _bit_, _feature_ ) m_cpuFeatures |=  (((_register_) & ( 1 << _bit_ )) || false) ? (1 << (_feature_)) : 0

    // Fetch vendor information.
    __get_cpuid( 0, &eax, &ebx, &ecx, &edx);
    char    vendorString[32];
    ((unsigned*) vendorString)[0] = ebx;
    ((unsigned*) vendorString)[1] = edx;
    ((unsigned*) vendorString)[2] = ecx;
    ((unsigned*) vendorString)[3] = 0;

    const unsigned maxBaseInformationValue = eax;
    const bool isIntel = hkString::strCmp( vendorString, "GenuineIntel" ) == 0;
    const bool isAMD = hkString::strCmp( vendorString, "AuthenticAMD" ) == 0;

    if( isIntel || isAMD )
    {
        //
        // Feature set common to both Intel and AMD.
        //

        __get_cpuid( 1, &eax, &ebx, &ecx, &edx);

        SET_FEATURE_BIT( edx, 8,    CMPXCHG8 );
        SET_FEATURE_BIT( edx, 15,   CMOV );
        SET_FEATURE_BIT( edx, 19,   CLFLUSH );
        SET_FEATURE_BIT( edx, 25,   SSE );
        SET_FEATURE_BIT( edx, 26,   SSE2 );
        SET_FEATURE_BIT( edx, 23,   MMX );

        SET_FEATURE_BIT( ecx, 0,    SSE3 );
        SET_FEATURE_BIT( ecx, 1,    PCLMULDQ );
        SET_FEATURE_BIT( ecx, 9,    SSSE3 );
        SET_FEATURE_BIT( ecx, 13,   CMPXCHG16B );
        SET_FEATURE_BIT( ecx, 19,   SSE41 );
        SET_FEATURE_BIT( ecx, 20,   SSE42 );
        SET_FEATURE_BIT( ecx, 22,   MOVBE );
        SET_FEATURE_BIT( ecx, 23,   POPCNT );
        SET_FEATURE_BIT( ecx, 25,   AES );
        SET_FEATURE_BIT( ecx, 12,   FMA );
        SET_FEATURE_BIT( ecx, 28,   AVX );
        SET_FEATURE_BIT( ecx, 29,   F16C );

        __get_cpuid( 0x80000001, &eax, &ebx, &ecx, &edx);

        SET_FEATURE_BIT( ecx, 5,    LZCNT );
    }

    if( isIntel )
    {
        //
        // Intel specific feature set.
        //

        if( maxBaseInformationValue >= 7 )
        {
            __get_cpuid( 7, &eax, &ebx, &ecx, &edx);

            SET_FEATURE_BIT( ebx, 5,    AVX2 );
            SET_FEATURE_BIT( ebx, 3,    BMI1 );
            SET_FEATURE_BIT( ebx, 8,    BMI2 );
        }
    }
    else if( isAMD )
    {
        //
        // AMD specific feature set.
        //
    }
    else
    {
        HK_WARN_ALWAYS(0x4FA4B98C, "CPU vendor information's ('" << vendorString << "') not recognized." );
    }

    #undef SET_FEATURE_BIT

#endif
}

int hkHardwareInfo::calcNumHardwareThreads(_Out_opt_ int* numThreadsPerCpuOut)
{
    int numHardwareCpus = 1;
    int numThreadsPerCpu = 1;

    #if defined(HK_PLATFORM_PSVITA)
        #if HK_CONFIG_THREAD==HK_CONFIG_MULTI_THREADED
            numHardwareCpus = 3;
        #else
            numHardwareCpus = 1;
        #endif

    #elif defined(HK_PLATFORM_PS4)
        numHardwareCpus = 6; // allowed 0x3f as user mask == first 6 cores

    #elif defined(HK_PLATFORM_WIIU)
        numHardwareCpus = HK_CONFIG_WIIU_NUM_THREADS;

    #elif defined(HK_PLATFORM_DURANGO)
        numHardwareCpus = 6; // OS reserves 2

    #elif defined(HK_PLATFORM_WIN32)
        // Use system info
        _SYSTEM_INFO lpSystemInfo;
        #if defined(HK_PLATFORM_WINRT)
            GetNativeSystemInfo(&lpSystemInfo);
        #else
            GetSystemInfo(&lpSystemInfo);
        #endif

        numHardwareCpus = lpSystemInfo.dwNumberOfProcessors;

        #if !defined(HK_PLATFORM_WINRT) && !defined( HK_PLATFORM_DURANGO )
            // Load this dynamically as GetLogicalProcessorInformation is only available on XP SP3 and later systems
            HINSTANCE hKernel32Dll;
            tGetLogicalProcessorInformation pGetLogicalProcessorInformation = HK_NULL;
            // Load kernel32.dll to see if GetLogicalProcessorInformation is available
            hKernel32Dll = LoadLibraryA( "kernel32.dll");

            if( hKernel32Dll )
            {
                pGetLogicalProcessorInformation = (tGetLogicalProcessorInformation) GetProcAddress(hKernel32Dll, "GetLogicalProcessorInformation");
            }

            if( pGetLogicalProcessorInformation )
            {
                // refine to dinstinguish hyperthreads and real cores
                DWORD length=0;

                pGetLogicalProcessorInformation( HK_NULL, &length );

                // Allocate on stack since heap may not be initialized yet - max 128 entries  = 3k
                char buffer[ 128 * sizeof( _SYSTEM_LOGICAL_PROCESSOR_INFORMATION ) ];
                if( length <= sizeof( buffer ) )
                {
                    pGetLogicalProcessorInformation( (PSYSTEM_LOGICAL_PROCESSOR_INFORMATION)buffer, &length );
                    PSYSTEM_LOGICAL_PROCESSOR_INFORMATION proc = (PSYSTEM_LOGICAL_PROCESSOR_INFORMATION)buffer;

                    // Count real cores
                    DWORD bufferOffset = 0;
                    int cores = 0;
                    while( bufferOffset < length )
                    {
                        if(proc->Relationship == RelationProcessorCore )
                        {
                            cores++;
                        }
                        bufferOffset += sizeof(SYSTEM_LOGICAL_PROCESSOR_INFORMATION);
                        proc++;
                    }
                    numThreadsPerCpu = numHardwareCpus / cores;
                    numHardwareCpus = cores;
                }
            }

            // We don't need this anymore
            if( hKernel32Dll )
            {
                FreeLibrary(hKernel32Dll);
            }

        #else

            // Currently no way to query this on WinRT
            // Need to implement our own cpuid calls, but with Core i7 etc that is tricky as it reports max num possible in package so the code to query actual cores is more involved.
            if( numHardwareCpus > 6 )
            {
                numThreadsPerCpu = 2;
                numHardwareCpus /= 2;
            }

        #endif

    #elif defined(HK_PLATFORM_LINUX) || defined(HK_PLATFORM_TIZEN)
        numHardwareCpus = sysconf(_SC_NPROCESSORS_ONLN);

    #elif defined(HK_PLATFORM_MAC) || ( (HK_CONFIG_THREAD==HK_CONFIG_MULTI_THREADED) && defined(HK_PLATFORM_IOS) )
        numHardwareCpus = 1;
        size_t size=sizeof(numHardwareCpus);
        sysctlbyname("hw.physicalcpu",&numHardwareCpus,&size,NULL,0);

    #elif (HK_CONFIG_THREAD==HK_CONFIG_MULTI_THREADED) && defined(HK_PLATFORM_ANDROID)
        numHardwareCpus = hkAndroidGetCpuCount();
        static bool reported = false;
        if( !reported )
        {
            __android_log_print(ANDROID_LOG_INFO, "Havok", "Found %d Cpu Cores (sysconf says %d)", numHardwareCpus, (int)sysconf(_SC_NPROCESSORS_ONLN) );
            //reported = true;
        }
    #elif defined(HK_PLATFORM_NX)
        unsigned numCores = 0;
        nn::Bit64 coreMask = nn::os::GetThreadAvailableCoreMask();
        while (coreMask != 0)
        {
            numCores += 0x01 & coreMask;
            coreMask >>= 1;
        }
        numHardwareCpus = numCores;

        #if defined(HK_PLATFORM_NX_WIN32) || defined(HK_PLATFORM_NX_X64)
        if (numHardwareCpus > 3)
        {
            numHardwareCpus = 3;
        }
        #endif

    #endif

    if (numThreadsPerCpuOut)
    {
        *numThreadsPerCpuOut = numThreadsPerCpu;
    }

    return numHardwareCpus;
}

const char* hkHardwareInfo::getArchitectureName()
{
    return HK_ARCH_STRING;
}

const char* hkHardwareInfo::getSimdSupportString()
{
    return HK_SIMD_STRING;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
