// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT    : COMMON
// VISIBILITY : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Types/hkVarArgs.h>
#include <Common/Base/PreProcessor/hkPreProcessorGetLocation.h>

/// Class with static methods to provide hooks to error reporting functions.
/// You can redirect all asserts, errors and warnings by setting the handlers
/// at run time. Asserts and Warnings are only compiled into debug builds.
/// Errors are compiled in all builds. See hkDefaultError.h for a sample implementation
/// of the handlers.
class HK_EXPORT_COMMON hkError : public hkReferencedObject
{
public:
    HK_DECLARE_CLASS(hkError, New, Singleton);
    enum Message
    {
        /// Pass to setMinimumMessageLevel to ensure all messages are handled.
        MESSAGE_ALL,

        /// Diagnostic reports
        MESSAGE_REPORT = MESSAGE_ALL,

        /// Warnings: maybe be important e.g. for performance but not critical
        MESSAGE_WARNING,

        /// Asserts: indicates an error condition. Ignoring these may crash soon afterwards or corrupt data.
        MESSAGE_ASSERT,

        /// Error: unable to reasonably continue. Can still be disabled or continued but generally crashes immediately.
        MESSAGE_ERROR,

        /// Fatal error: will always terminate after logging. Cannot be disabled.
        MESSAGE_FATAL_ERROR,

        /// Pass to setMinimumMessageLevel to ignore all messages.
        MESSAGE_NONE,
    };

    /// Return value indicates whether or not to trigger an HK_BREAKPOINT for errors and asserts.
    virtual int message(Message m, int id, _In_z_ const char* description, _In_z_ const char* file, int line) = 0;

    /// Enables/disables diagnostic by id.
    virtual void setEnabled(int id, hkBool enabled) = 0;

    /// Check whether diagnostic is enabled/disabled by id.
    virtual hkBool isEnabled(int id) = 0;

    /// Sets the minimum message level. Messages below this level are ignored.
    virtual void setMinimumMessageLevel(Message msg) {}

    /// Check whether diagnostic is enabled/disabled by Message type.
    virtual Message getMinimumMessageLevel() { return MESSAGE_ALL; }

    /// Force all diagnostics to be enabled.
    virtual void enableAll() = 0;

    /// Begin a new report section
    virtual void sectionBegin(int id, _In_z_ const char* sectionName) {}

    /// End the current report section
    virtual void sectionEnd() {}

    /// Out-of-line helper methods to reduce code size
    static HK_INLINE int HK_CALL messageReport(int id, _In_z_ const char* description, _In_z_ const char* file, int line) { return ::hkErrorFwd::messageReport(id, description, file, line); }
    static HK_INLINE int HK_CALL messageWarning(int id, _In_z_ const char* description, _In_z_ const char* file, int line) { return ::hkErrorFwd::messageWarning(id, description, file, line); }
    static HK_INLINE int HK_CALL messageAssert(int id, _In_z_ const char* description, _In_z_ const char* file, int line) { return ::hkErrorFwd::messageAssert(id, description, file, line); }
    static HK_INLINE int HK_CALL messageError(int id, _In_z_ const char* description, _In_z_ const char* file, int line) { return ::hkErrorFwd::messageError(id, description, file, line); }
};

/// Convenience class to disable one error ID within a scope.
class HK_EXPORT_COMMON hkDisableError
{
public:

    HK_DECLARE_CLASS(hkDisableError, New);

    hkDisableError(int id)
    {
        hkError& error = hkError::getInstance();
        m_id = id;
        m_wasEnabled = error.isEnabled(id);
        error.setEnabled(id, false);
    }
    ~hkDisableError()
    {
        hkError& error = hkError::getInstance();
        error.setEnabled(m_id, m_wasEnabled);
    }
public:
    int m_id;
    hkBool m_wasEnabled;
};

class HK_EXPORT_COMMON hkErrStream : public hkOstream
{
public:
    HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE);
    hkErrStream(_Inout_updates_bytes_(bufSize) void* buf, int bufSize);
    ~hkErrStream();
};

typedef void (HK_CALL *hkErrorReportFunction)(_In_z_ const char* msg, _Inout_opt_ void* errorReportObject);

#define HK_REPORT_SECTION_BEGIN(id, name)    hkError::getInstance().sectionBegin(id, name);

#define HK_REPORT_SECTION_END()              hkError::getInstance().sectionEnd();


// Asserts, errors, and warnings may be compiled out.
// Fatal errors are always present, even in Release, and terminate execution.

#if defined(_MANAGED)
    // Normally we want this aligned but we can't for managed code
#   define HK_ERROR_BUFFER_ALIGNMENT(...) __VA_ARGS__
#else
#   define HK_ERROR_BUFFER_ALIGNMENT(...) HK_ALIGN16(__VA_ARGS__)
#endif

#define HK_WARN_ALWAYS(id, TEXT)        HK_MULTILINE_MACRO_BEGIN                                                                                    \
                                        HK_ERROR_BUFFER_ALIGNMENT(char assertBuf[512]);                                                             \
                                        hkErrStream ostr(assertBuf, sizeof(assertBuf));                                                             \
                                        ostr << TEXT;                                                                                               \
                                        hkError::messageWarning(id, assertBuf, HK_CURRENT_FILE, __LINE__);                                          \
                                        HK_MULTILINE_MACRO_END

#define HK_FATAL_ERROR(id, ...)         hkErrorFwd::messageFatalError( id, HK_CURRENT_FILE, __LINE__, HK_VARARGS1_WRAP((__VA_ARGS__)) )

#define HK_ERROR(id, TEXT)              HK_MULTILINE_MACRO_BEGIN                                                                                    \
                                        HK_ERROR_BUFFER_ALIGNMENT(char assertBuf[512]);                                                             \
                                        hkErrStream ostr(assertBuf,sizeof(assertBuf));                                                              \
                                        ostr << TEXT;                                                                                               \
                                        HK_DETAIL_DIAG_MSVC_SUPPRESS(25041)                                                                         \
                                        if ( hkError::messageError(id, assertBuf, HK_CURRENT_FILE, __LINE__) )                                      \
                                        {                                                                                                           \
                                            HK_BREAKPOINT(id);                                                                                      \
                                        }                                                                                                           \
                                        HK_MULTILINE_MACRO_END

#define HK_REPORT2(id, TEXT)            HK_MULTILINE_MACRO_BEGIN                                                                                    \
                                        HK_ERROR_BUFFER_ALIGNMENT(char reportBuf[512]);                                                             \
                                        hkErrStream ostr(reportBuf,sizeof(reportBuf));                                                              \
                                        ostr << TEXT;                                                                                               \
                                        hkError::messageReport(id, reportBuf, HK_CURRENT_FILE, __LINE__);                                           \
                                        HK_MULTILINE_MACRO_END

#if defined HK_DEBUG

#    define HK_ASSERTV(id, a, ...)      HK_MULTILINE_MACRO_BEGIN                                                                                    \
                                        if ( HK_VERY_LIKELY(a) )                                                                                    \
                                        {                                                                                                           \
                                        }                                                                                                           \
                                        else                                                                                                        \
                                        {                                                                                                           \
                                            if ( hkErrorFwd::messageAssert(id, #a, HK_CURRENT_FILE, __LINE__, HK_VARARGS1_WRAP((__VA_ARGS__))) )    \
                                            {                                                                                                       \
                                                HK_BREAKPOINT(0);                                                                                   \
                                            }                                                                                                       \
                                        }                                                                                                           \
                                        HK_MULTILINE_MACRO_END

#    define HK_WARN(id, TEXT)           HK_WARN_ALWAYS(id, TEXT)

#    define HK_WARN_ONCE(id, TEXT)      HK_MULTILINE_MACRO_BEGIN                                                                                    \
                                        if ( HK_VERY_LIKELY(!hkError::getInstance().isEnabled(id)) )                                                \
                                        {                                                                                                           \
                                        }                                                                                                           \
                                        else                                                                                                        \
                                        {                                                                                                           \
                                            HK_ERROR_BUFFER_ALIGNMENT(char assertBuf[512]);                                                         \
                                            hkErrStream ostr( assertBuf, sizeof(assertBuf) );                                                       \
                                            ostr << TEXT;                                                                                           \
                                            hkError::messageWarning( id, assertBuf, HK_CURRENT_FILE, __LINE__);                                     \
                                            hkError::getInstance().setEnabled(id, false);                                                           \
                                        }                                                                                                           \
                                        HK_MULTILINE_MACRO_END

#else

#    define HK_WARN(id, a)
#    define HK_WARN_ONCE(id, a)
#    define HK_ASSERTV(id, a, ...)

#endif

#if (HK_NATIVE_ALIGNMENT==16) && !defined(HK_ALIGN_RELAX_CHECKS)
#    define HK_CHECK_ALIGN_NATIVE(ADDR) HK_ASSERT_NO_MSG(0xff00ff00, (hkUlong(ADDR) & 0xF) == 0 )
#elif (HK_NATIVE_ALIGNMENT==8) && !defined(HK_ALIGN_RELAX_CHECKS)
#    define HK_CHECK_ALIGN_NATIVE(ADDR) HK_ASSERT_NO_MSG(0xff00ff00, (hkUlong(ADDR) & 0x7) == 0 )
#else // 4
#    define HK_CHECK_ALIGN_NATIVE(ADDR) HK_ASSERT_NO_MSG(0xff00ff00, (hkUlong(ADDR) & 0x3) == 0 )
#endif
#define HK_CHECK_ALIGN16(ADDR) HK_ASSERT_NO_MSG(0xff00ff00, (hkUlong(ADDR) & HK_NATIVE_ALIGN_CHECK) == 0 )
#if defined(HK_ALIGN_RELAX_CHECKS) || defined(HK_PLATFORM_IOS_SIM)
// Some ARM platforms will only actually align on 8 bytes. This is ok as they also accept 8-byte aligned vectors
#    define HK_CHECK_ALIGN_REAL(ADDR) HK_ASSERT_NO_MSG(0xff00ff00, (hkUlong(ADDR) & (8-1)) == 0 )
#else
#    define HK_CHECK_ALIGN_REAL(ADDR) HK_ASSERT_NO_MSG(0xff00ff00, (hkUlong(ADDR) & (HK_REAL_ALIGNMENT-1)) == 0 )
#endif

#define HK_WARN_ON_DEBUG_IF(check,id,TEXT)         HK_ON_DEBUG(if (check) { HK_WARN(id,TEXT); } )
#define HK_WARN_ONCE_ON_DEBUG_IF(check,id,TEXT)    HK_ON_DEBUG(if (check) { HK_WARN_ONCE(id,TEXT); } )

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
