// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/System/Error/hkDefaultError.h>
#include <Common/Base/System/StackTracer/hkStackTracer.h>

hkDefaultError::hkDefaultError(_In_opt_ hkErrorReportFunction errorReportFunction, _Inout_opt_ void* errorReportObject)
    : m_errorFunction(errorReportFunction ? errorReportFunction : &hkBaseSystem::error),
    m_errorObject(errorReportObject),
    m_minimumMessageLevel(hkError::MESSAGE_ALL)
{
#if defined(HK_PLATFORM_IOS)
    setEnabled(0x46AEFCEE, false);
    setEnabled(0xF021D445, false);
    setEnabled(0x4314aad9, false);
    setEnabled(0x2db3d51a, false);
#elif defined(HK_PLATFORM_CTR) || defined(HK_PLATFORM_ANDROID)
    setEnabled(0x64211c2f, false);
#endif
}

void hkDefaultError::setEnabled(int id, hkBool enabled)
{
    hkCriticalSectionLock lock(&m_lock);

    if (enabled)
    {
        m_disabledAssertIds.remove(id);
    }
    else
    {
        if (id != 0)
        {
            m_disabledAssertIds.insert(id, 1);
        }
    }
}

hkBool hkDefaultError::isEnabled(int id)
{
    hkCriticalSectionLock lock(&m_lock);

    return m_disabledAssertIds.getWithDefault(id, 0) == 0;
}

void hkDefaultError::enableAll()
{
    hkCriticalSectionLock lock(&m_lock);

    m_minimumMessageLevel = hkError::MESSAGE_ALL;
    m_disabledAssertIds.clear();
}

void hkDefaultError::setMinimumMessageLevel(Message msg)
{
    m_minimumMessageLevel = msg;
}

hkError::Message hkDefaultError::getMinimumMessageLevel()
{
    return m_minimumMessageLevel;
}

#if defined(HK_PLATFORM_LINUX)
static const char s_showMessage_format[] = "%s:%d: [0x%08X] %s: %s\n";
static const char s_showMessage_formatNoId[] = "%s:%d: %s: %s\n";
#else
static const char s_showMessage_format[] = "%s(%d): [0x%08X] %s: %s\n";
static const char s_showMessage_formatNoId[] = "%s(%d): %s: %s\n";
#endif

void hkDefaultError::formMessage( hkError::Message msg, int id, _In_z_ const char* desc, _In_z_ const char* file, int line, hkStringBuf& messageOut ) const
{
    const char* what = "";
    getMessageTypeString( msg, &what );

    if ((id == 0) && (file == HK_NULL))
    {
        //Just a simple message
        messageOut.printf( "%s", desc );
    }
    else
    {
        if (id != (int)0xffffffff && id != 0)
        {
            messageOut.printf( s_showMessage_format, file, line, id, what, desc );
        }
        else
        {
            messageOut.printf( s_showMessage_formatNoId, file, line, what, desc );
        }
    }
    // make sure the string is always terminated properly
}

void hkDefaultError::showMessage(hkError::Message msg, int id, _In_z_ const char* desc, _In_z_ const char* file, int line) const
{
    hkStringBuf messageOut;
    formMessage( msg, id, desc, file, line, messageOut );

    ( *m_errorFunction )( messageOut, m_errorObject );

    if( shouldReportStackTrace( msg ) )
    {
        hkStackTracer tracer;

        hkUlong trace[128];
        int ntrace = tracer.getStackTrace(trace, HK_COUNT_OF(trace) );
        if (ntrace > 2)
        {
            (*m_errorFunction)("Stack trace is:\n", m_errorObject); // Marker for the build system in order to extract stack traces
            // first two frames are in this file.
            tracer.dumpStackTrace(trace + 2, ntrace - 2, m_errorFunction, m_errorObject);
            (*m_errorFunction)("Stack trace end\n", m_errorObject);
        }
    }
}

int hkDefaultError::message(hkError::Message msg, int id, _In_z_ const char* description, _In_opt_z_ const char* file, int line)
{
    hkCriticalSectionLock lock(&m_lock);

    if (id == -1 && m_sectionIds.getSize())
    {
        id = m_sectionIds.back();
    }

    if (msg < getMinimumMessageLevel())
    {
        return 0;
    }

    if (!isEnabled(id))
    {
        return 0;
    }

    showMessage(msg, id, description, file, line);

    return msg == MESSAGE_ASSERT || msg == MESSAGE_ERROR;
}

void hkDefaultError::getMessageTypeString(hkError::Message msg, _Outptr_result_z_ const char** msgType) const
{
    switch (msg)
    {
    case MESSAGE_REPORT:
        *msgType = "Report";
        break;
    case MESSAGE_WARNING:
        *msgType = "Warning";
        break;
    case MESSAGE_ASSERT:
        *msgType = "Assert";
        break;
    case MESSAGE_ERROR:
        *msgType = "Error";
        break;
    case MESSAGE_FATAL_ERROR:
        *msgType = "Fatal Error";
        break;
    case MESSAGE_NONE:
    default:
        HK_BREAKPOINT(0);
        break;
    }
}

bool hkDefaultError::shouldReportStackTrace( hkError::Message msg ) const
{
    switch ( msg )
    {
        case MESSAGE_ASSERT:
        case MESSAGE_ERROR:
            return true;
        default:
            return false;
    }
}

void hkDefaultError::sectionBegin(int id, _In_z_ const char* sectionName)
{
    hkCriticalSectionLock lock(&m_lock);

    m_sectionIds.pushBack(id);
}

void hkDefaultError::sectionEnd()
{
    hkCriticalSectionLock lock(&m_lock);

    m_sectionIds.popBack();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
