// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/PreProcessor/hkPreProcessorGetLocation.h>

/// \file

namespace hkAssert
{
    /// Function type for assert handlers.
    /// Called when an assertion fails.
    /// Should return true to trigger a breakpoint, false to ignore the assert.
    typedef bool (HK_CALL *AssertHandlerFunc)(int id, _In_z_ const char* file, unsigned int line, _In_z_ const char* function, _In_z_ const char* expression, _In_z_ const char* userErrorMsg);

    /// Called if the user disabled an assert manually (through the debugger) after waiting in the breakpoint.
    /// In this case no message should be shown, and only internal state should be set to ignore the assert in the future.
    typedef void (HK_CALL *DisableAssertFunc)(int id);

    HK_EXPORT_COMMON bool HK_CALL defaultAssertHandler( int id, _In_z_ const char* file, unsigned int line, _In_z_ const char* function, _In_z_ const char* expression, _In_z_ const char* userErrorMsg );

    HK_EXPORT_COMMON void HK_CALL defaultDisableAssert( int id );

    struct assertHandlers
    {
        AssertHandlerFunc m_assertHandler;
        DisableAssertFunc m_disableAssert;
    };

    /// This function allows to change the current assert handlers to custom handlers. Returns the previously active handlers.
    HK_EXPORT_COMMON assertHandlers HK_CALL setAssertHandlers( assertHandlers newHandlers );


    namespace Detail
    {

        /// Called by the assert macros whenever a check failed. Returns true if the user wants to trigger a breakpoint
        HK_EXPORT_COMMON bool HK_CALL assertFailed( int id, _In_z_ const char* file, unsigned int line, _In_z_ const char* function, _In_z_ const char* expression, _In_z_ const char* userErrorMsg, hkVarArgs::VaTypes argTypes, ... );

        /// Called by the assert macros whenever an assert has been disabled manually through the debugger
        HK_EXPORT_COMMON void HK_CALL disableAssert( int id );

#if (__cplusplus >= 201103L) || (defined(_MSC_VER) && (_MSC_VER >= 1900))
        constexpr int numPlaceholders(const char* fmt, int c = 0)
        {
            return fmt[0] == 0 ? c // check for end
                : fmt[0] != '{' ? numPlaceholders(fmt + 1, c) // not {, next char
                : numPlaceholders(fmt + 2, c + (fmt[1] != '{' ? 1 : 0)); // a placeholder {...} or quoted {{
        }
        #define HK_FORMAT_VARARG_CHECK(...) static_assert(HK_DETAIL_NUM_ARGS_MINUS1(__VA_ARGS__) == hkAssert::Detail::numPlaceholders(HK_DETAIL_FIRST(__VA_ARGS__)), "Number of '{}' placeholders does not match number of arguments")
#else
        #define HK_FORMAT_VARARG_CHECK(...)
#endif
    }
}

// If static code analysis is enabled, this should be changed to tag an expression as 'assume true for code analysis'
#define HK_CONDITION_IN_ASSERT(condition)

/// Macro to report a failure when that code is reached. This will ALWAYS be executed, even in release builds, therefore might crash the application (or trigger a debug break).
#define HK_REPORT_FAILURE(id, ...) \
    HK_MULTILINE_MACRO_BEGIN \
        HK_DETAIL_DIAG_MSVC_SUPPRESS(25041) \
        if (hkAssert::Detail::assertFailed(id, HK_CURRENT_FILE, HK_CURRENT_LINE, HK_CURRENT_FUNCTION, "", HK_VARARGS1_WRAP((__VA_ARGS__)) )) \
        { \
            HK_BREAKPOINT( id ); \
        } \
    HK_MULTILINE_MACRO_END

/// Macro to raise an error, if a condition is not met. Allows to write a message using printf style. This assert will be triggered, even in non-development builds and cannot be deactivated.
#define HK_ASSERT_IMPLEMENTATION(id, condition, ...) \
    HK_MULTILINE_MACRO_BEGIN \
        HK_CONDITION_IN_ASSERT( condition ); \
        if ( HK_VERY_LIKELY ( !!(condition) ) ) \
        { \
            /* More likely branch first, for better branch prediction */ \
        } \
        else \
        { \
            HK_FORMAT_VARARG_CHECK(__VA_ARGS__); \
            if ( hkAssert::Detail::assertFailed( id, HK_CURRENT_FILE, HK_CURRENT_LINE, HK_CURRENT_FUNCTION, #condition, HK_VARARGS1_WRAP((__VA_ARGS__)) ) ) \
            { \
                HK_BREAKPOINT( id ); \
            } \
        } \
    HK_MULTILINE_MACRO_END


/// This type of assert can be used to mark code as 'not (yet) implemented' and makes it easier to find it later on by just searching for these asserts.
#define HK_ASSERT_NOT_IMPLEMENTED(id) HK_REPORT_FAILURE(id, "Not implemented")

/// A version of HK_ASSERT that has an empty message
#define HK_ASSERT_NO_MSG(id, condition) HK_ASSERT(id, condition, "")
#define HK_ASSERT_SLOW_NO_MSG(id, condition) HK_ASSERT_SLOW(id, condition, "")


// Occurrences of HK_ASSERT_DEBUG are compiled out in non-debug (dev + release) builds
#if defined(HK_DEBUG_SLOW)
/// Macro to raise an error, if a condition is not met.
///
/// Compiled out in non-debug builds.
/// The condition is not evaluated, when this is compiled out, so do not execute important code in it.
#   define HK_ASSERT_DEBUG HK_ASSERT_IMPLEMENTATION

#   define HK_ASSERT_SLOW HK_ASSERT

#else
/// Macro to raise an error, if a condition is not met.
///
/// Compiled out in non-debug builds.
/// The condition is not evaluated, when this is compiled out, so do not execute important code in it.
#   define HK_ASSERT_DEBUG(id, condition, ...) HK_CONDITION_IN_ASSERT(condition)
/// For static code analysis, the assumption is that we will not ever need this in the 'slow' config.
#   define HK_ASSERT_SLOW(id, condition, ...)

#endif


// Occurrences of HK_ASSERT are compiled out in release builds
#if defined(HK_DEBUG) || defined(HK_DEBUG_SLOW)

/// Macro to raise an error, if a condition is not met.
///
/// Compiled out in non-development builds.
/// The condition is not evaluated, when this is compiled out, so do not execute important code in it.
#   define HK_ASSERT HK_ASSERT_IMPLEMENTATION

/// Macro to raise an error, if a condition is not met.
///
/// Compiled out in non-development builds, however the condition is always evaluated,
/// so you may execute important code in it.
#   define HK_VERIFY HK_ASSERT_IMPLEMENTATION

    #define HK_DETAIL_ASSERT_EXPR2(id,A)                       HK_ASSERT(id, A,                           #A "={}", A)
    #define HK_DETAIL_ASSERT_EXPR3(id,A,MSG)                   HK_ASSERT(id, A,                   MSG " " #A "={}", A)
    #define HK_DETAIL_ASSERT_EXPR4(id,A,op1,B)                 HK_ASSERT(id, A op1 B,                     #A "={} " #B "={}", A, B)
    #define HK_DETAIL_ASSERT_EXPR5(id,A,op1,B,MSG)             HK_ASSERT(id, A op1 B,             MSG " " #A "={} " #B "={}", A, B)
    #define HK_DETAIL_ASSERT_EXPR6(id,A,op1,B,op2,C)           HK_ASSERT(id, A op1 B op2 C,               #A "={} " #B "={}" #C "={}", A, B, C)
    #define HK_DETAIL_ASSERT_EXPR7(id,A,op1,B,op2,C,MSG)       HK_ASSERT(id, A op1 B op2 C,       MSG " " #A "={} " #B "={}" #C "={}", A, B, C)
    #define HK_DETAIL_ASSERT_EXPR8(id,A,op1,B,op2,C,op3,D)     HK_ASSERT(id, A op1 B op2 C op3 D,         #A "={} " #B "={}" #C "={} "#D "={}", A, B, C, D)
    #define HK_DETAIL_ASSERT_EXPR9(id,A,op1,B,op2,C,op3,D,MSG) HK_ASSERT(id, A op1 B op2 C op3 D, MSG " " #A "={} " #B "={}" #C "={}" #D "={}", A, B, C, D)

    /// Assert the given expression and displays the evaluated arguments on failure.
    /// e.g. HK_ASSERT_EXPR(0x123, indexOf(a), >=, 0, "'a' is missing")
    /// The expression may consist of 1 to 4 values separate by 0 to 3 operators. The macro
    /// simply pastes the expression without commas so the expression is evaluated using the
    /// normal precedence rules. i.e. HK_ASSERT_EXPR(0x123, a,+,b,==,10) -> evaluates (a+b)==10.
    /// Warning! Parts of the expression may be evaluated more than once upon failure!
    #define HK_ASSERT_EXPR(...) HK_PP_CALL_OVERLOAD(HK_DETAIL_ASSERT_EXPR,__VA_ARGS__)
#else

/// Macro to raise an error, if a condition is not met.
///
/// Compiled out in non-development builds.
/// The condition is not evaluated, when this is compiled out, so do not execute important code in it.
#   define HK_ASSERT(id, condition, ...) HK_CONDITION_IN_ASSERT(condition)

/// Macro to raise an error, if a condition is not met.
///
/// Compiled out in non-development builds, however the condition is always evaluated,
/// so you may execute important code in it.
#   define HK_VERIFY(id, condition, ...) \
        if (static_cast<bool>(condition) == true) { /* The condition is evaluated, even though nothing is done with it. */ }

#   define HK_ASSERT_EXPR(...)

#endif

/// A dedicated assert to be used in math functions.
#define HK_MATH_ASSERT HK_ASSERT

/// This is deprecated, do not use
#define HK_ASSERT_DEV HK_ASSERT

/// DEPRECATED in favour of direct use of static_assert
#define HK_COMPILE_TIME_ASSERT(...) static_assert(static_cast<bool>(__VA_ARGS__), "Compile time assert failed")

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
