// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/System/Dll/hkDllManager.h>
#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>
#include <Common/Base/System/Dll/DynamicLibrary/hkDynamicLibrary.h>
#include <Common/Base/System/Dll/hkDllModule.h>

#ifdef HK_DEBUG
#   include <Common/Base/System/StackTracer/hkStackTracer.h>
//# define REFRESH_SYMBOLS
#endif

#define DEBUG_LOG_IDENTIFIER "sys.DllManager"
#include <Common/Base/System/Log/hkLog.hxx>

//
//  The prototype of the module creation function

extern "C"
{
    typedef hkDllModule* (*CreateDllModule)();
}

//
//  Constructor

hkDllManager::hkDllManager()
:   m_preventUnload(false)
{}

//
//  Constructor

hkDllManager::Dll::Dll()
:   m_dllname(HK_NULL)
,   m_nativePath(HK_NULL)
,   m_dll(HK_NULL)
,   m_module(HK_NULL)
{}

//
//  Destructor

hkDllManager::~hkDllManager()
{
    unloadAll();
}

//
//  Adds a search path to the list of paths seen by this manager

void hkDllManager::addSearchPath(_In_z_ const char* dir, _In_z_ const char* ext)
{
    // check if have it already
    for (int s=0; s < m_searchPaths.getSize(); ++s)
    {
        if ( (hkString::strCmp( m_searchPaths[s].m_dir.cString(), dir) == 0) && ( hkString::strCmp( m_searchPaths[s].m_ext, ext ) == 0 ) )
        {
            return; // exact match, no need to add again (will just slow down search)
        }
    }

    DllSearchPath& sp = m_searchPaths.expandOne();
    sp.m_dir = dir;
    sp.m_ext = ext;
}

//
//  Loads a Dll (native)

_Ret_maybenull_ hkDynamicLibrary* hkDllManager::loadDllNative(_In_z_ const char* nativePath)
{
    // see if we have already loaded it
#ifdef REFRESH_SYMBOLS
    hkStackTracer sym;
#endif

    int dl = 0;
    for (; dl < m_dlls.getSize(); ++dl)
    {
        if (m_dlls[dl].m_nativePath == nativePath)
        {
            return m_dlls[dl].m_dll; // done already
        }
    }

    hkRefPtr<hkDynamicLibrary> dll = hkDynamicLibrary::load(nativePath);
    if ( dll )
    {
        Log_Info("Loaded native dll '{}'.", nativePath);
        Dll& de     = m_dlls.expandOne();
        de.m_dll    = dll;

        hkStringBuf dllname(nativePath);
        dllname.pathBasename();
        dllname.pathWithoutExtension();
        de.m_dllname = dllname;

        de.m_nativePath = nativePath;

        // Create the module if possible
        if ( CreateDllModule createDllModule = (CreateDllModule)dll->findSymbol("CreateDllInterface") )
        {
            de.m_module.setAndDontIncrementRefCount(createDllModule());

            // Initialize the module
            if ( de.m_module )
            {
                de.m_module->init();
            }
        }

        // note that will only load for given dll, so will not have all dependent dlls that may have loaded. As such, usually not enough, so leave full symbol refresh for call app normally
#ifdef REFRESH_SYMBOLS
        sym.refreshSymbols(nativePath);
#endif

        if (m_preventUnload)
        {
            dll->preventUnload(true);
        }
    }

    return dll.val();
}

//
//  Loads all Dlls found in the search paths

void hkDllManager::loadAll()
{
    HK_TIME_FUNCTION();
    
    const char* blocklist[] = {
        "hkeDestruction_dynamic.hkeplugin",
        "hkeAi_dynamic.hkeplugin",
        "hctFilterToolStub.dll"
    };

    hkFileSystem& fs = hkFileSystem::getInstance();
    for (int d = 0; d < m_searchPaths.getSize(); ++d)
    {
        hkFileSystem::Iterator dirIter(&fs, m_searchPaths[d].m_dir, m_searchPaths[d].m_ext);
        while (dirIter.advance())
        {
            const hkFileSystem::Entry entry(dirIter.current());
            if (entry.isFile())
            {
                hkStringBuf nativePath;
                fs.getOperatingSystemPath(entry.getPath(), nativePath);
                if (nativePath.getLength())
                {
                    bool blacklisted = false;
                    for (int i = 0; i < sizeof(blocklist) / sizeof(blocklist[0]); ++i)
                    {
                        if (nativePath.indexOf(blocklist[i]) != -1)
                        {
                            blacklisted = true;
                            break;
                        }
                    }
                    if (blacklisted)
                    {
                        Log_Dev("Skipping plugin '{}'.", nativePath);
                        continue;
                    }

                    loadDllNative(nativePath.cString());
                }
            }
        }
    }
}

//
//  Loads a single Dll

_Ret_maybenull_ hkDynamicLibrary* hkDllManager::loadDll(_In_z_ const char* name)
{
    HK_TIME_FUNCTION();
    hkFileSystem& fs = hkFileSystem::getInstance();
    for (int d = 0; d < m_searchPaths.getSize(); ++d)
    {
        hkFileSystem::Iterator dirIter(&fs, m_searchPaths[d].m_dir, m_searchPaths[d].m_ext);
        while (dirIter.advance())
        {
            const hkFileSystem::Entry entry(dirIter.current());
            if (entry.isFile())
            {
                hkStringBuf ename(entry.getPath());
                ename.pathBasename();
                int ext = ename.lastIndexOf('.');
                if (ext >= 0) ename.chompEnd( ename.getLength() - ext );
                if (ename.endsWith("_dynamic")) ename.chompEnd(8);

                if (ename.compareToIgnoreCase(name) == 0)
                {
                    // found
                    hkStringBuf nativePath;
                    fs.getOperatingSystemPath(entry.getPath(), nativePath);
                    if (nativePath.getLength())
                    {
                        return loadDllNative(nativePath.cString());
                    }
                }
            }
        }
    }

    Log_Warning("Couldn't find dll file '{}' for loading.", name);
    return HK_NULL;
}

_Ret_maybenull_ const hkReflect::Type* hkDllManager::load(_In_z_ const char* name, _In_opt_z_ const char* dllname)
{
    HK_TIME_FUNCTION();
    const hkReflect::Type* pluginType = hkReflect::typeFromName(name);

    if (!pluginType)
    {
        hkStringBuf dll;
        if (!dllname)
        {
            dllname = name;

            // assume something along the lines of hkeFxComponent, so stop once see second cap as dll name (hkeFx in this case)
            int numCaps = 0;
            const char* n = name;
            while ( (*n) && (numCaps < 2) )
            {
                if (*n <= 'Z') // Capital or number
                {
                    ++numCaps;
                }
                ++n;
            }

            if (*n) // not the end
            {
                int tlen = hkString::strLen(name);
                int nlen = int(n - name - 1);
                dll = dllname;
                dll.chompEnd( tlen - nlen );

                dllname = dll.cString();
            }
        }

        if ( loadDll( dllname ) )
        {
            // try again
            pluginType = hkReflect::typeFromName(name);
        }
    }

    return pluginType;
}

//
//  Debug. Prevents the unloading of Dll

void hkDllManager::preventUnload()
{
    m_preventUnload = true;
    for (int d=0; d < m_dlls.getSize(); ++d)
    {
        m_dlls[d].m_dll->preventUnload(true);
    }
}

//
//  Unloads a single Dll

hkResult hkDllManager::unload(_In_z_ const char* dllname)
{
    HK_TIME_FUNCTION();
    for (int d=0; d < m_dlls.getSize(); ++d)
    {
        if ( hkString::strCasecmp(m_dlls[d].m_dllname.cString(), dllname) == 0)
        {
            return unload(m_dlls[d].m_dll);
        }
    }

    Log_Warning("Couldn't find dll file '{}' for unloading.", dllname);
    return HK_FAILURE;
}

//
//  Unloads a single Dll

hkResult hkDllManager::unload(_Inout_ hkDynamicLibrary* dllIn)
{
    HK_TIME_FUNCTION();
    for (int d = 0; d < m_dlls.getSize(); ++d)
    {
        Dll& dll = m_dlls[d];

        if ( dll.m_dll == dllIn )
        {
            // Quit the module
            if ( hkDllModule* module = dll.m_module )
            {
                HK_ASSERT_NO_MSG(0x3f781d69, module->canQuit());
                module->quit();
            }

            m_dlls.removeAt(d); // should clean up
            return HK_SUCCESS;
        }
    }
    Log_Warning("Couldn't find dll file in dll manager list.");
    return HK_FAILURE;
}

//
//  Unloads all Dlls

void hkDllManager::unloadAll()
{
    m_dlls.clearAndDeallocate();// should clean up
}

_Ret_maybenull_ hkReferencedObject* hkDllManager::createInstance(_In_z_ const char* name, _In_opt_z_ const char* dllname)
{
    const hkReflect::Type* pluginType = load(name, dllname);
    if (pluginType)
    {
         return hkDynCast<hkReferencedObject>( pluginType->newInstance() );
    }
    return HK_NULL;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
