// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL !REFLECT
// PRODUCT    : COMMON
// VISIBILITY : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Reflect/Version/hkReflectVersionPatcher.h>
#include <Common/Base/Serialize/Detail/hkWriteBuffer.h>
#include <Common/Base/Serialize/Detail/hkReadBuffer.h>
#include <Common/Base/Reflect/Util/hkReflectClone.h>

class hkResource;
class hkObjectResource;
class hkStreamReader;

namespace hkReflect { namespace Version { class Patcher; class PatchSet; } }

namespace hkSerialize
{
    class Bundle;
    class BundleBuilder;
    class IdFromVar;
    class ReadFormat;
    class WriteFormat;
    class TagfileReadFormat;
    class NoteHandler;

        /// Fluent loading interface.
        /// Defaults are to use format:autodetect, version:true
        /// Example usage: hkRefPtr<hkObjectResource> res = hkSerialize::Load().toResource( ... );
        /// Example usage: hkUuid* uuid = hkSerialize::Load().toObject<hkUuid>( ... );
    class HK_EXPORT_COMMON Load
    {
        public:
            HK_DECLARE_CLASS(Load, New, NonCopyable);

                ///
            Load();

                ///
            ~Load();

                /// Override file type detection and use the given format.
            template<typename READ_FORMAT>
            HK_INLINE Load& withFormat();

                /// Advanced usage - Override file type detection and use the given format object.
            Load& withFormatState(ReadFormat* rfm);

            enum VersioningOptionsEnum
            {
                VERSIONING_DEFAULT = 0,
                VERSIONING_DISABLED = 1,
                FAIL_IF_VERSIONING_REQUIRED = 2,
            };

            struct VersioningOptions
            {
                VersioningOptions(bool oldVersioningEnable) : m_option(oldVersioningEnable ? VERSIONING_DEFAULT : VERSIONING_DISABLED) {}
                VersioningOptions(VersioningOptionsEnum opt = VERSIONING_DEFAULT) : m_option(opt) {}

                bool isEnabled() const { return m_option.noneIsSet(VERSIONING_DISABLED) != 0; }
                bool isDisabled() const { return m_option.allAreSet(VERSIONING_DISABLED) != 0; }
                bool isFailIfVersioning() const { return m_option.allAreSet(FAIL_IF_VERSIONING_REQUIRED) != 0; }

            private:
                hkFlags<VersioningOptionsEnum, hkUint32> m_option;
            };

                /// Configure versioning (enabled by default)
            Load& withVersioning(struct VersioningOptions options = VersioningOptions(), _In_opt_ hkReflect::Version::PatcherInterface* patcher = HK_NULL);

                /// Indicates to ignore invalid/unknown objects rather than fail.
            Load& withIgnoreInvalid(bool ignore = true);

                /// Use a custom type registry (uses the builtin one by default).
            Load& withTypeReg(_In_ const hkReflect::TypeReg* reg);

                /// Receive callbacks during cloning.
            Load& withCloneCallback(_In_ hkReflect::Cloner::Callback* cb);

                /// Specify the AfterReflectNewHandler to be used by cloning.
                /// If a handler is specified, the afterReflectNew function of
                /// newly created objects which implement this function won't be
                /// called immediately, but instead a reference to these
                /// functions is added to the handler. The user is then
                /// responsible for calling
                /// AfterReflectNewHandler::callAfterReflectNews at some later
                /// point. This is useful if there's additional work which needs
                /// to be done after the toResource/toVar/toObject has returned
                /// but before afterReflectNew can be called, for example if
                /// there pointers to objects not in the current bundle which
                /// need to be fixed up first.
            Load& withAfterReflectNewHandler(_In_ hkReflect::Cloner::AfterReflectNewHandler* handler);

                /// Specify an array in which to save objects that would die when the cloner
                /// release its reference to them.
            Load& withDyingObjectsSalvageArray(_In_ hkArray<hkRefPtr<hkReferencedObject>>* array);

                /// Add a note handler which will collect notes in the file.
            Load& withNoteHandler(NoteHandler* nh);

            /// Perform reflection validation on the received values
            Load& withValidation(bool val = true);

            

            

            /// \name Resource loading functions
            ///
            /// The various resource loading functions all have different
            /// overloads.
            ///
            /// The overload which takes a ReadBufferAdapter can be used to load
            /// a resource using an input from which a ReadBuffer can be
            /// constructed internally, like a file name or a
            /// \ref hkStreamReader (specifically, any type for which there is
            /// a createReaderImpl overload). The ReadBufferAdapter is intended
            /// to be constructed implicitly by passing the adaptable object
            /// to the resource loading function.
            ///
            /// The overload which takes a ReadBuffer can be used to load a
            /// resource from an existing ReadBuffer. This overload can be
            /// used to load multiple consecutive resources from the same source.
            ///
            /// The overload which takes a memory block loads the resource from
            /// the data in the specified memory block. The optional used output
            /// parameter gives the number of bytes of the input used to
            /// construct the output resource and can be used to know where the
            /// next resource starts in case the memory block contains multiple
            /// resources.


            /// \{

            // Create an on-heap resource
            hkRefNew<hkObjectResource> toResource(const hkIo::Detail::ReadBufferAdapter& rba);
            hkRefNew<hkObjectResource> toResource(hkIo::ReadBuffer& readBuffer);
            hkRefNew<hkObjectResource> toResource(_In_reads_bytes_(bufLen)const void* buf, hkUlong bufLen, _Out_opt_ hkUlong* used = HK_NULL);

            // Create a managed resource, the resource owns all of its contents and they are invalid once it is destroyed
            hkRefNew<hkResource> toOwnedResource(const hkIo::Detail::ReadBufferAdapter& rba);
            hkRefNew<hkResource> toOwnedResource(hkIo::ReadBuffer& readBuffer);
            hkRefNew<hkResource> toOwnedResource(_In_reads_bytes_(bufLen) const void* buf, hkUlong bufLen, _Out_opt_ hkUlong* used = HK_NULL);

            /// Return a new Var. Cleanup generically with Var::destroy(), or by dyn-casting to the right type and calling the destructor.
            hkReflect::Var toVar(const hkIo::Detail::ReadBufferAdapter& rba, const hkReflect::QualType type=HK_NULL);
            hkReflect::Var toVar(hkIo::ReadBuffer& readBuffer, const hkReflect::QualType type=HK_NULL);
            hkReflect::Var toVar(_In_reads_bytes_(bufLen) const void* buf, hkUlong bufLen, const hkReflect::QualType type = HK_NULL, _Out_opt_ hkUlong* used = HK_NULL);

            /// Return a new typed object or null.
            template<typename T> _Ret_maybenull_ T* toObject(const hkIo::Detail::ReadBufferAdapter& rba);
            template<typename T> _Ret_maybenull_ T* toObject(hkIo::ReadBuffer& readBuffer);
            template<typename T> _Ret_maybenull_ T* toObject(_In_reads_bytes_(bufLen) const void* buf, hkUlong bufSize, _Out_opt_ hkUlong* used = HK_NULL);

            /// Advanced use - load without cloning. Data is only usable via Var
            hkViewPtr<Bundle> toBundle(const hkIo::Detail::ReadBufferAdapter& rba);
            hkViewPtr<Bundle> toBundle(hkIo::ReadBuffer& readBuffer);
            hkViewPtr<Bundle> toBundle(_In_reads_bytes_(bufLen) const void* buf, hkUlong bufLen, _Out_opt_ hkUlong* used = nullptr);

            /// \}

            /// Load the given compendium.
            hkResult loadCompendium(const hkIo::Detail::ReadBufferAdapter& rba);

        private:

            hkRefNew<hkObjectResource> cloneToNativeResource(Bundle& bundle);
            hkRefNew<hkResource> cloneToNativeOwnedResource(Bundle& bundle);
            hkReflect::Var cloneToNativeVar(hkSerialize::Bundle& bundle, const hkReflect::QualType reqType);

            void construct();

            /// creates ReadFormat for provided buffer
            hkResult createFormat(_In_reads_bytes_(bufLen) const void * buf, const hkUlong bufLen);
        private:
            const hkReflect::TypeReg* m_typeReg;
            hkRefPtr<ReadFormat> m_format;
            hkRefPtr<hkReflect::Version::PatcherInterface> m_patcher;
            hkReflect::Cloner m_cloner;
            hkReflect::Cloner::Callback* m_cloneCallback;
            VersioningOptions m_versionOptions;
            NoteHandler* m_noteHandler;
    };


    /// Utility for inplace loading.
    class HK_EXPORT_COMMON InplaceLoad
    {
        public:

                ///
            InplaceLoad();
                ///
            ~InplaceLoad();

                /// Use the given compendium.
                /// The compendium must outlive any data loaded using the compendium.
            InplaceLoad& withCompendium(_Out_writes_bytes_(bufLen) void* buf, hkUlong bufLen);

                /// Use a custom type registry (uses the builtin one by default).
            InplaceLoad& withTypeReg(_In_ const hkReflect::TypeReg* reg);

                /// Create a managed resource, the resource owns all of its contents and they are invalid once it is destroyed.
            hkRefNew<hkResource> toOwnedResource(_Out_writes_bytes_(bufLen) void* buf, hkUlong bufLen);

                /// Attempt to load and return the contents as a var. If a type is
                /// specified, it's verified if the resulting var is of the given
                /// type and if not, and empty var will be returned instead. If no
                /// type is specified (ie. if \p type is HK_NULL), then the
                /// resulting var will always be returned. Calling this function
                /// with a type is equivalent to calling it with type = HK_NULL,
                /// followed by a call to hkReflect::Var::dynCast.
            hkReflect::Var toVar(_Out_writes_bytes_(bufLen) void* buf, hkUlong bufLen, const hkReflect::QualType type = HK_NULL);

                /// Attempt to load and return the contents as a typed object.
            template<typename T> _Ret_maybenull_ T* toObject(_Out_writes_bytes_(bufLen) void* buf, hkUlong bufLen);

                /// Destruct the objects in the buffer. Not needed if holding a reference to a returned hkResource.
                /// Returns HK_FAILURE if the buffer doesn't appear to be valid loaded data.
                /// Note that unload isn't the exact inverse of load, i.e. one cannot reuse the buffer for a new load operation.
            static hkResult unload(_Out_writes_bytes_(bufLen) void* buf, hkUlong bufLen);

        protected:

            hkScopedPtr<TagfileReadFormat> m_format;
            hkViewPtr<const hkReflect::TypeReg> m_typeReg;
    };


        /// Fluent saving interface
    class HK_EXPORT_COMMON Save
    {
        public:

            HK_DECLARE_CLASS(Save, New, NonCopyable);

                /// See withCallbacks method
            typedef void (HK_CALL *VarCallback)(const hkReflect::Var&, BundleBuilder&, void*);
                /// See withCallbacks method
            typedef hkReflect::Var (HK_CALL *PtrCallback)(const hkReflect::PointerVar&, BundleBuilder&, void*);

            Save();
            ~Save();

                /// Write using a format suitable for inplace loading on target "t".
                /// See hkReflect::TypeCopier::getTargetStrings() for a list of target strings.
                /// If "t" is null, configure for the host target.
            Save& withTarget(_In_z_ const char* t);

                /// Allow calling contents* multiple times.
                /// Each call should correspond to a Load::to* method.
            Save& withMultiBundle();

                /// Set callbacks for saving.
                /// The callbackData pointer is stored and passed as-is to the callback.
                ///
                /// VarCallback will be called when an object is about to be written.
                /// You can use the full BundleBuilder features there, for example:
                ///   - Call addImport to import the object,
                ///   - Call addEmpty to write an empty object instead,
                ///   - Call addNote to add a note to the object
                ///     ...
                /// If the object has not been "consumed" by the callback (consuming calls are typically
                /// addImport, addEmpty etc, not addNote for example) it will be written to the file
                /// afterwards.
                ///
                /// If the PtrCallback is not null, then after each object is written, the callback will be called for
                /// each non-null pointer encountered. The returned Var will be written in the file as the new target
                /// for the pointer. It can be the original object, another object (either added the Bundle or not), a
                /// note (previously or subsequently added to the Bundle through BundleBuilder::addNote) or an empty
                /// Var. If a note is returned, it will be attached to the pointer and will be available on load
                /// through Bundle::getNoteOnPointer.
                /// If the PtrCallback is null, the original contents will be used. The default behavior is equivalent
                /// to: Var callback(PointerVar ptr, ...) { return ptr.getValue(); }
                /// Objects added to the BundleBuilder in the PtrCallback will be written after the current object and
                /// must be kept alive until then.
            Save& withCallbacks(_In_opt_ void* callbackData, VarCallback vcb, PtrCallback pcb);

                /// Indicates to ignore invalid objects rather than fail.
                /// Sets the VarCallback so that non-serializable objects are skipped and pointers to them are set to
                /// HK_NULL.
            Save& withIgnoreInvalid();

                /// Choose a specific format to use.
            template<typename WRITE_FORMAT>
            Save& withFormat();

                /// Advanced usage - use a specific format object.
                /// Most people will want the templated withFormat<T> method instead.
                /// Note that WriteFormat is a stateful object.
            Save& withFormatState(_In_ WriteFormat* format);

            /// \name Resource saving functions
            ///
            /// The various resource saving functions all have various overloads
            /// which take a different type destination.
            ///
            /// The overloads which take a WriteBufferAdapter can be used to
            /// write a resource to a destination for which a WriteBuffer can
            /// be constructed.
            ///
            /// The overloads which take a WriteBuffer can be used to write a
            /// resource to an existing WriteBuffer. These overloads can be used
            /// to write multiple consecutive resources to the same destination.
            ///
            /// The overload which takes a memory block writes the resource to
            /// the specified memory block. The optional output parameter
            /// usedBufferSizeOut can be used to get the actual number of bytes
            /// of the memory block which were used.

            /// \{

            template<typename T>
            hkResult contentsPtr(_In_ const T* obj, const hkIo::Detail::WriteBufferAdapter& wba) { return contentsVar(hkReflect::Var(obj), wba); }

            template<typename T>
            hkResult contentsPtr(_In_reads_bytes_(bufferSize) const T* obj, void* buffer, hkLong bufferSize, _Out_opt_ hkLong* usedBufferSizeOut = HK_NULL) { return contentsVar(hkReflect::Var(obj), buffer, bufferSize, usedBufferSizeOut); }

            template<typename T>
            hkResult contentsPtr(_In_ const T* obj, hkIo::WriteBuffer& writeBuffer) { return contentsVar(hkReflect::Var(obj), writeBuffer); }

            hkResult contentsVar(const hkReflect::Var& v, const hkIo::Detail::WriteBufferAdapter& wba);
            hkResult contentsVar(const hkReflect::Var& v, _Out_writes_bytes_(bufferSize) void* buffer, hkLong bufferSize, _Inout_opt_ hkLong* usedBufferSizeOut = HK_NULL);
            hkResult contentsVar(const hkReflect::Var& v, hkIo::WriteBuffer& writeBuffer);

                ///
            void beginTypeCompendium(const hkIo::Detail::WriteBufferAdapter& sink);
                ///
            void endTypeCompendium();

        private:
            WriteFormat* getOrCreateFormat();
            hkRefPtr<WriteFormat> m_formatPtr;
            VarCallback m_varCallback;
            PtrCallback m_ptrCallback;
            void* m_callbackData;
    };
}

#include <Common/Base/Serialize/hkSerialize.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
