// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Serialize/hkSerialize.h>
#include <Common/Base/Serialize/Detail/hkSerializeDetail.h>
#include <Common/Base/Serialize/Detail/hkWriteBuffer.h>
#include <Common/Base/Reflect/Util/hkReflectClone.h>
#include <Common/Base/Reflect/Util/hkReflectVarMapOperations.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Container/StringView/hkStringView.h>

#include <Common/Base/Container/StringMap/hkStorageStringMap.h>
#include <Common/Base/Reflect/TypeReg/hkTypeReg.h>
#include <Common/Base/System/Io/Reader/Memory/hkMemoryStreamReader.h>
#include <Common/Base/Serialize/Resource/hkObjectResource.h>
#include <Common/Base/Container/LocalArray/hkLocalBuffer.h>
#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>
#include <Common/Base/Reflect/TypeReg/Detail/hkTypeRegDetail.h>
#include <Common/Base/Reflect/Version/hkReflectVersionPatcher.h>
#include <Common/Base/Reflect/Version/hkReflectPatchRegistry.h>
#include <Common/Base/Serialize/Format/Tagfile/hkTagfileWriteFormat.h>
#include <Common/Base/Serialize/Format/Tagfile/hkTagfileReadFormat.h>
#include <Common/Base/Container/Hash/hkHashMap.h>

#define DEBUG_LOG_IDENTIFIER "s11n.Serialize"
#include <Common/Base/System/Log/hkLog.hxx>

hkReflect::Version::PatcherInterface2014::CreatePatcherFn hkReflect::Version::PatcherInterface2014::s_createPatcher = HK_NULL;
hkReflect::Version::PatcherInterface::CreatePatcherFn hkReflect::Version::PatcherInterface::s_createPatcher = HK_NULL;

#define RETURN_UNLESS(COND, RET, ...) \
    if( !(COND) ) { \
        Log_Warning(__VA_ARGS__); \
        return RET; \
    }

namespace
{
    
    struct SimpleImportHandler : public hkSerialize::NoteHandler
    {
        virtual void addNotes(const hkArrayView<hkReflect::Var>& notes, hkReflect::Var& dst, hkReflect::Var& src) HK_OVERRIDE
        {
            using namespace hkSerialize;
            for (int nid = 0; nid < notes.getSize(); ++nid)
            {
                if (const Note::Export* exp = hkDynCast<Note::Export>(notes[nid]))
                {
                    m_exports.expandOne().set(dst, exp->m_name);
                }
            }
            m_allNotes.append(notes);
        }

        virtual bool atPointerNote(const hkReflect::Var& note, hkReflect::PointerVar& dst, hkReflect::PointerVar& src) HK_OVERRIDE
        {
            using namespace hkSerialize;
            if (const Note::Import* imp = hkDynCast<Note::Import>(note))
            {
                DLOG("Import \"{}\" pointed from {}", imp->m_name, dst);
                m_imports.expandOne().set(dst, imp->m_name);
            }
            return true;
        }

        ~SimpleImportHandler()
        {
            for (int i = 0; i < m_allNotes.getSize(); ++i)
            {
                m_allNotes[i].destroy();
            }
        }

        hkArray<hkResource::Import> m_imports;
        hkArray<hkResource::Export> m_exports;
        hkArray<hkReflect::Var> m_allNotes;
    };

    struct CloneNoteToNativeCallback : public hkReflect::Detail::CloneOnHeap
    {
        virtual const hkReflect::Type* typeFromType(const hkReflect::Type* srcType) HK_OVERRIDE
        {
            return hkReflect::typeFromType(srcType);
        }
    };
    struct CloneToNativeCallback : public hkReflect::Cloner::Callback
    {
        CloneToNativeCallback(const hkSerialize::Bundle& bundle,
            hkReflect::Cloner::Callback* parentCb,
            const hkReflect::TypeReg* typeReg,
            hkReflect::Cloner& cloner,
            hkSerialize::NoteHandler* noteHandler)
            : m_defaultCallback(typeReg)
            , m_parentCallback(parentCb ? parentCb : &m_defaultCallback)
            , m_cloner(cloner)
            , m_bundle(bundle)
            , m_noteHandler(noteHandler ? noteHandler : &m_defaultNoteHandler)
        {
            if (m_noteHandler)
            {
                using namespace hkSerialize;

                // copy notes now, they have no dependencies
                CloneNoteToNativeCallback notecb;
                hkArray<Bundle::Item> vars;
                bundle.getItems(vars);
                for (int nid = 0; nid < vars.getSize(); ++nid)
                {
                    if (vars[nid].isNote())
                    {
                        hkReflect::Var nativeNote = m_cloner.cloneVar(vars[nid].var(), notecb);
                        VarId annotatedId = vars[nid].getAnnotated();
                        if (annotatedId >= 0 && annotatedId < vars.getSize() && vars[annotatedId].isVar())
                        {
                            
                            hkReflect::Var annotatedVar = vars[annotatedId].var();
                            auto& currNotes = m_notes.getOrInsertKey(annotatedVar.getAddress(), hkArray<hkReflect::Var>());
                            currNotes.pushBack(nativeNote);
                        }
                    }
                }

                hkArray<hkReflect::Var> empty;
                auto& importNotes = m_notes.getWithDefault(HK_NULL, empty);
                if (importNotes.getSize())
                {
                    hkReflect::Var dummy;
                    m_noteHandler->addNotes(importNotes, dummy, dummy);
                }
            }
        }

        virtual hkResult beginVar(hkReflect::Var& dst, hkReflect::Var& src) HK_OVERRIDE
        {
            
            bool topLevel = dst.isValid() == false;

            if(m_parentCallback->beginVar(dst, src).isFailure())
            {
                return HK_FAILURE;
            }
            if (m_noteHandler && topLevel && dst.isValid())
            {
                hkArray<hkReflect::Var> empty;
                const auto& currNotes = m_notes.getWithDefault(src.getAddress(), empty);
                if (currNotes.getSize())
                {
                    m_noteHandler->addNotes(currNotes, dst, src);
                }
            }
            return HK_SUCCESS;
        }

        virtual hkResult beginArrayElements(hkReflect::ArrayVar& dst, hkReflect::ArrayVar& src) HK_OVERRIDE
        {
            return m_parentCallback->beginArrayElements(dst, src);
        }

        virtual hkResult atPointer(hkReflect::PointerVar& dst, hkReflect::PointerVar& src, _Out_ hkReflect::Cloner::PointerAction::Enum* action) HK_OVERRIDE
        {
            *action = hkReflect::Cloner::PointerAction::SKIP;

            if (m_parentCallback->atPointer(dst, src, action).isFailure())
            {
                return HK_FAILURE;
            }

            using namespace hkSerialize;

            hkReflect::Var var;
            if (src.getValue(&var).isFailure())
            {
                return HK_FAILURE;
            }

            if (!var && m_noteHandler)
            {
                if (hkReflect::Var note = m_bundle.getNoteOnPointer(src))
                {
                    hkReflect::Var nativeNote = m_cloner.getCachedClone(note);
                    *action = m_noteHandler->atPointerNote(nativeNote, dst, src) ? hkReflect::Cloner::PointerAction::CLONE : hkReflect::Cloner::PointerAction::SKIP;
                    return HK_SUCCESS;
                }
            }

            *action = hkReflect::Cloner::PointerAction::CLONE;
            return HK_SUCCESS;
        }

        virtual _Ret_maybenull_ const hkReflect::Type* typeFromType(_In_ const hkReflect::Type* srcType) HK_OVERRIDE
        {
            return m_parentCallback->typeFromType(srcType);
        }

        virtual void cloneStart() HK_OVERRIDE
        {
            m_parentCallback->cloneStart();
        }

        virtual void cloneEnd(hkResult r) HK_OVERRIDE
        {
            m_parentCallback->cloneEnd(r);
        }

        HK_ALWAYS_INLINE hkArray<hkResource::Export>& exports() { return m_defaultNoteHandler.m_exports; }
        HK_ALWAYS_INLINE hkArray<hkResource::Import>& imports() { return m_defaultNoteHandler.m_imports; }

        hkSerialize::Detail::CloneToRegistered m_defaultCallback;
        hkReflect::Cloner::Callback* m_parentCallback;
        hkHashMap<const void*, hkArray<hkReflect::Var> > m_notes;
        hkReflect::Cloner& m_cloner;
        const hkSerialize::Bundle& m_bundle;
        hkSerialize::NoteHandler* m_noteHandler;
        SimpleImportHandler m_defaultNoteHandler;
    };

    struct ObjectResource : public hkObjectResource
    {
        ObjectResource(const hkReflect::Var& top)
            : hkObjectResource(top)
        {
        }
        ~ObjectResource()
        {

        }
        void getImportsExports(hkArray<hkResource::Import>& impOut, hkArray<hkResource::Export>& expOut) const HK_OVERRIDE
        {
            impOut.append(m_imports);
            expOut.append(m_exports);
        }
        void addExport(const hkReflect::Var& var, _In_z_ const char* name)
        {
            m_exports.expandOne().set(var, name);
        }
        void addImport(const hkReflect::PointerVar& pvar, _In_z_ const char* name)
        {
            m_imports.expandOne().set(pvar, name);
        }

        hkArray<hkResource::Import> m_imports;
        hkArray<hkResource::Export> m_exports;
    };

    struct InplaceOwnedResource : public hkResource
    {
        InplaceOwnedResource( hkReflect::Var& top, _In_bytecount_(fromBufLen) void* fromBuf, hkUlong fromBufLen )
            : m_top( top ),
            m_fromBuf( fromBuf ),
            m_fromBufLen( fromBufLen )
        {}
        ~InplaceOwnedResource()
        {
            callDestructors();
        }

        virtual void getImportsExports( hkArray<Import>& impOut, hkArray<Export>& expOut ) const HK_OVERRIDE
        {
            
            HK_ASSERT_NO_MSG(0x60b2f09e, 0 );
        }

        virtual hkReflect::Var getContents() const HK_OVERRIDE
        {
            return m_top;
        }

        virtual void callDestructors() HK_OVERRIDE
        {
            if ( m_top )
            {
                hkSerialize::InplaceLoad::unload( m_fromBuf, m_fromBufLen );
                m_top.setAddress( (void*)HK_NULL );
            }
        }

        virtual _Ret_maybenull_z_ const char* getName() const HK_OVERRIDE
        {
            return HK_NULL;
        }

        hkReflect::Var m_top;
        void* m_fromBuf;
        hkUlong m_fromBufLen;
    };

    void HK_CALL s_ignoreInvalidVarCallback(const hkReflect::Var& var, hkSerialize::BundleBuilder& bb, _In_opt_ void*)
    {
        if (!var.getType()->isSerializable())
        {
            bb.addEmpty(var);
        }
    }
}

//////////////////////////////////////////////////////////////////////////
// Load
//////////////////////////////////////////////////////////////////////////

void hkSerialize::Load::construct()
{
    m_typeReg = HK_NULL;
    m_cloneCallback = HK_NULL;
    m_noteHandler = HK_NULL;
    m_cloner.setPerformValidation(false); 
}

hkSerialize::Load::Load()
{
    construct();
}

hkSerialize::Load::~Load()
{
}

hkSerialize::Load& hkSerialize::Load::withTypeReg(_In_ const hkReflect::TypeReg* reg)
{
    HK_ASSERT(0x7c625884, m_typeReg == HK_NULL || m_cloneCallback == HK_NULL, "Setting the clone callback and the type registry are mutually exclusive. (The clone callback has its own type registry)");
    m_typeReg = reg;
    return *this;
}

hkSerialize::Load& hkSerialize::Load::withCloneCallback(_In_ hkReflect::Cloner::Callback* cb)
{
    HK_ASSERT(0x490c216c, m_typeReg == HK_NULL || m_cloneCallback == HK_NULL, "Setting the clone callback and the type registry are mutually exclusive. (The clone callback has its own type registry)");
    m_cloneCallback = cb;
    return *this;
}

hkSerialize::Load& hkSerialize::Load::withAfterReflectNewHandler(_In_ hkReflect::Cloner::AfterReflectNewHandler* handler)
{
    m_cloner.setAfterReflectNewHandler(handler);
    return *this;
}

hkSerialize::Load& hkSerialize::Load::withDyingObjectsSalvageArray(_In_ hkArray<hkRefPtr<hkReferencedObject>>* array)
{
    m_cloner.setDyingObjectsSalvageArray(array);
    return *this;
}

/// Perform reflection validation on the received values
hkSerialize::Load& hkSerialize::Load::withValidation(bool val)
{
    m_cloner.setPerformValidation(val);
    return *this;
}

hkSerialize::Load& hkSerialize::Load::withNoteHandler(_In_ NoteHandler* nh)
{
    m_noteHandler = nh;
    return *this;
}

hkSerialize::Load& hkSerialize::Load::withFormatState(_In_ ReadFormat* rfm)
{
    HK_ASSERT(0x3322F14D, m_format == nullptr, "Format has already been set");
    m_format = rfm;
    return *this;
}

hkSerialize::Load& hkSerialize::Load::withVersioning(VersioningOptions options, _In_opt_ hkReflect::Version::PatcherInterface* patcher)
{
    m_versionOptions = options;
    if (m_versionOptions.isEnabled())
    {
        if(patcher)
        {
            m_patcher = patcher;
        }
    }
    else
    {
        m_patcher = HK_NULL;
    }
    return *this;
}

hkSerialize::Load& hkSerialize::Load::withIgnoreInvalid(bool ignore)
{
    m_cloner.setIgnoreInvalid(ignore);
    return *this;
}

hkViewPtr<hkSerialize::Bundle> hkSerialize::Load::toBundle(const hkIo::Detail::ReadBufferAdapter& rba)
{
    hkIo::ReadBuffer readBuffer(rba);
    return toBundle(readBuffer);
}

hkResult hkSerialize::Load::createFormat(_In_reads_bytes_(bufLen) const void* buf, const hkUlong bufLen)
{
    if (m_format == HK_NULL)
    {
        if (bufLen <= 0)
        {
            Log_Dev("Failed to read input").setId(Detail::ERRORID_READ_FAILED);
            return HK_FAILURE;
        }

        if (const hkSerialize::Detail::FileFormatRegNode* reg = hkSerialize::Detail::detectFormat(buf, bufLen))
        {
            m_format = reg->createReadFormat();
            if (m_format == HK_NULL)
            {
                Log_Warning("No registered loader was found for the format \"{}\". Have you included hkCompat as a product feature or used the asset conversion tool to convert your assets?", reg->getName()).setId(Detail::ERRORID_UNSUPPORTED_FORMAT);
                return HK_FAILURE;
            }
            // else fallthrough
        }
        else
        {
            Log_Warning("The format was not recognized").setId(Detail::ERRORID_UNSUPPORTED_FORMAT);
            return HK_FAILURE;
        }
    }
    return HK_SUCCESS;
}

hkViewPtr<hkSerialize::Bundle> hkSerialize::Load::toBundle(hkIo::ReadBuffer& readBuffer)
{
    HK_TIME_CODE_BLOCK_LIST( "ToBundle (ReadBuffer)", "Get read format" );
    DLOG_SCOPE("hkSerialize::Load::toBundle");
    if( m_format == HK_NULL )
    {
        // Peek at first few bytes to check if the file is binary
        hkUlong peekBufLen = readBuffer.prefetch(hkSerialize::Detail::FORMAT_HEADER_PEEK_LENGTH);
        const void* peekBuf = readBuffer.peekAt<void>(0,0);
        if(createFormat(peekBuf, peekBufLen).isFailure())
            return HK_NULL;
    }

    HK_TIMER_SPLIT_LIST( "Read bundle data" );
    hkViewPtr<hkSerialize::Bundle> bundle;
    bundle = m_format->read(readBuffer);
    RETURN_UNLESS(bundle, HK_NULL, "Unable to load from buffer");

    if(DLOG_ENABLED)
    {
        HK_TIMER_SPLIT_LIST( "Log loaded vars" );
        hkArray<Bundle::Item> vars; bundle->getItems(vars);
        DLOG_SCOPE("Loaded bundle has {} vars", vars.getSize());
        for(int vi = 0; vi < vars.getSize(); ++vi)
        {
            if (vars[vi].isNote())
            {
                DLOG("#{} NOTE {:*} on {}", vi, vars[vi].var(), vars[vi].getAnnotated());
            }
            else if (vars[vi].isVarN())
            {
                DLOG("#{} VARN {} count {}", vi, vars[vi].varn().getFirst(), vars[vi].getCount());
            }
            else
            {
                DLOG("#{} {}", vi, vars[vi].var());
            }
        }
    }

    if (!m_versionOptions.isDisabled())
    {
        HK_TIMER_SPLIT_LIST( "Versioning" );
        DLOG_SCOPE("Versioning");
        if(!m_patcher)
        {
            m_patcher = hkReflect::Version::PatcherInterface::create();
        }

        if(m_patcher)
        {
            bundle = m_patcher->applyPatchesTo( bundle, HK_NULL, HK_NULL, m_versionOptions.isFailIfVersioning() );
            RETURN_UNLESS( bundle, HK_NULL, "Versioning failed" );
        }
    }

    return bundle;
}

hkViewPtr<hkSerialize::Bundle> hkSerialize::Load::toBundle(_In_reads_bytes_(bufLen) const void* buf, hkUlong bufLen, _Out_opt_ hkUlong* used)
{
    HK_TIME_CODE_BLOCK_LIST( "ToBundle (memory view)", "Get read format" );

    hkUlong dumUsed;
    if(!used)
    {
        used = &dumUsed;
    }
    else
    {
        // Ensure non-garbage size in case of early out (failure)
        *used = 0;
    }

    DLOG_SCOPE("hkSerialize::Load::toBundle");
    if(createFormat(buf, bufLen).isFailure())
        return HK_NULL;

    HK_TIMER_SPLIT_LIST( "Read bundle data" );
    hkViewPtr<hkSerialize::Bundle> bundle = m_format->view(buf, bufLen, used);
    RETURN_UNLESS(bundle, HK_NULL, "Unable to load from buffer");

    if(DLOG_ENABLED)
    {
        HK_TIMER_SPLIT_LIST( "Log loaded vars" );
        hkArray<Bundle::Item> vars; bundle->getItems(vars);
        DLOG_SCOPE("Loaded bundle has {} vars", vars.getSize());
        for(int vi = 0; vi < vars.getSize(); ++vi)
        {
            if (vars[vi].isNote())
            {
                DLOG("#{} NOTE {:*} on {}", vi, vars[vi].var(), vars[vi].getAnnotated());
            }
            else if (vars[vi].isVarN())
            {
                DLOG("#{} VARN {} count {}", vi, vars[vi].varn().getFirst(), vars[vi].getCount());
            }
            else
            {
                DLOG("#{} {}", vi, vars[vi].var());
            }
        }
    }

    if (!m_versionOptions.isDisabled())
    {
        HK_TIMER_SPLIT_LIST( "Versioning" );
        DLOG_SCOPE("Versioning");
        if(!m_patcher)
        {
            m_patcher = hkReflect::Version::PatcherInterface::create();
        }

        if(m_patcher)
        {
            bundle = m_patcher->applyPatchesTo( bundle, HK_NULL, HK_NULL, m_versionOptions.isFailIfVersioning() );
            RETURN_UNLESS( bundle, HK_NULL, "Versioning failed" );
        }
    }

    return bundle;
}

hkRefNew<hkObjectResource> hkSerialize::Load::toResource(const hkIo::Detail::ReadBufferAdapter& rba)
{
    HK_TIME_CODE_BLOCK("hkSerialize::Load", HK_NULL);
    if( hkViewPtr<hkSerialize::Bundle> bundle = toBundle(rba) )
    {
        return cloneToNativeResource(*bundle);
    }

    return HK_NULL;
}

hkRefNew<hkObjectResource> hkSerialize::Load::toResource(hkIo::ReadBuffer& readBuffer)
{
    HK_TIME_CODE_BLOCK("hkSerialize::Load", HK_NULL);
    if( hkViewPtr<hkSerialize::Bundle> bundle = toBundle(readBuffer) )
    {
        return cloneToNativeResource(*bundle);
    }

    return HK_NULL;
}

hkRefNew<hkObjectResource> hkSerialize::Load::toResource(_In_reads_bytes_(bufLen) const void* buf, hkUlong bufLen, _Out_opt_ hkUlong* used)
{
    HK_TIME_CODE_BLOCK("hkSerialize::Load", HK_NULL);
    if( hkViewPtr<hkSerialize::Bundle> bundle = toBundle(buf, bufLen, used) )
    {
        return cloneToNativeResource(*bundle);
    }

    return HK_NULL;
}

hkRefNew<hkObjectResource> hkSerialize::Load::cloneToNativeResource(hkSerialize::Bundle& bundle)
{
    HK_TIME_CODE_BLOCK( "Clone to native (hkObjectResource)", HK_NULL );
    m_cloner.clearObjectCache();

    CloneToNativeCallback callback(bundle, m_cloneCallback, m_typeReg, m_cloner, m_noteHandler);
    if( hkReflect::Var top = m_cloner.cloneVar(bundle.getContents(), callback) )
    {
        
        if(const hkReferencedObject* ro = hkDynCast<hkReferencedObject>(top))
        {
            // Serialization will have created this with a 0 RC
            // We need it at 1 to manage the ownership correctly
            ro->addReference();
        }

        ObjectResource* res = new ObjectResource(top);

        for( int i = 0; i < callback.exports().getSize(); ++i )
        {
            hkResource::Export& e = callback.exports()[i];
            res->addExport( e.m_location, e.m_name );
        }

        for(int i = 0; i < callback.imports().getSize(); ++i)
        {
            hkResource::Import& imp = callback.imports()[i];
            res->addImport(imp.m_location, imp.m_name);
        }

        return static_cast<hkObjectResource*>(res);
    }
    else
    {
        Log_Warning("Unable to create native objects");
        return HK_NULL;
    }
}



hkRefNew<hkResource> hkSerialize::Load::toOwnedResource(const hkIo::Detail::ReadBufferAdapter& rba)
{
    
    HK_TIME_CODE_BLOCK("hkSerialize::Load::toOwnedResource", HK_NULL);
    if (hkViewPtr<hkSerialize::Bundle> bundle = toBundle(rba))
    {
        return cloneToNativeOwnedResource(*bundle);
    }

    return HK_NULL;
}

hkRefNew<hkResource> hkSerialize::Load::toOwnedResource(hkIo::ReadBuffer& readBuffer)
{
    
    HK_TIME_CODE_BLOCK("hkSerialize::Load::toOwnedResource", HK_NULL);
    if (hkViewPtr<hkSerialize::Bundle> bundle = toBundle(readBuffer))
    {
        return cloneToNativeOwnedResource(*bundle);
    }

    return HK_NULL;
}

hkRefNew<hkResource> hkSerialize::Load::toOwnedResource(_In_reads_bytes_(bufLen) const void* buf, hkUlong bufLen, _Out_opt_ hkUlong* used)
{
    
    HK_TIME_CODE_BLOCK("hkSerialize::Load::toOwnedResource", HK_NULL);
    if (hkViewPtr<hkSerialize::Bundle> bundle = toBundle(buf, bufLen, used))
    {
        return cloneToNativeOwnedResource(*bundle);
    }

    return HK_NULL;
}

hkRefNew<hkResource> hkSerialize::Load::cloneToNativeOwnedResource(hkSerialize::Bundle& bundle)
{
    HK_TIME_CODE_BLOCK( "Clone to native (hkResource)", HK_NULL );
    HK_ASSERT(0x92954bb, m_cloneCallback == HK_NULL, "toOwnedResource can only be used with the default callback");
    m_cloner.clearObjectCache();

    // We never want to delete objects in an owned resource. This stops the cloner
    // from deleting the objects before the owned resource has a change to take
    // ownership in addObject
    hkArray< hkRefPtr< hkReferencedObject > > dyingObjects;
    m_cloner.setDyingObjectsSalvageArray(&dyingObjects);

    CloneToNativeCallback callback(bundle, m_cloneCallback, m_typeReg, m_cloner, m_noteHandler);
    if( hkReflect::Var top = m_cloner.cloneVar(bundle.getContents(), callback) )
    {
        
        if(const hkReferencedObject* ro = hkDynCast<hkReferencedObject>(top))
        {
            // Reflection will have created this with a 0 RC
            // We need it at 1 to manage the ownership correctly
            ro->addReference();
        }

        Detail::OwnedResource* res = new Detail::OwnedResource();

        for (int i = 0; i < callback.m_defaultCallback.m_vars.getSize(); i++)
        {
            res->addObject(callback.m_defaultCallback.m_vars[i]);
        }

        for (int i = 0; i < callback.exports().getSize(); ++i)
        {
            hkResource::Export& e = callback.exports()[i];
            res->addExport(e.m_location, e.m_name);
        }

        for (int i = 0; i < callback.imports().getSize(); ++i)
        {
            hkResource::Import& imp = callback.imports()[i];
            res->addImport(imp.m_location, imp.m_name);
        }

        return static_cast<hkResource*>(res);
    }
    else
    {
        Log_Warning("Unable to create native type");
        return HK_NULL;
    }
}


hkResult hkSerialize::Load::loadCompendium(const hkIo::Detail::ReadBufferAdapter& rba)
{
    HK_TIME_CODE_BLOCK("hkSerialize::Load::loadCompendium", HK_NULL);
    hkViewPtr<hkSerialize::Bundle> bundle = toBundle(rba);
    if(bundle == HK_NULL)
    {
        Log_Dev("Unable to load bundle");
        return HK_FAILURE;
    }
    if(bundle->getContents())
    {
        Log_Dev("There should be no vars in a compendium");
        return HK_FAILURE;
    }
    return HK_SUCCESS;
}


hkReflect::Var hkSerialize::Load::toVar(const hkIo::Detail::ReadBufferAdapter& rba, const hkReflect::QualType reqType)
{
    HK_TIME_CODE_BLOCK("hkSerialize::Load::toVar", HK_NULL);

    if( hkViewPtr<hkSerialize::Bundle> bundle = toBundle(rba) )
    {
        return cloneToNativeVar(*bundle, reqType);
    }
    return hkReflect::Var();
}

hkReflect::Var hkSerialize::Load::toVar(hkIo::ReadBuffer& readBuffer, const hkReflect::QualType reqType)
{
    HK_TIME_CODE_BLOCK("hkSerialize::Load::toVar", HK_NULL);

    if( hkViewPtr<hkSerialize::Bundle> bundle = toBundle(readBuffer) )
    {
        return cloneToNativeVar(*bundle, reqType);
    }
    return hkReflect::Var();
}

hkReflect::Var hkSerialize::Load::toVar(_In_reads_bytes_(bufSize) const void* buf, hkUlong bufSize, const hkReflect::QualType reqType, _Out_opt_ hkUlong* used)
{
    HK_TIME_CODE_BLOCK("hkSerialize::Load::toVar", HK_NULL);
    m_cloner.clearObjectCache();

    if( hkViewPtr<hkSerialize::Bundle> bundle = toBundle(buf, bufSize, used) )
    {
        return cloneToNativeVar(*bundle, reqType);
    }
    return hkReflect::Var();
}

hkReflect::Var hkSerialize::Load::cloneToNativeVar(hkSerialize::Bundle& bundle, const hkReflect::QualType reqType)
{
    HK_TIME_CODE_BLOCK( "Clone to native (hkReflect::Var)", HK_NULL );
    m_cloner.clearObjectCache();

    CloneToNativeCallback callback(bundle, m_cloneCallback, m_typeReg, m_cloner, m_noteHandler);
    hkReflect::Var srcContents = bundle.getContents();
    if( reqType )
    {
        // we can save a copy to native if the types don't match
        hkReflect::QualType actualType = callback.typeFromType(srcContents.getType());
        RETURN_UNLESS(actualType, hkReflect::Var(),
            "Loaded type mismatch, loaded type {} does not correspond to any native type and cannot be compared to requested type {}",
            srcContents.getType(), reqType);
        RETURN_UNLESS( actualType->extendsOrEquals(reqType), hkReflect::Var(),
            "Loaded type mismatch, requested type {} from a stream containing {}", reqType, actualType);
    }
    if( srcContents )
    {
        HK_TIME_CODE_BLOCK("Clone", HK_NULL);
        if( hkReflect::Var top = m_cloner.cloneVar(srcContents, callback) )
        {
            
            if(const hkReferencedObject* ro = hkDynCast<hkReferencedObject>(top))
            {
                // Serialization will have created this with a 0 RC
                // We need it at 1 to manage the ownership correctly
                ro->addReference();
            }
            return top;
        }
        else
        {
            Log_Warning("Unable to clone contents");
        }
    }

    return hkReflect::Var();
}

//////////////////////////////////////////////////////////////////////////
// InplaceLoad
//////////////////////////////////////////////////////////////////////////

hkSerialize::InplaceLoad::InplaceLoad()
{
}

hkSerialize::InplaceLoad::~InplaceLoad()
{
}

hkRefNew<hkResource> hkSerialize::InplaceLoad::toOwnedResource(_Out_writes_bytes_(bufLen) void* buf, hkUlong bufLen)
{
    hkReflect::Var top = toVar(buf, bufLen);
    RETURN_UNLESS(top, HK_NULL, "Unable to load resource in-place");
    return new InplaceOwnedResource( top, buf, bufLen );
}

hkReflect::Var hkSerialize::InplaceLoad::toVar(_Out_writes_bytes_(bufLen) void* buf, hkUlong bufLen, const hkReflect::QualType reqType)
{
    HK_TIME_CODE_BLOCK("hkSerialize::Load::toVarInplace", HK_NULL);
    if( buf )
    {
        if(m_format == HK_NULL)
        {
            m_format.reset(new TagfileReadFormat());
        }
        hkUlong discard;
        if( const Bundle* bundle = m_format->inplace(buf, bufLen, &discard, m_typeReg) )
        {
            hkReflect::Var r = bundle->getContents();
            if( reqType.isNull() || r.isInstanceOf(reqType) )
            {
                return r;
            }
            Log_Warning("Type mismatch - requested {}, but got {}", reqType, r.getType());
            
        }
        else
        {
            Log_Warning("Failed to load");
        }
    }
    return hkReflect::Var();
}

hkSerialize::InplaceLoad& hkSerialize::InplaceLoad::withCompendium(_Out_writes_bytes_(bufLen) void* buf, hkUlong bufLen)
{
    if(m_format == HK_NULL)
    {
        m_format.reset(new TagfileReadFormat());
    }
    hkUlong discard = 0;
    m_format->inplace(buf, bufLen, &discard, m_typeReg);
    return *this;
}

hkSerialize::InplaceLoad& hkSerialize::InplaceLoad::withTypeReg(_In_ const hkReflect::TypeReg* reg)
{
    m_typeReg = reg;
    return *this;
}

hkResult hkSerialize::InplaceLoad::unload(_Out_writes_bytes_(bufLen) void* buf, hkUlong bufLen)
{
    return TagfileReadFormat::unload(buf, bufLen);
}


//////////////////////////////////////////////////////////////////////////
// Save
//////////////////////////////////////////////////////////////////////////

hkSerialize::Save::Save()
    : m_varCallback(HK_NULL)
    , m_ptrCallback(HK_NULL)
    , m_callbackData(HK_NULL)
{
}

hkSerialize::Save::~Save()
{
}

hkSerialize::Save& hkSerialize::Save::withFormatState(_In_ WriteFormat* format)
{
    HK_ASSERT(0x1322A14F, m_formatPtr == nullptr, "Format has already been set");
    m_formatPtr = format;
    return *this;
}

hkSerialize::Save& hkSerialize::Save::withTarget(_In_z_ const char* target)
{
    RETURN_UNLESS(m_formatPtr == HK_NULL, *this, "Format has already been initialized");
    m_formatPtr.setAndDontIncrementRefCount(new TagfileWriteFormat(target));
    return *this;
}

hkSerialize::Save& hkSerialize::Save::withMultiBundle()
{
    auto* format = getOrCreateFormat();
    format->enableMultiBundle(); 
    return *this;
}

hkSerialize::Save& hkSerialize::Save::withCallbacks(_In_opt_ void* callbackData, VarCallback vcb, PtrCallback pcb)
{
    HK_ASSERT(0x1493bd24, m_varCallback != &s_ignoreInvalidVarCallback, "Cannot use 'withCallbacks' and 'withIgnoreInvalid' together");
    m_varCallback = vcb;
    m_ptrCallback = pcb;
    m_callbackData = callbackData;
    return *this;
}

hkSerialize::Save& hkSerialize::Save::withIgnoreInvalid()
{
    HK_ASSERT(0x1493bd24, m_varCallback == HK_NULL, "Cannot use 'withCallbacks' and 'withIgnoreInvalid' together");
    m_varCallback = &s_ignoreInvalidVarCallback;
    return *this;
}

hkResult hkSerialize::Save::contentsVar(const hkReflect::Var& v, const hkIo::Detail::WriteBufferAdapter& wba)
{
    hkIo::WriteBuffer writeBuffer(wba);
    return contentsVar(v, writeBuffer);
}

hkResult hkSerialize::Save::contentsVar(const hkReflect::Var& v, _Out_writes_bytes_(bufferSize) void* buffer, hkLong bufferSize, _Inout_opt_ hkLong* usedBufferSizeOut)
{
    hkIo::WriteBuffer writeBuffer(buffer, bufferSize, usedBufferSizeOut);
    return contentsVar(v, writeBuffer);
}

hkResult hkSerialize::Save::contentsVar(const hkReflect::Var& v, hkIo::WriteBuffer& writeBuffer)
{
    DLOG_SCOPE("hkSerialize::Save");
    HK_TIME_CODE_BLOCK("hkSerialize::Save", HK_NULL);

    RETURN_UNLESS(v.isValid(), HK_FAILURE, "Cannot save, contents is null");
    RETURN_UNLESS(writeBuffer.isAttached(), HK_FAILURE, "Cannot save, writeBuffer is closed");

    auto* format = getOrCreateFormat();
    BundleBuilder bb(format, &writeBuffer);
    bb.setCallbacks(m_callbackData, m_varCallback, m_ptrCallback);
    bb.add(v);
    return bb.recursiveAdd();
}

void hkSerialize::Save::beginTypeCompendium(const hkIo::Detail::WriteBufferAdapter& sink)
{
    HK_ASSERT(0x1493bd23, sink, "Cannot write type compendium to invalid NULL output buffer.");
    auto* format = getOrCreateFormat();
    format->beginTypeCompendium(sink);
}

void hkSerialize::Save::endTypeCompendium()
{
    auto* format = getOrCreateFormat();
    format->endTypeCompendium();
}

hkSerialize::WriteFormat* hkSerialize::Save::getOrCreateFormat()
{
    if (m_formatPtr == nullptr)
    {
        m_formatPtr = WriteFormat::createDefault();
    }
    HK_ASSERT(0x7449d8cd, m_formatPtr, "Format cannot be null");
    return m_formatPtr;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
