// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>

#include <Common/Base/Fwd/hkcstdarg.h>
#include <Common/Base/Serialize/Version/hkVersionPatchChecker.h>
#include <Common/Base/Serialize/Version/hkVersionBundle.h>
#include <Common/Base/Reflect/TypeReg/hkTypeReg.h>
#include <Common/Base/Reflect/Version/hkModifiableTypeSet.h>
#include <Common/Base/Reflect/TypeVm/hkTypeVmCompiler.h>
#include <Common/Base/Reflect/TypeVm/hkTypeVmCompilerPasses.h>
#include <Common/Base/Serialize/Version/hkVersionBundleUtil.h>
#include <Common/Base/Types/hkVarArgs.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Reflect/Util/hkReflectUtil.h>
#include <Common/Base/Reflect/Builder/hkTypeBuilder.h>

#define DEBUG_LOG_IDENTIFIER "reflect.patchchecker"
#include <Common/Base/System/Log/hkLog.hxx>

#define REPORT_WARNING(...) output.logWarning(HK_VARARGS1_WRAP((__VA_ARGS__)))
#define REPORT_ERROR(...) output.logError(HK_VARARGS1_WRAP((__VA_ARGS__)))
#define REPORT_PATCH_SUGGEST(...) output.addPatchSuggestion(HK_VARARGS1_WRAP((__VA_ARGS__)))
#define REPORT_PATCH_REQUIRED(...) output.addPatchRequired(HK_VARARGS1_WRAP((__VA_ARGS__)))

namespace HK_UNITY_ANONYMOUS_NAMESPACE
{
    bool isVersionableType(_In_ const hkReflect::Type* t)
    {
        // We can't version unnamed things. We don't want to version pointers or basic types
        return t->getName() && !t->asInteger() && !t->asPointer() && !t->asString() && !t->asFloat() && !t->asBool() && !t->asArray();
    }

    struct ReportOutput
    {
        enum WarningAction
        {
            WARNINGS_AS_WARNINGS = 0,
            WARNINGS_AS_ERRORS = 1
        };

        ReportOutput(WarningAction wa = WARNINGS_AS_WARNINGS) : m_numWarnings(0), m_numErrors(0), m_warningAction(wa) {}

        void logWarning(_In_opt_z_ const char* fmt, hkVarArgs::VaTypes argTypes, ...)
        {
            hkVarArgs::FixedArray<10> args;
            HK_VARARGS_UNPACK(args, argTypes);
            m_warningBuf.appendFormatV(fmt, args);
            m_numWarnings++;
        }

        void logError(_In_opt_z_ const char* fmt, hkVarArgs::VaTypes argTypes, ...)
        {
            hkVarArgs::FixedArray<10> args;
            HK_VARARGS_UNPACK(args, argTypes);
            m_errorBuf.appendFormatV(fmt, args);
            m_numErrors++;
        }

        void addPatchSuggestion(_In_opt_z_ const char* fmt, hkVarArgs::VaTypes argTypes, ...)
        {
            hkVarArgs::FixedArray<10> args;
            HK_VARARGS_UNPACK(args, argTypes);
            m_suggestedPatchBuf.appendFormatV(fmt, args);
        }

        void addPatchRequired(_In_opt_z_ const char* fmt, hkVarArgs::VaTypes argTypes, ...)
        {
            hkVarArgs::FixedArray<10> args;
            HK_VARARGS_UNPACK(args, argTypes);
            m_requiredPatchBuf.appendFormatV(fmt, args);
            // If a patch is required, we should return an error
            m_numErrors++;
        }

        void addPatchRequired(_In_z_ const char* fmt)
        {
            m_requiredPatchBuf.append(fmt);
        }
        hkStringView getWarnings() const { return m_warningBuf; }
        hkStringView getErrors() const { return m_errorBuf; }
        hkStringView getSuggestedPatches() const { return m_suggestedPatchBuf; }
        hkStringView getRequiredPatches() const { return m_requiredPatchBuf; }

        int getErrorCount() const { return m_numErrors + ((m_warningAction == WARNINGS_AS_ERRORS) ? m_numWarnings : 0); }

    private:
        hkStringBuf m_warningBuf;
        hkStringBuf m_errorBuf;
        hkStringBuf m_suggestedPatchBuf;
        hkStringBuf m_requiredPatchBuf;
        int m_numWarnings;
        int m_numErrors;
        WarningAction m_warningAction;
    };

    struct PatchBuilder
    {
        PatchBuilder(hkReflect::Version::PatchRegistry& patchReg, _In_z_ const char* oldName, int oldVersion, _In_z_ const char* newName, int newVersion) : m_patchReg(patchReg), m_oldName(oldName), m_newName(newName), m_oldVersion(oldVersion), m_newVersion(newVersion) {}

        _Ret_z_ const char* str()
        {
            if(m_oldName)
            {
                m_patchBuf.appendPrintf("BEGIN%s(\"%s\", %d, ", m_componentBuf.getLength() ? "" : "_EMPTY", m_oldName, m_oldVersion);
            }
            else
            {
                m_patchBuf.appendPrintf("BEGIN%s(HK_NULL, HK_CLASS_ADDED, ", m_componentBuf.getLength() ? "" : "_EMPTY");
            }

            if (m_newName)
            {
                m_patchBuf.appendPrintf("\"%s\", %d)\n", m_newName, m_newVersion);
            }
            else
            {
                m_patchBuf.appendPrintf("HK_NULL, HK_CLASS_REMOVED)\n");
            }

            m_patchBuf.append(m_componentBuf);
            if (m_depends.getSize())
            {
                for (const hkStringPtr& line : m_depends)
                {
                    m_patchBuf.append(line);
                }
            }

            m_patchBuf.append(m_componentBuf.getLength() ? "END()\n\n" : "END_EMPTY()\n\n");

            return m_patchBuf.cString();
        }

        void appendStructDescription(hkUint64 uid)
        {
            // Run forward to get the final name / version
            int patchIndex;
            do
            {
                patchIndex = m_patchReg.getPatchIndex(uid);
                if (patchIndex >= 0)
                {
                    const hkReflect::Version::PatchInfo* thisPatch = m_patchReg.getPatch(patchIndex);
                    if (thisPatch->newName)
                    {
                        uid = m_patchReg.getUid(thisPatch->newName, thisPatch->newVersion);
                    }
                    else
                    {
                        patchIndex = -1;
                    }
                }
            }
            while (patchIndex >= 0);

            const char* structName = m_patchReg.getClassName(uid);
            const hkInt32 version = m_patchReg.getClassVersion(uid);
            m_componentBuf.append(structName);

            if ((!m_oldName || (hkString::strCmp(structName, m_oldName) != 0)) && (!m_newName || (hkString::strCmp(structName, m_newName) != 0)))
            {
                addDepends(structName, version);
            }
        }

        void appendTypeDescription(_Inout_ hkVersionBundleUtil::Field* field)
        {
            switch (field->m_typeKind)
            {
                default:
                case hkVersionBundleUtil::Field::TYPE_INVALID:
                {
                    m_componentBuf.append("ERROR");
                    return;
                }
                case hkVersionBundleUtil::Field::TYPE_BASIC:
                {
                    hkStringBuf tName;
                    const hkReflect::Type* basicType = field->getTypeAsBasic();
                    const char* typeFullName = basicType->getFullName(tName);
                    if ((hkString::strNcmp(typeFullName, "hkEnum<", 7) == 0) || (hkString::strNcmp(typeFullName, "hkFlags<", 8) == 0))
                    {
                        hkStringBuf stripEnumBuilder(typeFullName);
                        hkArray<const char*>::Temp bits;
                        stripEnumBuilder.setLength(stripEnumBuilder.lastIndexOf('>'));
                        stripEnumBuilder.split(',', bits);
                        typeFullName = hkStringBuf(bits.back()).trimEnd().trimStart();
                    }
                    else if (basicType->asInteger())
                    {
                        switch (basicType->getFormat().get())
                        {
                            case (hkUint32)hkReflect::Format::OfInt<hkInt8>::Value: typeFullName = hkReflect::getName<hkReflect::Typedef::hkInt8_Tag>(); break;
                            case (hkUint32)hkReflect::Format::OfInt<hkUint8>::Value: typeFullName = hkReflect::getName<hkReflect::Typedef::hkUint8_Tag>(); break;
                            case (hkUint32)hkReflect::Format::OfInt<hkInt16>::Value: typeFullName = hkReflect::getName<hkReflect::Typedef::hkInt16_Tag>(); break;
                            case (hkUint32)hkReflect::Format::OfInt<hkUint16>::Value: typeFullName = hkReflect::getName<hkReflect::Typedef::hkUint16_Tag>(); break;
                            case (hkUint32)hkReflect::Format::OfInt<hkInt32>::Value: typeFullName = hkReflect::getName<hkReflect::Typedef::hkInt32_Tag>(); break;
                            case (hkUint32)hkReflect::Format::OfInt<hkUint32>::Value: typeFullName = hkReflect::getName<hkReflect::Typedef::hkUint32_Tag>(); break;
                            case (hkUint32)hkReflect::Format::OfInt<hkInt64>::Value: typeFullName = hkReflect::getName<hkReflect::Typedef::hkInt64_Tag>(); break;
                            case (hkUint32)hkReflect::Format::OfInt<hkUint64>::Value: typeFullName = hkReflect::getName<hkReflect::Typedef::hkUint64_Tag>(); break;
                            default:
                            {
                                typeFullName = "Unknown";
                                Log_Error("Unknown integer type in enum");
                            }
                        }
                    }
                    m_componentBuf.appendFormat("{}", typeFullName);
                    return;
                }
                case hkVersionBundleUtil::Field::TYPE_POINTER:
                {
                    if(field->getTypeAsField())
                    {
                        appendTypeDescription(field->getTypeAsField());
                        m_componentBuf.append("*");
                    }
                    else
                    {
                        m_componentBuf.append("hkReflect::Var");
                    }
                    return;
                }
                case hkVersionBundleUtil::Field::TYPE_TYPE_POINTER:
                {
                    m_componentBuf.append("hkReflect::Type*");
                    return;
                }
                case hkVersionBundleUtil::Field::TYPE_STRUCT:
                {
                    hkStringBuf sb;
                    const hkUint64 fieldUid = field->getTypeAsStruct<hkUint64>();
                    appendStructDescription(fieldUid);
                    return;
                }
                case hkVersionBundleUtil::Field::TYPE_HKARRAY:
                {
                    if(hkVersionBundleUtil::Field* thisField = field->getTypeAsField())
                    {
                        m_componentBuf.append("hkArray< ");
                        appendTypeDescription(thisField);
                        m_componentBuf.append(" >");
                    }
                    else
                    {
                        m_componentBuf.append("hkVariantArray");
                    }
                    return;
                }
                case hkVersionBundleUtil::Field::TYPE_CARRAY:
                {
                    appendTypeDescription(field->getTypeAsField());
                    m_componentBuf.appendFormat("[{}]", field->m_tuples);
                    return;
                }
            }
        }

        void addMemberRemoved(_Inout_ hkVersionBundleUtil::Field* field)
        {
            m_componentBuf.appendFormat("    REMOVE(\"{}\", \"", field->m_name);
            appendTypeDescription(field);
            m_componentBuf.append("\")\n");
        }

        void addParentSet(_In_opt_z_ const char* oldName, int oldVersion, _In_opt_z_ const char* newName, int newVersion)
        {
            m_componentBuf.append("    SET_PARENT(");
            if (oldName)
            {
                addDepends(oldName, oldVersion);
                m_componentBuf.appendFormat("\"{}\", ", oldName);
            }
            else
            {
                m_componentBuf.append("HK_NULL, ");
            }
            if (newName)
            {
                addDepends(newName, newVersion);
                m_componentBuf.appendFormat("\"{}\")\n", newName);
            }
            else
            {
                m_componentBuf.append("HK_NULL)\n");
            }
        }

        void addMemberAdded(_Inout_ hkVersionBundleUtil::Field* field)
        {
            m_componentBuf.appendFormat("    ADD(\"{}\", \"", field->m_name);
            appendTypeDescription(field);
            m_componentBuf.append("\")\n");
        }

        void addMemberTypeChanged(_Inout_ hkVersionBundleUtil::Field* newField, _Inout_ hkVersionBundleUtil::Field* oldField)
        {
            // TEMP
            addMemberAdded(newField);
            m_componentBuf.append("    // Function needed here\n");
            addMemberRemoved(oldField);
        }
        void addDepends(const char* name, int version)
        {
            hkStringBuf newDependsLine;
            newDependsLine.appendFormat("    DEPENDS(\"{}\", {})\n", name, version);
            for (const hkStringPtr& line : m_depends)
            {
                if (line == newDependsLine.cString())
                {
                    return;
                }
            }
            m_depends.emplaceBack(newDependsLine);
        }

        hkReflect::Version::PatchRegistry& m_patchReg;
        const char* m_oldName;
        const char* m_newName;
        int m_oldVersion;
        int m_newVersion;
        hkStringBuf m_patchBuf;
        hkStringBuf m_componentBuf;
        hkArray<hkStringPtr> m_depends;
    };


    bool shouldConsiderType(_In_ const hkReflect::Type* thisType)
    {
        return !thisType->isDynamicType() && !thisType->findAttribute<hk::ExcludeFromVersionCheck>();
    }

    struct TypesOnlyBundle : public hkSerialize::Bundle
    {
        TypesOnlyBundle() {}

        virtual hkReflect::Var getContents() const HK_OVERRIDE
        {
            return hkReflect::Var();
        }

        virtual int getItems(hkArray<Item>& out) const HK_OVERRIDE
        {
            out.clear();
            return 0;
        }

        virtual hkReflect::Var getNoteOnPointer(const hkReflect::PointerVar& ptr) const HK_OVERRIDE
        {
            return hkReflect::Var();
        }

        void addTypes(hkArrayView<const hkReflect::Type*> types)
        {
            m_types.append(types);
        }
    };

    struct VersionedField
    {
        hkStringPtr m_name;
        const hkReflect::Type* m_type;

        void set(_In_z_ const char* name, _In_ const hkReflect::Type* t) { m_name = name; m_type = t; }
    };

    class TypeCb : public hkVersionBundleUtil::TypeFromTypeCb
    {
    public:
        TypeCb(hkReflect::Version::PatchRegistry& patchReg) : m_patchReg(patchReg) {}
        virtual _Ret_notnull_ void* getTypeFromType(_In_ const hkReflect::Type* type) override
        {
            // TEMP: Need to track renames!
            // However, we know the version we are removing so we just need to track future versions
            // Should just write the uid and track them forward

            //return (void*)type->getName();
            hkStringBuf sb;
            const hkUint64 uid = m_patchReg.getUid(type->getFullName(sb), type->getVersion());
            // What do we do for 32-bit??! needs to be hkUint64 and not void*
            return (void*)uid;
        }

        hkReflect::Version::PatchRegistry& m_patchReg;
    };

    class NameCb : public hkVersionBundleUtil::TypeFromNameCb
    {
    public:
        enum
        {
            TypeIsNotFound = 0,
            TypeIsNotBasic = 8
        };

        NameCb(hkReflect::Version::PatchRegistry& patchReg) : m_patchReg(patchReg) {}
        virtual _Ret_notnull_ void* getTypeFromName(_In_z_ const char* name) override
        {
            const int version = m_deps->getVersion(name);
            const hkUint64 uid = m_patchReg.getUid(name, version);
            // What do we do for 32-bit??! needs to be hkUint64 and not void*
            return (void*)uid;
        }

        _Ret_maybenull_ const hkReflect::Type* checkForBasicType(_In_z_ const char* strName) override
        {
            hkStringBuf str(strName);
            str.trimEnd();

            hkStringMap<const hkReflect::Type*>::Iterator it = m_basicTypeCache.findOrInsertKey(hkReflectUtil::internCopy(str), (const hkReflect::Type*)TypeIsNotFound);
            const hkReflect::Type* lookedUpType = m_basicTypeCache.getValue(it);
            if (lookedUpType == (const hkReflect::Type*)TypeIsNotFound)
            {
                // First time we have seen this type
                if (const hkReflect::Type* builtInType = hkReflect::typeFromName(str))
                {
                    // Int, float, bool, strings are builtin and never change
                    if (builtInType->asInteger() || builtInType->asFloat() || builtInType->asBool() || builtInType->asString())
                    {
                        m_basicTypeCache.setValue(it, builtInType);
                        return builtInType;
                    }

                    // Fixed size arrays of floats are builtin and never change
                    if(hkVersionBundleUtil::isSpecialFloatArray(builtInType))
                    {
                        m_basicTypeCache.setValue(it, builtInType);
                        return builtInType;
                    }
                }

                // Everything else is not basic
                m_basicTypeCache.setValue(it, (const hkReflect::Type*)TypeIsNotBasic);
                return HK_NULL;
            }
            else if (lookedUpType == (const hkReflect::Type*)TypeIsNotBasic)
            {
                return HK_NULL;
            }
            else
            {
                // It it basic and we have seen it before
                return lookedUpType;
            }
        }

        hkStringMap<const hkReflect::Type*> m_basicTypeCache;
        hkPatchDependencies* m_deps;
        hkReflect::Version::PatchRegistry& m_patchReg;
    };

    void updateToMostRecentVersion(hkReflect::Version::PatchRegistry& patchReg, const char*& name, int& version)
    {
        hkUint64 uid = patchReg.getUid(name, version);
        int patchIndex = patchReg.getPatchIndex(uid);

        while (patchIndex >= 0)
        {
            const hkReflect::Version::PatchInfo* thisPatch = patchReg.getPatch(patchIndex);
            if (thisPatch->newName)
            {
                name = thisPatch->newName;
                version = thisPatch->newVersion;

                uid = patchReg.getUid(name, version);
                patchIndex = patchReg.getPatchIndex(uid);
            }
            else
            {
                return;
            }
        }
    }

    hkUint64 getMostRecentUid(hkReflect::Version::PatchRegistry& patchReg, hkUint64 uid)
    {
        int patchIndex = patchReg.getPatchIndex(uid);

        while (patchIndex >= 0)
        {
            const hkReflect::Version::PatchInfo* thisPatch = patchReg.getPatch(patchIndex);
            if (thisPatch->newName)
            {
                uid = patchReg.getUid(thisPatch->newName, thisPatch->newVersion);
                patchIndex = patchReg.getPatchIndex(uid);
            }
            else
            {
                patchIndex = -1;
            }
        }

        return uid;
    }

    int checkSizeOfFixedFloatArray(const hkVersionBundleUtil::Field* typeIn)
    {
        if (typeIn->m_typeKind == hkVersionBundleUtil::Field::TYPE_CARRAY)
        {
            hkVersionBundleUtil::Field* subType = typeIn->getTypeAsField();
            if (const hkReflect::Type* basicSubType = subType->getTypeAsBasic())
            {
                if (basicSubType->asFloat())
                {
                    return typeIn->m_tuples;
                }
            }
        }
        else
        {
            if (const hkReflect::Type* basicType = typeIn->getTypeAsBasic())
            {
                if (const hkReflect::ArrayType* basicArrayType = basicType->asArray())
                {
                    const int fixedCount = basicArrayType->getFixedCount();
                    if ((fixedCount > 0) && basicArrayType->getSubType() && basicArrayType->getSubType()->asFloat())
                    {
                        return fixedCount;
                    }
                }
            }
        }
        return -1;
    }

    // Some conversions are possible, we are not really checking that here yet
    hkResult checkTypesMatch(hkReflect::Version::PatchRegistry& patchReg, _Inout_ hkVersionBundleUtil::Field* fieldFromPatch, _Inout_ hkVersionBundleUtil::Field* fieldFromClass)
    {
        // Need some special handling for float arrays
        int patchFixedFloat = checkSizeOfFixedFloatArray(fieldFromPatch);
        int classFixedFloat = checkSizeOfFixedFloatArray(fieldFromClass);
        if ((patchFixedFloat > 0) && (patchFixedFloat == classFixedFloat))
        {
            // Any fixed float type with the same structure is considered ok
            return HK_SUCCESS;
        }

        if (fieldFromPatch->m_typeKind != fieldFromClass->m_typeKind)
        {
            return HK_FAILURE;
        }

        if (fieldFromPatch->m_typeKind == hkVersionBundleUtil::Field::TYPE_STRUCT)
        {
            hkUint64 patchUid = getMostRecentUid(patchReg, fieldFromPatch->getTypeAsStruct<hkUint64>());
            hkUint64 classUid = getMostRecentUid(patchReg, fieldFromPatch->getTypeAsStruct<hkUint64>());

            if (patchUid != classUid)
            {
                return HK_FAILURE;
            }
        }

        return HK_SUCCESS;
    }

    // TODO: This needs to be a class with more complex error handling (warnings / errors)
    int checkVersionedType(_In_ const hkReflect::Type* versionedType, const hkStringMap<const hkReflect::Type*>& registeredTypeFromName, hkSet<const hkReflect::Type*>& registeredTypesMatched, hkReflect::Version::PatchRegistry& patchReg, ReportOutput& output, bool startsFromAddedType)
    {
        // TODO: Check parents (just need string name!)
        hkTransientAllocator allocator(hkMemHeapAllocator());
        TypeCb typeCb(patchReg);
        NameCb nameCb(patchReg);
        hkStringBuf sb;

        if(isVersionableType(versionedType))
        {
            hkArray<hkVersionBundleUtil::Field*> fields;
            for (const hkReflect::FieldDecl& field : hkReflect::TypeDetail::localGetFields(versionedType))
            {
                if(field.getName())
                {
                    if(hkVersionBundleUtil::Field* f = hkVersionBundleUtil::Field::createFromStaticType(typeCb, field.getName(), field.getType()))
                    {
                        fields.pushBack(f);
                    }
                }
            }

            const char* parentName = HK_NULL;
            int parentVersion = -1;
            hkStringBuf parentNameBuf;
            if (const hkReflect::Type* parentRt = versionedType->getParent())
            {
                parentName = parentRt->getFullName(parentNameBuf);
                parentVersion = parentRt->getVersion();
            }

            hkStringBuf startNameBuf;
            const char* currentName = versionedType->getFullName(startNameBuf);
            int currentVersion = versionedType->getVersion();
            const hkUint64 dependsUid = patchReg.getUid(currentName, currentVersion);
            int patchIndex = -1;
            if (startsFromAddedType)
            {
                patchIndex = patchReg.getPatchAddedIndexFromLaterVersion(dependsUid); // ??
            }
            else
            {
                patchIndex = patchReg.getPatchIndex(dependsUid);
            }
            bool wasDeleted = false;


            while (patchIndex >= 0)
            {
                const hkReflect::Version::PatchInfo* thisPatch = patchReg.getPatch(patchIndex);
                hkPatchDependencies pd(thisPatch);
                nameCb.m_deps = &pd;

                for (int componentIndex = 0; componentIndex < thisPatch->numComponent; componentIndex++)
                {
                    const hkReflect::Version::PatchInfo::Component& component = thisPatch->component[componentIndex];

                    if (const hkReflect::Version::PatchInfo::SetParentPatch* setParentPatch = component.asSetParentPatch())
                    {
                        if (setParentPatch->newParent)
                        {
                            parentName = setParentPatch->newParent;
                            parentVersion = pd.getVersion(parentName);
                        }
                        else
                        {
                            parentName = HK_NULL;
                            parentVersion = -1;
                        }
                    }
                    else if (const hkReflect::Version::PatchInfo::MemberRemovedPatch* memberRemovedPatch = component.asMemberRemovedPatch())
                    {
                        if ((memberRemovedPatch->type != hkReflect::Version::TYPE_FROMNAME) || (memberRemovedPatch->tuples != 0))
                        {
                            REPORT_ERROR("{}({}) -> {}({}) has invalid member removed component ({})\n", thisPatch->oldName, thisPatch->oldVersion, thisPatch->newName, thisPatch->newVersion, memberRemovedPatch->name);
                        }
                        else
                        {
                            hkVersionBundleUtil::Field* fieldToRemove = hkVersionBundleUtil::Field::createFromTypeDescription(nameCb, memberRemovedPatch->name, memberRemovedPatch->typeName);

                            bool found = false;
                            for (int i = 0; (i < fields.getSize()) && !found; i++)
                            {
                                if (fields[i]->nameEquals(memberRemovedPatch->name))
                                {
                                    if (checkTypesMatch(patchReg, fieldToRemove, fields[i]).isSuccess())
                                    {
                                        delete fields[i];
                                        fields.removeAtAndCopy(i);
                                    }
                                    found = true;
                                }
                            }

                            if (!found)
                            {
                                REPORT_ERROR("{}({}) -> {}({}) is trying to remove field {} that does not exist\n", thisPatch->oldName, thisPatch->oldVersion, thisPatch->newName, thisPatch->newVersion, memberRemovedPatch->name);
                            }
                            delete fieldToRemove;
                        }
                    }
                    else if (const hkReflect::Version::PatchInfo::MemberAddedPatch* memberAddedPatch = component.asMemberAddedPatch())
                    {
                        // Only type_fromname is allowed!!!
                        if ((memberAddedPatch->type != hkReflect::Version::TYPE_FROMNAME) || (memberAddedPatch->tuples != 0))
                        {
                            REPORT_ERROR("{}({}) -> {}({}) has invalid member added component ({})\n", thisPatch->oldName, thisPatch->oldVersion, thisPatch->newName, thisPatch->newVersion, memberAddedPatch->name);
                        }
                        else
                        {
                            hkVersionBundleUtil::Field* fieldToAdd = hkVersionBundleUtil::Field::createFromTypeDescription(nameCb, memberAddedPatch->name, memberAddedPatch->typeName);

                            bool found = false;
                            for (int i = 0; (i < fields.getSize()) && !found; i++)
                            {
                                if (fields[i]->nameEquals(memberAddedPatch->name))
                                {
                                    found = true;
                                }
                            }

                            if (found)
                            {
                                REPORT_ERROR("{}({}) -> {}({}) is trying to add field {} that already exists\n", thisPatch->oldName, thisPatch->oldVersion, thisPatch->newName, thisPatch->newVersion, memberAddedPatch->name);
                            }

                            fields.pushBack(fieldToAdd);
                        }
                    }
                    else if (const hkReflect::Version::PatchInfo::MemberRenamedPatch* memberRenamedPatch = component.asMemberRenamedPatch())
                    {
                        bool found = false;
                        for (int i = 0; (i < fields.getSize()) && !found; i++)
                        {
                            if (fields[i]->nameEquals(memberRenamedPatch->oldName))
                            {
                                fields[i]->m_name = memberRenamedPatch->newName;
                                found = true;
                            }
                        }

                        if (!found)
                        {
                            REPORT_ERROR("{}({}) -> {}({}) is trying to rename field {} that does not exist\n", thisPatch->oldName, thisPatch->oldVersion, thisPatch->newName, thisPatch->newVersion, memberRenamedPatch->oldName);
                        }
                    }
                    else if (const hkReflect::Version::PatchInfo::TypeChangedFunctionPatch* typeChangedPatch = component.asTypeChangedFunctionPatch())
                    {
                        // Old type changed
                        if ((typeChangedPatch->m_oldType != hkReflect::Version::TYPE_FROMNAME) || (typeChangedPatch->m_oldTuples != 0))
                        {
                            REPORT_ERROR("{}({}) -> {}({}) has invalid type changed component ({})\n", thisPatch->oldName, thisPatch->oldVersion, thisPatch->newName, thisPatch->newVersion, typeChangedPatch->m_name);
                        }
                        else
                        {
                            REPORT_ERROR("{}({}) -> {}({}) has old version type changed component ({})\n", thisPatch->oldName, thisPatch->oldVersion, thisPatch->newName, thisPatch->newVersion, typeChangedPatch->m_name);
                        }
                    }
                    else if (const hkReflect::Version::PatchInfo::VarTypeChangedFunctionPatch* varTypeChangedPatch = component.asVarTypeChangedFunctionPatch())
                    {
                        // Remove old, add new
                        // This is a bit cumbersome (could just change the type) but it matches what happens in hkVersionBundle so leaving it this way
                        {
                            hkVersionBundleUtil::Field* fieldToRemove = hkVersionBundleUtil::Field::createFromTypeDescription(nameCb, varTypeChangedPatch->m_name, varTypeChangedPatch->m_oldType);

                            bool found = false;
                            for (int i = 0; (i < fields.getSize()) && !found; i++)
                            {
                                if (fields[i]->nameEquals(varTypeChangedPatch->m_name))
                                {
                                    if (checkTypesMatch(patchReg, fieldToRemove, fields[i]).isSuccess())
                                    {
                                        delete fields[i];
                                        fields.removeAtAndCopy(i);
                                    }
                                    found = true;
                                }
                            }

                            if (!found)
                            {
                                REPORT_ERROR("{}({}) -> {}({}) is trying to change type of field {} that does not exist\n", thisPatch->oldName, thisPatch->oldVersion, thisPatch->newName, thisPatch->newVersion, varTypeChangedPatch->m_name);
                            }
                            delete fieldToRemove;
                        }
                        {
                            hkVersionBundleUtil::Field* fieldToAdd = hkVersionBundleUtil::Field::createFromTypeDescription(nameCb, varTypeChangedPatch->m_name, varTypeChangedPatch->m_newType);

                            bool found = false;
                            for (int i = 0; (i < fields.getSize()) && !found; i++)
                            {
                                if (fields[i]->nameEquals(varTypeChangedPatch->m_name))
                                {
                                    found = true;
                                }
                            }

                            if (found)
                            {
                                REPORT_ERROR("{}({}) -> {}({}) is trying to add field {} that already exists\n", thisPatch->oldName, thisPatch->oldVersion, thisPatch->newName, thisPatch->newVersion, varTypeChangedPatch->m_name);
                            }

                            fields.pushBack(fieldToAdd);
                        }
                    }
                    else if (component.asVarFunctionPatch() || component.asSetValuePatch() || component.asDependsPatch())
                    {
                        // We don't care about these
                    }
                    else
                    {
                        REPORT_ERROR("{}({}) -> {}({}) has invalid component (type unknown)\n", thisPatch->oldName, thisPatch->oldVersion, thisPatch->newName, thisPatch->newVersion);
                    }
                }

                currentName = thisPatch->newName;
                currentVersion = thisPatch->newVersion;

                if (thisPatch->newName) // Not deleted
                {
                    const hkUint64 newUid = patchReg.getUid(thisPatch->newName, thisPatch->newVersion);

                    patchIndex = patchReg.getPatchIndex(newUid);
                }
                else
                {
                    // This was a deleted patch, check for any members left

                    if(fields.getSize())
                    {
                        for (int i = 0; i < fields.getSize(); i++)
                        {
                            REPORT_ERROR("{}({}) was deleted but member {} was not removed\n", thisPatch->oldName, thisPatch->oldVersion, fields[i]->m_name);
                        }
                    }

                    patchIndex = -1;
                    wasDeleted = true;
                }
            }

            if (!wasDeleted)
            {
                //const char* lastName = thisPatch->newName;
                if (const hkReflect::Type* nativeType = registeredTypeFromName.getWithDefault(currentName, HK_NULL))
                {
                    registeredTypesMatched.insert(nativeType);

                    if (nativeType->findAttribute<hk::ExcludeFromVersionCheck>())
                    {
                        // Patched but the native says exclude from check, do what the user asked for here...
                        for (int i = 0; i < fields.getSize(); i++)
                        {
                            delete fields[i];
                        }

                        return 0;
                    }

                    if (nativeType->getVersion() != currentVersion)
                    {
                        REPORT_WARNING("{}({}) is last patched version but native version is {}\n", currentName, currentVersion, nativeType->getVersion());
                    }

                    if(isVersionableType(nativeType))
                    {
                        // Check we match the native type
                        hkArray<hkVersionBundleUtil::Field*> nativeFields;
                        hkStringMap<int> fieldIndexFromName;
                        hkSet<int> nativeFieldsMatched;

                        hkStringBuf npName;
                        const char* nativeParentName = HK_NULL;
                        int nativeParentVersion = -1;

                        if(const hkReflect::Type* nativeParent = nativeType->getParent())
                        {
                            nativeParentName = nativeParent->getFullName(npName);
                            nativeParentVersion = nativeParent->getVersion();
                        }

                        hkArray<hkTuple<hkVersionBundleUtil::Field*, hkVersionBundleUtil::Field*> > mismatchedFields;
                        for (const hkReflect::FieldDecl& field : hkReflect::TypeDetail::localGetFields(nativeType))
                        {
                            if (field.getName())
                            {
                                if (hkVersionBundleUtil::Field* f = hkVersionBundleUtil::Field::createFromStaticType(typeCb, field.getName(), field.getType()))
                                {
                                fieldIndexFromName.insert(field.getName(), nativeFields.getSize());
                                    nativeFields.pushBack(f);
                                }
                            }
                        }

                        for (int i = 0; i < fields.getSize(); i++)
                        {
                            const int matchingFieldIndex = fieldIndexFromName.getWithDefault(fields[i]->m_name, -1);
                            if (matchingFieldIndex >= 0)
                            {
                                nativeFieldsMatched.insert(matchingFieldIndex);
                                if (checkTypesMatch(patchReg, fields[i], nativeFields[matchingFieldIndex]).isSuccess() || !nativeFields[matchingFieldIndex]->isSerializable())
                                {
                                    // Ok
                                    delete fields[i];
                                }
                                else
                                {
                                    REPORT_ERROR("{}({}) native type mismatch in field {}\n", currentName, currentVersion, fields[i]->m_name);
                                    mismatchedFields.pushBack(hkTupleT::make(nativeFields[matchingFieldIndex], fields[i]));
                                }
                                fields.removeAtAndCopy(i);
                                i--;
                            }
                        }

                        bool suggestPatchNeeded = false;
                        for (int i = 0; i < fields.getSize(); i++)
                        {
                            REPORT_WARNING("{}({})::{} versioned field not matched (needs remove)\n", currentName, currentVersion, fields[i]->m_name);
                            suggestPatchNeeded = true;
                        }

                        for (int i = 0; i < nativeFields.getSize(); i++)
                        {
                            if (!nativeFieldsMatched.contains(i) && nativeFields[i]->isSerializable())
                            {
                                REPORT_WARNING("{}({})::{} native field not matched (needs add)\n", currentName, currentVersion, nativeFields[i]->m_name);
                                suggestPatchNeeded = true;
                            }
                        }

                        bool parentsDiffer = false;

                        if (parentName)
                        {
                            updateToMostRecentVersion(patchReg, parentName, parentVersion);
                            if (nativeParentName)
                            {
                                parentsDiffer = hkString::strCmp(parentName, nativeParentName) != 0;
                            }
                            else
                            {
                                parentsDiffer = true;
                            }
                        }

                        if (mismatchedFields.getSize() || parentsDiffer)
                        {
                            PatchBuilder builder(patchReg, currentName, currentVersion, currentName, currentVersion + 1);

                            if (parentsDiffer)
                            {
                                if(parentName)
                                {
                                    builder.addParentSet(parentName, parentVersion, nativeParentName, nativeParentVersion);
                                }
                                else
                                {
                                    builder.addParentSet(HK_NULL, 0, nativeParentName, nativeParentVersion);
                                }
                            }

                            for (int i = 0; i < mismatchedFields.getSize(); i++)
                            {
                                builder.addMemberTypeChanged(mismatchedFields[i].m_0, mismatchedFields[i].m_1);
                                delete mismatchedFields[i].m_1; // This was the patched field, we are done with it now
                            }

                            for (int i = 0; i < fields.getSize(); i++)
                            {
                                builder.addMemberRemoved(fields[i]);
                            }

                            for (int i = 0; i < nativeFields.getSize(); i++)
                            {
                                if (!nativeFieldsMatched.contains(i) && nativeFields[i]->isSerializable())
                                {
                                    builder.addMemberAdded(nativeFields[i]);
                                }
                            }
                            // Required patch here
                            REPORT_PATCH_REQUIRED("{}", builder.str());
                        }
                        else if (suggestPatchNeeded)
                        {
                            PatchBuilder builder(patchReg, currentName, currentVersion, currentName, currentVersion + 1);

                            if (parentName)
                            {
                                if(nativeParentName)
                                {
                                    if (hkString::strCmp(parentName, nativeParentName) != 0)
                                    {
                                        builder.addParentSet(parentName, parentVersion, nativeParentName, nativeParentVersion);
                                    }
                                }
                                else
                                {
                                    builder.addParentSet(parentName, parentVersion, HK_NULL, 0);
                                }
                            }
                            else if (nativeParentName)
                            {
                                builder.addParentSet(HK_NULL, 0, nativeParentName, nativeParentVersion);
                            }
                            for (int i = 0; i < fields.getSize(); i++)
                            {
                                builder.addMemberRemoved(fields[i]);
                            }

                            for (int i = 0; i < nativeFields.getSize(); i++)
                            {
                                if (!nativeFieldsMatched.contains(i) && nativeFields[i]->isSerializable())
                                {
                                    builder.addMemberAdded(nativeFields[i]);
                                }
                            }

                            // Suggested patch
                            REPORT_PATCH_SUGGEST("{}", builder.str());
                        }

                        for (int i = 0; i < nativeFields.getSize(); i++)
                        {
                            delete nativeFields[i];
                        }
                    }
                }
                else
                {
                    

                    if((hkString::strCmp(currentName, "hkHashBase< hkHashMapDetail::MapTuple< hkUuid, hkndMeshShapeRegistry::Entry* > >") != 0) &&
                        (hkString::strCmp(currentName, "hkHashMap< hkUuid, hkndMeshShapeRegistry::Entry* >") != 0))
                    {
                        REPORT_ERROR("{}({}) not deleted but not registered\n", currentName, currentVersion);

                        PatchBuilder builder(patchReg, currentName, currentVersion, 0, 0);

                        if (parentName)
                        {
                            updateToMostRecentVersion(patchReg, parentName, parentVersion);
                            builder.addParentSet(parentName, parentVersion, HK_NULL, 0);
                        }
                        for (int i = 0; i < fields.getSize(); i++)
                        {
                            builder.addMemberRemoved(fields[i]);
                        }
                        REPORT_PATCH_REQUIRED("{}", builder.str());
                    }
                }
            }


            for (int i = 0; i < fields.getSize(); i++)
            {
                delete fields[i];
            }
        }

        return 0;
    }
}

namespace hkVersionPatchChecker
{

    int HK_CALL reportPatchProblems(hkStreamWriter& patchesOut, hkStreamWriter& infoOut, _In_reads_opt_(numTypesIn) const hkReflect::Type** typesIn, int numTypesIn, hkReflect::Version::PatchRegistry* patchRegIn)
    {
        HK_UNITY_USING_ANONYMOUS_NAMESPACE;
        hkStringBuf sb;

        hkTransientAllocator tempTypeRegAlloc(hkMemHeapAllocator());
        hkReflect::MutableTypeReg& typeReg = *hkReflect::getTypeReg();

        hkReflect::Version::PatchRegistry* patchReg = patchRegIn ? patchRegIn : hkReflect::Version::PatchRegistry::getInstancePtr();
        hkRefPtr<hkSerialize::VersionBundle> vb = hkAutoRemoveReference(new hkSerialize::VersionBundle(*patchReg, typeReg));

        // Compat Patch Types
        hkReflect::MutableTypeReg* compatTypeReg = vb->createBackwardCompatibleTypeReg();
        hkReflect::Version::PatchSet ps(*patchReg, compatTypeReg);

        // Everything!
        ps.addAllPatchesFromRegistry();
        hkModifiableTypeSet ts(ps, compatTypeReg, hkMemHeapAllocator(), &hkReflect::Detail::hkArrayImpl::s_instance, &hkReflect::Detail::hkArrayImpl::s_instance, &hkReflect::Detail::RawPointerImpl::s_instance, &hkReflect::Detail::HavokStringImpl::s_instance, hkModifiableTypeSet::BEST_EFFORT_ON_INCOMPLETE_TYPES | hkModifiableTypeSet::GENERATE_TYPES_FROM_PATCHES_ONLY);

        // Is this needed??
        ts.createAllTypes();
        ts.resolveAndFinalize(false);
        hkArray<hkVersionedRecordType*> allPatchedRecordsFromCompat;
        ts.appendAllTypesToArray(allPatchedRecordsFromCompat);


        // Registered Types
        hkArray<const hkReflect::Type*> allRegisteredTypes;
        typeReg.appendAllTypesToArray(allRegisteredTypes);

        // These go into a bundle to get versioned
        hkArray<const hkReflect::Type*> compatTypes;
        compatTypeReg->appendAllTypesToArray(compatTypes);
        for (hkVersionedRecordType* t : allPatchedRecordsFromCompat)
        {
            //const char* tn = t->getName();
            const hkReflect::Type* completedType = t->getCompletedType();
            if(completedType->getVersion() != 0x80000000) // Deleted marker in old types
            {
                compatTypes.pushBack(t->getCompletedType());
            }
        }
        hkArray<hkStringPtr> typeNames;

        hkStringMap<const hkReflect::Type*> registeredTypeFromName;
        for (const hkReflect::Type* registeredType : allRegisteredTypes)
        {
            typeNames.pushBack(registeredType->getFullName(sb));
            const char* nn = typeNames.back();
            registeredTypeFromName.insert(nn, registeredType);
        }

        hkSet<const hkReflect::Type*> registeredTypesMatched;
        // For each versioned type, walk forward through patches

        ReportOutput output(ReportOutput::WARNINGS_AS_ERRORS);

        for (const hkReflect::Type* versionedType : compatTypes)
        {
            checkVersionedType(versionedType, registeredTypeFromName, registeredTypesMatched, *patchReg, output, false);
        }

        // Also need to do this for types newly added in new patches
        for (const hkReflect::Version::PatchInfo* pi : patchReg->getPatches())
        {
            if (pi->isV2Patch() && pi->isClassAdded())
            {
                hkReflect::TypeBuilder tb;
                tb.beginRecord(pi->newName, HK_NULL);
                tb.addItem<hkReflect::Opt::VERSION>(pi->newVersion);
                const hkReflect::Type* tempInputType = tb.allocate(&tempTypeRegAlloc);

                // TODO: Combine these some way maybe? We are creating a hkReflect::Type just to pass in a name / version
                checkVersionedType(tempInputType, registeredTypeFromName, registeredTypesMatched, *patchReg, output, true);
            }
        }

        TypeCb typeCb(*patchReg);
        hkArray<const hkReflect::Type*> notMentionedRegisteredTypes;
        for (const hkReflect::Type* registeredType : allRegisteredTypes)
        {
            // isSerializable maybe temporary, getTemplate should be temporary
            if(isVersionableType(registeredType) && shouldConsiderType(registeredType) && registeredType->isSerializable() && !registeredType->getTemplate() && !registeredType->isAttribute())
            {
                if (!registeredTypesMatched.contains(registeredType))
                {
                    notMentionedRegisteredTypes.pushBack(registeredType);
                    hkStringBuf typeNameBuf;
                    hkStringBuf parentNameBuf;
                    hkReflect::TypeName tn(registeredType->getFullName());
                    REPORT_WARNING("Type {} is reflected but it does not have any patches. You should add a HK_CLASS_ADDED patch\n", tn);

                    PatchBuilder builder(*patchReg, HK_NULL, 0, tn.toString(typeNameBuf), registeredType->getVersion());

                    if (const hkReflect::Type* parent = registeredType->getParent())
                    {
                        builder.addParentSet(HK_NULL, 0, parent->getFullName().toString(parentNameBuf), parent->getVersion());
                    }

                    // Local fields only, we don't want parent fields, even if the parent relationship is just a typedef
                    for (const hkReflect::FieldDecl& field : hkReflect::TypeDetail::localGetFields(registeredType))
                    {
                        if (field.getName())
                        {
                            if(hkVersionBundleUtil::Field* newField = hkVersionBundleUtil::Field::createFromStaticType(typeCb, field.getName(), field.getType()))
                            {
                            builder.addMemberAdded(newField);

                            delete newField;
                            }
                        }
                    }

                    REPORT_PATCH_SUGGEST("{}", builder.str());
                }
            }
        }

        hkStringView warnings = output.getWarnings();
        hkStringView errors = output.getErrors();
        hkStringView suggestedPatches = output.getSuggestedPatches();
        hkStringView requiredPatches = output.getRequiredPatches();
        hkStringBuf header;

        if(errors.getSize())
        {
            header = "ERRORS:\n";
            infoOut.write(header.cString(), header.getLength());
            infoOut.write(errors.begin(), errors.getSize());
            Log_Error("{}", errors);
        }

        if(warnings.getSize())
        {
            header = "\nWARNINGS:\n";
            infoOut.write(header.cString(), header.getLength());
            infoOut.write(warnings.begin(), warnings.getSize());
            Log_Warning("{}", warnings);
        }

        if(suggestedPatches.getSize())
        {
            header = "\nSUGGESTED PATCHES:\n";
            infoOut.write(header.cString(), header.getLength());
            patchesOut.write(suggestedPatches.begin(), suggestedPatches.getSize());
        }

        if(requiredPatches.getSize())
        {
            header = "\nREQUIRED PATCHES:\n";
            patchesOut.write(header.cString(), header.getLength());
            patchesOut.write(requiredPatches.begin(), requiredPatches.getSize());
        }

        return output.getErrorCount();
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
