// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Serialize/Version/hkVersionBundleUtil.h>
#include <Common/Base/Reflect/Version/hkModifiableTypeSet.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Reflect/Core/hkReflectType.h>

namespace
{
    bool isAlpha(char c)
    {
        return (c >= '0') && (c <= '9');
    }

    bool checkAndStripPointerName(hkStringBuf& nameInOut)
    {
        if (nameInOut.startsWith("hkRefPtr"))
        {
            nameInOut.chompStart(8); // hkRefPtr
        }
        else if (nameInOut.startsWith("hkPtr"))
        {
            nameInOut.chompStart(5); // hkPtr
        }
        else if (nameInOut.startsWith("hkViewPtr"))
        {
            nameInOut.chompStart(9); // hkViewPtr
        }
        else if (nameInOut.startsWith("hkStrongRefPtr"))
        {
            nameInOut.chompStart(14); // hkStrongRefPtr
        }
        else if (nameInOut.startsWith("hkWeakPtr"))
        {
            nameInOut.chompStart(9); // hkWeakPtr
        }
        else
        {
            return false;
        }
        nameInOut.trimStartToFirst('<');
        nameInOut.trimEndFromLast('>');
        nameInOut.trimStart();
        nameInOut.trimEnd();
        return true;
    }
}

namespace hkVersionBundleUtil
{
    Field::~Field()
    {
        if (Field* f = getTypeAsField())
        {
            delete f;
        }
    }

    _Ret_maybenull_ Field* Field::createFromTypeDescription(TypeFromNameCb& cb, _In_z_ const char* name, _In_z_ const char* typeNameStr)
    {
        Field* newField = new Field();
        newField->m_name = name;
        newField->m_typeName = typeNameStr;
        newField->m_isSerializable = true;
        // Order of looking up things: First check for hkArray*, *[n] -> hkarray, carray
        // Then check for * -> Pointer
        // After that it's either struct or basic

        hkStringBuf typeName(typeNameStr);
        typeName.trimStart();
        typeName.trimEnd();

        // Check endswith first, * and [] take precedence. e.g. hkRefPtr<xx>*[] is an array of pointers to refptrs

        if (typeName.endsWith("]"))
        {
            // Some form of array
            typeName.chompEnd(1);
            typeName.trimEnd();

            // C Array
            int numCharsInCount = 0;
            // XXX this is simplistic, allow hex etc
            while (isAlpha(typeName[typeName.getLength() - 1 - numCharsInCount]) && (numCharsInCount < typeName.getLength()))
            {
                numCharsInCount++;
            }

            int tupleCount = 1;
            if (numCharsInCount)
            {
                tupleCount = hkString::atoi(typeName.cString() + typeName.getLength() - numCharsInCount);

                typeName.chompEnd(numCharsInCount);
                typeName.trimEnd();
            }
            else
            {
                HK_WARN_ALWAYS(0x5b3ff9fd, "Invalid array count in field " << name << " ( " << typeNameStr << ")");
                return HK_NULL;
            }

            if (typeName.endsWith("["))
            {
                typeName.chompEnd(1); // Remove the [
            }
            else
            {
                HK_WARN_ALWAYS(0x5b3ff9fe, "Invalid array count in field " << name << " ( " << typeNameStr << ")");
                return HK_NULL;
            }

            newField->m_typeKind = Field::TYPE_CARRAY;
            if (Field* fieldType = createFromTypeDescription(cb, HK_NULL, typeName))
            {
                newField->m_type = fieldType;
                newField->m_tuples = tupleCount;
            }
            else
            {
                delete newField;
                return HK_NULL;
            }
        }
        else if (typeName.endsWith("*") && (typeName != "const char*"))
        {
            // Some form of pointer
            typeName.chompEnd(1);

            //XXX Type / Field pointer XXX
            if ((typeName == "hkReflect::Type"))
            {
                newField->m_typeKind = Field::TYPE_TYPE_POINTER;
                newField->m_type = HK_NULL;
                newField->m_tuples = 0;
            }
            else
            {
                newField->m_typeKind = Field::TYPE_POINTER;
                if (Field* fieldType = createFromTypeDescription(cb, HK_NULL, typeName))
                {
                    newField->m_type = fieldType;
                    newField->m_tuples = 0;
                }
                else
                {
                    delete newField;
                    return HK_NULL;
                }
            }
        }
        else if (typeName.endsWith("hkVariantArray") || typeName.endsWith("hkReflect::Any"))
        {
            newField->m_typeKind = Field::TYPE_HKARRAY;
            newField->m_type = HK_NULL;
            newField->m_tuples = 0;
        }
        else if (typeName.endsWith("hkVariant") || typeName.endsWith("hkReflect::Var"))
        {
            newField->m_typeKind = Field::TYPE_POINTER;
            newField->m_type = HK_NULL;
            newField->m_tuples = 0;
        }
        else if (typeName.startsWith("hkArray"))
        {
            
            typeName.chompStart(7); // "hkArray"
            typeName.trimStartToFirst('<');
            typeName.trimEndFromLast('>');
            typeName.trimStart();
            typeName.trimEnd();
            newField->m_typeKind = Field::TYPE_HKARRAY;
            if (Field* fieldType = createFromTypeDescription(cb, HK_NULL, typeName))
            {
                newField->m_type = fieldType;
                newField->m_tuples = 0;
            }
            else
            {
                delete newField;
                return HK_NULL;
            }
        }
        else if (checkAndStripPointerName(typeName))
        {
            newField->m_typeKind = Field::TYPE_POINTER;
            if (Field* fieldType = createFromTypeDescription(cb, HK_NULL, typeName))
            {
                newField->m_type = fieldType;
                newField->m_tuples = 0;
            }
            else
            {
                delete newField;
                return HK_NULL;
            }
        }
        else
        {
            // Struct or basic type
            if (const hkReflect::Type* basicType = cb.checkForBasicType(typeName))
            {
                // This is a basic type, these never change
                newField->m_typeKind = Field::TYPE_BASIC;
                newField->m_type = const_cast<void*>(reinterpret_cast<const void*>(basicType));
                newField->m_tuples = 0;
            }
            else
            {
                newField->m_typeKind = Field::TYPE_STRUCT;

                newField->m_type = cb.getTypeFromName(typeName);//m_owner->getTypeForNameVer(typeName, version); // TODO
                newField->m_tuples = 0;
            }
        }

        return newField;
    }

    bool isSpecialFloatArray(_In_ const hkReflect::Type* staticTypeIn)
    {
        const hkReflect::Type* staticType = staticTypeIn;
        if (staticType->asArray() && staticType->asArray()->getSubType() && staticType->asArray()->getSubType()->asFloat())
        {
            if (const int fixedCount = staticType->asArray()->getFixedCount())
            {
                const bool isSpecialFixedSize = ((fixedCount == 4) || (fixedCount == 8) || (fixedCount == 12) || (fixedCount == 16));
                return isSpecialFixedSize;
            }
        }
        return false;
    }
    _Ret_notnull_ Field* Field::createFromStaticType(TypeFromTypeCb& cb, _In_z_ const char* name, _In_ const hkReflect::Type* staticType, bool typeIsSerializable)
    {
        hkVersionBundleUtil::Field* newField = nullptr;
        hkReflect::Decl decl(staticType);
        if (decl && !decl.isSerializable())
        {
            newField = new Field();
            newField->m_name = name;
            hkStringBuf sb;
            newField->m_typeName = staticType->getFullName().toString(sb);
            newField->m_tuples = 0;

            newField->m_isSerializable = false;

            // This decl just needs to pass through.
            newField->setType<hkVersionBundleUtil::Field::TYPE_BASIC>( hkReflect::getType<hkReflect::Detail::Opaque>() );
            return newField;
        }
        else if (!staticType->asVoid() && !staticType->isSerializable()) // Void is treated specially below
        {
            return nullptr;
        }

        newField = new Field();
        newField->m_name = name;
        hkStringBuf sb;
        newField->m_typeName = staticType->getFullName().toString(sb);
        newField->m_tuples = 0;
        newField->m_isSerializable = typeIsSerializable;
        // Special case m_foreignUtil
        // Just reuse value types
        if (staticType->asValue() || staticType->asVoid() || isSpecialFloatArray(staticType))
        {
            // Skip over any property elements, we just want the underlying data field
            while (staticType && staticType->isProperty())
            {
                staticType = staticType->getParent();
            }
            HK_ASSERT(0x1ef20e8f, staticType, "Couldn't find a fundamental type for property field");
            newField->setType<hkVersionBundleUtil::Field::TYPE_BASIC>(staticType);
            return newField;
        }
        else if (const hkReflect::PointerType* pt = staticType->asPointer())
        {
            if ( hkReflect::Detail::isTypeOfType( pt->getSubType() ) )
            {
                newField->setType<hkVersionBundleUtil::Field::TYPE_TYPE_POINTER>(HK_NULL);
                return newField;
            }
            else
            {
                if(pt->getSubType())
                {
                    newField->setType<hkVersionBundleUtil::Field::TYPE_POINTER>(createFromStaticType(cb, HK_NULL, pt->getSubType()));
                }
                else
                {
                    newField->setType<hkVersionBundleUtil::Field::TYPE_POINTER>(HK_NULL);
                }
                return newField;
            }
        }
        else if (const hkReflect::ArrayType* at = staticType->asArray())
        {
            // XXX Todo vector
            if (const int count = at->getFixedCount())
            {
                // CArray
                newField->setType<hkVersionBundleUtil::Field::TYPE_CARRAY>(createFromStaticType(cb, HK_NULL, at->getSubType()));
                newField->m_tuples = count;
            }
            else
            {
                // hkArray
                newField->setType<hkVersionBundleUtil::Field::TYPE_HKARRAY>(at->getSubType() ? createFromStaticType(cb, HK_NULL, at->getSubType()) : HK_NULL);
            }
        }
        else if (const hkReflect::RecordType* rt = staticType->asRecord())
        {
            // Record?
            newField->setType<hkVersionBundleUtil::Field::TYPE_STRUCT>(cb.getTypeFromType(rt));
        }
        else
        {
            HK_ASSERT_NOT_IMPLEMENTED(0x6ae11bd3);
        }

        return newField;
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
