// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Serialize/Version/hkVersionBundle.h>
#include <Common/Base/Reflect/Version/hkReflectVersionPatches.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Reflect/Version/hkModifiableTypeSet.h>
#include <Common/Base/Reflect/TypeReg/hkTypeReg.h>
#include <Common/Base/Reflect/Builder/hkTypeBuilder.h>
#include <Common/Base/Reflect/Builder/hkRecordLayout.h>
#include <Common/Base/Serialize/Core/hkSerializeCore.h>
#include <Common/Base/Reflect/TypeVm/hkTypeVmCompiler.h>
#include <Common/Base/Reflect/TypeVm/hkTypeVmFastCopyInterpreter.h>
#include <Common/Base/Serialize/Detail/hkIndexedBundle.h>
#include <Common/Base/Reflect/TypeVm/hkTypeVmCompilerPasses.h>
#include <Common/Base/Serialize/Version/hkVersionBundleUtil.h>
#include <Common/Base/Reflect/Util/hkReflectUtil.h>
#include <Common/Base/Types/Color/hkColor.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Container/PointerMultiMap/hkMultiMap.h>

#define DEBUG_LOG_IDENTIFIER "ver.VersionBundle"
#include <Common/Base/System/Log/hkLog.hxx>

namespace HK_UNITY_ANONYMOUS_NAMESPACE
{
    int hkReflect_Type_getTypeSizeInBytes(hkUlong optional)
    {
        int count = 0;
        for (hkUlong c = optional; c; c &= c - 1)
        {
            count += 1;
        }

        return sizeof(hkReflect::Type) + sizeof(hkUlong) * count;
    }

    bool isVersionableType(_In_ const hkReflect::Type* t)
    {
        if (t->getName() && (hkString::strCmp(t->getName(), "hkReflect::Detail::ForeignUtil") == 0))
        {
            return false;
        }
        // We can't version unnamed things. We don't want to version pointers or basic types
        return t->getName() && !t->asInteger() && !t->asPointer() && !t->asString() && !t->asFloat() && !t->asBool() && !t->asArray() && !t->asOpaque();
    }

    void adjustFieldsForNameMismatches(_In_ const hkReflect::Type* srcType, _Inout_ hkReflect::Type* dstType)
    {
        hkArrayView<const hkReflect::FieldDecl> srcFields = srcType->getDataFields();
        hkArrayView<const hkReflect::FieldDecl> dstFields = dstType->getDataFields();
        hkStringMap<int> dstFieldIndex; dstFieldIndex.reserve(dstFields.getSize());

        for (int i = 0; i < dstFields.getSize(); i++)
        {
            if (const char* dstName = dstFields[i].getName())
            {
                dstFieldIndex.insert(dstName, i);
            }
        }

        for (int i = 0; i < srcFields.getSize(); i++)
        {
            if (const char* srcName = srcFields[i].getName())
            {
                if (!dstFieldIndex.hasKey(srcName))
                {
                    hkStringBuf fieldToBeAdded("#");
                    fieldToBeAdded.append(srcName);
                    hkStringMap<int>::Iterator it = dstFieldIndex.findKey(fieldToBeAdded.cString());
                    if (dstFieldIndex.isValid(it))
                    {
                        Log_Info("Type {} adds a field named \"{}\" in versioning but that field is already present in the file, adjusting types to load from the file anyway. Either the file is invalid or the patches are incorrect", srcType->getName(), srcName);
                        hkReflect::TypeDetail::setFieldName(dstFields[dstFieldIndex.getValue(it)], hkReflectUtil::internCopy(srcName));
                    }
                    //                  else
                    //                  {
                    //                      HK_WARN_ALWAYS(0xda1c72f4, "Type " << srcType->getName() << " has field " << srcName << " in the file but that field has no corresponding field at runtime. The value in the file will be ignored. Either the file is invalid or the patches are incorrect");
                    //                      // We would need to add a field here;
                    //                  }
                }
            }
        }
    }
}

namespace hkSerialize
{
    class VersionedTypeBuilder;
    class ForwardingPointerImpl;

    struct VersionedTypeStorage
    {
        typedef hkStringMap<const hkReflect::Type*> BasicTypeCache;

        struct CompoundTypeCache
        {
            public:
                CompoundTypeCache( VersionedTypeStorage& owner ) : m_owner( owner ) {}

                
                
                _Ret_notnull_ const hkReflect::PointerType* getVersionedPointerType(_In_z_ const char* baseName, _In_ const hkReflect::Type* subType, bool addTemplateArgs);
                _Ret_notnull_ const hkReflect::ArrayType* getVersionedArrayType(_In_z_ const char* baseName, _In_ const hkReflect::Type* subType, bool addTemplateArgs, int fixedCount);

                _Ret_notnull_ const hkReflect::PointerType* getVersionedPointerType(_In_ const hkReflect::PointerType* referenceType, _In_ const hkReflect::Type* subType);
                _Ret_notnull_ const hkReflect::ArrayType* getVersionedArrayType(_In_ const hkReflect::ArrayType* referenceType, _In_ const hkReflect::Type* subType );

            protected:
                typedef hkTuple<const hkReflect::Type*, hkUlong, int> SubtypeNameAndCount;
                hkHashMap<SubtypeNameAndCount, const hkReflect::PointerType*> m_versionedPointerTypesMap;
                hkHashMap<SubtypeNameAndCount, const hkReflect::ArrayType*> m_versionedArrayTypesMap;
                VersionedTypeStorage& m_owner;

        };

        virtual hkMemoryAllocator& getAllocator() = 0;
        virtual VersionedTypeBuilder* getTypeForNameVer(_In_z_ const char* name, int version) = 0;
        virtual hkReflect::Var getWorkVar(int id) = 0;
        virtual hkSerialize::VarN getWorkExtra(int id) = 0;
        virtual _Ret_maybenull_ hkReflect::Detail::ArrayImpl* getArrayImpl() = 0;
        virtual _Ret_maybenull_ hkReflect::Detail::StringImpl* getStringImpl() = 0;
        virtual _Ret_maybenull_ hkReflect::Detail::PointerImpl* getPointerImpl() = 0;
        virtual _Ret_notnull_ VersionedTypeBuilder* finishTypeBuilder(_Inout_ VersionedTypeBuilder* typeIn) const = 0;
        virtual BasicTypeCache& getBasicTypeCache() = 0;
        virtual CompoundTypeCache& getCompoundTypeCache() = 0;
    };

    template<typename T>
    void grow(hkArray<T>& a, int i, const T& fill = T())
    {
        a.setSize(hkMath::max2(i + 1, a.getSize()), fill);
    }

    struct WorkBundle : public hkSerialize::Detail::DefaultBundle
    {
        WorkBundle(_In_opt_ VersionBundle::Impl* parent)
            : m_parent(parent)
            , m_allocator(hkMemHeapAllocator())
        {
            m_tidFromType.insert(HK_NULL, 0);
        }

        void setCopied(int index)
        {
            const int numToExpand = 1 + index - m_copied.getSize();
            if (numToExpand > 0)
            {
                const bool notCopied = false;
                m_copied.expandBy(numToExpand, notCopied);
            }
            m_copied[index] = true;
        }

        bool wasCopied(int index)
        {
            if (index < m_copied.getSize())
            {
                return m_copied[index];
            }

            return false;
        }

        int addType(_In_ const hkReflect::Type* t)
        {
            const int newVal = m_tidFromType.getSize();
            hkMap<const hkReflect::Type*, int>::Iterator it = m_tidFromType.findOrInsertKey(t, newVal);
            const int ret = m_tidFromType.getValue(it);
            if(ret == newVal)
            {
                setType(ret, const_cast<hkReflect::Type*>(t)); // ??
            }
            return ret;
        }

        virtual hkReflect::Var getNoteOnPointer(const hkReflect::PointerVar& ptr) const override;
        hkArray<bool> m_copied;
        hkMap<const hkReflect::Type*, int> m_tidFromType;
        VersionBundle::Impl* m_parent;
        hkTransientAllocator m_allocator;
    };

    struct VersionBundle::Impl : public VersionedTypeStorage
    {
        HK_DECLARE_CLASS(Impl, New);
        class CopyInputObjectsInterpreter;

        Impl(hkReflect::Version::PatchRegistry& patchReg, hkReflect::MutableTypeReg& typeReg)
            : m_patchReg(patchReg)
            , m_typeReg(typeReg)
            , m_fieldAllocator(hkMemHeapAllocator())
            , m_pointerImpl(*this)
            , m_arrayImpl(*this)
            , m_functionHelper(*this)
            , m_stringImpl(&m_fieldAllocator)
            , m_workBundle(this)
            , m_finishedRunningPatches(false)
            , m_compoundTypeCache(*this)
        {
            hkTypeVm::addDefaultPasses(m_compiler);
            m_compiler.addPass<hkTypeVm::IntAndFloatConversionPass>();
            m_compiler.excludeNonSerializableFields = true;
            m_compiler.addInstrWithNoSource = false;
        }

        virtual ~Impl();

        hkViewPtr<hkSerialize::Bundle> applyPatchesToBundle(hkViewPtr<hkSerialize::Bundle> bundleIn, bool failIfVersioningRequired);
        void recursivelyAddTypesToSet(hkSet<const hkReflect::Type*>& alreadyDone, _In_opt_ const hkReflect::Type* t);
        //void applyPatchToTypes(const hkReflect::Version::PatchInfo* p);
        void finalizeAndCopyTypes();
        void copyInputObjects(hkViewPtr<hkSerialize::Bundle> bundleIn);
        _Ret_notnull_ hkReflect::Type* chainingCopyType(_In_ const hkReflect::Type* srcType);
        void makeVarNCopy(int index, const hkSerialize::VarN& varN, _Inout_opt_ VersionedTypeBuilder* builder = HK_NULL);
        void makeVarCopy(int index, const hkReflect::Var& thisVar, _Inout_opt_ VersionedTypeBuilder* builder = HK_NULL);
        void makeNoteCopy(int varIndex, const hkSerialize::Bundle::Item& item, _Inout_ VersionedTypeBuilder* builder, hkArray<hkSerialize::Bundle::Item>& items);

        _Ret_maybenull_ const hkReflect::Type* getOrCreate(_In_ const hkReflect::Type* type);

        int getVarIndexFromSourceAddress(_In_ void* sourceAddress);
        // Reserve a new index
        int getNewVarIndex();
        int getArrayIndexFromSourceAddress(_In_ void* sourceAddress);

        hkResult applyPatchToObjects(_In_ const hkReflect::Version::PatchInfo* p);

        hkReflect::Version::PatchRegistry& m_patchReg;
        hkReflect::MutableTypeReg& m_typeReg;
        hkMap<hkUint64, const hkReflect::Type*> m_startingUidsInFile;
        hkMap<hkUint64, hkUint64> m_parentUidFromStartingUid;
        // Anything in this set is a foreign type
        hkSet<hkUint64> m_foreignUids;
        hkSet<hkUint64> m_dynamicUids;
        hkSet<hkUint64> m_subsequentUidsInFile;
        // Does this need to be a member
        hkSet<hkUint64> m_startingUidsFromPatches;
        hkSet<hkUint64> m_subsequentUidsFromPatches;
        hkMap<hkUint64, int> m_typeIndexFromUid; // Needed out here??
        hkArray<VersionedTypeBuilder*> m_typeBuilders;
        _Ret_maybenull_ VersionedTypeBuilder* getTypeBuilderForCompletedType(_In_ const hkReflect::Type* completedType) const { return m_typeBuilderFromCompletedType.getWithDefault(completedType, HK_NULL); }

        hkTransientAllocator m_fieldAllocator;
        hkPointerMap<const hkReflect::Type*, VersionedTypeBuilder*> m_typeBuilderFromCompletedType;

        struct PointerImpl : public hkReflect::Detail::PointerImpl
        {
            PointerImpl(VersionBundle::Impl& parent) : m_parent(parent)
            {
            }

            virtual hkResult setValue(_Inout_ void* self, _In_ const hkReflect::PointerType* ptrType, const hkReflect::Var& var) const HK_OVERRIDE
            {
                if (var.getAddress())
                {
                    if (const hkReflect::Type* t = hkReflect::Detail::asType(var))
                    {
                        // Type pointers are stored directly
                        const hkReflect::Type* subType = ptrType->getSubType();

                        if (subType && hkReflect::Detail::isTypeOfType(subType))
                        {
                            *reinterpret_cast<const hkReflect::Type**>(self) = t;
                            return HK_SUCCESS;
                        }
                        return HK_FAILURE;
                    }
                    else
                    {
                        int id;
                        if (m_parent.m_indexFromAddressMap.get(var.getAddress(), &id).isFailure())
                        {
                            return HK_FAILURE;
                        }

                        *reinterpret_cast<hkUint32Le*>(self) = id;
                    }
                }
                else
                {
                    *reinterpret_cast<hkUint32Le*>(self) = 0;
                }
                return HK_SUCCESS;
            }

            virtual hkResult getValue(_In_ const void* self, _In_ const hkReflect::PointerType* ptrType, _Out_ hkReflect::Var* val) const HK_OVERRIDE
            {
                const hkReflect::Type* ptd = ptrType->getSubType();

                if (ptd && hkReflect::Detail::isTypeOfType( ptd ))
                {
                    // Type pointers are just stored directly
                    // This could be a different impl
                    if(const hkReflect::Type* pointedToType = *reinterpret_cast<const hkReflect::Type* const *>(self))
                    {
                        if (pointedToType->isField())
                        {
                            hkReflect::FieldDecl fd(pointedToType);
                            // Null field name = field deleted, decl context = -1 -> deleted type. Either way we return null
                            if ((!fd.getName()) || (fd.getDeclContext()->getVersion() == -1))
                            {
                                *val = hkReflect::Var();
                                return HK_SUCCESS;
                            }
                            *val = hkReflect::Var(pointedToType, hkReflect::typeFromKind(pointedToType->getKind()));
                            return HK_SUCCESS;
                        }
                        else
                        {
                            if (pointedToType->getVersion() == -1)
                            {
                                *val = hkReflect::Var((void*)HK_NULL, hkReflect::getType<hkReflect::Type>());
                                return HK_SUCCESS;
                            }
                            else
                            {
                                *val = hkReflect::Var(pointedToType, hkReflect::typeFromKind(pointedToType->getKind()));
                                return HK_SUCCESS;
                            }
                        }
                    }
                    else
                    {
                        *val = hkReflect::Var((void*)HK_NULL, ptd);
                        return HK_SUCCESS;
                    }
                }
                else
                {
                    hkUint32 id = 0;
                    if (self)
                    {
                        id = *reinterpret_cast<const hkUint32Le*>(self);
                    }

                    if(id)
                    {
                        if (int annotated = m_parent.m_workBundle.item(id).getAnnotated())
                        {
                            *val = m_parent.m_workBundle.var(annotated);
                            return HK_SUCCESS;
                        }
                        else
                        {
                            // We don't want to do this when we are at the output
                            if(!m_parent.allPatchesHaveBeenApplied())
                            {
                                m_parent.makeVarCopy(id, m_parent.m_workBundle.var(id));
                            }
                            *val = m_parent.m_workBundle.var(id);
                            return HK_SUCCESS;
                        }
                    }

                    *val = hkReflect::Var((void*)HK_NULL, ptd); // Typed Null pointer
                    return HK_SUCCESS;
                }
            }

            _Ret_maybenull_ virtual void* queryInterfaceImpl(_In_ const hkReflect::Type* type, const hkReflect::Var& self) const HK_OVERRIDE
            {
                if (type->extendsOrEquals<hkSerialize::Detail::IdFromPointer>())
                {
                    if (self)
                    {
                        hkUint32 id = *reinterpret_cast<const hkUint32Le*>(self.getAddress());
                        if (id)
                        {
                            return reinterpret_cast<hkSerialize::Detail::IdFromPointer*>(self.getAddress());
                        }
                    }
                }
                return HK_NULL;
            }
            VersionBundle::Impl& m_parent;
        } m_pointerImpl;

        struct ArrayImpl : public hkReflect::Detail::ArrayImpl
        {
            ArrayImpl(VersionBundle::Impl& parent) : m_parent(parent)
            {
            }

            ~ArrayImpl()
            {
            }

            virtual hkResult getValue(_In_opt_ const void* arrAddr, _In_opt_ const hkReflect::ArrayType* arrType, _Out_ hkReflect::ArrayValue* val) const HK_OVERRIDE
            {
                if (arrAddr)
                {
                    const hkUint32 id = *reinterpret_cast<const hkUint32Le*>(arrAddr);
                    if (id > 0)
                    {
                        // We don't want to do this when we are at the output
                        if (!m_parent.allPatchesHaveBeenApplied())
                        {
                            m_parent.makeVarNCopy(id, m_parent.m_workBundle.varn(id));
                        }

                        VarN vn = m_parent.m_workBundle.varn(id);
                        *val = hkReflect::ArrayValue(vn.getAddress(), vn.getCount(), vn.getType());
                        return HK_SUCCESS;
                    }
                }
                *val = hkReflect::ArrayValue();
                return HK_SUCCESS;
            }

            virtual hkResult setNumElements(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, int newSize) const HK_OVERRIDE
            {
                hkUint32 id = 0;
                if (arrAddr)
                {
                    id = *reinterpret_cast<const hkUint32Le*>(arrAddr);
                    if(id)
                    {
                        HK_ASSERT_NO_MSG(0x427b0574, m_parent.m_workBundle.wasCopied(id)); // TODO.ntm for now, although we should handle this
                    }
                }
                int oldElementCount = 0;
                void* oldData = HK_NULL;
                if (id)
                {
                    //m_parent.removeObjectFromTrackingMap(m_parent.m_bundle.extra(id).getFirst());
                    oldElementCount = m_parent.m_workBundle.varn(id).getCount();
                    oldData = m_parent.m_workBundle.varn(id).getAddress();
                }

                if (arrType->getSubType()->asVoid())
                {
                    Log_Error("Attempting to set the size of an untyped array");
                    return HK_FAILURE;
                }

                const int elementSize = arrType->getSubType()->getSizeOf();
                void* newData = m_parent.m_fieldAllocator.blockAlloc(elementSize * newSize);//m_parent.getStorage()->m_allocator.blockAlloc(elementSize * newSize);

                hkString::memSet(hkAddByteOffset(newData, elementSize * oldElementCount), 0, (newSize > oldElementCount) ? (elementSize * (newSize - oldElementCount)) : 0);

                if (id)
                {
                    hkString::memCpy(newData, oldData, (newSize > oldElementCount ? oldElementCount : newSize) * elementSize);

                    int tid = m_parent.m_workBundle.addType(arrType->getSubType());
                    m_parent.m_workBundle.setVarN(id, newData, tid, newSize);
                }
                else
                {
                    const int newId = m_parent.m_workBundle.getSize();
                    int tid = m_parent.m_workBundle.addType(arrType->getSubType());
                    m_parent.m_workBundle.setVarN(newId, newData, tid, newSize);
                    m_parent.m_workBundle.setCopied(newId);

                    *reinterpret_cast<hkUint32Le*>(arrAddr) = newId; // Stash the id here so the array impl can find it
                }

                return HK_SUCCESS;
            }

            virtual hkResult spliceInto(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, int index, int numToDel, const hkReflect::ArrayValue& val) const HK_OVERRIDE;

            virtual AllocResult allocateElements(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, hkReflect::QualType elemType, int newSize) const HK_OVERRIDE
            {
                return setNumElements(arrAddr, arrType, newSize).isSuccess() ? ALLOC_SUCCESS : ALLOC_FAILURE;
            }
            VersionBundle::Impl& m_parent;
        } m_arrayImpl;

        struct FunctionHelper : public hkSerialize::PatchFunctionHelper
        {
            FunctionHelper(VersionBundle::Impl& owner) : m_owner(owner) {}

            virtual hkReflect::ArrayValue newArray(_In_ const hkReflect::Type* t, int cardinality) HK_OVERRIDE;
            virtual hkReflect::ArrayValue newArray(_In_z_ const char* typeName, int cardinality) HK_OVERRIDE;

            hkReflect::ArrayValue newArray(_Inout_ VersionedTypeBuilder* builder, int cardinality);

            void setDependencies(_In_ hkPatchDependencies* dependencies)
            {
                m_dependencies = dependencies;
            }

            virtual PointerValue getPointerValue(hkReflect::Var v) override;
            virtual hkResult setPointerValue(hkReflect::Var v, PointerValue pv) override;
            virtual _Ret_notnull_ const hkReflect::PointerType* getPointerTypeTo(_In_z_ const char* basename, _In_ const hkReflect::Type* pointedType) override;
            virtual _Ret_notnull_ const hkReflect::ArrayType* getArrayTypeTo(_In_z_ const char* arrayName, _In_ const hkReflect::Type* subType, int fixedCount ) override;

            VersionBundle::Impl& m_owner;
            hkPatchDependencies* m_dependencies;
        } m_functionHelper;

        hkReflect::Detail::HavokStringImpl m_stringImpl;

        // TODO : cleanup
        hkResult recursivelyProcessFieldsAndParents(_Inout_ VersionedTypeBuilder* thisBuilder, hkSet<VersionedTypeBuilder*>& alreadyDone, hkSet<VersionedTypeBuilder*>& currentlyProcessing);
        hkResult recursivelyProcessField(_Inout_ hkVersionBundleUtil::Field* fieldType, hkSet<VersionedTypeBuilder*>& alreadyDone, hkSet<VersionedTypeBuilder*>& currentlyProcessing);

        _Ret_notnull_ hkReflect::Type* copyAndAddOptional(_In_ const hkReflect::Type* srcType, hkReflect::Optional opt, _In_ const void* val);

        int getIndexForNoteOnPointer(hkReflect::PointerVar ptr);

        hkTypeVm::Compiler m_compiler;
        //hkSerialize::Detail::DefaultIndexedBundle m_retBundle;
        hkViewPtr<hkSerialize::Bundle> m_bundleIn;
        WorkBundle m_workBundle;
        hkPointerMap<void*, int> m_indexFromAddressMap;
        hkPointerMap<const hkReflect::Type*, hkReflect::Type*> m_chainedCopiedTypesFromSrc;

        // VersionedTypeStorage interface
        virtual hkMemoryAllocator& getAllocator() override { return m_fieldAllocator; }
        virtual _Ret_notnull_ VersionedTypeBuilder* getTypeForNameVer(_In_z_ const char* name, int version) override;
        virtual hkReflect::Var getWorkVar(int id) override { return m_workBundle.var(id); }
        virtual hkSerialize::VarN getWorkExtra(int id) override { return m_workBundle.varn(id); }
        virtual _Ret_notnull_ hkReflect::Detail::ArrayImpl* getArrayImpl() override { return &m_arrayImpl; }
        virtual _Ret_notnull_ hkReflect::Detail::StringImpl* getStringImpl() override { return &m_stringImpl; }
        virtual _Ret_notnull_ hkReflect::Detail::PointerImpl* getPointerImpl() override { return &m_pointerImpl; }
        virtual _Ret_notnull_ VersionedTypeBuilder* finishTypeBuilder(_Inout_ VersionedTypeBuilder* builder) const override { return builder; }
        virtual BasicTypeCache& getBasicTypeCache() override { return m_basicTypeCache; }
        virtual CompoundTypeCache& getCompoundTypeCache() override { return m_compoundTypeCache; }
        // End

        class CompatTypeReg : public hkReflect::MutableTypeReg, public VersionedTypeStorage
        {
        public:
            HK_DECLARE_CLASS(CompatTypeReg, New);

            CompatTypeReg(_Inout_ VersionBundle::Impl* parent);
            ~CompatTypeReg();


            // MutableTypeReg
            virtual _Ret_maybenull_ const hkReflect::Type* typeFromName(_In_z_ const char* name) const override;
            virtual _Ret_maybenull_ const hkReflect::Type* typeFromType(_In_ const hkReflect::Type* type) const override
            {
                hkStringBuf fn;
                const char* n = type->getFullName(fn);
                const hkReflect::Type* ret = typeFromName(n);

                return ret;
            }

            virtual hkResult add(_In_ hkReflect::Type* type) override { return HK_FAILURE; }
            virtual hkResult remove(_In_ hkReflect::Type* type) override { return HK_FAILURE;  }
            virtual Subscription subscribeForChange(hkReflect::MutableTypeReg::TypesChangedFunc cbFunc, _In_opt_ void* cbData) override { return Subscription(); }
            virtual void notifyTypesMutated() override {}
            virtual void appendAllTypesToArray(hkArray<const hkReflect::Type*>& types) const override;
            // End


            virtual hkMemoryAllocator& getAllocator() override { return m_parent->m_fieldAllocator; }
            virtual _Ret_maybenull_ VersionedTypeBuilder* getTypeForNameVer(_In_opt_z_ const char* name, int version) override;
            virtual hkReflect::Var getWorkVar(int id) override { return m_parent->getWorkVar(id); }
            virtual hkSerialize::VarN getWorkExtra(int id) override { return m_parent->getWorkExtra(id); }
            virtual _Ret_notnull_ hkReflect::Detail::ArrayImpl* getArrayImpl() override { return m_parent->getArrayImpl(); }
            virtual _Ret_notnull_ hkReflect::Detail::StringImpl* getStringImpl() override { return m_parent->getStringImpl(); }
            virtual _Ret_notnull_ hkReflect::Detail::PointerImpl* getPointerImpl() override { return m_parent->getPointerImpl(); }
            virtual _Ret_notnull_ VersionedTypeBuilder* finishTypeBuilder(_Inout_ VersionedTypeBuilder* builder) const override;
            virtual VersionedTypeStorage::BasicTypeCache& getBasicTypeCache() override { return m_basicTypeCache; }
            virtual VersionedTypeStorage::CompoundTypeCache& getCompoundTypeCache() override { return m_compoundTypeCache; }

            VersionBundle::Impl* m_parent;
            hkArray<int> m_startingPatchIndices;
            hkStringMap<int> m_typeBuilderIndexFromName;
            hkMap<hkUint64, int> m_typeIndexFromUid; // Needed out here??
            mutable hkArray<VersionedTypeBuilder*> m_typeBuilders;
            VersionedTypeStorage::BasicTypeCache m_basicTypeCache;
            VersionedTypeStorage::CompoundTypeCache m_compoundTypeCache;
        };
        hkRefPtr<CompatTypeReg> m_compatTypeReg;

        _Ret_notnull_ hkReflect::MutableTypeReg* createBackwardCompatibleTypeReg()
        {
            if (!m_compatTypeReg)
            {
                m_compatTypeReg = hkAutoRemoveReference(new CompatTypeReg(this));
            }
            return m_compatTypeReg;
        }

        bool allPatchesHaveBeenApplied() const { return m_finishedRunningPatches; }
        bool m_finishedRunningPatches;

        BasicTypeCache m_basicTypeCache;
        CompoundTypeCache m_compoundTypeCache;
    };

    class VersionedTypeBuilder
    {
    public:
        HK_DECLARE_CLASS(VersionedTypeBuilder, New);
        VersionedTypeBuilder(_In_ VersionedTypeStorage* owner)
            : m_owner(owner)
            , m_version(-1)
            , m_startingVersion(-1)
            , m_lastParentFromNativeType(false)
            , m_currentParentIndex(-1)
            , m_nextType(HK_NULL)
            , m_startedFromFile(false)
            , m_typeIsForeign(false)
            , m_typeIsDynamic(false)
            , m_completedType(HK_NULL)
            , m_templateParametersObject(HK_NULL)
            , m_localFieldsRequireCopying(false)
        {
        }

        // TODO: Make a constructor, we always have to call this
        void setFromPatchInfo(_In_z_ const char* startingName, int startingVersion)
        {
            m_name = m_startingName = startingName;
            m_version = m_startingVersion = startingVersion;
        }

        bool startedFromFileType() const { return m_startedFromFile; }
        bool isForeign() const { return m_typeIsForeign; }
        bool isDynamic() const { return m_typeIsDynamic; }

        _Ret_maybenull_ const hkReflect::Type* turnFieldIntoRealType(_Inout_ hkVersionBundleUtil::Field* field, _Inout_ VersionedTypeStorage* owner)
        {
            hkMemoryAllocator& alloc = owner->getAllocator();
            const hkReflect::Type* subType = HK_NULL;

            hkStringBuf baseName(field->m_typeName);
            int paramStart = baseName.indexOf('<');
            bool hasTemplateParams = paramStart >= 0;
            if (hasTemplateParams)
            {
                baseName.setLength(paramStart);
            }

            switch (field->m_typeKind)
            {
            case hkVersionBundleUtil::Field::TYPE_BASIC:
            {
                if (field->getTypeAsBasic()->asString())
                {
                    hkReflect::BasicStringType* newSt = alloc.create<hkReflect::BasicStringType>(hkSizeOf(hkStringPtr), (int)HK_ALIGN_OF(hkStringPtr)
                        , owner->getStringImpl(), hkReflectUtil::internCopy(field->m_typeName)
                        , true, hkReflect::Type::TYPE_TEMPORARY); // const char*
                    return newSt;
                }
                else
                {
                    return field->getTypeAsBasic();
                }
            }

            case hkVersionBundleUtil::Field::TYPE_POINTER:
            {
                if (hkVersionBundleUtil::Field* subField = field->getTypeAsField())
                {
                    subType = turnFieldIntoRealType(subField, (owner));
                }

                return m_owner->getCompoundTypeCache().getVersionedPointerType( baseName, subType, hasTemplateParams );
            }

            case hkVersionBundleUtil::Field::TYPE_TYPE_POINTER:
            {
                return m_owner->getCompoundTypeCache().getVersionedPointerType( baseName, hkReflect::getType<hkReflect::Type>(), hasTemplateParams );
            }

            case hkVersionBundleUtil::Field::TYPE_STRUCT:
            {
                VersionedTypeBuilder* builder = field->getTypeAsStruct<VersionedTypeBuilder*>();
                builder = m_owner->finishTypeBuilder(builder);
                return builder->m_completedType;
            }

            case hkVersionBundleUtil::Field::TYPE_HKARRAY:
            case hkVersionBundleUtil::Field::TYPE_CARRAY:
            {
                if ( hkVersionBundleUtil::Field* subField = field->getTypeAsField() )
                {
                    subType = turnFieldIntoRealType( subField, ( owner ) );
                }
                return m_owner->getCompoundTypeCache().getVersionedArrayType( baseName, subType, hasTemplateParams, field->m_tuples );
            }

            case hkVersionBundleUtil::Field::TYPE_SPECIAL:
            {
                return field->getTypeAsSpecial();
            }

            default:
                HK_ERROR(0x33b1b823, "Invalid field type");
                return HK_NULL;
            }
        }

        class DescriptionCB : public hkVersionBundleUtil::TypeFromNameCb
        {
        public:
            DescriptionCB(hkPatchDependencies& deps, VersionedTypeStorage* owner)
                : m_deps(deps), m_owner(owner), m_basicTypeCache(owner->getBasicTypeCache()) {}
            enum
            {
                TypeIsNotFound = 0,
                TypeIsNotBasic = 8
            };

            virtual _Ret_maybenull_ void* getTypeFromName(_In_z_ const char* typeName) override
            {
                const int version = m_deps.getVersion(typeName);
                return m_owner->getTypeForNameVer(typeName, version);
            }

            // TODO: There should be a better way to do this
            _Ret_maybenull_ const hkReflect::Type* checkForBasicType(_In_z_ const char* strName) override
            {
                hkStringBuf str(strName);
                str.trimEnd();

                VersionedTypeStorage::BasicTypeCache::Iterator it = m_basicTypeCache.findOrInsertKey( hkReflectUtil::internCopy(str), (const hkReflect::Type*)TypeIsNotFound);
                const hkReflect::Type* lookedUpType = m_basicTypeCache.getValue(it);
                if (lookedUpType == (const hkReflect::Type*)TypeIsNotFound)
                {
                    // First time we have seen this type
                    if(const hkReflect::Type* builtInType = hkReflect::typeFromName(str))
                    {
                        // Int, float, bool, strings are builtin and never change
                        if (builtInType->asInteger() || builtInType->asFloat() || builtInType->asBool() || builtInType->asString())
                        {
                            m_basicTypeCache.setValue(it, builtInType);
                            return builtInType;
                        }

                        // Fixed size arrays of floats are builtin and never change
                        if (const hkReflect::ArrayType* at = builtInType->asArray())
                        {
                            if(hkVersionBundleUtil::isSpecialFloatArray(at))
                            {
                                m_basicTypeCache.setValue(it, builtInType);
                                return builtInType;
                            }
                        }
                    }

                    // Everything else is not basic
                    m_basicTypeCache.setValue(it, (const hkReflect::Type*)TypeIsNotBasic);
                    return HK_NULL;
                }
                else if (lookedUpType == (const hkReflect::Type*)TypeIsNotBasic)
                {
                    return HK_NULL;
                }
                else
                {
                    // It it basic and we have seen it before
                    return lookedUpType;
                }
            }

            hkPatchDependencies& m_deps;
            VersionedTypeStorage* m_owner;
            hkSerialize::VersionedTypeStorage::BasicTypeCache& m_basicTypeCache;
        };

        void addFieldFromTypeDescription(hkPatchDependencies& dependencies, _In_z_ const char* name, _In_z_ const char* typeName, int position = -1)
        {
            DescriptionCB cb(dependencies, m_owner);
            if(hkVersionBundleUtil::Field* newField = hkVersionBundleUtil::Field::createFromTypeDescription(cb, name, typeName))
            {
                if (position >= 0)
                {
                    m_fields.expandAt(position, 1)->reset(newField);
                }
                else
                {
                    m_fields.expandOne().reset(newField);
                }
            }
            else
            {
                Log_Warning("Versioning was unable to create field {} in type {} ({})", name, m_name, typeName);
            }
        }

        class StaticTypeCB : public hkVersionBundleUtil::TypeFromTypeCb
        {
        public:
            StaticTypeCB(_In_ VersionedTypeStorage* owner) : m_owner(owner) {}

            virtual _Ret_maybenull_ void* getTypeFromType(_In_ const hkReflect::Type* rt) override
            {
                hkStringBuf sb;
                return m_owner->getTypeForNameVer(rt->getFullName(sb), rt->getVersion());
            }

            VersionedTypeStorage* m_owner;
        };

        void addFieldFromStaticType(_In_z_ const char* name, _In_ const hkReflect::Type* staticType, int position = -1)
        {
            StaticTypeCB cb(m_owner);
            if(hkVersionBundleUtil::Field* newField = hkVersionBundleUtil::Field::createFromStaticType(cb, name, staticType))
            {
                if (position >= 0)
                {
                    m_fields.expandAt(position, 1)->reset(newField);
                }
                else
                {
                    m_fields.expandOne().reset(newField);
                }
            }
        }

        _Ret_maybenull_ hkVersionBundleUtil::Field* findField(_In_z_ const char* name)
        {
            for (auto& f : m_fields)
            {
                if (f->m_name && (hkString::strCmp(f->m_name, name) == 0)) // TODO NOT ALL THESE STRCMPS GET A PROPER INTERFACE
                {
                    return f.get();
                }
            }
            return HK_NULL;
        }

        enum PatchingBehavior
        {
            DEFAULT_BEHAVIOR = 0,
            REMOVE_ADDED_MEMBERS = 1
        };
        hkResult applyPatchToType(_In_ const hkReflect::Version::PatchInfo* patch, PatchingBehavior removeAddedMembers);
        void addFieldsFromNativeType(_In_ const hkReflect::Type* nativeType, bool includeSerializeFalseFields);

        void processField(_Inout_ hkVersionBundleUtil::Field* f, _In_ const hkReflect::Type* completedType, int varId, int offset, int count)
        {
            if (f->m_typeKind == hkVersionBundleUtil::Field::TYPE_STRUCT)
            {
                //const int fieldOffset = completedType.getOffset();
                f->getTypeAsStruct<VersionedTypeBuilder*>()->addTrackedField(varId, offset, count);
            }
            else if (f->m_typeKind == hkVersionBundleUtil::Field::TYPE_CARRAY)
            {
                //XXX Make this recursive. Probably wants to be a method on the field, not here XXX

                const hkReflect::ArrayType* at = completedType->asArray();
                const int fieldSize = at->getSubType()->getSizeOf();
                for (int i = 0; i < f->m_tuples; i++)
                {
                    processField(f->getTypeAsField(), at->getSubType(), varId, offset + (i * fieldSize), count);
                }
            }

        }

        // I don't like this
        void recursivelyAddMembersToTracking(int varId, int offset, int count)
        {
            //m_workVarIds.expandOne() = { varId, offset };
            for (hkScopedPtr<hkVersionBundleUtil::Field>& f : m_fields)
            {
                if(hkReflect::FieldDecl fd = m_completedType->findField(f->m_name, false))
                {
                    processField(f.get(), fd.getType(), varId, offset + fd.getOffset(), count);
                }
                // Else we don't care
            }
        }

        // Add a complete Var to tracking
        void addTrackedVar(int varId)
        {
            bool found = false;
            for (auto i = m_workObjectsFromId.findKey(varId); m_workObjectsFromId.isValid(i); i = m_workObjectsFromId.getNext(i, varId))
            {
                m_workObjectsFromId.getValueRef(i).setVar(varId, 0);
                found = true;
            }

            if(!found)
            {
                TrackedObject obj;
                obj.setVar(varId, 0);
                m_workObjectsFromId.insert(varId, obj);
            }

            if (m_currentParentIndex >= 0)
            {
                if(m_parents[m_currentParentIndex])
                {
                    m_parents[m_currentParentIndex]->addTrackedVar(varId);
                }
            }

            // We have added the outer struct, now we need to look at members
            recursivelyAddMembersToTracking(varId, 0, -1);
        }

        void addTrackedField(int varId, int offset, int count)
        {
            TrackedObject obj;  obj.setVarN(varId, offset, count);
            m_workObjectsFromId.insert(varId, obj);

            // Also add the parent fields
            if (m_currentParentIndex >= 0)
            {
                if (m_parents[m_currentParentIndex])
                {
                    m_parents[m_currentParentIndex]->addTrackedField(varId, offset, count);
                }
            }

            if(count >= 0)
            {
                // m_owner->getWorkExtra(varId) hasn't been set yet
                recursivelyAddMembersToTracking(varId, offset, count);
            }
            else
            {
                recursivelyAddMembersToTracking(varId, offset, -1);
            }
        }

        void addTrackedExtra(int extraId, int offset, int count) // Offset into EACH member
        {
            bool found = false;
            for (auto i = m_workObjectsFromId.findKey(extraId); m_workObjectsFromId.isValid(i); i = m_workObjectsFromId.getNext(i, extraId))
            {
                m_workObjectsFromId.getValueRef(i).setVarN(extraId, offset, count);;
                found = true;
            }

            if(!found)
            {
                TrackedObject obj;  obj.setVarN(extraId, offset, count);
                m_workObjectsFromId.insert(extraId, obj);
            }

            if (m_currentParentIndex >= 0)
            {
                if (m_parents[m_currentParentIndex])
                {
                    m_parents[m_currentParentIndex]->addTrackedExtra(extraId, offset, count);
                }
            }

            recursivelyAddMembersToTracking(extraId, offset, count);
        }

        void getAllTrackedVars(hkArray<hkReflect::Var>::Temp& varsOut)
        {
            for(auto it = m_workObjectsFromId.getIterator(); m_workObjectsFromId.isValid(it); it = m_workObjectsFromId.getNext(it))
            {
                const TrackedObject& to = m_workObjectsFromId.getValueRef(it);
                if (to.m_count >= 0)
                {
                    const hkSerialize::VarN& vn = m_owner->getWorkExtra(to.m_id);
                    
                    HK_ASSERT_NO_MSG(0xc747fd5c, vn.getCount() == to.m_count); // Redundant??
                    for (int i = 0; i < vn.getCount(); i++)
                    {
                        if(vn.get(i).getAddress())
                        {
                            varsOut.expandOne() = hkReflect::Var(hkAddByteOffset(vn.get(i).getAddress(), to.m_offset), m_completedType);
                        }
                    }
                }
                else
                {
                    if(m_owner->getWorkVar(to.m_id).getAddress())
                    {
                        varsOut.expandOne() = hkReflect::Var(hkAddByteOffset(m_owner->getWorkVar(to.m_id).getAddress(), to.m_offset), m_completedType);
                    }
                }
            }
        }

        void addChildrenToTracking(_In_ VersionedTypeBuilder* childType)
        {
            for (auto i = childType->m_workObjectsFromId.getIterator(); childType->m_workObjectsFromId.isValid(i); i = childType->m_workObjectsFromId.getNext(i))
            {
                const TrackedObject& to = childType->m_workObjectsFromId.getValueRef(i);

                m_workObjectsFromId.insert(to.m_id, to);
            }
        }

        void removeChildrenFromTracking(_Inout_ VersionedTypeBuilder* childType)
        {
            for (auto i = childType->m_workObjectsFromId.getIterator(); childType->m_workObjectsFromId.isValid(i); i = childType->m_workObjectsFromId.getNext(i))
            {
                const auto& childObj = childType->m_workObjectsFromId.getValueRef(i);
                while (m_workObjectsFromId.remove(childObj.m_id, childObj).isSuccess())
                {
                    // Do nothing, just keep removing
                }
            }

        }

        void addCombinedType(_In_ VersionedTypeBuilder* typeBeingAdded)
        {
            VersionedTypeBuilder* nt = this;
            for (; nt->m_nextType; nt = nt->m_nextType)
            {
                if (nt == typeBeingAdded)
                {
                    return;
                }
            }
            if (nt == typeBeingAdded)
            {
                return; // Should this be an error?
            }

            nt->m_nextType = typeBeingAdded;
        }

        void addPatchIndex(int index)
        {
            m_patchIds.pushBack(index);
        }


        void allocateRecord()
        {
            hkReflect::TypeBuilder builder;

            if (m_templateParameters.getSize())
            {
                hkStringBuf nameWithoutParams(m_startingName);
                int paramStart = nameWithoutParams.indexOf('<');
                if (paramStart >= 0)
                {
                    nameWithoutParams.setLength(paramStart);
                    m_startingName = nameWithoutParams;
                }
            }
            builder.beginRecord(hkReflectUtil::internCopy(m_startingName), HK_NULL); // Start from the FIRST type name, not the last
                                                                       //builder.addItem<hkReflect::Opt::ALLOC_IMPL>(&m_allocImpl);
            builder.addItem<hkReflect::Opt::VERSION>(m_startingVersion);
            builder.addItem<hkReflect::Opt::IMPL>(&hkReflect::Detail::HavokRecordImpl::s_instance);
            builder.setTypeWorld(m_owner);
            builder.setItem<hkReflect::Opt::EXACT_TYPE>(HK_NULL);
            builder.addFlags(hkReflect::Type::TYPE_TEMPORARY);

            if (isForeign())
            {
                builder.addFlags(hkReflect::Type::TYPE_FOREIGN);
                builder.addFlags(hkReflect::Type::TYPE_DYNAMIC);
            }
            else if (isDynamic())
            {
                builder.addFlags(hkReflect::Type::TYPE_DYNAMIC);
            }

            if (m_templateParameters.getSize())
            {
                builder.addItem<hkReflect::Opt::TEMPLATE>(HK_NULL);
            }

            hkReflect::Type* rec = builder.allocate(&m_owner->getAllocator());
            m_completedType = rec;
            m_nextType = HK_NULL; // Start again for object patching
        }

        void allocateFieldsAndParameters()
        {
            hkArray<hkReflect::Detail::RecordFieldPod> fieldTypes;
            for (hkScopedPtr<hkVersionBundleUtil::Field>& field : m_fields)
            {
                if(field->m_name)
                {
                    if (field->m_typeKind == hkVersionBundleUtil::Field::TYPE_TYPE_POINTER)
                    {
                        // Anything with a type pointer needs to be copied
                        m_localFieldsRequireCopying = true;
                    }
                    hkReflect::TypeBuilder fieldBuilder;

                    const hkReflect::Type* realType = turnFieldIntoRealType( field.get(), m_owner );
                    fieldBuilder.beginDerived(realType);

                    // Opaque fields are non-serializable Decls which have passed through versioning.
                    fieldBuilder.setField(field->m_name, 0, realType->asOpaque() ?
                        (hkReflect::Decl::DECL_PROPERTY_FIELD | hkReflect::Decl::DECL_NOT_SERIALIZABLE) : 0); // offset?
                    fieldBuilder.addItem<hkReflect::Opt::ALLOC_IMPL>(HK_NULL); // prevent allocation of field types
                    fieldBuilder.setItem<hkReflect::Opt::DECL_CONTEXT>(m_completedType);

                    hkReflect::Detail::RecordFieldPod fieldPod = { fieldBuilder.allocate(&m_owner->getAllocator()) };
                    fieldTypes.pushBack(fieldPod);
                }
            }

            hkReflect::Detail::DeclsArray* fieldArray = hkReflect::Detail::DeclsArray::create(fieldTypes.begin(), fieldTypes.getSize(), 0, m_owner->getAllocator());
            hkReflect::TypeDetail::localSetOptional<hkReflect::Opt::DECLS>(m_completedType, fieldArray);

            if (m_templateParameters.getSize())
            {
                m_templateParametersObject = hkReflect::Template::create(m_templateParameters.getSize(), m_owner->getAllocator());
                for (int i = 0; i < m_templateParameters.getSize(); i++)
                {
                    // I created this, why do I have to const_cast it?
                    const_cast<hkReflect::Template::Parameter*>(m_templateParametersObject->getParam(i))->m_kindAndName = m_templateParameters[i].m_kindAndName;
                    if (m_templateParameters[i].m_field)
                    {
                        const hkReflect::Type* paramType = turnFieldIntoRealType(m_templateParameters[i].m_field.get(), m_owner);
                        HK_ASSERT_NO_MSG(0xc747fd5d, paramType->getName());
                        const_cast<hkReflect::Template::Parameter*>(m_templateParametersObject->getParam(i))->m_storage = hkUlong(paramType);
                    }
                    else
                    {
                        const_cast<hkReflect::Template::Parameter*>(m_templateParametersObject->getParam(i))->m_storage = m_templateParameters[i].m_value;
                    }
                }
                hkReflect::TypeDetail::localSetOptional<hkReflect::Opt::TEMPLATE>(m_completedType, m_templateParametersObject);
            }
        }

        hkResult setParentAndFinish(hkSet<VersionedTypeBuilder*>& currentlyProcessing)
        {
            if(currentlyProcessing.insert(this))
            {
                if (!(m_completedType->getSizeOf()))
                {
                    const char* nn = m_completedType->getName();
                    (void)nn;

                    hkSet<int> overriddenFields;

                    for (int i = 0; i < m_fields.getSize(); i++)
                    {
                        if (VersionedTypeBuilder* fieldType = m_fields[i]->getTypeAsStruct<VersionedTypeBuilder*>())
                        {
                            m_owner->finishTypeBuilder(fieldType);
                            if (fieldType->setParentAndFinish(currentlyProcessing).isFailure())
                            {
                                overriddenFields.insert(i);
                                
                                
                                
                                hkReflect::TypeDetail::setFieldType(m_completedType, m_fields[i]->m_name, hkReflect::getType<char[1024]>());
                            }
                            else
                            {
                                if (!m_localFieldsRequireCopying)
                                {
                                    m_localFieldsRequireCopying = fieldType->recursivelyCheckForFieldsNeedingCopy();
                                }
                            }
                        }
                    }

                    if (m_parents.getSize())
                    {
                        int biggestParentIndex = -1;
                        int biggestParentSize = -1;
                        bool foundOutsizedParent = false;
                        // This will not handle dependencies properly XXXXXXXX
                        for (int i = 0; i < m_parents.getSize(); i++)
                        {
                            if(m_parents[i])
                            {
                                m_owner->finishTypeBuilder(m_parents[i]);
                                if (m_parents[i]->setParentAndFinish(currentlyProcessing).isFailure())
                                {
                                    foundOutsizedParent = true;
                                    biggestParentSize = 1024;
                                    biggestParentIndex = i;
                                }
                                else
                                {
                                    if (m_parents[i]->m_completedType->getSizeOf() > biggestParentSize)
                                    {
                                        biggestParentSize = m_parents[i]->m_completedType->getSizeOf();
                                        biggestParentIndex = i;
                                    }
                                }

                                // If any parents are patched, we are patched
                                if (m_parents[i]->needsToBeCopied())
                                {
                                    m_localFieldsRequireCopying = true;
                                }
                            }
                        }

                        // Allocate enough space for the biggest parent, or reserve 1k if there is a dependency loop
                        if (foundOutsizedParent)
                        {
                            hkReflect::TypeDetail::setParent(m_completedType, hkReflect::getType<char[1024]>());
                        }
                        else
                        {
                            hkReflect::TypeDetail::setParent(m_completedType, m_parents[biggestParentIndex]->m_completedType);
                        }
                    }

                    hkReflect::RecordLayout::Options opts;
                    opts.setNative().withMinimumRecordSize( HK_POINTER_SIZE );
                    opts.m_reuseParentPadding = false;
                    hkReflect::RecordLayout::recompute( m_completedType, opts );
                    if (m_parents.getSize() && (m_parents[0]))
                    {
                        hkReflect::TypeDetail::setParent(m_completedType, m_parents[0]->m_completedType);
                    }
                    else
                    {
                        hkReflect::TypeDetail::setParent(m_completedType, HK_NULL); // Needed??
                    }

                    if (m_parents.getSize())
                    {
                        m_currentParentIndex = 0;
                    }

                    for (hkSet<int>::Iterator it = overriddenFields.getIterator(); overriddenFields.isValid(it); it = overriddenFields.getNext(it))
                    {
                        const int idx = overriddenFields.getElement(it);
                        hkReflect::TypeDetail::setFieldType(m_completedType, m_fields[idx]->m_name, m_fields[idx]->getTypeAsStruct<VersionedTypeBuilder*>()->m_completedType);
                    }

                    // set name and version
                    hkReflect::TypeDetail::localSetOptional<hkReflect::Opt::VERSION>(m_completedType, m_startingVersion);
                    hkReflect::TypeDetail::localSetOptional<hkReflect::Opt::NAME>(m_completedType, m_startingName.cString());
                }

                currentlyProcessing.remove(this);
                return HK_SUCCESS;
            }
            else
            {
                return HK_FAILURE;
            }
        }

        ~VersionedTypeBuilder()
        {
            for (int i = 0; i < m_fields.getSize(); i++)
            {
            }

            for (int i = 0; i < m_templateParameters.getSize(); i++)
            {
                if (m_templateParameters[i].m_field)
                {
                }
            }
        }

        // Current name, current version -- can change!
        VersionedTypeStorage* m_owner;
        hkStringPtr m_name;
        int m_version;

        hkStringPtr m_startingName;
        int m_startingVersion;

        hkArray< hkScopedPtr<hkVersionBundleUtil::Field> > m_fields;
        hkArray<VersionedTypeBuilder*> m_parents; // Most recent LAST
        bool m_lastParentFromNativeType;
        int m_currentParentIndex; // -1 = no parent
        class VersionedTypeBuilder* m_nextType; // When we have combined types, this points to the next type in the combined list
        
        bool m_startedFromFile;
        bool m_typeIsForeign;
        bool m_typeIsDynamic;

        hkReflect::Type* m_completedType;
        hkArray<int> m_patchIds;
        struct TemplateParameter
        {
            hkScopedPtr<hkVersionBundleUtil::Field> m_field;
            hkUlong m_value;
            hkStringPtr m_kindAndName;
        };
        hkArray<TemplateParameter> m_templateParameters; // template parameters while building
        hkReflect::Template* m_templateParametersObject; // final template parameters object

        bool needsToBeCopied() const { return m_localFieldsRequireCopying; }
        bool recursivelyCheckForFieldsNeedingCopy()
        {
            if (m_localFieldsRequireCopying) { return true; }
            for (auto& field : m_fields)
            {
                if (VersionedTypeBuilder* vtb = field->getTypeAsStruct<VersionedTypeBuilder*>())
                {
                    if (vtb->recursivelyCheckForFieldsNeedingCopy())
                    {
                        m_localFieldsRequireCopying = true;
                        return true;
                    }
                }
            }

            return false;
        }
    private:
        bool m_localFieldsRequireCopying;
        struct TrackedObject
        {
            TrackedObject() {}
            TrackedObject(const TrackedObject& other) : m_id(other.m_id), m_offset(other.m_offset), m_count(other.m_count) {}
            TrackedObject& operator=(const TrackedObject& other) { m_id = other.m_id; m_offset = other.m_offset; m_count = other.m_count; return *this; }
            bool operator==(const TrackedObject& other) const { return (m_id == other.m_id) && (m_offset == other.m_offset) && (m_count == other.m_count); }
            void setVar(int id, int offset) { m_id = id; m_offset = offset; m_count = -1; }
            void setVarN(int id, int offset, int count) { m_id = id; m_offset = offset; m_count = count; }

            int m_id;
            int m_offset;
            int m_count; // -1 = var, >=0 = extra
        };
        hkMultiMap<int, TrackedObject> m_workObjectsFromId;

        // Patches required for this type
    };

    hkViewPtr<hkSerialize::Bundle> VersionBundle::Impl::applyPatchesToBundle(hkViewPtr<hkSerialize::Bundle> bundleIn, bool failIfVersioningRequired)
    {
        hkSet<int> patchIndicesToApply;
        DLOG("Apply patches\n");

        m_bundleIn = bundleIn;
        // Lock patch registry
        hkReadLock thisLock(m_patchReg.getLock());
        // First, we find all the patches we will need to run

        hkArrayView<const hkReflect::Type*> bundleTypes = bundleIn->types();

        hkSet<const hkReflect::Type*> typesAlreadyAdded;
        for (int bundleTypeIndex = 0; bundleTypeIndex < bundleTypes.getSize(); bundleTypeIndex++)
        {
            const hkReflect::Type* t = bundleTypes[bundleTypeIndex];

            {
                DLOG_IF(const char* n = t ? t->getName() : "<none>");
                //hkStringBuf sb; // can't use fullname -> unresolved pointers??
                DLOG("Adding type {} = {}({}) parent is {} ({})\n", bundleTypeIndex, n, t ? t->getVersion() : -1, t && t->getParent() ? t->getParent()->getName() : "<none>", t && t->getParent() ? t->getParent()->getVersion() : 0);
            }
            recursivelyAddTypesToSet(typesAlreadyAdded, t);
        }

        // Now m_startingUidsInFile contains the start of every type that is in the file.
        // Type combines may produce more implicit dependencies
        hkArray<hkUint64> trackingUidsNeededAfterFileScan;
        hkArray<hkUint64> currentlyProcessingFileUids;
        for (auto it = m_startingUidsInFile.getIterator(); m_startingUidsInFile.isValid(it); it = m_startingUidsInFile.getNext(it))
        {
            currentlyProcessingFileUids.pushBack(m_startingUidsInFile.getKey(it));
        }

        while(currentlyProcessingFileUids.getSize())
        {
            for(const hkUint64 startingFileUid: currentlyProcessingFileUids)
            {
                hkArray<int> patchIndices;
                int newTypeBuilderIndex = -1;
                VersionedTypeBuilder* thisTypeBuilder = 0;

                // Need to run fwd and see if we already have it, we could have it from an older dependency
                int patchIndex = m_patchReg.getPatchIndex(startingFileUid);
                while (patchIndex >= 0)
                {
                    patchIndicesToApply.insert(patchIndex);
                    //thisTypeBuilder->addPatchIndex(patchIndex);
                    patchIndices.pushBack(patchIndex);
                    const hkReflect::Version::PatchInfo* thisPatch = m_patchReg.getPatch(patchIndex);

                    DLOG("Applying patch {} {} -> {} {} to type {} / {}", thisPatch->oldName, thisPatch->oldVersion, thisPatch->newName, thisPatch->newVersion, m_patchReg.getClassName(startingFileUid), m_patchReg.getClassVersion(startingFileUid));

                    if (thisPatch->newName)
                    {
                        const hkUint64 newUid = m_patchReg.getUid(thisPatch->newName, thisPatch->newVersion);
                        m_subsequentUidsInFile.insert(newUid); 
                        patchIndex = m_patchReg.getPatchIndex(newUid);
                    }
                    else
                    {
                        patchIndex = -1;
                    }
                }

                if (newTypeBuilderIndex == -1)
                {
                    newTypeBuilderIndex = m_typeBuilders.getSize();
                    m_typeIndexFromUid.insert(startingFileUid, newTypeBuilderIndex);
                    thisTypeBuilder = new VersionedTypeBuilder(this);
                    m_typeBuilders.pushBack(thisTypeBuilder);
                }

                const char* cn = m_patchReg.getClassName(startingFileUid);
                const int cv = m_patchReg.getClassVersion(startingFileUid);

                DLOG("File contains {} {}", cn, cv);

                // Really just sets the name / version
                thisTypeBuilder->setFromPatchInfo(cn, cv);
                thisTypeBuilder->m_startedFromFile = true;
                thisTypeBuilder->m_typeIsForeign = m_foreignUids.contains(startingFileUid);
                thisTypeBuilder->m_typeIsDynamic = m_dynamicUids.contains(startingFileUid);
                if (thisTypeBuilder->m_typeIsForeign)
                {
                    // Foreign types must have a parent. Push back an empty space for it
                    thisTypeBuilder->m_parents.pushBack(HK_NULL);
                }
                //int patchIndex = m_patchReg.getPatchIndex(startingFileUid);
                //while (patchIndex >= 0)
                for (int thisPatchIndex : patchIndices)
                {
                    thisTypeBuilder->addPatchIndex(thisPatchIndex);
                    const hkReflect::Version::PatchInfo* thisPatch = m_patchReg.getPatch(thisPatchIndex);

                    if (thisPatch->newName)
                    {
                        const hkUint64 newUid = m_patchReg.getUid(thisPatch->newName, thisPatch->newVersion);
                        if (thisPatch->isCombine())
                        {
                            if(m_typeIndexFromUid.getWithDefault(newUid, -1) == -1)
                            {
                                // If we haven't seen this yet, we need to add it to the list of things we are going to create
                                // Need to treat it slightly differently, the type we are combining to should also be created
                                trackingUidsNeededAfterFileScan.pushBack(newUid);
                            }
                        }
                        else
                        {
                            m_typeIndexFromUid.insert(newUid, newTypeBuilderIndex);
                        }
                    }
                }
            }
            currentlyProcessingFileUids.clear();

            for (const hkUint64 uid: trackingUidsNeededAfterFileScan)
            {
                if(!m_startingUidsInFile.hasKey(uid))
                {
                    currentlyProcessingFileUids.pushBack(uid);
                    m_startingUidsInFile.insert(uid, nullptr);
                }
            }
        }

        // Now each file type has a type builder allocated with the correct start name / version
        // and m_typeIndexFromUid is set for all of the file types future versions

        // Copy it as we will add new patch indices
        hkSet<int> patchIndicesToApplyFromFileTypes(patchIndicesToApply);
        // We do this in a separate phase to be sure all of the file types are done first
        for (auto it = patchIndicesToApplyFromFileTypes.getIterator(); patchIndicesToApplyFromFileTypes.isValid(it); it = patchIndicesToApplyFromFileTypes.getNext(it))
        {
            const int startingPatchIndex = patchIndicesToApplyFromFileTypes.getElement(it);
            const hkReflect::Version::PatchInfo* thisPatch = m_patchReg.getPatch(startingPatchIndex);

//          if (thisPatch->newName && (thisPatch->newVersion != -1))
//          {
                // STORE dependencies, don't run them yet
                for (int componentIndex = thisPatch->numComponent - 1; componentIndex >= 0; componentIndex--)
                {
                    const hkReflect::Version::PatchInfo::Component& component = thisPatch->component[componentIndex];
                    if (const hkReflect::Version::PatchInfo::DependsPatch* dependsPatch = component.asDependsPatch())
                    {
                        const hkUint64 dependsUid = m_patchReg.getUid(dependsPatch->name, dependsPatch->version);
                        //m_startingUidsFromPatches.insert(dependsUid); // Just the starting UID, we will walk this forward later
                        if (!m_typeIndexFromUid.hasKey(dependsUid))
                        {
                            // This type / version does not already exist
                            hkArray<int> patchIds;
                            hkArray<hkUint64> uids; uids.pushBack(dependsUid);
                            const int newTypeBuilderIndex = m_typeBuilders.getSize();
                            bool typeWasFound = false;

                            // Run forward to see if a future version of this type already exists
                            int patchIndex = m_patchReg.getPatchIndex(dependsUid);
                            while (patchIndex >= 0)
                            {
                                const hkReflect::Version::PatchInfo* laterPatch = m_patchReg.getPatch(patchIndex);
                                //m_typeIndexFromUid.insert(newUid, newTypeBuilderIndex);
                                patchIndicesToApply.insert(patchIndex);
                                patchIds.pushBack(patchIndex);

                                if(laterPatch->newName) // Not deleted
                                {
                                    const hkUint64 newUid = m_patchReg.getUid(laterPatch->newName, laterPatch->newVersion);

                                    //m_subsequentUidsInFile.insert(newUid); // Needed ?
                                    patchIndex = m_patchReg.getPatchIndex(newUid);
                                    const int typeIndex = m_typeIndexFromUid.getWithDefault(newUid, -1);
                                    if (typeIndex >= 0)
                                    {
                                        if(m_typeBuilders[typeIndex]->startedFromFileType())
                                        {
                                            // We have found a future version, this cannot be a file type, our dependencies are messed up
                                            Log_Warning("{} ({}) is in the file but there is a patch dependency on {} ({}) in patch {} ({}). This usually indicates patch dependencies are not correct.", m_typeBuilders[typeIndex]->m_startingName, m_typeBuilders[typeIndex]->m_startingVersion, dependsPatch->name, dependsPatch->version, thisPatch->oldName, thisPatch->oldVersion);
                                        }
                                        else
                                        {
                                            for (int idx = 0; idx < patchIds.getSize(); idx++)
                                            {
                                                m_typeBuilders[typeIndex]->m_patchIds.insertAt(idx, patchIds[idx]);
                                            }

                                            // Reset the starting point to here
                                            m_typeBuilders[typeIndex]->setFromPatchInfo(m_patchReg.getClassName(dependsUid), m_patchReg.getClassVersion(dependsUid));

                                            // Link from these uids to the existing type
                                            for (hkUint64 uid : uids)
                                            {
                                                m_typeIndexFromUid.insert(uid, typeIndex);
                                            }
                                        }

                                        // Early out
                                        typeWasFound = true;
                                        patchIndex = -1;
                                    }

                                    uids.pushBack(newUid); // Why here??
                                }
                                else
                                {
                                    patchIndex = -1;
                                }
                            }

                            if (!typeWasFound)
                            {

                                for(hkUint64 uid: uids)
                                {
                                    m_typeIndexFromUid.insert(uid, newTypeBuilderIndex);
                                }
                                VersionedTypeBuilder* thisTypeBuilder = new VersionedTypeBuilder(this);
                                m_typeBuilders.pushBack(thisTypeBuilder);

                                thisTypeBuilder->setFromPatchInfo(m_patchReg.getClassName(dependsUid), m_patchReg.getClassVersion(dependsUid));
                                thisTypeBuilder->m_startedFromFile = false;
                                for (int patchIdx : patchIds)
                                {
                                    thisTypeBuilder->addPatchIndex(patchIdx);
                                }
                            }
                        }
//                  }
                }

            }
        }

        // We need this later to apply to objects.
        hkArray<int> sortedPatchIndicesToApply(patchIndicesToApply.getSize());
        {
            int sortedPatchCount = 0;
            for (auto it = patchIndicesToApply.getIterator(); patchIndicesToApply.isValid(it); it = patchIndicesToApply.getNext(it)) { sortedPatchIndicesToApply[sortedPatchCount++] = patchIndicesToApply.getElement(it); }
        }

        hkAlgorithm::quickSort(sortedPatchIndicesToApply.begin(), sortedPatchIndicesToApply.getSize());

        // When parents change, we only keep the current one. We allocate storage for the LARGEST and switch in-place
        // allocate (temporary) storage for intermediate parents, only the final one goes into the final object
        // The intermediate parent objects all die before we reach the end anyway so they can go in some temp storage

        // We can add new builders inside the loop
        for (int typeBuilderIndex = 0; typeBuilderIndex < m_typeBuilders.getSize(); typeBuilderIndex++)
        {
            VersionedTypeBuilder* builder = m_typeBuilders[typeBuilderIndex];
            const char* finalName = builder->m_name;
            int finalVersion = builder->m_version;

            const char* startingName = builder->m_startingName;
            (void)startingName;

            if (builder->m_patchIds.getSize())
            {
                const hkReflect::Version::PatchInfo* thisPatch = m_patchReg.getPatch(builder->m_patchIds.back());
                finalName = thisPatch->newName;
                finalVersion = thisPatch->newVersion;
            }

            const hkReflect::Type* finalNativeParent = HK_NULL;

            if ((finalVersion != 0x80000000) && (finalVersion != -1) && !builder->isForeign() && !builder->isDynamic())
            {
                // The type wasn't deleted, we need to add in the types from the last native version
                const hkReflect::Type* nativeType = m_typeReg.typeFromName(finalName);
                if (!nativeType)
                {
                    Log_Warning("Type {} was seen while loading but it is not registered. Most likely patches are not registered or missing", finalName);
                }
                else if (nativeType->getVersion() != finalVersion)
                {
                    Log_Warning("Type {} version {} was seen while loading but the native version is {}. Most likely patches are not registered or missing", finalName, finalVersion, nativeType->getVersion());
                }
                else
                {
                    builder->addFieldsFromNativeType(nativeType, true);
                    finalNativeParent = nativeType->getParent();
                }
            }

            for (int patchIndexCount = builder->m_patchIds.getSize() - 1; patchIndexCount >= 0; patchIndexCount--)
            {
                const int patchIdx = builder->m_patchIds[patchIndexCount];
                const hkReflect::Version::PatchInfo* thisPatch = m_patchReg.getPatch(patchIdx);
                builder->applyPatchToType(thisPatch, VersionedTypeBuilder::DEFAULT_BEHAVIOR); // Reversed patch
            }

            if ((builder->m_parents.getSize() == 1) && (finalNativeParent || builder->isForeign())) // If no patches, then we already have the native version
            {
                const int finalParentIndex = builder->m_parents.getSize() - 1; // Always 0 ??
                const hkUint64 startingUid = m_patchReg.getUid(builder->m_startingName, builder->m_startingVersion);
                const hkUint64 startingParentUid = m_parentUidFromStartingUid.getWithDefault(startingUid, 0xffffffffffffffff);
                if(startingParentUid != 0xffffffffffffffff)
                {
                    const int startingParentIndex = m_typeIndexFromUid.getWithDefault(startingParentUid, -1);
                    VersionedTypeBuilder* parentBuilder = m_typeBuilders[startingParentIndex];
                    if(builder->m_parents[finalParentIndex] != parentBuilder)
                    {
                        builder->m_parents[finalParentIndex] = parentBuilder;
                    }
                }
            }
        }

        // Now run the patches over these types, expanding as we go


        // Now do final version of types
        // If the type was deleted OR it started with a file type, this is already done
        // If it started with a file type and was deleted, we can verify the structure (for consistency, not a hard failure)
        // If we started in a patch and ended up at a native type then we are missing info and, we need to copy it in
        finalizeAndCopyTypes();

        const int oldNumTypes = m_typeBuilders.getSize();
        // Now copy input objects, where needed
        copyInputObjects(bundleIn);
        // Now run the patches over the input objects

        if (oldNumTypes != m_typeBuilders.getSize())
        {
            // All of the input types should be in the input bundle and already processed. Types being added here means dependencies aren't correct
            Log_Warning("Types were added during versioning input handling, this usually indicates patches are not set up correctly");
        }

        for (int patchIndex : sortedPatchIndicesToApply)
        {
            const hkReflect::Version::PatchInfo* thisPatch = m_patchReg.getPatch(patchIndex);

            applyPatchToObjects(thisPatch);
        }
        // Now we're done
        // Create output bundle
        hkHashMap<const hkReflect::Type*, TypeId> tidFromType;

        m_finishedRunningPatches = true;
        return &m_workBundle;
    }

    void VersionBundle::Impl::recursivelyAddTypesToSet(hkSet<const hkReflect::Type*>& alreadyDone, _In_opt_ const hkReflect::Type* t)
    {
        HK_UNITY_USING_ANONYMOUS_NAMESPACE;

        if (t)
        {
            while (t->isDecorator() && !hkReflect::TypeDetail::localHasOptional(t, hkReflect::Opt::NAME)) { t = t->getParent(); }
            // A bit hacky, we don't want to add FieldDecl or Decl
            //if (t->extendsOrEquals<hkReflect::Decl>()) // this won't work if we have type world pointers
            if(t->getName() && ((hkString::strCmp(t->getName(), "hkReflect::Decl") == 0) || (hkString::strCmp(t->getName(), "hkReflect::FieldDecl") == 0)))
            {
                return;
            }
            // New type
            if (isVersionableType(t))
            {
                if(alreadyDone.insert(t))
                {
                    hkStringBuf sb;
                    const char* typeName = t->getFullName(sb);
                    const int typeVersion = t->getVersion();
                    const hkUint64 uid = m_patchReg.getUid(typeName, typeVersion);
                    if (t->isForeign())
                    {
                        m_foreignUids.insert(uid);
                        HK_ASSERT(0xb338f77, t->getParent(), "Foreign types must have a parent");
                    }
                    if (t->isDynamicType())
                    {
                        m_dynamicUids.insert(uid);
                    }
                    m_startingUidsInFile.insert(uid, t);

                    if (const hkReflect::Type* parentType = t->getParent())
                    {
                        const char* parentName = parentType->getFullName(sb);
                        const int parentVersion = parentType->getVersion();
                        const hkUint64 parentUid = m_patchReg.getUid(parentName, parentVersion);
                        m_parentUidFromStartingUid.insert(uid, parentUid);
                    }

                    for (hkReflect::DeclIter<hkReflect::DataFieldDecl> fieldIterator(t); fieldIterator.advance();)
                    {
                        const hkReflect::DataFieldDecl f = fieldIterator.current();
                        //DLOG("hkVersionBundleUtil::Field {}", f);
                        recursivelyAddTypesToSet(alreadyDone, f.getType());
                    }

                    if (const hkSerialize::CompatTypeParentInfo* parentInfoAttr = hkReflect::TypeDetail::decoratorFindAttribute<hkSerialize::CompatTypeParentInfo>(t))
                    {
                        hkUint64 currentParentUid = uid;
                        for(hkSerialize::CompatTypeParentInfo::Parent* parentInfo = parentInfoAttr->m_firstParent; parentInfo; parentInfo = parentInfo->m_next)
                        {
                            // If the previous stage has passed in custom parent info we should use it
                            const char* parentName = parentInfo->m_name;
                            const int parentVersion = parentInfo->m_version;
                            if (parentName && (parentVersion != -1))
                            {
                                const hkUint64 parentUid = m_patchReg.getUid(parentName, parentVersion);
                                m_parentUidFromStartingUid.insert(currentParentUid, parentUid);
                                currentParentUid = parentUid;
                                m_startingUidsInFile.insert(parentUid, t);
                            }
                        }
                    }
                    else
                    {
                        if (const hkReflect::Type* thisTypeParent = t->getParent())
                        {
                            //DLOG("Parent {}", thisTypeParent);
                            recursivelyAddTypesToSet(alreadyDone, thisTypeParent);
                        }
                    }
                }
            }
        }
    }

    _Ret_notnull_ VersionedTypeBuilder* VersionBundle::Impl::getTypeForNameVer(_In_z_ const char* name, int version)
    {
        const hkUint64 uid = m_patchReg.getUid(name, version);
        const int thisTypeIndex = m_typeIndexFromUid.getWithDefault(uid, -1);

        if (thisTypeIndex >= 0)
        {
            return m_typeBuilders[thisTypeIndex];
        }
        else
        {
            VersionedTypeBuilder* newBuilder = new VersionedTypeBuilder(this);
            newBuilder->setFromPatchInfo(name, version);

            const int newBuilderIndex = m_typeBuilders.getSize();
            DLOG("Adding {}/{} from dependencies, index {}", name, version, newBuilderIndex);

            m_typeIndexFromUid.insert(uid, newBuilderIndex);
            m_typeBuilders.pushBack(newBuilder);
            return newBuilder;
        }
    }

    hkResult VersionBundle::Impl::recursivelyProcessField(_Inout_ hkVersionBundleUtil::Field* fieldType, hkSet<VersionedTypeBuilder*>& alreadyDone, hkSet<VersionedTypeBuilder*>& currentlyProcessing)
    {
        hkResult res = HK_SUCCESS;
        if (VersionedTypeBuilder* fieldTypeStruct = fieldType->getTypeAsStruct<VersionedTypeBuilder*>())
        {
            if (recursivelyProcessFieldsAndParents(fieldTypeStruct, alreadyDone, currentlyProcessing).isFailure())
            {
                res = HK_FAILURE;
            }
        }
        if (fieldType->m_tuples > 0)
        {
            if (hkVersionBundleUtil::Field* subField = fieldType->getTypeAsField())
            {
                if (recursivelyProcessField(subField, alreadyDone, currentlyProcessing).isFailure())
                {
                    res = HK_FAILURE;
                }
            }
        }

        return res;
    }

    hkResult VersionBundle::Impl::recursivelyProcessFieldsAndParents(_Inout_ VersionedTypeBuilder* thisBuilder, hkSet<VersionedTypeBuilder*>& alreadyDone, hkSet<VersionedTypeBuilder*>& currentlyProcessing)
    {
        hkResult res = HK_SUCCESS;
        if (alreadyDone.contains(thisBuilder))
        {
            return HK_SUCCESS;
        }

        if (currentlyProcessing.insert(thisBuilder))
        {
            for (int i = 0; i < thisBuilder->m_parents.getSize(); i++)
            {
                if (thisBuilder->m_parents[i])
                {
                    if (recursivelyProcessFieldsAndParents(thisBuilder->m_parents[i], alreadyDone, currentlyProcessing).isFailure())
                    {
                        res = HK_FAILURE;
                    }
                }
            }

            for (int i = 0; i < thisBuilder->m_fields.getSize(); i++)
            {
                if (recursivelyProcessField(thisBuilder->m_fields[i].get(), alreadyDone, currentlyProcessing).isFailure())
                {
                    res = HK_FAILURE;
                }
            }

            hkSet<VersionedTypeBuilder*> setParentProc;
            thisBuilder->setParentAndFinish(setParentProc);
            alreadyDone.insert(thisBuilder);
            currentlyProcessing.remove(thisBuilder);
        }
        else
        {
            // Loop in processing
            Log_Error("Loop in field or parents detected while trying to process {}/{}", thisBuilder->m_startingName, thisBuilder->m_startingVersion);
            res = HK_FAILURE;
        }

        return res;
    }

    void VersionBundle::Impl::finalizeAndCopyTypes()
    {
        HK_UNITY_USING_ANONYMOUS_NAMESPACE;

        for (int idx = 0; idx < m_typeBuilders.getSize(); idx++)
        {
            VersionedTypeBuilder* thisBuilder = m_typeBuilders[idx];
            thisBuilder->allocateRecord();
            m_typeBuilderFromCompletedType.insert(thisBuilder->m_completedType, thisBuilder);
        }

        // Use the builder's relocToThisType and resolveRelocs to create real types for us to use
        for (int idx = 0; idx < m_typeBuilders.getSize(); idx++)
        {
            VersionedTypeBuilder* thisBuilder = m_typeBuilders[idx];

            thisBuilder->allocateFieldsAndParameters();
        }

        // Need to set parents / fields FIRST then calculate field sizes
        hkSet<VersionedTypeBuilder*> alreadyDone;
        hkSet<VersionedTypeBuilder*> currentlyProcessing;

        for (int idx = 0; idx < m_typeBuilders.getSize(); idx++)
        {
            VersionedTypeBuilder* thisBuilder = m_typeBuilders[idx];
            recursivelyProcessFieldsAndParents(thisBuilder, alreadyDone, currentlyProcessing);
        }

        for (int idx = 0; idx < m_typeBuilders.getSize(); idx++)
        {
            if (m_typeBuilders[idx]->m_startedFromFile)
            {
                const hkUint64 uid = m_patchReg.getUid(m_typeBuilders[idx]->m_startingName, m_typeBuilders[idx]->m_startingVersion);
                if(const hkReflect::Type* fileType = m_startingUidsInFile.getWithDefault(uid, HK_NULL))
                {
                    adjustFieldsForNameMismatches(fileType, m_typeBuilders[idx]->m_completedType);
                }
            }
        }
    }

    // These perform the mapping from pointers in the source objects to object indices in the versioning objects
    int VersionBundle::Impl::getVarIndexFromSourceAddress(_In_ void* sourceAddress)
    {
        hkPointerMap<void*, int>::Iterator it = m_indexFromAddressMap.findOrInsertKey(sourceAddress, m_workBundle.getSize());
        int index = m_indexFromAddressMap.getValue(it);

        if (index == m_workBundle.getSize())
        {
            // Resizes the array and inserts a placeholder. We still need
            // to add it to the tracking map once it is completed

            m_workBundle.setVar(index, HK_NULL, 0);
        }
        return index;
    }

    // Sometimes we need a create a space for a null
    int VersionBundle::Impl::getNewVarIndex()
    {
        const int index = m_workBundle.getSize();
        m_workBundle.setVar(index, HK_NULL, 0);
        return index;
    }

    int VersionBundle::Impl::getArrayIndexFromSourceAddress(_In_ void* sourceAddress)
    {
        hkPointerMap<void*, int>::Iterator it = m_indexFromAddressMap.findOrInsertKey(sourceAddress, m_workBundle.getSize());
        int index = m_indexFromAddressMap.getValue(it);

        if (index == m_workBundle.getSize())
        {
            // Resizes the array and inserts a placeholder. We still need
            // to add it to the tracking map once it is completed
            m_workBundle.setVarN(index, HK_NULL, 0, 0);
        }

        return index;
    }

    _Ret_maybenull_ const hkReflect::Type* VersionBundle::Impl::getOrCreate(_In_ const hkReflect::Type* pointedToType)
    {
        hkStringBuf sb;

        if (hkReflect::FieldDecl fd = pointedToType->isField())
        {
            const hkReflect::Type* declContext = fd.getDeclContext();

            const char* declContextName = declContext->getFullName(sb);
            const int declContextVersion = declContext->getVersion();

            const hkUint64 declContextUid = m_patchReg.getUid(declContextName, declContextVersion);
            const int declContextTypeIndex = m_typeIndexFromUid.getWithDefault(declContextUid, -1);

            if (declContextTypeIndex >= 0)
            {
                if (hkReflect::FieldDecl versionedFd = m_typeBuilders[declContextTypeIndex]->m_completedType->findField(fd.getName(), false))
                {
                    return versionedFd.getType();
                }
                else
                {
                    // If we can't find it, clear it
                    return HK_NULL;
                }
            }
            else
            {
                return HK_NULL;
            }
        }
        else if (const hkReflect::PointerType* pt = pointedToType->asPointer())
        {
            const hkReflect::Type* subType = HK_NULL;
            if(const hkReflect::Type* st = pt->getSubType())
            {
                subType = getOrCreate(st);
            }
            return m_compoundTypeCache.getVersionedPointerType( pt, subType );
        }
        else if ( const hkReflect::ArrayType* at = pointedToType->asArray() )
        {
            const hkReflect::Type* subType = HK_NULL;
            if ( const hkReflect::Type* st = at->getSubType() )
            {
                subType = getOrCreate( st );
            }
            return m_compoundTypeCache.getVersionedArrayType( at, subType );
        }
        else if (const char* name = pointedToType->getFullName(sb)) // Type
        {
            const int version = pointedToType->getVersion();
            const hkUint64 uid = m_patchReg.getUid(name, version);
            const int thisTypeIndex = m_typeIndexFromUid.getWithDefault(uid, -1);

            if (thisTypeIndex >= 0)
            {
                return m_typeBuilders[thisTypeIndex]->m_completedType;
            }
            else
            {
                // Basic type, just pass it through
                return pointedToType;
            }
        }
        else
        {
            // Basic type, just pass it through. This should be int, float etc
            return pointedToType;
        }

    }

    class VersionBundle::Impl::CopyInputObjectsInterpreter : public hkTypeVm::FastCopyInterpreter
    {
    public:
        void handleTypePointer(_In_ const hkReflect::Type* pointedToType, _Inout_ void* address)
        {
            *reinterpret_cast<const hkReflect::Type**>( address ) = m_parent->getOrCreate( pointedToType );
        }

        HK_INLINE hkResult execPointer(hkReflect::PointerVar dst, hkReflect::PointerVar src, int dstStride, int srcStride, int numElems)
        {
            for (int i = 0; i < numElems; i++, src = incrementAddress(src, srcStride), dst = incrementAddress(dst, dstStride))
            {
                // Copy the old pointer value to the new object
                hkReflect::Var oldVar = src.getValue();

                //const hkReflect::Type* subType = oldVar.getType();
                if (const hkReflect::Type* pointedToType = hkReflect::Detail::asType(oldVar))
                {
                    handleTypePointer(pointedToType, dst.getAddress());
                }
                else
                {
                    //DLOG("PTR {0} at {1} (from {2})\n", oldVar, dst, src);
                    void* addr = oldVar.getAddress();

                    if (int noteIndex = m_parent->getIndexForNoteOnPointer(src))
                    {
                        *reinterpret_cast<hkUint32Le*>(dst.getAddress()) = noteIndex; // Stash the value here so that the pointer impl can see it and get to the object
                    }
                    else if (addr)
                    {
                        int newIndex = m_parent->getVarIndexFromSourceAddress(addr);
                        *reinterpret_cast<hkUint32Le*>(dst.getAddress()) = newIndex; // Stash the value here so that the pointer impl can see it and get to the object
                    }
                    else
                    {
                        *reinterpret_cast<hkUint32Le*>(dst.getAddress()) = 0;
                    }
                }
            }
            return HK_SUCCESS;
        }

        HK_INLINE hkResult execCString(hkReflect::StringVar dst, hkReflect::StringVar src)
        {
            return dst.setValue(src.getValue());
        }


        HK_INLINE hkResult execArray(hkReflect::ArrayVar dst, hkReflect::ArrayVar src)
        {
            void* addr = src.getDataPointer();

            const int newId = m_parent->getArrayIndexFromSourceAddress(addr);
            //DLOG("Array reloc {0} -> {1} at {2} (from {3})", addr, newId, dst, src);

            *reinterpret_cast<hkUint32Le*>(dst.getAddress()) = newId;// Stash the value here so that the array impl can see it and get to the object
            return HK_SUCCESS;
        }

        VersionBundle::Impl* m_parent;
    };

    void VersionBundle::Impl::makeVarNCopy(int extraIndex, const hkSerialize::VarN& thisVarN, _Inout_opt_ VersionedTypeBuilder* builder)
    {
        HK_UNITY_USING_ANONYMOUS_NAMESPACE;

        if (!m_workBundle.wasCopied(extraIndex))
        {
            // We need to copy the object
            m_workBundle.setCopied(extraIndex);

            // Make a versioned copy of the type.
            const hkReflect::Type* sourceType = thisVarN.getType();

            if (!sourceType)
            {
                return;
            }

            const void* srcPtr = thisVarN.getAddress();
            const int numElements = thisVarN.getCount();

            // This will contain the final converted type.
            const hkReflect::Type* convertedTypeOut = nullptr;

            // If the type is a decorator, we need to make a copy of the whole chain of decorators and of the concrete type.
            // Save the place where we need to write the pointer of the concrete copy. If the type is not a decorator,
            // write the pointer in convertedTypeOut.
            const hkReflect::Type** currentAddressOfTypeOut = &convertedTypeOut;
            const hkReflect::Type* sourceConcreteType = sourceType;
            while (sourceConcreteType->isDecorator())
            {
                // Make a shallow copy of the decorator.
                hkReflect::TypeBuilder tb;
                tb.beginShallowClone( sourceConcreteType );
                hkReflect::Type* decoratorCopy = tb.allocate( &m_fieldAllocator );

                // Write the pointer to the decorator copy in currentAddressOfTypeOut (will point to convertedTypeOut
                // in the first iteration, the parent pointer of the last decorator from the second one on).
                *currentAddressOfTypeOut = decoratorCopy;

                // Update currentAddressOfTypeOut to the address of the parent pointer of the current decorator.
                currentAddressOfTypeOut = const_cast<const hkReflect::Type**>(hkReflect::TypeDetail::addressParent(decoratorCopy));

                // Go to next.
                sourceConcreteType = sourceConcreteType->getParent();
            }

            const hkReflect::Type* convertedConcreteType;
            if (isVersionableType(sourceConcreteType))
            {
                if (!builder)
                {
                    hkStringBuf sb;
                    const char* startName = sourceConcreteType->getFullName(sb);
                    int startVersion = sourceConcreteType->getVersion();

                    builder = getTypeForNameVer(startName, startVersion);
                }

                convertedConcreteType = builder->m_completedType;
                builder->addTrackedExtra(extraIndex, 0, numElements);
            }
            else if (const hkReflect::PointerType* pt = sourceConcreteType->asPointer())
            {
                // The name of the pointer type needs to be preserved in order to find the native element type
                // when cloning a variant array to native - in this case, the element type cannot be inferred
                // from the array type.
                const hkReflect::Type* subType = pt->getSubType();
                // Do we need a recursive thing here? Or do we rely on this happening at copy of the target extra anyway?
                if (subType && isVersionableType(subType))
                {
                    {
                        hkStringBuf sb;
                        const char* startName = subType->getFullName(sb);
                        int startVersion = subType->getVersion();

                        builder = getTypeForNameVer(startName, startVersion);
                    }

                    subType = builder->m_completedType;
                }

                convertedConcreteType = m_compoundTypeCache.getVersionedPointerType( pt, subType );
            }
            else if (const hkReflect::ArrayType* at = sourceConcreteType->asArray())
            {
                if(hkVersionBundleUtil::isSpecialFloatArray(at))
                {
                    // hkVector4 etc
                    convertedConcreteType = at;
                }
                else
                {
                    // The name of the pointer type needs to be preserved in order to find the native element type
                    // when cloning a variant array to native - in this case, the element type cannot be inferred
                    // from the array type.
                    const hkReflect::Type* subType = at->getSubType();

                    // Do we need a recursive thing here? Or do we rely on this happening at copy of the target extra anyway?
                    if (subType && isVersionableType(subType))
                    {
                        {
                            hkStringBuf sb;
                            const char* startName = subType->getFullName(sb);
                            int startVersion = subType->getVersion();

                            builder = getTypeForNameVer(startName, startVersion);
                        }

                        subType = builder->m_completedType;
                    }

                    convertedConcreteType = m_compoundTypeCache.getVersionedArrayType(at, subType);
                }
            }
            else if (const hkReflect::StringType* st = sourceConcreteType->asString())
            {
                hkReflect::BasicStringType* newSt = m_fieldAllocator.create<hkReflect::BasicStringType>(hkSizeOf(int), (int)HK_ALIGN_OF(int), &m_stringImpl, hkReflectUtil::internCopy(st->getName()), hkReflect::Format::StringValue(st->getFormat()).isImmutable(), hkReflect::Type::TYPE_TEMPORARY);
                convertedConcreteType = newSt;
            }
            else
            {
                // it's a basic type, just copy it directly
                convertedConcreteType = sourceConcreteType;
            }

            // Save the concrete type in convertedTypeOut (if there were no decorators) or as the parent of the last
            // decorator we created.
            *currentAddressOfTypeOut = convertedConcreteType;

            const int sizeToAllocate = convertedConcreteType->getSizeOf() * numElements;

            // Could use a different allocator??
            void* dstObjectPtr = HK_NULL;
            if (sizeToAllocate)
            {
                dstObjectPtr = m_fieldAllocator.blockAlloc(sizeToAllocate);
                hkString::memSet(dstObjectPtr, 0, sizeToAllocate);
            }

            const int tid = m_workBundle.addType(convertedTypeOut);
            m_workBundle.setVarN(extraIndex, dstObjectPtr, tid, numElements);
            m_workBundle.setCopied(extraIndex);

            if (numElements)
            {
                const hkTypeVm::Program* program = m_compiler.compile(sourceConcreteType, convertedConcreteType);
                m_indexFromAddressMap.insert(dstObjectPtr, extraIndex);

                CopyInputObjectsInterpreter newMachine;
                newMachine.m_parent = this;
                {
                    HK_VERIFY(0x6a6edfb3, hkTypeVm::FastCopyInterpreter::execN(newMachine, program,
                        dstObjectPtr, sizeToAllocate,
                        srcPtr, sourceConcreteType->getSizeOf() * numElements,
                        convertedConcreteType->getSizeOf(), sourceConcreteType->getSizeOf(), numElements).isSuccess(),
                        "Failed to copy array for versioning");
                }
            }
        }
    }

    void VersionBundle::Impl::makeVarCopy(int varIndex, const hkReflect::Var& thisVar, _Inout_opt_ VersionedTypeBuilder* builder)
    {
        if(!m_workBundle.wasCopied(varIndex))
        {
            // We need to copy the object
            m_workBundle.setCopied(varIndex);
            if (!builder)
            {
                hkStringBuf sb;
                const char* startName = thisVar.getType()->getFullName(sb);
                int startVersion = thisVar.getType()->getVersion();

                builder = getTypeForNameVer(startName, startVersion);
            }

            builder->addTrackedVar(varIndex);

            void* srcPtr = thisVar.getAddress();
            const hkReflect::Type* sourceType = thisVar.getType();

            const hkReflect::Type* convertedType = builder->m_completedType;
            const int sizeToAllocate = convertedType->getSizeOf();
            
            void* dstObjectPtr = m_fieldAllocator.blockAlloc(sizeToAllocate);
            hkString::memSet(dstObjectPtr, 0, sizeToAllocate);
            int tid = m_workBundle.addType(convertedType);

            m_workBundle.setVar(varIndex, dstObjectPtr, tid);

            const hkTypeVm::Program* program = m_compiler.compile(sourceType, convertedType);
            m_indexFromAddressMap.insert(dstObjectPtr, varIndex);

            CopyInputObjectsInterpreter newMachine;
            newMachine.m_parent = this;
            {
                HK_VERIFY(0x6898982e, hkTypeVm::FastCopyInterpreter::execN(newMachine, program,
                    dstObjectPtr, sizeToAllocate,
                    srcPtr, sourceType->getSizeOf(),
                    sizeToAllocate, 0, 1).isSuccess(),
                    "Failed to copy object for versioning");
            }
        }
    }

    void VersionBundle::Impl::makeNoteCopy(int varIndex, const hkSerialize::Bundle::Item& item, _Inout_ VersionedTypeBuilder* builder, hkArray<hkSerialize::Bundle::Item>& items)
    {
        hkReflect::Var thisVar = item.var();

        // We need to copy the object
        m_workBundle.setCopied(varIndex);
        builder->addTrackedVar(varIndex);

        void* srcPtr = thisVar.getAddress();
        const hkReflect::Type* sourceType = thisVar.getType();

        const hkReflect::Type* convertedType = builder->m_completedType;
        const int sizeToAllocate = convertedType->getSizeOf();
        
        void* dstObjectPtr = m_fieldAllocator.blockAlloc(sizeToAllocate);
        hkString::memSet(dstObjectPtr, 0, sizeToAllocate);
        int tid = m_workBundle.addType(convertedType);

        const int inputAnnotatedIndex = item.getAnnotated();
        int outputAnnotatedIndex = -1;
        if ((inputAnnotatedIndex < items.getSize()) && items[inputAnnotatedIndex].var().getAddress())
        {
            // The index it will be placed at
            outputAnnotatedIndex = getVarIndexFromSourceAddress(items[inputAnnotatedIndex].var().getAddress());
        }
        else
        {
            outputAnnotatedIndex = getNewVarIndex();
        }
        m_workBundle.setNote(varIndex, outputAnnotatedIndex, dstObjectPtr, tid);

        const hkTypeVm::Program* program = m_compiler.compile(sourceType, convertedType);
        m_indexFromAddressMap.insert(dstObjectPtr, varIndex);

        CopyInputObjectsInterpreter newMachine;
        newMachine.m_parent = this;
        {
            HK_VERIFY(0x350fbf9, hkTypeVm::FastCopyInterpreter::execN(newMachine, program,
                dstObjectPtr, sizeToAllocate,
                srcPtr, sourceType->getSizeOf(),
                sizeToAllocate, 0, 1).isSuccess(),
                "Failed to copy object for versioning");
        }
    }

    void VersionBundle::Impl::copyInputObjects(hkViewPtr<hkSerialize::Bundle> bundleIn)
    {
        HK_UNITY_USING_ANONYMOUS_NAMESPACE;

        m_workBundle.setVar(0, HK_NULL, 0);
        hkArray<hkSerialize::Bundle::Item> items;
        bundleIn->getItems(items);

        for (int itemIndex = 0; itemIndex < items.getSize(); itemIndex++)
        {
            const hkSerialize::Bundle::Item& item = items[itemIndex];
            if ((item.m_addr == HK_NULL) && (item.m_type.isNull()))
            {
                // Skip
            }
            else if (item.isVar() || item.isNote())
            {
                const hkReflect::Var thisVar = item.var();
                void* srcPtr = thisVar.getAddress();
                const int varIndex = getVarIndexFromSourceAddress(srcPtr);

                bool copied = false;

                if (const hkReflect::Type* sourceType = thisVar.getType())
                {
                    if (isVersionableType(sourceType))
                    {
                        hkStringBuf sb;
                        const char* startName = sourceType->getFullName(sb);
                        int startVersion = sourceType->getVersion();

                        VersionedTypeBuilder* builder = getTypeForNameVer(startName, startVersion);
                        if (builder->needsToBeCopied())
                        {
                            copied = true;

                            if(item.isVar())
                            {
                                makeVarCopy(varIndex, thisVar, builder);
                            }
                            else
                            {
                                makeNoteCopy(varIndex, item, builder, items);
                            }
                        }

                    }
                }

                if (!copied)
                {
                    if(thisVar.getType())
                    {
                        hkReflect::Type* newType = chainingCopyType(thisVar.getType());
                        int tid = m_workBundle.addType(newType);
                        //m_workBundle.setVar(varIndex, thisVar.getAddress(), tid);
                        if (item.isVar())
                        {
                            m_workBundle.setVar(varIndex, thisVar.getAddress(), tid);
                        }
                        else
                        {
                            const int inputAnnotatedIndex = item.getAnnotated();
                            int outputAnnotatedIndex = -1;
                            if ((inputAnnotatedIndex < items.getSize()) && items[inputAnnotatedIndex].var().getAddress())
                            {
                                // The index it will be placed at
                                outputAnnotatedIndex = getVarIndexFromSourceAddress(items[inputAnnotatedIndex].var().getAddress());
                            }
                            else
                            {
                                outputAnnotatedIndex = getNewVarIndex();
                            }
                            m_workBundle.setNote(varIndex, outputAnnotatedIndex, thisVar.getAddress(), tid);
                        }
                    }
                    else
                    {
                        m_workBundle.setVar(m_workBundle.getSize(), HK_NULL, 0);
                    }
                }
            }
            else if (item.isVarN())
            {
                // Assuming we never have null extras for now
                hkSerialize::VarN thisVarN = item.varn();
                if (const hkReflect::Type* sourceType = thisVarN.getType())
                {
                    // NOT USING THE CORRECT TYPE FOR THIS
                    void* srcPtr = thisVarN.getAddress();
                    const int extraIndex = getArrayIndexFromSourceAddress(srcPtr);

                    bool copied = false;
                    if (isVersionableType(sourceType))
                    {
                        hkStringBuf sb;
                        const char* startName = sourceType->getFullName(sb);
                        int startVersion = sourceType->getVersion();

                        // Temp, previous phase should be stripping these XXX
                        if (startVersion != -1)
                        {
                            VersionedTypeBuilder* builder = getTypeForNameVer(startName, startVersion);

                            if (builder->needsToBeCopied())
                            {
                                copied = true;
                                //const hkReflect::Type* convertedType = builder->m_completedType;
                                //int tid = m_workBundle.addType(convertedType);
                                //m_workBundle.setVarN(extraIndex, thisVarN.getAddress(), tid, thisVarN.getCount());

                                makeVarNCopy(extraIndex, thisVarN, builder);
                            }
                        }
                    }
                    else if (const hkReflect::PointerType* pt = sourceType->asPointer())
                    {
                        copied = true;
                        makeVarNCopy(extraIndex, thisVarN);
                    }

                    if(!copied)
                    {
                        hkReflect::Type* newType = chainingCopyType(thisVarN.getType());

                        int tid = m_workBundle.addType(newType);
                        m_workBundle.setVarN(extraIndex, thisVarN.getAddress(), tid, thisVarN.getCount());
                    }
                }
            }
        }
    }

    hkResult VersionBundle::Impl::ArrayImpl::spliceInto(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, int index, int numToDel, const hkReflect::ArrayValue& val) const
    {
        HK_UNITY_USING_ANONYMOUS_NAMESPACE;

        // Delete numToDel elements (may be zero) from index, then insert val.m_numElem (may be zero) at the same point

        hkUint32 id = 0;
        if (arrAddr)
        {
            id = *reinterpret_cast<const hkUint32Le*>(arrAddr);
            if (id)
            {
                m_parent.makeVarNCopy(id, m_parent.m_workBundle.varn(id));
            }
        }

        int oldElementCount = 0;
        void* oldData = HK_NULL;
        const hkReflect::Type* elementType = HK_NULL;

        if (id && m_parent.m_workBundle.varn(id).getType())
        {
            oldElementCount = m_parent.m_workBundle.varn( id ).getCount();
            if ( arrType->getSubType() != HK_NULL ||
                elementType == val.getSubType() ) 
            {
                // Normal case.
                oldData = m_parent.m_workBundle.varn( id ).getAddress();
                elementType = m_parent.m_workBundle.varn( id ).getType();
            }
            else
            {
                // Non-empty variant array: only completely replacing the contents is allowed.
                if ( numToDel != -1 && numToDel != oldElementCount )
                {
                    return HK_FAILURE;
                }

                // Consider the array as empty (clear the id) and proceed normally.
                id = 0;
                oldElementCount = 0;
                elementType = val.getSubType();
            }
        }
        else
        {
            elementType = val.getSubType();
        }
        numToDel = ( numToDel == -1 ) ? oldElementCount : numToDel;

        int newElementCount = oldElementCount - numToDel + val.getCount();
        const int elementSize = elementType ? elementType->getSizeOf() : 0;

        void* newData = m_parent.m_fieldAllocator.blockAlloc(elementSize * newElementCount);//m_parent.getStorage()->m_allocator.blockAlloc(elementSize * newSize);

        // TODO
        //hkString::memSet(hkAddByteOffset(newData, elementSize * oldElementCount), 0, (newSize > oldElementCount) ? (elementSize * (newSize - oldElementCount)) : 0);

        if (index == -1)
        {
            index = oldElementCount;
        }

        if (index > 0)
        {
            // Copy existing elements before the insert / delete point
            hkString::memCpy(newData, oldData, elementSize * index);
            //hkViewPtr<hkTypeVm::Program> program = m_parent.m_compiler.compile(oldElementType, arrType->getSubType());
            //CopyInputObjectsInterpreter newMachine;
            //newMachine.m_parent = &m_parent;
            //{
            //  hkTypeVm::FastCopyInterpreter::execN(newMachine, program,
            //      newData, elementSize * newElementCount,
            //      oldData, oldElementType->getSizeOf() * index, elementSize, oldElementType->getSizeOf(), index);
            //}
        }

        if (val.getCount())
        {
            // Insert new elements
            // Can't just memcpy, these could be different representations of the same thing
            hkViewPtr<hkTypeVm::Program> program = m_parent.m_compiler.compile(val.getSubType(), elementType);
            CopyInputObjectsInterpreter newMachine;
            newMachine.m_parent = &m_parent;

            {
                HK_VERIFY(0x2dc3b7f3, hkTypeVm::FastCopyInterpreter::execN(newMachine, program,
                    hkAddByteOffset(newData, elementSize * index), elementSize * val.getCount(),
                    val.getAddress(), val.getStride() * val.getCount(), elementSize, val.getStride(), val.getCount())
                    .isSuccess(), "Failed to copy array for versioning");;
            }
        }

        if (index + numToDel < oldElementCount)
        {
            // Copy the existing elements after the delete point across
            hkString::memCpy(hkAddByteOffset(newData, elementSize * (index + val.getCount())), hkAddByteOffsetConst(oldData, elementSize * (index + numToDel)), elementSize * (oldElementCount - index - numToDel));
        }

        if (id)
        {
            int tid = m_parent.m_workBundle.addType(elementType);
            m_parent.m_workBundle.setVarN(id, newData, tid, newElementCount);
            m_parent.m_workBundle.setCopied(id);
            m_parent.m_indexFromAddressMap.insert(newData, id);
        }
        else if (val.getCount())
        {
            const int newId = m_parent.m_workBundle.getSize();
            int tid = m_parent.m_workBundle.addType(elementType);
            m_parent.m_workBundle.setVarN(newId, newData, tid, newElementCount);
            m_parent.m_indexFromAddressMap.insert(newData, newId);

            *reinterpret_cast<hkUint32Le*>(arrAddr) = newId; // Stash the id here so the array impl can find it
            m_parent.m_workBundle.setCopied(newId);

            // This only happens if it's a versionable type
            if(isVersionableType(elementType))
            {
                hkStringBuf sb;
                const char* startName = elementType->getFullName(sb);
                int startVersion = elementType->getVersion();

                VersionedTypeBuilder* builder = m_parent.getTypeForNameVer(startName, startVersion);
                builder->addTrackedExtra(newId, 0, newElementCount);
            }
        }

        return HK_SUCCESS;
    }

    _Ret_notnull_ hkReflect::Type* VersionBundle::Impl::copyAndAddOptional(_In_ const hkReflect::Type* srcType, hkReflect::Optional opt, _In_ const void* val)
    {
        HK_UNITY_USING_ANONYMOUS_NAMESPACE;

        // Enlarge the type size for the new opt if it isn't present
        hkUlong newOptMask = hkReflect::TypeDetail::getOptMask(srcType) | opt;
        const int newTypeSize = hkReflect_Type_getTypeSizeInBytes(hkReflect::TypeDetail::getOptMask(srcType) | opt);
        hkReflect::Type* newType = reinterpret_cast<hkReflect::Type*>(m_fieldAllocator.blockAlloc(newTypeSize));

        hkString::memCpy(newType, srcType, sizeof(hkReflect::Type));
        hkReflect::TypeDetail::setOptMask(newType, newOptMask);

        for (int i = 0; i < 32; ++i)
        {
            const hkReflect::Optional o = static_cast<hkReflect::Optional>(1 << i);
            if (hkReflect::TypeDetail::localHasOptional(srcType, o))
            {
                hkUlong v = hkReflect::TypeDetail::localGetOptionalUlong(srcType, o);
                hkReflect::TypeDetail::localSetOptionalUlong(newType, o, v);
            }
        }

        // Insert or overwrite the new opt
        hkReflect::TypeDetail::localSetOptionalUlong(newType, opt, hkUlong(val));

        return newType;
    }

    hkResult applyMemberRemoved(_In_ const hkReflect::Version::PatchInfo* p, _Inout_ hkReflect::Type* completedType, _In_z_ const char* name)
    {
        hkReflect::FieldDecl f = completedType->findField(name, false);
        if (f)
        {
            hkReflect::TypeDetail::setFieldName(f, HK_NULL);
            return HK_SUCCESS;
        }
        else
        {
            Log_Error("Field {} not found applying patch {}/{}->{}/{}", name, p->oldName, p->oldVersion, p->newName, p->newVersion);
            return HK_FAILURE;
        }
    }

    hkResult applyMemberAdded(_In_ const hkReflect::Version::PatchInfo* p, _Inout_ VersionedTypeBuilder* typeBuilder, _Inout_ hkReflect::Type* completedType, _In_z_ const char* name)
    {
        // The field should be present, but with a '#' so it is not used yet
        hkStringBuf nameWithHash("#");
        nameWithHash.append(name);
        hkReflect::FieldDecl f = completedType->findField(nameWithHash, false);
        if (f)
        {
            // Remove the '#'
            hkReflect::TypeDetail::setFieldName(f, name);

            hkArray<hkReflect::Var>::Temp vars;
            typeBuilder->getAllTrackedVars(vars);

            for (hkReflect::Var& v : vars)
            {
                if(hkReflect::Var addedField = v[name])
                {
                    addedField.writeDefault();
                }
            }

            return HK_SUCCESS;
        }
        else
        {
            if (hkReflect::FieldDecl fieldAlreadyAdded = completedType->findField(name, false))
            {
                Log_Info("Field {} was already present when applying patch {}/{}->{}/{}. This often indicates version mismatches or missing patches.", name, p->oldName, p->oldVersion, p->newName, p->newVersion);
                return HK_SUCCESS;
            }
            else
            {
                Log_Error("Field {} not found applying patch {}/{}->{}/{}. This often indicates version mismatches or missing patches.", name, p->oldName, p->oldVersion, p->newName, p->newVersion);
                return HK_FAILURE;
            }
        }
    }

    hkResult applyMemberRenamed(_In_ const hkReflect::Version::PatchInfo* p, _Inout_ hkReflect::Type* completedType, _In_z_ const char* oldName, _In_z_ const char* newName)
    {
        hkReflect::FieldDecl f = completedType->findField(oldName, false);
        if (f)
        {
            hkReflect::TypeDetail::setFieldName(f, newName);
            return HK_SUCCESS;
        }
        else
        {
            Log_Error("Field {} not found applying patch {}/{}->{}/{}", oldName, p->oldName, p->oldVersion, p->newName, p->newVersion);
            return HK_FAILURE;
        }
    }

    hkResult VersionBundle::Impl::applyPatchToObjects(_In_ const hkReflect::Version::PatchInfo* p)
    {
        VersionedTypeBuilder* rootBuilder;
        VersionedTypeBuilder* typeToCombine = HK_NULL;

        hkResult res = HK_SUCCESS;

        // Could be added or deleted, we just need to get the type builder. They should always be the same
        if (p->oldName)
        {
            rootBuilder = getTypeForNameVer(p->oldName, p->oldVersion);
        }
        else
        {
            rootBuilder = getTypeForNameVer(p->newName, p->newVersion);
        }

        for (VersionedTypeBuilder* typeBuilder = rootBuilder; typeBuilder; typeBuilder = typeBuilder->m_nextType)
        {
            HK_ASSERT(0xa8dec740, typeBuilder, "Inconsistent versioned type setup");
            hkReflect::Type* completedType = typeBuilder->m_completedType;
            HK_ASSERT(0xa8dec741, completedType, "Inconsistent versioned type setup");

            DLOG("Applying {} ({}) -> {} ({}) to type {} ({})\n", p->oldName, p->oldVersion, p->newName, p->newVersion, completedType->getName(), completedType->getVersion());

            hkPatchDependencies dependencies(p);

            for (int componentIndex = 0; componentIndex < p->numComponent; componentIndex++)
            {
                const hkReflect::Version::PatchInfo::Component& component = p->component[componentIndex];

                if (const hkReflect::Version::PatchInfo::SetParentPatch* setParentPatch = component.asSetParentPatch())
                {
                    if (setParentPatch->oldParent)
                    {
                        VersionedTypeBuilder* parentType = typeBuilder->m_parents[0];

                        parentType->removeChildrenFromTracking(typeBuilder);
                    }

                    if(setParentPatch->newParent)
                    {
                        if (typeBuilder->m_parents.getSize() < 2) { res = HK_FAILURE; }
                        else
                        {
                            VersionedTypeBuilder* parentType = typeBuilder->m_parents[1];
                            hkArray<hkReflect::Var>::Temp vars;
                            typeBuilder->getAllTrackedVars(vars);

                            for (hkReflect::Var& v : vars)
                            {
                                hkReflect::Var parentVar(v.getAddress(), parentType->m_completedType);
                                hkString::memSet(parentVar.getAddress(), 0, parentVar.getType()->getSizeOf());
                                parentVar.writeDefault();
                            }
                            // TODO: Also need to track these for the parents, subsequent patches may be applied to them too

                            parentType->addChildrenToTracking(typeBuilder);
                        }

                    }
                    //else
                    typeBuilder->m_parents.removeAtAndCopy(0);
                    if(typeBuilder->m_parents.getSize())
                    {
                        hkReflect::TypeDetail::setParent(typeBuilder->m_completedType, typeBuilder->m_parents[0]->m_completedType);
                    }
                    else
                    {
                        hkReflect::TypeDetail::setParent(typeBuilder->m_completedType, HK_NULL);
                    }
                }
                else if (const hkReflect::Version::PatchInfo::MemberRemovedPatch* memberRemovedPatch = component.asMemberRemovedPatch())
                {
                    // Only type_fromname is allowed!!!
                    HK_ASSERT(0x56e69c60, memberRemovedPatch->type == hkReflect::Version::TYPE_FROMNAME, "Invalid patch {}/{}->{}/{}", p->oldName, p->oldVersion, p->newName, p->newVersion);
                    HK_ASSERT(0x56e69c61, memberRemovedPatch->tuples == 0, "Invalid patch {}/{}->{}/{}", p->oldName, p->oldVersion, p->newName, p->newVersion);

                    if (applyMemberRemoved(p, completedType, memberRemovedPatch->name).isFailure()) { res = HK_FAILURE; }
                }
                else if (const hkReflect::Version::PatchInfo::MemberAddedPatch* memberAddedPatch = component.asMemberAddedPatch())
                {
                    // Only type_fromname is allowed!!!
                    HK_ASSERT(0x56e69c62, memberAddedPatch->type == hkReflect::Version::TYPE_FROMNAME, "Invalid patch {}/{}->{}/{}", p->oldName, p->oldVersion, p->newName, p->newVersion);
                    HK_ASSERT(0x56e69c63, memberAddedPatch->tuples == 0, "Invalid patch {}/{}->{}/{}", p->oldName, p->oldVersion, p->newName, p->newVersion);

                    if (applyMemberAdded(p, typeBuilder, completedType, memberAddedPatch->name).isFailure()) { res = HK_FAILURE; }
                }
                else if (const hkReflect::Version::PatchInfo::MemberRenamedPatch* memberRenamedPatch = component.asMemberRenamedPatch())
                {
                    if(applyMemberRenamed(p, completedType, memberRenamedPatch->oldName, memberRenamedPatch->newName).isFailure()) { res = HK_FAILURE; }
                }
                else if (const hkReflect::Version::PatchInfo::VarFunctionPatch* varPatch = component.asVarFunctionPatch())
                {
                    // TODO: We do this a few times, split it out
                    hkArray<hkReflect::Var>::Temp vars;
                    typeBuilder->getAllTrackedVars(vars);
                    m_functionHelper.setDependencies(&dependencies);

                    for(hkReflect::Var& var: vars)
                    {
                        varPatch->function(var, m_functionHelper); // TODO: Probably better to call the function with a list of vars, it may want to precompute things
                    }
                }
                else if (const hkReflect::Version::PatchInfo::SetValuePatch* setValuePatch = component.asSetValuePatch())
                {
                    hkArray<hkReflect::Var>::Temp vars;

                    hkReflect::FieldDecl fd = typeBuilder->m_completedType->findField(setValuePatch->m_name, false);
                    HK_ASSERT(0x114c8644, fd, "Field {} not found", setValuePatch->m_name);
                    const hkReflect::Type* fieldType = fd.getType();

                    //if (!fieldType->getDefault())
                    {
                        fieldType = copyAndAddOptional(fieldType, hkReflect::Opt::DEFAULT, setValuePatch->m_value);
                        hkReflect::TypeDetail::setFieldType(typeBuilder->m_completedType, setValuePatch->m_name, fieldType);
                    }
                    // Can't do this if we reuse basic types from the input
//                  else
//                  {
//                      // Already had a default, just change it
//                      hkReflect::TypeDetail::localSetOptional(const_cast<hkReflect::Type*>(fieldType), hkReflect::Opt::DEFAULT, (hkUlong)setValuePatch->m_value);
//                  }

                    typeBuilder->getAllTrackedVars(vars);
                    for (hkReflect::Var& v : vars)
                    {
                        v = v[setValuePatch->m_name];
                    }

                    hkReflect::Var sourceVar(setValuePatch->m_value, setValuePatch->m_type());
                    for (hkReflect::Var& var : vars)
                    {
                        
                        var.assign(sourceVar);
                    }
                }
                else if (const hkReflect::Version::PatchInfo::VarTypeChangedFunctionPatch* typeChangedPatch = component.asVarTypeChangedFunctionPatch())
                {
                    hkStringBuf nameOld(typeChangedPatch->m_name);
                    nameOld.append("#OLD");
                    if(applyMemberRenamed(p, completedType, typeChangedPatch->m_name, nameOld).isSuccess())
                    {
                        if(applyMemberAdded(p, typeBuilder, completedType, typeChangedPatch->m_name).isSuccess())
                        {
                            // Function

                            hkArray<hkReflect::Var>::Temp vars;
                            typeBuilder->getAllTrackedVars(vars);
                            m_functionHelper.setDependencies(&dependencies);

                            
                            for (hkReflect::Var& var : vars)
                            {
                                typeChangedPatch->m_function(var, m_functionHelper, nameOld, typeChangedPatch->m_name);
                            }

                            if(applyMemberRemoved(p, completedType, nameOld).isFailure()) { res = HK_FAILURE; }
                        }
                        else
                        {
                            res = HK_FAILURE;
                        }
                    }
                    else
                    {
                        res = HK_FAILURE;
                    }
                }
                else if (component.asDependsPatch())
                {
                    // Do nothing
                }
                else
                {
                    res = HK_FAILURE;
                    HK_ERROR(0x56e69c67, "Invalid patch component");
                }
            }

            // TODO: Check for deletes etc
            hkReflect::TypeDetail::localSetOptional<hkReflect::Opt::VERSION>(completedType, p->newVersion);
            // We don't want template parameters here, only the base name
            // Changing template parameters is not currently supported
            hkStringBuf baseName(p->newName);
            int paramStart = baseName.indexOf('<');
            if (paramStart >= 0)
            {
                baseName.setLength(paramStart);
            }
            hkReflect::TypeDetail::localSetOptional<hkReflect::Opt::NAME>(completedType, hkReflectUtil::internCopy(baseName));

            if (p->newName && (p->newVersion != -1))
            {
                VersionedTypeBuilder* newBuilder = getTypeForNameVer(p->newName, p->newVersion);
                if (newBuilder != typeBuilder)
                {
                    DLOG("Types have combined {}/{} -> {}/{}", p->oldName, p->oldVersion, p->newName, p->newVersion);
                    typeToCombine = newBuilder;
                }
            }
        }

        if (typeToCombine)
        {
            typeToCombine->addCombinedType(rootBuilder);
        }

        return res;
    }

#define FAIL_IF_NOT(cond) if(!(cond)) { /*HK_BREAKPOINT(0);*/ return HK_FAILURE; }

    // Remove added members = true -> actually run removes so we end up with the starting version and not the entire future form
    hkResult VersionedTypeBuilder::applyPatchToType(_In_ const hkReflect::Version::PatchInfo* patch, VersionedTypeBuilder::PatchingBehavior removeAddedMembers)
    {
        m_localFieldsRequireCopying = true;
        DLOG("{}({})->{}({}) [{}]\n", patch->oldName ? patch->oldName : "<none>", patch->oldVersion, patch->newName ? patch->newName : "<none>", patch->newVersion, m_name);
        // Need to update this as I go along???
        FAIL_IF_NOT(patch->m_isNewPatch);
        // Apply reversed patch here

        hkPatchDependencies dependencies(patch);
        for (int componentIndex = patch->numComponent - 1; componentIndex >= 0; componentIndex--)
        {
            const hkReflect::Version::PatchInfo::Component& component = patch->component[componentIndex];

            if (const hkReflect::Version::PatchInfo::SetParentPatch* setParentPatch = component.asSetParentPatch())
            {
                if ((m_parents.getSize() == 1) && m_lastParentFromNativeType)
                {
                    if (setParentPatch->newParent)
                    {
                        const int version = dependencies.getVersion(setParentPatch->newParent);
                        VersionedTypeBuilder* parentBuilder = m_owner->getTypeForNameVer(setParentPatch->newParent, version);
                        m_parents[0] = parentBuilder;
                    }
                }

                if (setParentPatch->oldParent) // NULL = no parent
                {
                    const int version = dependencies.getVersion(setParentPatch->oldParent);
                    VersionedTypeBuilder* parentBuilder = m_owner->getTypeForNameVer(setParentPatch->oldParent, version);
                    m_parents.insertAt(0, parentBuilder);
                }
                else
                {
                    m_parents.insertAt(0, HK_NULL);
                }
                //if(m_parents.getSize())

            }
            else if (const hkReflect::Version::PatchInfo::MemberRemovedPatch* memberRemovedPatch = component.asMemberRemovedPatch())
            {
                // Only type_fromname is allowed!!!
                FAIL_IF_NOT(memberRemovedPatch->type == hkReflect::Version::TYPE_FROMNAME);
                FAIL_IF_NOT(memberRemovedPatch->tuples == 0);

                for (int i = 0; i < m_fields.getSize(); i++)
                {
                    if (m_fields[i]->m_name && (hkString::strCmp(m_fields[i]->m_name, memberRemovedPatch->name) == 0))
                    {
                        DLOG("{} ({}) to {} ({}) is adding (removing) field {} that is already present", patch->oldName, patch->oldVersion, patch->newName, patch->newVersion, memberRemovedPatch->name);
                        return HK_FAILURE;
                    }
                }
                // Remove field, we add it as we are going in reverse
                addFieldFromTypeDescription(dependencies, memberRemovedPatch->name, memberRemovedPatch->typeName, 0);
            }
            else if (const hkReflect::Version::PatchInfo::MemberAddedPatch* memberAddedPatch = component.asMemberAddedPatch())
            {
                // Only type_fromname is allowed!!!
                FAIL_IF_NOT(memberAddedPatch->type == hkReflect::Version::TYPE_FROMNAME);
                FAIL_IF_NOT(memberAddedPatch->tuples == 0);

                if (removeAddedMembers == REMOVE_ADDED_MEMBERS)
                {
                    bool memberFound = false;
                    for (int i = 0; i < m_fields.getSize(); i++)
                    {
                        if (m_fields[i]->m_name && (hkString::strCmp(memberAddedPatch->name, m_fields[i]->m_name) == 0))
                        {
                            m_fields[i]->m_name = HK_NULL;
                            //m_fields[i]->m_typeName = HK_NULL; // ??
                            memberFound = true;
                            break;
                        }
                    }

                    FAIL_IF_NOT(memberFound);
                }
                else
                {
                    int membersFound = 0;
                    for (int i = 0; i < m_fields.getSize(); i++)
                    {
                        if (m_fields[i]->m_name && (hkString::strCmp(memberAddedPatch->name, m_fields[i]->m_name) == 0))
                        {
                            //m_fields[i]->m_name = HK_NULL;
                            hkStringBuf nameWithHash("#");
                            nameWithHash.append(m_fields[i]->m_name);
                            m_fields[i]->m_name = nameWithHash;
                            membersFound++;
                            break;
                        }
                    }

                    // it should be there once, not present or more than one is a bug
                    FAIL_IF_NOT(membersFound == 1);
                }
            }
            else if (const hkReflect::Version::PatchInfo::MemberRenamedPatch* memberRenamedPatch = component.asMemberRenamedPatch())
            {
                bool memberFound = false;
                // Reverse the rename to get the starting name
                for (int i = 0; i < m_fields.getSize(); i++)
                {
                    if (m_fields[i]->m_name && (hkString::strCmp(memberRenamedPatch->newName, m_fields[i]->m_name) == 0))
                    {
                        m_fields[i]->m_name = memberRenamedPatch->oldName;
                        memberFound = true;
                        break;
                    }
                }

                for (int i = 0; i < m_fields.getSize(); i++)
                {
                    if (m_fields[i]->m_name && (hkString::strCmp(m_fields[i]->m_name, memberRenamedPatch->newName) == 0))
                    {
                        DLOG("{} ({}) to {} ({}) is renaming to field {} that is already present", patch->oldName, patch->oldVersion, patch->newName, patch->newVersion, memberRenamedPatch->newName);
                        return HK_FAILURE;
                    }
                }

                FAIL_IF_NOT(memberFound);
            }
            else if (const hkReflect::Version::PatchInfo::VarTypeChangedFunctionPatch* typeChangedPatch = component.asVarTypeChangedFunctionPatch())
            {
                // Remove the new field, add the old one
                if (removeAddedMembers == REMOVE_ADDED_MEMBERS)
                {
                    bool memberFound = false;
                    for (int i = 0; i < m_fields.getSize(); i++)
                    {
                        if (m_fields[i]->m_name && (hkString::strCmp(typeChangedPatch->m_name, m_fields[i]->m_name) == 0))
                        {
                            m_fields[i]->m_name = HK_NULL;
                            memberFound = true;
                            break;
                        }
                    }

                    FAIL_IF_NOT(memberFound);
                }
                else
                {
                    int membersFound = 0;
                    for (int i = 0; i < m_fields.getSize(); i++)
                    {
                        if (m_fields[i]->m_name && (hkString::strCmp(typeChangedPatch->m_name, m_fields[i]->m_name) == 0))
                        {
                            hkStringBuf nameWithHash("#");
                            nameWithHash.append(m_fields[i]->m_name);
                            m_fields[i]->m_name = nameWithHash;
                            membersFound++;
                            break;
                        }
                    }
                }

                addFieldFromTypeDescription(dependencies, typeChangedPatch->m_name, typeChangedPatch->m_oldType, 0);
            }
            else if (component.asVarFunctionPatch() || component.asDependsPatch() || component.asSetValuePatch())
            {
                // Nothing
            }
            else
            {
                HK_ERROR(0x2e53f5e2, "Unknown Component");
                return HK_FAILURE;
            }
        }

        m_name = patch->newName;
        m_version = patch->newVersion;

        if (removeAddedMembers == DEFAULT_BEHAVIOR)
        {
            if (patch->newName && (patch->newVersion != -1) && patch->oldName && (hkString::strCmp(patch->newName, patch->oldName) != 0))
            {
                VersionedTypeBuilder* newBuilder = m_owner->getTypeForNameVer(patch->newName, patch->newVersion);
                if (newBuilder != this)
                {
                    DLOG("Types have combined {} ({}) -> {} ({})", patch->oldName, patch->oldVersion, patch->newName, patch->newVersion);
                    newBuilder->addCombinedType(this);
                }
            }
        }

        return HK_SUCCESS;
    }

    void VersionedTypeBuilder::addFieldsFromNativeType(_In_ const hkReflect::Type* nativeType, bool includeSerializeFalseFields)
    {
        const char* ntn = nativeType->getName();
        (void)ntn;

        if (const hkReflect::Type* parent = nativeType->getParent())
        {
            m_currentParentIndex = m_parents.getSize(); // Always 0 ??
            hkStringBuf sb;
            m_parents.pushBack(m_owner->getTypeForNameVer(parent->getFullName(sb), parent->getVersion()));
            m_lastParentFromNativeType = true;
        }


        // Local members only, both fields and serializable properties
        for(hkReflect::FieldDecl const& f: hkReflect::TypeDetail::localGetFields(nativeType))
        {
            if(includeSerializeFalseFields || f.isSerializable())
            {
                addFieldFromStaticType(f.getName(), f.getType());
            }
        }

        if (const hkReflect::Template* templ = nativeType->getTemplate())
        {
            for (int paramIdx = 0; paramIdx < templ->getNumParams(); paramIdx++)
            {
                TemplateParameter& newParam = m_templateParameters.expandOne();
                const hkReflect::Template::Parameter* oldParam = templ->getParam(paramIdx);

                newParam.m_kindAndName = oldParam->m_kindAndName;
                if (oldParam->isType())
                {
                    StaticTypeCB cb(m_owner);
                    // Encode the typename in the field name so we can reconstruct it properly later
                    hkVersionBundleUtil::Field* newField = hkVersionBundleUtil::Field::createFromStaticType(cb, oldParam->getAsType()->getName(), oldParam->getAsType());
                    newParam.m_field.reset(newField);
                    newParam.m_value = 0;
                }
                else
                {
                    newParam.m_value = oldParam->getAsValue();
                    newParam.m_field.reset();
                }
            }
        }
    }

    class ForwardingPointerImpl : public hkReflect::Detail::PointerImpl
    {
    public:
        ForwardingPointerImpl(_In_ const hkReflect::Detail::PointerImpl* chainedImpl, _In_ VersionBundle::Impl* parent) : m_chainedImpl(chainedImpl), m_parent(parent) {}

        /// Sets the pointer value from the given var.
        /// Returns false on failure (usually type incompatibility)
        virtual hkResult setValue(_Inout_ void* self, _In_ const hkReflect::PointerType* ptd, const hkReflect::Var& var) const HK_OVERRIDE
        {
            // Disallowed for now
            HK_ASSERT_NOT_IMPLEMENTED(0xcd3e62f);
            return HK_FAILURE;
        }
        /// Gets the pointed to object and type.
        virtual hkResult getValue(_In_ const void* self, _In_ const hkReflect::PointerType* ptrType, _Out_ hkReflect::Var* val) const HK_OVERRIDE
        {
            hkReflect::Var chainedVal;
            m_chainedImpl->getValue(self, ptrType, &chainedVal);

            if (const hkReflect::Type* pointedToType = hkReflect::Detail::asType(chainedVal))
            {
                hkStringBuf sb;
                const hkReflect::Type* ret = HK_NULL;

                if (hkReflect::FieldDecl fd = pointedToType->isField())
                {
                    const hkReflect::Type* declContext = fd.getDeclContext();

                    const char* declContextName = declContext->getFullName(sb);
                    const int declContextVersion = declContext->getVersion();

                    const hkUint64 declContextUid = m_parent->m_patchReg.getUid(declContextName, declContextVersion);
                    const int declContextTypeIndex = m_parent->m_typeIndexFromUid.getWithDefault(declContextUid, -1);

                    if (declContextTypeIndex >= 0)
                    {
                        if (hkReflect::FieldDecl versionedFd = m_parent->m_typeBuilders[declContextTypeIndex]->m_completedType->findField(fd.getName(), false))
                        {
                            ret = versionedFd.getType();
                        }
                    }
                }
                
                else if (pointedToType->asRecord() || (pointedToType->asPointer() && pointedToType->asPointer()->getSubType()->asRecord())) // Type
                {
                    const hkReflect::PointerType* asPointer = pointedToType->asPointer();
                    if (asPointer)
                    {
                        pointedToType = asPointer->getSubType().get();
                    }

                    const char* name = pointedToType->getFullName(sb);
                    const int version = pointedToType->getVersion();
                    // Patch up type pointer

                    const hkUint64 uid = m_parent->m_patchReg.getUid(name, version);
                    const int thisTypeIndex = m_parent->m_typeIndexFromUid.getWithDefault(uid, -1);

                    if (thisTypeIndex >= 0)
                    {
                        ret = m_parent->m_typeBuilders[thisTypeIndex]->m_completedType;
                    }
                    else
                    {
                        // Basic type, just pass it through
                        ret = pointedToType;
                    }

                    if (asPointer)
                    {
                        ret = m_parent->m_compoundTypeCache.getVersionedPointerType(asPointer, ret);
                    }
                }
                else
                {
                    // Basic type, just pass it through. This should be int, float etc
                    ret = pointedToType;
                }

                if(ret)
                {
                    *val = hkReflect::Var(ret, hkReflect::typeFromKind(ret->getKind()));
                    return HK_SUCCESS;
                }
                else
                {
                    *val = hkReflect::Var();
                    return HK_SUCCESS;
                }
            }
            else if (void* addr = chainedVal.getAddress())
            {
                int index = m_parent->m_indexFromAddressMap.getWithDefault(addr, -1);

                *val = m_parent->m_workBundle.var(index);
                return HK_SUCCESS;
            }

            *val = hkReflect::Var();
            return HK_SUCCESS;
        }

        virtual _Ret_maybenull_ void* queryInterfaceImpl(_In_ const hkReflect::Type* type, const hkReflect::Var& self) const HK_OVERRIDE
        {
            return m_chainedImpl->queryInterfaceImpl(type, self);
        }

        const hkReflect::Detail::PointerImpl* m_chainedImpl;
        VersionBundle::Impl* m_parent;
    };

    class ForwardingArrayImpl : public hkReflect::Detail::ArrayImpl
    {
    public:
        ForwardingArrayImpl(_In_ const hkReflect::Detail::ArrayImpl* chainedImpl, _In_ VersionBundle::Impl* parent) : m_chainedImpl(chainedImpl), m_parent(parent) {}

        virtual hkReflect::ContainerValue getContainerValue(_In_ const void* addr, _In_ const hkReflect::ContainerType* type) const HK_OVERRIDE
        {
            // TODO.ntm, container value has no addr??
            return m_chainedImpl->getContainerValue(addr, type);
        }
        ///
        virtual hkResult getValue(_In_ const void* arrAddr, _In_ const hkReflect::ArrayType* arrType, _Out_ hkReflect::ArrayValue* val) const HK_OVERRIDE
        {
            hkReflect::ArrayValue chainedVal;
            m_chainedImpl->getValue(arrAddr, arrType, &chainedVal);

            if (arrType->getFixedCount() > 0)
            {
                *val = chainedVal;
                return HK_SUCCESS;
            }
            else
            {
                if(void* addr = chainedVal.getAddress())
                {
                    int index = m_parent->m_indexFromAddressMap.getWithDefault(addr, -1);

                    // We don't want to do this when we are at the output
                    if (!m_parent->allPatchesHaveBeenApplied())
                    {
                        m_parent->makeVarNCopy(index, m_parent->m_workBundle.varn(index));
                    }

                    hkSerialize::VarN newVal = m_parent->m_workBundle.varn(index);
                    *val = hkReflect::ArrayValue(newVal.getAddress(), newVal.getCount(), newVal.getType());
                    return HK_SUCCESS;
                }
            }
            *val = chainedVal;
            return HK_SUCCESS;
        }

        /// Resize the array
        virtual hkResult setNumElements(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, int nelem) const HK_OVERRIDE
        {
            return m_chainedImpl->setNumElements(arrAddr, arrType, nelem);
        }

        /// Set the element type. Not all arrays support this operation.
        virtual AllocResult allocateElements(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, hkReflect::QualType elemType, int nelem) const HK_OVERRIDE
        {
            return m_chainedImpl->allocateElements(arrAddr, arrType, elemType, nelem);
        }
        /// Default implementation replaces the elements in arrInOut with the elements in source if
        /// numToDel and numToInserts are equal, otherwise it returns HK_FAILURE.
        virtual hkResult spliceInto(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, int index, int numToDel, const hkReflect::ArrayValue& val) const HK_OVERRIDE
        {
            return m_chainedImpl->spliceInto(arrAddr, arrType, index, numToDel, val);
        }

        const hkReflect::Detail::ArrayImpl* m_chainedImpl;
        VersionBundle::Impl* m_parent;
    };

    int VersionBundle::Impl::getIndexForNoteOnPointer(hkReflect::PointerVar ptr)
    {
        // This is forwardingpointerimpl
        if (ptr.getType()->getTypeWorld() == this)
        {
            // We cannot pass the Var as it is because the Impl does not correspond to the original one: unwrap it
            const hkReflect::Detail::PointerImpl* chainedImpl = static_cast<const ForwardingPointerImpl*>(ptr.getImpl())->m_chainedImpl;
            ptr = hkReflect::PointerVar(ptr.getAddress(), ptr.getType(), chainedImpl);
        }
        else if (ptr.getImpl() == &m_pointerImpl)
        {
            if (hkReflect::Var noteVar = m_workBundle.getNoteOnPointer(ptr))
            {
                return getVarIndexFromSourceAddress(noteVar.getAddress());
            }
            return 0;
        }

        // What happens for the versioning impl? Do we even need that?
        if (hkReflect::Var noteVar = m_bundleIn->getNoteOnPointer(ptr))
        {
            return getVarIndexFromSourceAddress(noteVar.getAddress());
        }
        return 0;
    }
    // Copy the input type, just replacing the pointer and array impls to forward
    _Ret_notnull_ hkReflect::Type* VersionBundle::Impl::chainingCopyType(_In_ const hkReflect::Type* srcType)
    {
        if (hkReflect::Type* lookedUpType = m_chainedCopiedTypesFromSrc.getWithDefault(srcType, HK_NULL))
        {
            return lookedUpType;
        }

        // Just a dumb copy for now
        hkReflect::TypeBuilder tb;
        tb.beginShallowClone(srcType);
        tb.addFlags(hkReflect::Type::TYPE_TEMPORARY);

        // Only records need this, not basic types
        if(srcType->asRecord() || srcType->asPointer())
        {
            tb.setTypeWorld(this);
            tb.setItem<hkReflect::Opt::EXACT_TYPE>(HK_NULL);
        }
        else
        {
            // Override the type world, for everything else we want to treat it as native
            // Eventually we will not copy types we identify as native and everything else
            // will get copied and the type world set
            tb.setTypeWorld(HK_NULL);
        }

        if (const hkReflect::PointerType* pt = srcType->asPointer())
        {
            ForwardingPointerImpl* newImpl = m_fieldAllocator.create<ForwardingPointerImpl>(pt->getImpl(), this);
            tb.setItem<hkReflect::Opt::IMPL>(newImpl);
            if (pt->getSubType())
            {
                tb.setItem<hkReflect::Opt::SUBTYPE>(chainingCopyType(pt->getSubType()));
            }
        }
        else if (const hkReflect::ArrayType* at = srcType->asArray())
        {
            ForwardingArrayImpl* newImpl = m_fieldAllocator.create<ForwardingArrayImpl>(at->getImpl(), this);
            tb.setItem<hkReflect::Opt::IMPL>(newImpl);
            if (at->getSubType())
            {
                tb.setItem<hkReflect::Opt::SUBTYPE>(chainingCopyType(at->getSubType()));
            }
        }



        if (const hkReflect::Type* pt = srcType->getParent())
        {
            tb.overrideParent(chainingCopyType(pt));
        }

        hkReflect::Type* ret = tb.allocate(&m_fieldAllocator);
        m_chainedCopiedTypesFromSrc.insert(srcType, ret);

        const hkReflect::Detail::DeclsArray* srcFields = hkReflect::TypeDetail::localGetOptional<hkReflect::Opt::DECLS>(srcType);
        if (srcFields)
        {
            hkArray<const hkReflect::Type*> dstFields;
            for (int i = 0; i < srcFields->getNumFields(); i++)
            {
                const hkReflect::FieldDecl srcField = srcFields->getField(i);
                // size/align and offsets are computed after
                if (srcField.getName() && srcField.isSerializable())
                {
                    const hkReflect::Type* baseType = chainingCopyType(srcField.getType()->getParent()); // skip the field itself
                    const char* newName = hkString::strDup(srcField.getName(), m_fieldAllocator);
                    hkReflect::TypeBuilder fieldBuilder;
                    //fieldBuilder.setTypeWorld(baseType->getTypeWorld());
                    fieldBuilder.beginDerived(baseType);
                    fieldBuilder.setField(newName, srcField.getOffset(), 0);
                    fieldBuilder.setItem<hkReflect::Opt::DEFAULT>(HK_NULL);
                    fieldBuilder.setItem<hkReflect::Optional::DECL_CONTEXT>(ret);
                    hkReflect::Type* fieldType = fieldBuilder.allocate(&m_fieldAllocator);
                    dstFields.pushBack(fieldType);
                }
            }
            hkReflect::Detail::DeclsArray* fieldsArray = hkReflect::Detail::DeclsArray::create(dstFields.begin(), dstFields.getSize(), 0, m_fieldAllocator);
            hkReflect::TypeDetail::localSetOptional<hkReflect::Opt::DECLS>(ret, fieldsArray);
        }

        return ret;
    }

    hkReflect::ArrayValue VersionBundle::Impl::FunctionHelper::newArray(_In_ const hkReflect::Type* t, int cardinality )
    {
        if ( t->asRecord() )
        {
            if ( VersionedTypeBuilder* builder = m_owner.getTypeBuilderForCompletedType( t ) )
            {
                return newArray( builder, cardinality );
            }
            else
            {
                // Fall back to named lookup, that can find types that we are seeing
                // here for the first time
                return newArray( t->getName(), cardinality );
            }
        }
        else
        {
            HK_ASSERT_NO_MSG(0x2a1ca6c5, !t->asVoid() && !t->asOpaque() );

            // All non-records don't need any special handling, just allocate a plain memory block.
            int num = ( cardinality == -1 ) ? 1 : cardinality;
            void* data = m_owner.m_fieldAllocator.blockAlloc( t->getSizeOf() * num );
            hkString::memSet( data, 0, t->getSizeOf() * num );
            return hkReflect::ArrayValue( data, num, t );
        }
    }

    hkReflect::ArrayValue VersionBundle::Impl::FunctionHelper::newArray(_In_z_ const char* typeName, int cardinality)
    {
        const int version = m_dependencies->getVersion( typeName );
        if ( VersionedTypeBuilder* builder = m_owner.getTypeForNameVer( typeName, version ) )
        {
            return newArray( builder, cardinality );
        }
        else
        {
            Log_Warning( "Patch function asked to create type {} but no type builder was found. Patch may be missing dependencies", typeName ? typeName : "<Unnamed>");
            return hkReflect::ArrayValue();
        }

    }

    hkReflect::ArrayValue VersionBundle::Impl::FunctionHelper::newArray(_Inout_ VersionedTypeBuilder* builder, int cardinality)
    {
        HK_ASSERT( 0x32af6ba5, builder->m_completedType, "Type created in patch function, are you missing a dependency?" );

        int num = ( cardinality == -1 ) ? 1 : cardinality;

        const int objectSize = builder->m_completedType->getSizeOf();
        void* data = m_owner.m_fieldAllocator.blockAlloc( objectSize * num );
        hkString::memSet( data, 0, objectSize * num );
        hkReflect::ArrayValue res( data, num, builder->m_completedType );
        for ( int i = 0; i < num; ++i )
        {
            res[i].writeDefault();
        }

        const int newId = m_owner.m_workBundle.getSize();
        int tid = m_owner.m_workBundle.addType( builder->m_completedType );

        if ( cardinality == -1 )
        {
            m_owner.m_workBundle.setVar( newId, data, tid );
            builder->addTrackedVar( newId );
        }
        else
        {
            m_owner.m_workBundle.setVarN( newId, data, tid, num );
            builder->addTrackedExtra( newId, 0, num );
        }

        m_owner.m_workBundle.setCopied( newId );

        if ( data )
        {
            m_owner.m_indexFromAddressMap.insert( data, newId );
        }

        return res;
    }

    VersionBundle::Impl::FunctionHelper::PointerValue VersionBundle::Impl::FunctionHelper::getPointerValue(hkReflect::Var v)
    {
        VersionBundle::Impl::FunctionHelper::PointerValue ret;
        ret.m_id = 0;
        hkReflect::PointerVar pv(v);
        if (pv)
        {
            const hkUint32Le* self = reinterpret_cast<const hkUint32Le*>(pv.getAddress());
            hkUint32 id = 0;
            if (self)
            {
                id = *reinterpret_cast<const hkUint32Le*>(self);
            }

            ret.m_id = id;
        }
        return ret;
    }

    hkReflect::Var hkSerialize::WorkBundle::getNoteOnPointer(const hkReflect::PointerVar& ptr) const
    {
        if (ptr.getType()->getTypeWorld() == m_parent)
        {
            // We cannot pass the Var as it is because the Impl does not correspond to the original one: unwrap it
            const hkReflect::Detail::PointerImpl* chainedImpl = static_cast<const ForwardingPointerImpl*>(ptr.getImpl())->m_chainedImpl;
            hkReflect::PointerVar pvNew(ptr.getAddress(), ptr.getType(), chainedImpl);
            if (hkReflect::Var noteVar = m_parent->m_bundleIn->getNoteOnPointer(pvNew))
            {
                int fwdNoteIndex = m_parent->getVarIndexFromSourceAddress(noteVar.getAddress());

                if (fwdNoteIndex > 0)
                {
                    const Item& note = m_items[fwdNoteIndex];
                    HK_ASSERT(0x110ae889, note.isNote(), "Forwarding index did not point to a note");
                    return note.var();
                }

            }
        }

        // Types can't be notes
        if (hkReflect::Detail::isTypeOfType(ptr.getType()->getSubType()))
        {
            return hkReflect::Var();
        }

        hkUint32 index = *reinterpret_cast<hkUint32Le*>(ptr.getAddress());
        if (m_items[index].isNote())
        {
            return m_items[index].var();
        }
        else if (m_items[index].m_addr == HK_NULL && m_items[index].m_extra > 0)
        {
            // todo.notes old-style imports
            // If the index corresponds to an import with a note, m_vars[index].m_count will contain the note id
            const Item& importNote = m_items[m_items[index].m_extra];
            HK_ASSERT(0x110ae88a, importNote.isNote(), "Index did not point to a note");
            return importNote.var();
        }
        return hkReflect::Var();
    }

    hkResult VersionBundle::Impl::FunctionHelper::setPointerValue(hkReflect::Var v, PointerValue pval)
    {
        hkReflect::PointerVar pv(v);
        if (pv)
        {
            hkUint32Le* self = reinterpret_cast<hkUint32Le*>(pv.getAddress());

            *self = pval.m_id;
        }

        return HK_SUCCESS;
    }

    _Ret_notnull_ const hkReflect::PointerType* VersionedTypeStorage::CompoundTypeCache::getVersionedPointerType(_In_ const hkReflect::PointerType* referenceType, _In_ const hkReflect::Type* newSubType)
    {
        if ( referenceType->getTemplate() )
        {
            HK_ASSERT(0x9abc5fa, referenceType->getTemplate()->getNumParams() == 1 &&
                referenceType->getTemplate()->getParam( 0 )->getAsType() == referenceType->getSubType(),
                "Only pointers in the form Ptr<T> are supported" );
            if(referenceType->getSubType())
            {
            HK_ASSERT_NO_MSG(0x3b737754, referenceType->getSubType()->getFullName() == newSubType->getFullName() );
            }
        }
        return getVersionedPointerType( referenceType->getName(), newSubType, referenceType->getTemplate() != HK_NULL );
    }

    _Ret_notnull_ const hkReflect::ArrayType* VersionedTypeStorage::CompoundTypeCache::getVersionedArrayType(_In_ const hkReflect::ArrayType* referenceType, _In_ const hkReflect::Type* newSubType )
    {
        if ( referenceType->getTemplate() )
        {
            HK_ASSERT(0x41e88c5e, referenceType->getTemplate()->getNumParams() == 2 &&
                referenceType->getTemplate()->getParam( 0 )->getAsType()->getFullName() == newSubType->getFullName() &&
                referenceType->getTemplate()->getParam( 1 )->isType(),
                "Only arrays in the form Arr<T, Allocator> are supported" );
            HK_ASSERT_NO_MSG(0x125def06, referenceType->getSubType()->getFullName() == newSubType->getFullName() );
        }
        return getVersionedArrayType( referenceType->getName(), newSubType, referenceType->getTemplate() != HK_NULL, referenceType->getFixedCount() );
    }

    _Ret_notnull_  const hkReflect::PointerType* VersionedTypeStorage::CompoundTypeCache::getVersionedPointerType(_In_z_ const char* baseName, _In_ const hkReflect::Type* subType, bool addTemplateArgs)
    {
        // Lookup in the map, create if not found.
        const char* internedName = hkReflectUtil::internCopy( baseName );
        SubtypeNameAndCount key( subType, reinterpret_cast<hkUlong>( internedName ), -1 );
        auto it = m_versionedPointerTypesMap.findOrInsertKeyWithFunctor( key, [this, subType, internedName, addTemplateArgs] ()
        {
            auto newPt = addTemplateArgs ?
                m_owner.getAllocator().create<hkReflect::BasicPointerType>( subType, internedName, hkReflect::Type::TYPE_TEMPORARY ) :
                m_owner.getAllocator().create<hkReflect::BasicPointerType>( subType, internedName, (hkReflect::Detail::TemplateParameterArray*)HK_NULL, hkReflect::Type::TYPE_TEMPORARY );
            newPt->m_impl = m_owner.getPointerImpl();
            return newPt;
        } );
        return m_versionedPointerTypesMap.getValue(it);
    }

    _Ret_notnull_ const hkReflect::ArrayType* VersionedTypeStorage::CompoundTypeCache::getVersionedArrayType(_In_z_ const char* baseName, _In_ const hkReflect::Type* subType, bool addTemplateArgs, int fixedCount)
    {
        // Lookup in the map, create if not found.
        const char* internedName = hkReflectUtil::internCopy( baseName );
        SubtypeNameAndCount key( subType, reinterpret_cast<hkUlong>( internedName ), fixedCount );
        auto it = m_versionedArrayTypesMap.findOrInsertKeyWithFunctor( key, [this, subType, internedName, addTemplateArgs, fixedCount] ()
        {
            if ( fixedCount )
            {
                int elementStorageSize = HK_NEXT_MULTIPLE_OF( subType->getAlignOf(), subType->getSizeOf() ); // Align could be more than size?
                hkReflect::ArrayType* newAt = static_cast<hkReflect::ArrayType*>(
                    m_owner.getAllocator().create<hkReflect::RepeatArrayType>( subType, fixedCount,
                        elementStorageSize * fixedCount, subType->getAlignOf(),
                        &hkReflect::Detail::RepeatImpl::s_instance, internedName, hkReflect::Type::TYPE_TEMPORARY));
                return newAt;

            }
            else if ( addTemplateArgs )
            {
                hkReflect::ArrayType* newAt = static_cast<hkReflect::ArrayType*>(
                    m_owner.getAllocator().create<hkReflect::BasicArrayTypeWithParams>( subType
                    , hkSizeOf( hkReflect::Detail::HomogeneousArrayImpl::InMemory )
                    , (int)HK_ALIGN_OF( hkReflect::Detail::HomogeneousArrayImpl::InMemory ), m_owner.getArrayImpl()
                    , internedName, hkReflect::Type::TYPE_TEMPORARY ) );
                return newAt;
            }
            else
            {
                hkReflect::ArrayType* newAt = static_cast<hkReflect::ArrayType*>(
                    m_owner.getAllocator().create<hkReflect::BasicArrayType>( subType
                    , hkSizeOf( hkReflect::Detail::HomogeneousArrayImpl::InMemory )
                    , (int)HK_ALIGN_OF( hkReflect::Detail::HomogeneousArrayImpl::InMemory ), m_owner.getArrayImpl()
                    , internedName, hkReflect::Type::TYPE_TEMPORARY ) );
                return newAt;
            }

        } );
        return m_versionedArrayTypesMap.getValue( it );
    }

    _Ret_notnull_ const hkReflect::PointerType* VersionBundle::Impl::FunctionHelper::getPointerTypeTo(_In_z_ const char* basename, const _In_ hkReflect::Type* pointedType)
    {
        const hkReflect::Type* subType = m_owner.getOrCreate( pointedType );
        return m_owner.m_compoundTypeCache.getVersionedPointerType( basename, subType, true );
    }

    _Ret_notnull_ const hkReflect::ArrayType* VersionBundle::Impl::FunctionHelper::getArrayTypeTo(_In_z_ const char* basename, const _In_ hkReflect::Type* elemType, int fixedCount)
    {
        const hkReflect::Type* subType = m_owner.getOrCreate( elemType );
        return m_owner.m_compoundTypeCache.getVersionedArrayType( basename, subType, true, fixedCount );
    }

    VersionBundle::Impl::CompatTypeReg::CompatTypeReg(_Inout_ VersionBundle::Impl* parent) : m_parent(parent), m_compoundTypeCache(*this)
    {
        // TODO: Lock?
        hkArray<int>::Temp patchIndices;
        hkArrayView<const hkReflect::Version::PatchInfo*> allPatches = m_parent->m_patchReg.getPatches();
        for (int i = 0; i < allPatches.getSize(); i++)
        {
            const hkReflect::Version::PatchInfo* patch = allPatches[i];
            if (patch->isV2Patch() && !patch->isClassAdded()) // We don't want to create extra types from added patches
            {
                patchIndices.pushBack(i);
            }
        }

        hkSet<int> removedPatchIndices;

        for (int patchListIndex = 0; patchListIndex < patchIndices.getSize(); patchListIndex++)
        {
            const int patchIndex = patchIndices[patchListIndex];
            const hkReflect::Version::PatchInfo* patch = allPatches[patchIndex];
            if (removedPatchIndices.contains(patchIndex))
            {
                //patches.removeAt(patchIndex);
                //patchIndex--;
            }
            else
            {
                // Walk forward, remove anything we see
                while (patch->newVersion != -1)
                {
                    const hkUint64 nextUid = m_parent->m_patchReg.getUid(patch->newName, patch->newVersion);
                    const int index = m_parent->m_patchReg.getPatchIndex(nextUid);
                    if (index >= 0)
                    {
                        if (!removedPatchIndices.insert(index))
                        {
                            // Already did this chain
                            break;
                        }
                        patch = m_parent->m_patchReg.getPatch(index);
                    }
                    else
                    {
                        break;
                    }
                }
            }
        }

        for (int patchListIndex = 0; patchListIndex < patchIndices.getSize(); patchListIndex++)
        {
            int patchIndex = patchIndices[patchListIndex];

            if (!removedPatchIndices.contains(patchIndex))
            {
                const hkReflect::Version::PatchInfo* patch = allPatches[patchIndex];
                m_startingPatchIndices.pushBack(patchIndex);


                const char* className = patch->oldName ? patch->oldName : patch->newName;
                const int classVersion = patch->oldName ? patch->oldVersion : patch->newVersion;

                const hkUint64 startingUid = m_parent->m_patchReg.getUid(className, classVersion);
                // Might not actually need this
                const int newTypeBuilderIndex = m_typeBuilders.getSize();
                m_typeIndexFromUid.insert(startingUid, newTypeBuilderIndex);
                m_typeBuilderIndexFromName.insert(className, newTypeBuilderIndex);
                VersionedTypeBuilder* thisTypeBuilder = new VersionedTypeBuilder(this);
                m_typeBuilders.pushBack(thisTypeBuilder);

                // Really just sets the name / version
                thisTypeBuilder->setFromPatchInfo(className, classVersion);
                thisTypeBuilder->m_startedFromFile = true;
                while (patchIndex >= 0)
                {
                    thisTypeBuilder->addPatchIndex(patchIndex);
                    const hkReflect::Version::PatchInfo* thisPatch = m_parent->m_patchReg.getPatch(patchIndex);
                    if (thisPatch->newName)
                    {
                        const hkUint64 nextUid = m_parent->m_patchReg.getUid(thisPatch->newName, thisPatch->newVersion);
                        m_typeIndexFromUid.insert(nextUid, newTypeBuilderIndex); // fix here??

                        patchIndex = m_parent->m_patchReg.getPatchIndex(nextUid);
                    }
                    else
                    {
                        patchIndex = -1; // Removed Patch
                    }
                }
            }
        }

        hkArray<const hkReflect::Type*> types;
        // Now make sure we have covered all registered types
        m_parent->m_typeReg.appendAllTypesToArray(types);
        for (const auto& t : types)
        {
            if(t->asRecord() && t->isSerializable())
            {
                hkStringBuf classNameBuf;
                const char* className = t->getName();

                if (t->getTemplate())
                {
                    className = t->getFullName(classNameBuf);
                }
                const int classVersion = t->getVersion();

                const hkUint64 startingUid = m_parent->m_patchReg.getUid(className, classVersion);
                const int newTypeBuilderIndex = m_typeBuilders.getSize();
                auto typeIndexIt = m_typeIndexFromUid.findOrInsertKey(startingUid, newTypeBuilderIndex);
                if (m_typeIndexFromUid.getValue(typeIndexIt) == newTypeBuilderIndex)
                {
                    m_typeBuilderIndexFromName.insert(className, newTypeBuilderIndex);
                    VersionedTypeBuilder* thisTypeBuilder = new VersionedTypeBuilder(this);
                    m_typeBuilders.pushBack(thisTypeBuilder);

                    // Really just sets the name / version
                    thisTypeBuilder->setFromPatchInfo(className, classVersion);
                    thisTypeBuilder->m_startedFromFile = true;
                    // There are no patches to apply
                }
            }
        }
    }

    _Ret_notnull_ VersionedTypeBuilder* VersionBundle::Impl::CompatTypeReg::finishTypeBuilder(_Inout_ VersionedTypeBuilder* builder) const
    {
        if (builder->m_completedType)
        {
            return builder;
        }

        const char* finalName = builder->m_name;
        int finalVersion = builder->m_version;

        if (builder->m_patchIds.getSize())
        {
            const hkReflect::Version::PatchInfo* thisPatch = m_parent->m_patchReg.getPatch(builder->m_patchIds.back());
            finalName = thisPatch->newName;
            finalVersion = thisPatch->newVersion;
        }

        if (finalVersion != -1)
        {
            // The type wasn't deleted, we need to add in the types from the last native version
            const hkReflect::Type* nativeType = m_parent->m_typeReg.typeFromName(finalName);
            if (nativeType)
            {
                builder->addFieldsFromNativeType(nativeType, false);
            }
        }

        for (int patchIndexCount = builder->m_patchIds.getSize() - 1; patchIndexCount >= 0; patchIndexCount--)
        {
            const int patchIdx = builder->m_patchIds[patchIndexCount];
            const hkReflect::Version::PatchInfo* thisPatch = m_parent->m_patchReg.getPatch(patchIdx);
            if (thisPatch->oldName) // Skip class added patches??
            {
                builder->applyPatchToType(thisPatch, VersionedTypeBuilder::REMOVE_ADDED_MEMBERS); // Reversed patch

                builder->m_name = thisPatch->oldName;
                builder->m_version = thisPatch->oldVersion;
            }
        }

        builder->allocateRecord();
        builder->allocateFieldsAndParameters();
        hkSet<VersionedTypeBuilder*> currentlyProcessing;
        builder->setParentAndFinish(currentlyProcessing);

        return builder;
    }

    VersionBundle::Impl::CompatTypeReg::~CompatTypeReg()
    {
        for (VersionedTypeBuilder* tb : m_typeBuilders)
        {
            delete tb;
        }
    }

    _Ret_maybenull_ const hkReflect::Type* VersionBundle::Impl::CompatTypeReg::typeFromName(_In_z_ const char* name) const
    {
        

        int typeIndex = m_typeBuilderIndexFromName.getWithDefault(name, -1);
        if (typeIndex >= 0)
        {
            // We really need to reverse the patches
            VersionedTypeBuilder* builder = finishTypeBuilder(m_typeBuilders[typeIndex]);
            hkReflect::Type* ret = builder->m_completedType;

            return ret;
        }
        else
        {
            // We don't modify it, just return the original
            const hkReflect::Type* ret = m_parent->m_typeReg.typeFromName(name);

            return ret;
        }
    }

    void VersionBundle::Impl::CompatTypeReg::appendAllTypesToArray(hkArray<const hkReflect::Type*>& typesOut) const
    {
        hkArray<const hkReflect::Type*> subRegTypes;
        m_parent->m_typeReg.appendAllTypesToArray(subRegTypes);

        for (const hkReflect::Type* type : subRegTypes)
        {
            if(!type->findAttribute<hk::ExcludeFromVersionCheck>())
            {
                const char* typeName = type->getName();
                const int typeVersion = type->getVersion();
                const hkUint64 startingUid = m_parent->m_patchReg.getUid(typeName, typeVersion);
                const int thisTypeIndex = m_typeIndexFromUid.getWithDefault(startingUid, -1);

                if (thisTypeIndex >= 0)
                {
                    finishTypeBuilder(m_typeBuilders[thisTypeIndex]);
                    typesOut.pushBack(m_typeBuilders[thisTypeIndex]->m_completedType);
                }
                else
                {
                    //typesOut.pushBack(type);
                }
            }
        }
    }


    _Ret_maybenull_ VersionedTypeBuilder* VersionBundle::Impl::CompatTypeReg::getTypeForNameVer(_In_opt_z_ const char* name, int version)
    {
        if (!name || (version == -1))
        {
            return HK_NULL;
        }

        const hkUint64 startingUid = m_parent->m_patchReg.getUid(name, version);
        const int thisTypeIndex = m_typeIndexFromUid.getWithDefault(startingUid, -1);

        if (thisTypeIndex >= 0)
        {
            // Don't finalize here, we may not actually use the type
            return m_typeBuilders[thisTypeIndex];
        }
        else
        {
            VersionedTypeBuilder* newBuilder = new VersionedTypeBuilder(this);
            newBuilder->setFromPatchInfo(name, version);

            const int newBuilderIndex = m_typeBuilders.getSize();
            DLOG("Adding {}/{} from dependencies, index {}", name, version, newBuilderIndex);

            m_typeIndexFromUid.insert(startingUid, newBuilderIndex);
            m_typeBuilders.pushBack(newBuilder);

            newBuilder->m_startedFromFile = true;

            int patchIndex = m_parent->m_patchReg.getPatchIndex(startingUid);

            while (patchIndex >= 0)
            {
                newBuilder->addPatchIndex(patchIndex);
                const hkReflect::Version::PatchInfo* thisPatch = m_parent->m_patchReg.getPatch(patchIndex);
                if (thisPatch->newName)
                {
                    const hkUint64 nextUid = m_parent->m_patchReg.getUid(thisPatch->newName, thisPatch->newVersion);
                    patchIndex =  m_parent->m_patchReg.getPatchIndex(nextUid);
                }
                else
                {
                    newBuilder->addPatchIndex(patchIndex); // Removed Patch
                    patchIndex = -1;
                }
            }
            return newBuilder;
        }
    }

    VersionBundle::Impl::~Impl()
    {
        for (auto i : m_typeBuilders)
        {
            delete i;
        }
    }

    VersionBundle::VersionBundle(hkReflect::Version::PatchRegistry& patchReg, hkReflect::MutableTypeReg& typeReg)
    {
        m_impl = new Impl(patchReg, typeReg);
    }

    VersionBundle::~VersionBundle()
    {
        delete m_impl;
    }

    hkViewPtr<hkSerialize::Bundle> VersionBundle::applyPatchesTo(hkViewPtr<hkSerialize::Bundle> bundleIn, bool failIfVersioningRequired)
    {
        return m_impl->applyPatchesToBundle(bundleIn, failIfVersioningRequired);
    }

    _Ret_notnull_ hkReflect::MutableTypeReg* VersionBundle::createBackwardCompatibleTypeReg()
    {
        return m_impl->createBackwardCompatibleTypeReg();
    }

    int VersionBundle::appendAllTypes(hkArray<const hkReflect::Type*>& typesOut)
    {
        int retVal = 0;
        for (VersionedTypeBuilder* tb : m_impl->m_typeBuilders)
        {
            if (tb->m_completedType)
            {
                typesOut.pushBack(tb->m_completedType);
                retVal++;
            }
        }

        return retVal;
    }

}


#include <Common/Base/Container/PointerMap/hkMap.hxx>
#include <Common/Base/Container/PointerMultiMap/hkMultiMap.hxx>

template class hkMultiMap<int, hkSerialize::VersionedTypeBuilder::TrackedObject>;
template class hkMapBase<hkUint64, int>;
template class hkMap<hkUint64, int>;

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
